/*****************************************************************
 * Quoridor 2-player version
 * CS441/541		1/2/98
 *
 * Implementation of quoridor.h
 *
 * Feel free to use/study any portions of this code that you
 * like for your own program.
 *
 * Here white is 'O' and black is 'X'.  When you make a move,
 * enter 'u', 'd', 'l', or 'r' to move up down left or right.
 * To place a fence, enter the X and Y coordinates of the cell
 * where you'd like to put the fence, followed by 'r' to put the
 * fence to the right side (going down 2 squares), or enter 'b' to
 * put the fence on the bottom (going to the right 2 squares).
 *
 * Kenrick Mock
 *****************************************************************/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "globals.h"
#include "quoridor.h"

/*****************************************************************
 * Initialize the board for a new game.
 *****************************************************************/
void InitBoard(cell board[XDIM][YDIM])
{
 int i,j;

 i_aryFences[WHITE]=NUMFENCES;
 i_aryFences[BLACK]=NUMFENCES;
 for (i=0; i<XDIM; i++)				/* Blank the board */
	for (j=0; j<YDIM; j++) 
	{
  		board[i][j].state=EMPTY;				 
  		board[i][j].r=OPEN;
  		board[i][j].b=OPEN;
  		board[i][j].rvalid=VALID;
  		board[i][j].bvalid=VALID;
  		board[i][j].fsource=OPEN;
	} 
 board[4][0].state=BLACK;			/* Put black piece up top */
 board[4][8].state=WHITE;			/* Put white piece on bottom */
 i_aryXLoc[WHITE]=4;
 i_aryXLoc[BLACK]=4;
 i_aryYLoc[WHITE]=8;
 i_aryYLoc[BLACK]=0;
 i_aryLastXLoc[WHITE]=4;
 i_aryLastXLoc[BLACK]=4;
 i_aryLastYLoc[WHITE]=8;
 i_aryLastYLoc[BLACK]=0;
 i_aryLastWasMove[WHITE]=0;
 i_aryLastWasMove[BLACK]=0;
}


/*****************************************************************
 * Checks if a fence could be placed here .
 * Returns 1 if its valid for player iColor to put a fence
 * at location x,y at side cSide (either 'r' or 'b').
 *
 * Otherwise 0 is returned if a fence may not be placed here.
 *****************************************************************/
int ValidFenceSpot(cell boardOrig[XDIM][YDIM], int iColor, int x, int y, char cSide)
{
 cell board[XDIM][YDIM];

 /* Copy original board to a temporary board */
 memcpy((char *) &board[0], (char *) &boardOrig[0], XDIM*YDIM*sizeof(cell));

 if ((x<0) || (x>=XDIM-1)) return(0);
 if ((y<0) || (y>=YDIM-1)) return(0);
 if ((cSide!='b') && (cSide!='r')) return(0);
 if (cSide=='b') 		/* Valid to put fence to bottom? */
 {
	if ((board[x][y].b==BLOCKED) || (board[x+1][y].b==BLOCKED)) return(0);
	if (board[x][y].bvalid==INVALID) return(0);
   	if (x>0) 
 	{
		board[x-1][y].bvalid=INVALID;
	}
	board[x][y].rvalid=INVALID;
	board[x][y].bvalid=INVALID;
	board[x][y].b=BLOCKED;
	board[x+1][y].b=BLOCKED;
	board[x][y].fsource=BOTTOM;
	if (i_aryFences[WHITE]+i_aryFences[BLACK] > 19) return(1);
	/* See if this would block access to the goal */
	if (FencePathNecessary(board,x,y,cSide)) {
 	  if ((PathExists(board, i_aryXLoc[WHITE], i_aryYLoc[WHITE], 0)) &&
	      (PathExists(board, i_aryXLoc[BLACK], i_aryYLoc[BLACK], YDIM-1)))
	  {
		return(1);
	  }
        }
	else {
		return(1);
	}
 }
 else if (cSide=='r') 		/* Valid to put fence to right? */
 {
	if ((board[x][y].r==BLOCKED) || (board[x][y+1].r==BLOCKED)) return(0);
	if (board[x][y].rvalid==INVALID) return(0);
   	if (y>0) 
 	{
		board[x][y-1].rvalid=INVALID;
	}
	board[x][y].rvalid=INVALID;
	board[x][y].bvalid=INVALID;
	board[x][y].r=BLOCKED;
	board[x][y+1].r=BLOCKED;
	board[x][y].fsource=RIGHT;
	/* See if this would block access to the goal */
	if (i_aryFences[WHITE]+i_aryFences[BLACK] > 19) return(1);
	if (FencePathNecessary(board,x,y,cSide)) {
 	  if ((PathExists(board, i_aryXLoc[WHITE], i_aryYLoc[WHITE], 0)) &&
	      (PathExists(board, i_aryXLoc[BLACK], i_aryYLoc[BLACK], YDIM-1)))
	  {
		return(1);
	  }
	}
	else {
		return(1);
	}
 }
 return(0);
}

/*** Check to see if we need to do the fence path stuff ***/
int FencePathNecessary(cell board[XDIM][YDIM], int x, int y, char side)
{
 if (side=='r') 
 {
	if (y>0) {
		if (board[x][y-1].b==BLOCKED) return(1);
		if (board[x][y-1].r==BLOCKED) return(1);
		if (x<XDIM-1) {
			if (board[x+1][y-1].b==BLOCKED) return(1);
		}
	}
	if (board[x][y].b==BLOCKED) return(1);
	if (y<YDIM-1) {
		if (board[x][y+1].b==BLOCKED) return(1);
		if (x<XDIM-1) {
			if (board[x+1][y+1].b==BLOCKED) return(1);
		}
	}
	if (x<XDIM-1) {
		if (board[x+1][y].b==BLOCKED) return(1);
	}
	if (y<YDIM-2) {
		if (board[x][y+2].r==BLOCKED) return(1);
	}
	return(0);
 }
 else
 { 
 	if (x>0) {
		if (board[x-1][y].r==BLOCKED) return(1);
		if (board[x-1][y].b==BLOCKED) return(1);
		if (y<YDIM-1) {
			if (board[x-1][y+1].r==BLOCKED) return(1);
		}
	}
	if (board[x][y].r==BLOCKED) return(1);
	if (y<YDIM-1) {
		if (board[x][y+1].r==BLOCKED) return(1);
	}
	if (x<XDIM-1) {
		if (board[x+1][y].r==BLOCKED) return(1);
		if (y<YDIM-1) {
			if (board[x+1][y+1].r==BLOCKED) return(1);
		}
	}
	if (x<XDIM-2) {
		if (board[x+2][y].b==BLOCKED) return(1);
	}
	return(0);
 }
 return(1);
}

/*****************************************************************
 * Adds a fence to location x,y for player iColor
 * to side cSide.
 *
 * Does NOT check if a fence may be placed here.
 * You should make sure that ValidFenceSpot is verified
 * before attempting to add a fence.
 *****************************************************************/
void AddFence(cell board[XDIM][YDIM], int iColor, int x, int y, char cSide)
{
 if (cSide=='b') 
 {
   	if (x>0) 
 	{
		board[x-1][y].bvalid=INVALID;
	}
	board[x][y].rvalid=INVALID;
	board[x][y].bvalid=INVALID;
	board[x][y].b=BLOCKED;
	board[x+1][y].b=BLOCKED;
	board[x][y].fsource=BOTTOM;
	i_aryFences[iColor]-=1;
 }
 else if (cSide=='r') 
 {
   	if (y>0) 
 	{
		board[x][y-1].rvalid=INVALID;
	}
	board[x][y].rvalid=INVALID;
	board[x][y].bvalid=INVALID;
	board[x][y].r=BLOCKED;
	board[x][y+1].r=BLOCKED;
	board[x][y].fsource=RIGHT;
	i_aryFences[iColor]-=1;
 }
}


/*****************************************************************
 * Displays the game board to the screen.
 * Not a very pretty display, currently.
 *****************************************************************/
void ShowBoard(cell board[XDIM][YDIM])
{
 int i,j;

 printf("\n");
 printf("   ");
 for (i=0; i<XDIM; i++) 
 {
  printf("%d ",i);
 }
 printf("\n");
 for (j=0; j<YDIM; j++)
 {
	printf("%2d ",j);
	for (i=0; i<XDIM; i++) 
	{
		if (board[i][j].state==EMPTY) printf(".");
		else if (board[i][j].state==WHITE) printf("O");	
		else if (board[i][j].state==BLACK) printf("X");	
		if (i<XDIM-1)
		{
			if (board[i][j].r==OPEN) printf(" ");
			else if (board[i][j].r==BLOCKED) printf("|");
		}
	}
	printf("\n   ");
	if (j<YDIM-1) 
	{
		for (i=0; i<XDIM; i++) 
		{
			if (board[i][j].fsource==OPEN) 
			{
				if (board[i][j].b==OPEN) printf("  ");
				else if (board[i][j].b==BLOCKED) printf("- ");
			}
			else if (board[i][j].fsource==BOTTOM)
			{
				printf("--");
			}
			else if (board[i][j].fsource==RIGHT)
			{
				if (board[i][j].b==BLOCKED) printf("-|");
				else printf(" |");
			}
		}
	}
	 printf("\n");
 }
 printf("White Fences (O): %d    Black Fences (X): %d\n",i_aryFences[WHITE], i_aryFences[BLACK]);
}


/*****************************************************************
 * RecursivePathCheck returns 1 if a path exists from x,y
 * to the destination Y coordinate.  This is used to determine
 * if a fence would block all access to the goal -- these moves
 * are invalid.
 *
 * The routine exhaustively searches for a path to the goal.
 * It is not very efficient.
 *****************************************************************/
int RecursivePathCheck(cell board[XDIM][YDIM], int board_visited[XDIM][YDIM], int x, int y, int desty, char cDir)
{
 int iCounter=0;
 int iResult;
 char cNewDir;

 /* are we done? */
 if (y==desty) return 1;

 /* Set current node to visited */
 board_visited[x][y]=TRUE;
 cNewDir=cDir;
 
 /* start checking neighboring nodes, starting with cNewDir */
 while (iCounter<4)
 {
	if (cNewDir=='t')
	{
		if ((y>0) && (board_visited[x][y-1]==FALSE) && (board[x][y-1].b==OPEN))
		{
			iResult=RecursivePathCheck(board,board_visited,x,y-1,desty,cDir);
			if (iResult>0) return(iResult);
		}
		cNewDir='r';
		iCounter++;
	}
	else if (cNewDir=='r')
	{
		if ((x<XDIM-1) && (board_visited[x+1][y]==FALSE) && (board[x][y].r==OPEN))
		{
			iResult=RecursivePathCheck(board,board_visited,x+1,y,desty,cDir);
			if (iResult>0) return(iResult);
		}
		cNewDir='b';
		iCounter++;
	}
	else if (cNewDir=='b')
	{
		if ((y<YDIM-1) && (board_visited[x][y+1]==FALSE) && (board[x][y].b==OPEN))
		{
			iResult=RecursivePathCheck(board,board_visited,x,y+1,desty,cDir);
			if (iResult>0) return(iResult);
		}
		cNewDir='l';
		iCounter++;
	}
	else if (cNewDir=='l')
	{
		if ((x>0) && (board_visited[x-1][y]==FALSE) && (board[x-1][y].r==OPEN))
		{
			iResult=RecursivePathCheck(board,board_visited,x-1,y,desty,cDir);
			if (iResult>0) return(iResult);
		}
		cNewDir='t';
		iCounter++;
	}
 }

 /* Nothing found from this node */
 return(0); 
}


/*****************************************************************
 * Returns 1 if a path exists from x,y to the destination row
 * desty.  The real work is done in the above function. 0 is
 * returned if no path exists to the destination row.  This
 * means that access is blocked.
 *****************************************************************/
int PathExists(cell board[XDIM][YDIM],int x, int y, int desty)
{
 int board_visited[XDIM][YDIM];
 int i,j;

 if ((y==0) && (desty==0)) return 1;
 if ((y==YDIM-1) && (desty==YDIM-1)) return 1;

 /* keep array of places we've visited already */
 for (i=0; i<XDIM; i++) 
 {
	for (j=0; j<YDIM; j++) 
	{
		board_visited[i][j]=FALSE;
	}
 }
 if (desty==0) i=RecursivePathCheck(board, board_visited, x, y, desty, 't');
 else i=RecursivePathCheck(board, board_visited, x, y, desty, 'b');
 return i;
}


/*****************************************************************
 * Returns 1 if a move is valid, 0 if it is invalid for
 * any reason.
 *
 * The move should be in the form: 'u','d','l','r' to move
 * or   'x y r' to add a fence to the right/down of cell x,y 
 * or   'x y b' to add a fence to the bottom/right of cell x,y 
 *
 *****************************************************************/
int IsValidMove(cell board[XDIM][YDIM], char *sMove, int iWhoseMove)
{
 int x,y;
 char cDirection;
 int iOpponent;
 
 if (iWhoseMove==WHITE) iOpponent=BLACK;
 else iOpponent=WHITE;

 if (sMove[0]=='u') 
 {
	x=i_aryXLoc[iWhoseMove];
	y=i_aryYLoc[iWhoseMove];
	if (y==0) return(0);
	if (board[x][y-1].b==BLOCKED) return(0);
	if ( (y==1) &&
	     (x==i_aryXLoc[iOpponent]) &&
	     (y-1==i_aryYLoc[iOpponent]) ) return(0);
	if ( (y>1) &&
	     (x==i_aryXLoc[iOpponent]) &&
	     (y-1==i_aryYLoc[iOpponent]) &&
	     (board[x][y-2].b==BLOCKED)) return(0);
	return(1);
 }
 else if (sMove[0]=='d')
 {
	x=i_aryXLoc[iWhoseMove];
	y=i_aryYLoc[iWhoseMove];
	if (y==YDIM-1) return(0);
	if (board[x][y].b==BLOCKED) return(0);
	if ( (y==YDIM-2) &&
	     (x==i_aryXLoc[iOpponent]) &&
	     (y+1==i_aryYLoc[iOpponent]) ) return(0);
	if ( (y<YDIM-2) &&
	     (x==i_aryXLoc[iOpponent]) &&
	     (y+1==i_aryYLoc[iOpponent]) &&
	     (board[x][y+1].b==BLOCKED)) return(0);
	return(1);
 }
 else if (sMove[0]=='l')
 {
	x=i_aryXLoc[iWhoseMove];
	y=i_aryYLoc[iWhoseMove];
	if (x==0) return(0);
	if (board[x-1][y].r==BLOCKED) return(0);
	if ( (x==1) &&
	     (y==i_aryYLoc[iOpponent]) &&
	     (x-1==i_aryXLoc[iOpponent]) ) return(0);
	if ( (x>1) &&
	     (y==i_aryYLoc[iOpponent]) &&
	     (x-1==i_aryXLoc[iOpponent]) &&
	     (board[x-2][y].r==BLOCKED)) return(0);
	return(1);
 }
 else if (sMove[0]=='r')
 {
	x=i_aryXLoc[iWhoseMove];
	y=i_aryYLoc[iWhoseMove];
	if (x==XDIM-1) return(0);
	if (board[x][y].r==BLOCKED) return(0);
	if ( (x==XDIM-2) &&
	     (y==i_aryYLoc[iOpponent]) &&
	     (x+1==i_aryXLoc[iOpponent]) ) return(0);
	if ( (x<XDIM-2) &&
	     (y==i_aryYLoc[iOpponent]) &&
	     (x+1==i_aryXLoc[iOpponent]) &&
	     (board[x+1][y].r==BLOCKED)) return(0);
	return(1);
 }
 else 	/* adding a fence */
 {
	if (i_aryFences[iWhoseMove]==0) 
	{
		return(0);
	}
	sscanf(sMove,"%d %d %c",&x,&y,&cDirection);
	if (ValidFenceSpot(board, iWhoseMove, x, y, cDirection)==1) 
	{
		return(1);
	}
 }
 return(0);
}


/*****************************************************************
 * Implements a move.  Returns 1 if the move is valid and was
 * done, and 0 otherwise.
 *****************************************************************/
int MakeMove(cell board[XDIM][YDIM], char *sMove, int iWhoseMove)
{
 int x,y;
 char cDirection;
 int iOpponent;
 
 if (iWhoseMove==WHITE) iOpponent=BLACK;
 else iOpponent=WHITE;

 if (IsValidMove(board,sMove,iWhoseMove))
 {
	x=i_aryXLoc[iWhoseMove];
	y=i_aryYLoc[iWhoseMove];

	if (sMove[0]=='u') 
	{
		board[x][y].state=EMPTY;
		if ((x==i_aryXLoc[iOpponent]) && (y-1==i_aryYLoc[iOpponent])) y-=2;	/* hop over opponent */
		else y--;
		board[x][y].state=iWhoseMove;
		i_aryYLoc[iWhoseMove]=y;
	}
	else if (sMove[0]=='d') 
	{
		board[x][y].state=EMPTY;
		if ((x==i_aryXLoc[iOpponent]) && (y+1==i_aryYLoc[iOpponent])) y+=2;	/* hop over opponent */
		else y++;
		board[x][y].state=iWhoseMove;
		i_aryYLoc[iWhoseMove]=y;
	}
	else if (sMove[0]=='l') 
	{
		board[x][y].state=EMPTY;
		if ((x-1==i_aryXLoc[iOpponent]) && (y==i_aryYLoc[iOpponent])) x-=2;	/* hop over opponent */
		else x--;
		board[x][y].state=iWhoseMove;
		i_aryXLoc[iWhoseMove]=x;
	}
	else if (sMove[0]=='r') 
	{
		board[x][y].state=EMPTY;
		if ((x+1==i_aryXLoc[iOpponent]) && (y==i_aryYLoc[iOpponent])) x+=2;	/* hop over opponent */
		else x++;
		board[x][y].state=iWhoseMove;
		i_aryXLoc[iWhoseMove]=x;
	}
	else
	{
	 sscanf(sMove,"%d %d %c",&x,&y,&cDirection);
	 AddFence(board,iWhoseMove,x,y,cDirection);
	}
 }
 else
 { 
 	return(0);
 }
}


/*****************************************************************
 * Used to check and see if someone won.
 * Returns EMPTY if the board is in a state where nobody has won.
 * Returns WHITE if the board is in a state where white has won.
 * Returns BLACK if the board is in a state where black has won.
 *****************************************************************/
int CheckForWin()
{
 if (i_aryYLoc[WHITE]==0) 
 {
	return(WHITE);
 }
 else if (i_aryYLoc[BLACK]==8) 
 {
	return(BLACK);
 }
 else 
 {
	return(EMPTY);
 }
}

void MoveGen(cell board[XDIM][YDIM], int iWhoseMove, char mlist[150][7], int p)
{
 int i,j,iMoveCount=0;
 int xdist1,ydist1,xdist2,ydist2;
 char s[50];

 /* Check up first for white, down first for black */
 if (iWhoseMove==WHITE)
 {
  if (IsValidMove(board,"u",iWhoseMove)) 
  {
   strcpy(mlist[iMoveCount],"u");
   iMoveCount++;
  }
  if (IsValidMove(board,"d",iWhoseMove)) 
  {
   strcpy(mlist[iMoveCount],"d");
   iMoveCount++;
  }
 }
 else
 {
  if (IsValidMove(board,"d",iWhoseMove)) 
  {
   strcpy(mlist[iMoveCount],"d");
   iMoveCount++;
  }
  if (IsValidMove(board,"u",iWhoseMove)) 
  {
   strcpy(mlist[iMoveCount],"u");
   iMoveCount++;
  }
 }
 if (IsValidMove(board,"r",iWhoseMove)) 
 {
   strcpy(mlist[iMoveCount],"r");
   iMoveCount++;
 }
 if (IsValidMove(board,"l",iWhoseMove)) 
 {
   strcpy(mlist[iMoveCount],"l");
   iMoveCount++;
 }
 
 if (i_aryFences[iWhoseMove]>0) 
 {
  for (i=0; i<XDIM-1; i++)
   for (j=0; j<YDIM-1; j++)
   {
	if (p==1) xdist1=1;
	else {
 		if (i>i_aryXLoc[WHITE]) xdist1=i-i_aryXLoc[WHITE];
			else xdist1=i_aryXLoc[WHITE]-i;
		if (i>i_aryXLoc[BLACK]) xdist2=i-i_aryXLoc[BLACK];
		else xdist2=i_aryXLoc[BLACK]-i;
		if (j>i_aryYLoc[WHITE]) ydist1=j-i_aryYLoc[WHITE];
		else ydist1=i_aryYLoc[WHITE]-j;
		if (j>i_aryYLoc[BLACK]) ydist2=j-i_aryYLoc[BLACK];
		else ydist2=i_aryYLoc[BLACK]-j;
	}
        if (((xdist1<=iXNeighborhood) && (ydist1<=iYNeighborhood)) || ((xdist2<=iXNeighborhood) && (ydist2<=iYNeighborhood)))
 	{
	  sprintf(s,"%d %d r",i,j);
 	  if (IsValidMove(board,s,iWhoseMove)) 
	  {
	   	sprintf(s,"%d %d r",i,j);	
		strcpy(mlist[iMoveCount],s);
	  	iMoveCount++;
          }
	  sprintf(s,"%d %d b",i,j);
	  if (IsValidMove(board,s,iWhoseMove)) 
	  {
	   	sprintf(s,"%d %d b",i,j);	
		strcpy(mlist[iMoveCount],s);
	  	iMoveCount++;
          }
 	}
   }
 }
 strcpy(mlist[iMoveCount],"null");  /* end of list */
}

int astar(cell board[XDIM][YDIM], int curx, int cury, int desty)
{
 int openX[81], openY[81], openH[81], openG[81];
 int closedX[81], closedY[81], closedH[81], closedG[81];
 int numopen=0, numclosed=0, x, y, h, g;
 int i,j,min,minindex,childx,childy,onopen=0,onclosed=0;
 int childX[4],childY[4];
 int numchildren,curg,bestx,besty,besth;
 int OpenOrClosed[XDIM][YDIM];

 for (i=0; i<XDIM; i++)
  for (j=0; j<YDIM; j++)
   OpenOrClosed[i][j]=0;
 openX[0]=curx; openY[0]=cury; openH[0]=0; openG[0]=0;
 OpenOrClosed[curx][cury]=1;
 numopen=1;
 while (numopen>0) 
 {
  /* Find min */
  for (i=0,min=999,minindex=0; i<numopen; i++) 
  {
   x=openX[i]; y=openY[i]; h=openH[i];
   if (h<min) { min=h; minindex=i; }
  }
  /* Remove best from open */
  x=openX[minindex]; y=openY[minindex]; h=openH[minindex]; curg=openG[minindex];
/*printf("pulled off %d %d  h=%d g=%d\n",x,y,h,curg);*/
  bestx=x; besty=y; besth=h;
  for (i=minindex; i<numopen-1; i++)
  {
   openX[i]=openX[i+1]; openY[i]=openY[i+1]; 
   openH[i]=openH[i+1]; openG[i]=openG[i+1];
  } 
  numopen--;
  if (y==desty) return(h);
  numchildren=0;
  /* Generate children */
  if (y>0) {
   if (board[x][y-1].b==OPEN) { 
	childX[numchildren]=x; childY[numchildren]=y-1;
	numchildren++;
   }
  }
  if (x>0) {
   if (board[x-1][y].r==OPEN) { 
	childX[numchildren]=x-1; childY[numchildren]=y;
	numchildren++;
   }
  }
  if (x<XDIM) {
   if (board[x][y].r==OPEN) { 
	childX[numchildren]=x+1; childY[numchildren]=y;
	numchildren++;
   }
  }
  if (y<XDIM) {
   if (board[x][y].b==OPEN) { 
	childX[numchildren]=x; childY[numchildren]=y+1;
	numchildren++;
   }
  }
  for (i=0; i<numchildren; i++)
  {
   childx=childX[i]; childy=childY[i];
   /* See if its on open */
   onopen=OpenOrClosed[childx][childy];
   /* Add to open if not on open or closed */
   if (onopen==0)
   {
    if (childy<desty) h=desty-childy;
    else h=childy-desty;
    openX[numopen]=childx; openY[numopen]=childy;
    openH[numopen]=h+curg+1; openG[numopen]=curg+1;
    numopen++;
    OpenOrClosed[childx][childy]=1;
   }
   /* If already on open or closed ignore, dont need to update in this case */
  }
  closedX[numclosed]=bestx; closedY[numclosed]=besty;
  closedH[numclosed]=besth; closedG[numclosed]=curg;
  OpenOrClosed[childx][childy]=1;
  numclosed++;
 }
 return(999);
}

int astarOld(cell board[XDIM][YDIM], int curx, int cury, int desty)
{
 char open[81][20], closed[81][20];
 int numopen=0, numclosed=0, x, y, h, g;
 int i,j,min,minindex,childx,childy,onopen,onclosed;
 char children[4][20];
 int numchildren,curg,bestx,besty,besth;

 sprintf(open[0],"%d %d 0 0",curx,cury);
 numopen=1;
 while (numopen>0) 
 {
  /* Find min */
  for (i=0,min=999,minindex=0; i<numopen; i++) 
  {
   sscanf(open[i],"%d %d %d %d",&x,&y,&h,&g);
   if (h<min) { min=h; minindex=i; }
  }
  /* Remove best from open */
  sscanf(open[minindex],"%d %d %d %d",&x,&y,&h,&curg);
/*printf("pulled off %d %d  h=%d g=%d\n",x,y,h,curg);*/
  bestx=x; besty=y; besth=h;
  for (i=minindex; i<numopen-1; i++)
  {
   strcpy(open[i],open[i+1]);
  } 
  numopen--;
  if (y==desty) return(h);
  numchildren=0;
  /* Generate children */
  if (y>0) {
   if (board[x][y-1].b==OPEN) { 
	sprintf(children[numchildren],"%d %d",x,y-1); numchildren++;
   }
  }
  if (x>0) {
   if (board[x-1][y].r==OPEN) { 
	sprintf(children[numchildren],"%d %d",x-1,y); numchildren++;
   }
  }
  if (x<XDIM) {
   if (board[x][y].r==OPEN) { 
	sprintf(children[numchildren],"%d %d",x+1,y); numchildren++;
   }
  }
  if (y<XDIM) {
   if (board[x][y].b==OPEN) { 
	sprintf(children[numchildren],"%d %d",x,y+1); numchildren++;
   }
  }
  for (i=0; i<numchildren; i++)
  {
   sscanf(children[i],"%d %d",&childx,&childy);
   /* See if its on open */
   for (onopen=0,j=0; ((onopen==0) && (j<numopen)); j++) 
   {
    sscanf(open[j],"%d %d %d %d",&x,&y,&h,&g);
    if ((childx==x) && (childy==y)) onopen=1;
   }
   /* See if its on clsoed */
   for (onclosed=0,j=0; ((onclosed==0) && (j<numclosed)); j++) 
   {
    sscanf(closed[j],"%d %d %d %d",&x,&y,&h,&g);
    if ((childx==x) && (childy==y)) onclosed=1;
   }
   /* Add to open if not on open or closed */
   if ((onopen==0) && (onclosed==0))
   {
    if (childy<desty) h=desty-childy;
    else h=childy-desty;
    sprintf(open[numopen],"%d %d %d %d",childx,childy,h+curg+1,curg+1);
    numopen++;
   }
   /* If already on open or closed ignore, dont need to update in this case */
  }
  sprintf(closed[numclosed],"%d %d %d %d",bestx,besty,besth,curg);
  numclosed++;
 }
 return(999);
}

int heuristic(cell board[XDIM][YDIM], int iMyColor)
{
 int awhite,ablack;
 int i,hiscolor;

 i=CheckForWin();
 if (iMyColor==WHITE) hiscolor=BLACK;
 else hiscolor=WHITE;
 if (i==iMyColor) return -9999;
 else if (i==hiscolor) return 9999;

 awhite=astar(board,i_aryXLoc[WHITE],i_aryYLoc[WHITE],0);
 ablack=astar(board,i_aryXLoc[BLACK],i_aryYLoc[BLACK],8);
 if (iMyColor==WHITE) 
  {
   if (ablack<awhite) 
     return 2*(2*awhite-2*ablack)-4*(i_aryFences[WHITE]-i_aryFences[BLACK]);
   else
     return 3*(2*awhite-2*ablack)-4*(i_aryFences[WHITE]-i_aryFences[BLACK]);
  }
 else 
  {
   if (ablack<awhite) 
     return 2*(2*ablack-2*awhite)-4*(i_aryFences[BLACK]-i_aryFences[WHITE]);
   else
     return 3*(2*ablack-2*awhite)-4*(i_aryFences[BLACK]-i_aryFences[WHITE]);
  }
}

int maxvalue(cell board[XDIM][YDIM], int iComputerColor, int iWhoseMove, int alpha, int beta,
	int depth, char *bestmove)
{
 char mlist[150][7];
 int i,temp;
 cell tempboard[XDIM][YDIM]; 
 int iNextMove;
 char sopponentmove[10];
 int wFence,bFence,wX,wY,bX,bY;
 int h;
 int hiscolor;
 FILE *f;
 int uval=0, dval=0;

 i=CheckForWin();
 if (iComputerColor==WHITE) hiscolor=BLACK;
 else hiscolor=WHITE;
 if (i==iComputerColor) return -9999;
 else if (i==hiscolor) return 9999;

 strcpy(bestmove,"n");
 if (depth>=iDepth) 
 {
  h=(heuristic(board,iComputerColor));
  return h;
 } 
 if (depth==0) {
    iXNeighborhood=3;
    iYNeighborhood=3;
 }
 else if (depth==1) {
   iXNeighborhood=2;
   iYNeighborhood=2;
 }
 else {
   iXNeighborhood=2;
   iYNeighborhood=1;
 }
 MoveGen(board, iWhoseMove, mlist,0);
 if (!strcmp(mlist[0],"null")) {
    MoveGen(board, iWhoseMove, mlist, 1);
    strcpy(mlist[10],"null");
 }
 for (i=0; strcmp(mlist[i],"null"); i++)
 {
   memcpy((char *) &tempboard[0], (char *) &board[0], XDIM*YDIM*sizeof(cell));
   wFence=i_aryFences[WHITE]; bFence=i_aryFences[BLACK];
   wX=i_aryXLoc[WHITE]; wY=i_aryYLoc[WHITE];
   bX=i_aryXLoc[BLACK]; bY=i_aryYLoc[BLACK];
   MakeMove(tempboard, mlist[i], iWhoseMove);
   if (iWhoseMove==WHITE) iNextMove=BLACK; else iNextMove=WHITE;
   temp = minvalue(tempboard, iComputerColor, iNextMove, alpha, beta, depth+1, sopponentmove);
/*
	if (iWhoseMove==WHITE) {
	  if (!strcmp(mlist[i],"u"))  { temp+=1; uval=temp; }
	  else if (!strcmp(mlist[i],"d")) { temp-=1; dval=temp; }
	}
	else {
	  if (!strcmp(mlist[i],"u")) { temp-=1; uval=temp; }
	  else if (!strcmp(mlist[i],"d")) { temp+=1; dval=temp; }
	}
*/
if ((DEBUG) && (iMoveNum>25)) {
printf("d=%d move=%s temp=%d alpha=%d beta=%d\n",depth,mlist[i],temp,alpha,beta);
}

   i_aryFences[WHITE]=wFence; i_aryFences[BLACK]=bFence;
   i_aryXLoc[WHITE]=wX; i_aryYLoc[WHITE]=wY;
   i_aryXLoc[BLACK]=bX; i_aryYLoc[BLACK]=bY;
   if (temp>alpha) { alpha=temp; strcpy(bestmove,mlist[i]); }
   else if (RANDOMIT && (temp==alpha) && ((random() % 100) < 50) && (iMoveNum>25))
   	 { alpha=temp; strcpy(bestmove,mlist[i]); }
   if (alpha>=beta) return alpha;
 }
if ((DEBUG) && (iMoveNum>25)) {
printf("d=%d returning %d\n", depth, alpha);
}
 return alpha;
}

int minvalue(cell board[XDIM][YDIM], int iComputerColor, int iWhoseMove, int alpha, int beta,
	int depth, char *bestmove)
{
 char mlist[150][7];
 int i,temp;
 cell tempboard[XDIM][YDIM]; 
 int iNextMove;
 char sopponentmove[10];
 int wFence,bFence,wX,wY,bX,bY;
 int hiscolor;
 FILE *f;

 i=CheckForWin();
 if (iComputerColor==WHITE) hiscolor=BLACK;
 else hiscolor=WHITE;
 if (i==iComputerColor) return -9999;
 else if (i==hiscolor) return 9999;

 strcpy(bestmove,"n");
 if (depth>=iDepth) 
 {
  return (heuristic(board,iComputerColor));
 } 
 if (depth==0) {
    iXNeighborhood=3;
    iYNeighborhood=3;
 }
 else if (depth==1) {
   iXNeighborhood=2;
   iYNeighborhood=2;
 }
 else {
   iXNeighborhood=2;
   iYNeighborhood=1;
 }
 MoveGen(board, iWhoseMove, mlist,0);
 if (!strcmp(mlist[0],"null")) {
    MoveGen(board, iWhoseMove, mlist, 1);
    strcpy(mlist[10],"null");
 }
 for (i=0; strcmp(mlist[i],"null"); i++)
 {
   memcpy((char *) &tempboard[0], (char *) &board[0], XDIM*YDIM*sizeof(cell));
   wFence=i_aryFences[WHITE]; bFence=i_aryFences[BLACK];
   wX=i_aryXLoc[WHITE]; wY=i_aryYLoc[WHITE];
   bX=i_aryXLoc[BLACK]; bY=i_aryYLoc[BLACK];
   MakeMove(tempboard, mlist[i], iWhoseMove);
   if (iWhoseMove==WHITE) iNextMove=BLACK; else iNextMove=WHITE;
   temp = maxvalue(tempboard, iComputerColor, iNextMove, alpha, beta, depth+1, sopponentmove);
  /* Try to not move back to where we came from if we're just moving around */
   if (depth==0) {
     if ((mlist[i][0]=='u') || (mlist[i][0]=='d') || (mlist[i][0]=='l') || (mlist[i][0]=='r')) {
	if ((i_aryXLoc[iWhoseMove]==i_aryLastXLoc[iWhoseMove]) &&
	    (i_aryYLoc[iWhoseMove]==i_aryLastYLoc[iWhoseMove]) &&
	    (i_aryLastWasMove[WHITE]) && (i_aryLastWasMove[BLACK])) {
		temp+=200;
	}
     } 
   }
if ((DEBUG) && (iMoveNum>25)) {
printf("d=%d move=%s temp=%d alpha=%d beta=%d\n",depth,mlist[i],temp,alpha,beta);
}
   i_aryFences[WHITE]=wFence; i_aryFences[BLACK]=bFence;
   i_aryXLoc[WHITE]=wX; i_aryYLoc[WHITE]=wY;
   i_aryXLoc[BLACK]=bX; i_aryYLoc[BLACK]=bY;
/*
	if (iWhoseMove==WHITE) {
	  if (!strcmp(mlist[i],"u")) temp-=1;
	  else if (!strcmp(mlist[i],"d")) temp+=1;
	}
	else {
	  if (!strcmp(mlist[i],"u")) temp+=1;
	  else if (!strcmp(mlist[i],"d")) temp-=1;
	}
*/
   if (temp<beta) { beta=temp; strcpy(bestmove,mlist[i]); }
   else if ((RANDOMIT) && (temp==beta) && ((random() % 100) < 50) && (iMoveNum>25))
   	{ 
	 if (DEBUG) printf("changing %s b=%d to %s t=%d\n",bestmove,beta,mlist[i],temp); 
  	 beta=temp; strcpy(bestmove,mlist[i]); 
 	}
   if (alpha>=beta) return beta;
 }

if ((DEBUG) && (iMoveNum>25)) {
printf("d=%d returning beta=%d\n",depth,beta);
}

 return beta;
}

