import java.awt.*;
import java.awt.event.*;
import java.math.BigInteger;
import javax.swing.*;
import javax.swing.border.*;
import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;

import java.util.Vector;

/**
Date: 2/21/04
Author: Anastasia Mironova

The purpose of this class is to represent part the landscape model for the PMPSim system
*/

public class LandscapeSegment extends JPanel implements MouseListener, MouseMotionListener{
	//the coordinates of this segment in the Landscape grid
	private Location coordinatesInLandscapeGrid = null;
	
	//the reference to the Landscape object
	private Landscape landscape = null;
	
	//Strings that contain the ids of the objects in the scene
	private Vector plants = new Vector();
	private Vector moose = new Vector();
	private Vector predators = new Vector();
	
	//distribution patterns for all the user-specified types of plants, moose, predators
	//objects that are added to this vector are of type SpeciesDistributionSpecifications
	private Vector plantsDistributionPatterns;
	private Vector mooseDistributionPatterns;
	private Vector predatorsDistributionPatterns;
	
	//dimensional specificaions of a single segment panel
	public static final int defaultSideIncrementLength = 10;
	private static int sideLengthIncrement = 1;
	
	//elevation parameter is in meters from the sea level
	private int elevationValue = 0;
	private boolean isWaterPatch = false;
	private boolean isRiparianHabitat = false;
	private boolean isForestHabitat = false;
	
	
	/**Constructors for a new instance of a Landscape
	*/
	public LandscapeSegment(Landscape l, Location coordinatesInLandscapeIn, Vector plantsIn, Vector mooseIn, Vector predatorsIn){
		super();
		
		//set the landscape
		landscape = l;
		
		setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
		setBorder(new BevelBorder(BevelBorder.LOWERED));
		
		coordinatesInLandscapeGrid = coordinatesInLandscapeIn;
		plants = plantsIn;
		moose = mooseIn;
		predators = predatorsIn;
		
		/**
		This is a mouse listener to allow the user to specify elevation or water
		*/
		addMouseListener(this);
		addMouseMotionListener(this);
	}
	/*
	Sets the default side length increment, this value is used to set the size of the segment, - 
	here the defaultSideLength is multiplied by this value in order to create a segment of desired dimensions
	*/
	public static void setLengthIncrement(int inc){
		sideLengthIncrement = inc;
	}
	
	public static int getLengthIncrement(){
		return sideLengthIncrement;
	}
	
	//returns the dimensions of the single cell
	public static Dimension getDimensions(){
		return new Dimension(sideLengthIncrement*defaultSideIncrementLength, sideLengthIncrement*defaultSideIncrementLength);
	}
	
	/** Methods to get/set parameters of this class
	*/
	//returns location of this segment in the grid
	public Location getLocationInLandscapeGrid(){
		return coordinatesInLandscapeGrid;
	}
	
	//returns the vector containting plants
	public Vector getPlants(){
		return plants;
	}
	
	//returns the vector containing moose
	public Vector getMoose(){
		return moose;
	}
	
	//returns the vector containing predators
	public Vector getPredators(){
		return predators;
	}
	
	//sets the vector containting plants for this object
	public void setPlants(Vector plantsIn){
		plants = plantsIn;
	}
	
	//sets the vector containing moose
	public void setMoose(Vector mooseIn){
		moose = mooseIn;
	}
	
	//sets the vector containing predators
	public void setPredators(Vector predatorsIn){
		predators = predatorsIn;
	}
	
	//returns a copy of the object of this class
	public LandscapeSegment copy(){
		LandscapeSegment copy = new LandscapeSegment(this.getLandscape(), this.getLocationInLandscapeGrid(),
								this.getPlants(),
								this.getMoose(),
								this.getPredators());
		return copy;
	}
	
	
	/*
	Get the parent landscape of this segment
	*/
	public Landscape getLandscape(){
		return landscape;
	}
	
	//makes the object adopt the properties of the given one
	public void makeLike(LandscapeSegment s){
		this.coordinatesInLandscapeGrid = s.getLocationInLandscapeGrid();
		this.plants = s.getPlants();
		this.moose = s.getMoose();
		this.predators = s.getPredators();
	} 
	
	/** Methods to add new agents to this part of the landscape
	*/
	//add a new plant to this part of the landscape
	public void addPlant(String newPlant){
		plants.addElement(newPlant);
		repaint();
	}
	
	//add a new moose to this part of the landscape
	public void addMoose(String newMoose){
		moose.addElement(newMoose);
		repaint();
	}
	
	//add a new predator to this part of the landscape
	public void addPredator(String newPredator){
		predators.addElement(newPredator);
		repaint();
	}
	
	
	/** Methods to remove new agents to this part of the landscape
	*/
	//remove a plant from this part of the landscape
	public void removePlant(String plantToRemove){
		if (plants.contains(plantToRemove)){
			plants.remove(plantToRemove);
			repaint();
		}
	}
	
	//remove a moose from this part of the landscape
	public void removeMoose(String mooseToRemove){
		if (moose.contains(mooseToRemove)){
			moose.remove(mooseToRemove);
			repaint();
		}
	}
	
	//remove a predator from this part of the landscape
	public void removePredator(String predatorToRemove){
		if (predators.contains(predatorToRemove)){
			predators.remove(predatorToRemove);
			repaint();
		}
	}
	
	//definition of properties occurs only when the plandscape is in the state of being customized
	//the two methods below handle the user-defined modifications and set the appropriate values for the overall setup of the 
	//landscape of the simulation
	public void mousePressed(MouseEvent evt) {
		if (landscape.areModificationsEnabled() == true){
			setBackground(landscape.getCurrentColor());
			landscape.setMouseToPressed(true);
		}
	}
	
	public void mouseReleased(MouseEvent evt) {
		if (landscape.areModificationsEnabled() == true){
			landscape.setMouseToPressed(false);
		}
	}
	
	
	public void mouseDragged(MouseEvent evt) {
	}
	
	
	public void mouseEntered(MouseEvent evt) {
		if (landscape.areModificationsEnabled() == true){
			if (landscape.isMousePressed() == true){
				setBackground(landscape.getCurrentColor());
			}
		}
	}
	
	public void mouseExited(MouseEvent evt) { }    //    (Required by the MouseListener
	
	public void mouseClicked(MouseEvent evt) { }   //    and MouseMotionListener
	
	public void mouseMoved(MouseEvent evt) {
	}
	
	//sets the value of elevation in meters specified for this segment
	public void setElevation(int value){
		System.out.println("Setting elevation value for segment at: " + coordinatesInLandscapeGrid.getXCoordinate() + " " + coordinatesInLandscapeGrid.getYCoordinate());
		elevationValue = value;
	}
	
	//returns the value of elevation in meters specified for this segment
	public int getElevation(){
		return elevationValue;
	}
	
	//defines a bog/wetland area at this segment
	public void makeWaterPatch(){
		System.out.println("Makind a wetland patch for segment at: " + coordinatesInLandscapeGrid.getXCoordinate() + " " + coordinatesInLandscapeGrid.getYCoordinate());
		isWaterPatch = true;
	}
	
	//checks if this segment is a bog/wetland area
	public boolean isWaterPatch(){
		return isWaterPatch;
	}
	
	//defines a riparian habitat at this segment
	public void makeRiparianPatch(){
		System.out.println("Makind a riparian patch for segment at: " + coordinatesInLandscapeGrid.getXCoordinate() + " " + coordinatesInLandscapeGrid.getYCoordinate());
		isRiparianHabitat = true;
	}
	
	//checks if this segment is a riparian habitat
	public boolean isRiparianPatch(){
		return isRiparianHabitat;
	}
	
	//defines a forest habitat at this segment
	public void makeForestPatch(){
		System.out.println("Makind a forest habitat for segment at: " + coordinatesInLandscapeGrid.getXCoordinate() + " " + coordinatesInLandscapeGrid.getYCoordinate());
		isForestHabitat = true;
	}
	
	//checks if this segment is a forest habitat
	public boolean isForestPatch(){
		return isForestHabitat;
	}
	
	
	/**
	Overriding the default to draw the components
	Outlines the Location of this segment in the overall landscape
	*/
	public void paintComponent(Graphics g){
		Graphics2D g2 = (Graphics2D) g;
		super.paintComponent(g2);
		g2.drawString("" + coordinatesInLandscapeGrid.getXCoordinate() + " " +
				coordinatesInLandscapeGrid.getYCoordinate(), 5, 15);
		System.out.println("Segment " + coordinatesInLandscapeGrid.getXCoordinate()+ ", " + coordinatesInLandscapeGrid.getYCoordinate()  + " repainted");
	}
}
