// 
//	DIB.C
//
//	display bitonal, grayscale and color DIB images
//
///////////////////////////////////////////////
#include "stdafx.h"
#include "dib.h"
#include <math.h>

// defined in <wifview.c>, it is updated each time a tifread() is called
BOOL BitonalFlag;

static int ColorMode;		// we support grayscale image & 24-bit RGB color image
static int nNumColors;		// size of the color table in DIB header struct

static OPENFILENAME    QKofN;
static FILE *fp, *fdbg;

// writes an image onto a window
int CreateDIBDisplay(HWND hWnd, 
	unsigned char * hpAcquireBuf, unsigned long ulHeight, 
	unsigned long ulWidth, unsigned long ulPixelDepth, 
	unsigned long ulBufferSize, double stretch)
{
	HDC hdc;
	HPALETTE hColorPal;
	HGLOBAL hPal = NULL;
	LPLOGPALETTE lpPal = NULL;
	HGLOBAL hDIBdata = NULL;
	unsigned char * hpDIBdata = NULL;
	HGLOBAL hDIBheader = NULL;
	BITMAPINFO * lpDIBheader = NULL; 
	POINTERLIST PointerList = {0};		// used strictly for cleanup
	unsigned char * hpTempAcquireBuf = NULL;
	unsigned char * hpTempDIBdata = NULL;  
	int scanlines;

 	int DIBdataSize, ZeroPaddedPerRow, i = 0;


	// to see which color mode we should work with
	switch(ulPixelDepth)
	{
	case 1:
		if(BitonalFlag==FALSE)
		{
			ColorMode = GRAY_SCALE;
			nNumColors = 256;
		}
		else
		{
			ColorMode = BITONAL;
			nNumColors = 2;
		}
		break;
	case 3:
		ColorMode = RGB_COLOR24;
		nNumColors = 0;
		break;
	default:
		return -1;
		break;
	}

    // Allocate space for a DIB header  
	hDIBheader = GlobalAlloc(GHND, sizeof(BITMAPINFO) + (nNumColors * sizeof(RGBQUAD)));
	PointerList.hDIBheader = hDIBheader;
	if (!hDIBheader)
	{
		CleanupDIBPointers(&PointerList);
		return -1; 
	}

    // Allocate space for a logical palette
	hPal = GlobalAlloc(GHND, sizeof(LOGPALETTE) + (MAXPALETTE * sizeof(PALETTEENTRY)));
	PointerList.hPal = hPal;
	if (!hPal)
	{                          
		CleanupDIBPointers(&PointerList);
		return -1;
	}

	// Allocate space for the actual DIB data (rows padded to 4-byte boundaries)
	if(ColorMode==BITONAL)
	{
		DIBdataSize=(int)ceil(1.0*ulWidth/32)*4*ulHeight;
		ZeroPaddedPerRow=(int)ceil(1.0*ulWidth/32)*4-(int)ceil(1.0*ulWidth/8);
	}
	else
	{
		DIBdataSize=ulBufferSize;
	}
	hDIBdata = GlobalAlloc(GHND, DIBdataSize);  
	PointerList.hDIBdata = hDIBdata;
	if (!hDIBdata )
	{
		CleanupDIBPointers(&PointerList);
		return -1; 
	}


	// Lock the space so we can get pointers to the data
	lpDIBheader = (tagBITMAPINFO *)GlobalLock(hDIBheader);  
	PointerList.lpDIBheader = lpDIBheader;
	if (!lpDIBheader)
	{
		CleanupDIBPointers(&PointerList);
		return -1; 
	}

	lpPal = (tagLOGPALETTE *)GlobalLock(hPal);             
	PointerList.lpPal = lpPal;
	if (!lpPal)
	{
		CleanupDIBPointers(&PointerList);
		return -1; 
	}

	hpDIBdata = (unsigned char *) GlobalLock(hDIBdata);
	PointerList.hpDIBdata = hpDIBdata;
	if (!hDIBdata)
	{
		CleanupDIBPointers(&PointerList);
		return -1; 
	}

    // Fill in the static parts of the DIB header
	FillDIBheader(lpDIBheader, ulHeight, ulWidth, ulPixelDepth, ulBufferSize);

	// Fill in the static parts of the logical palette
	FillLogicalPalette(lpPal);

    // Try and create a palette
	hColorPal = CreatePalette(lpPal);
	PointerList.hGrayscalePal = hColorPal;
	if ( !hColorPal )
	{
		CleanupDIBPointers(&PointerList);
		return -1;
	}
	
	if(BitonalFlag==FALSE)
	{
		// use temporary pointers to move through acquire and bitmap data
		hpTempDIBdata = hpDIBdata;
		hpTempAcquireBuf = hpAcquireBuf;
		hpTempAcquireBuf += (ulHeight - 1) * ulWidth * ulPixelDepth; 
	
		// Load bitmap top to bottom starting with the bottom row of the acquire data.
		// Since the function assumes an 8 bit DIB, ulPixelDepth is assumed to be 1.
		// It is just used here to be a reminder for future implementation issues.
         
		for (i = 0; i < (int) ulHeight; i++)
		{
			_fmemcpy(hpTempDIBdata, hpTempAcquireBuf, (size_t) (ulWidth * ulPixelDepth));
			hpTempDIBdata += ulWidth * ulPixelDepth;		
			hpTempAcquireBuf -= ulWidth * ulPixelDepth;
		}
	}
	else
	{
		// use temporary pointers to move through acquire and bitmap data
		hpTempDIBdata = hpDIBdata;
		hpTempAcquireBuf = hpAcquireBuf;
		hpTempAcquireBuf += (ulHeight - 1) * (ulBufferSize/ulHeight); 

		for (i = 0; i < (int) ulHeight; i++)
		{
			_fmemcpy(hpTempDIBdata, hpTempAcquireBuf, (size_t)(ulBufferSize/ulHeight));
			hpTempDIBdata += (DIBdataSize/ulHeight);		
			hpTempAcquireBuf -= (ulBufferSize/ulHeight);
		}
	}


	// Lets Select and Realize the palette we created earlier (this has to be done
	// before creating the device dependent bitmap from the DIB info
	
	hdc = GetDC(hWnd);		//GetActiveWindow();

	// save the old palette to restore later
	SelectPalette(hdc, hColorPal, FALSE);
	RealizePalette(hdc);

	// blast the bitmap bits directly to the device context surface (our client area)

//	scanlines = SetDIBitsToDevice(
//		hdc,			// handle of device context
//		0,				// x-coordinate origin of destination rect
//		0,				// y-coordinate origin of destination rect
//		(int)ulWidth,	// rectangle width
//		(int)ulHeight,	// rectangle height
//		0,				// x-coordinate origin of source rect
//		0,				// y-coordinate origin of source rect
//		0,				// number of first scan line in array
//		(UINT)ulHeight,	// number of scan lines
//		hpDIBdata,		// address of array with DIB bits
//		lpDIBheader,	// address of structure with bitmap info
//		DIB_RGB_COLORS);// RGB or palette indices
	
	scanlines = StretchDIBits(
	  hdc,					// handle to device context
	  0,					// x-coordinate of upper-left corner of dest. rectangle
	  0,					// y-coordinate of upper-left corner of dest. rectangle
	  (int)(stretch*ulWidth),	// width of destination rectangle
	  (int)(stretch*ulHeight),	// height of destination rectangle
	  0,					// x-coordinate of upper-left corner of source rectangle
	  0,					// y-coordinate of upper-left corner of source rectangle
	  (int)ulWidth,         // width of source rectangle
	  (int)ulHeight,        // height of source rectangle
	  hpDIBdata,            // address of bitmap bits
	  lpDIBheader,			// address of bitmap data
	  DIB_RGB_COLORS,       // usage flags
	  SRCCOPY);				// raster operation code
	

 	// clean up
	ReleaseDC(hWnd,hdc);
    CleanupDIBPointers(&PointerList);

	return 1;
}

//=====================================================================================
//=====================================================================================

/*	Function:	CleanupDIBPointers

	Description:
		This function frees up any allocated memory or resources used in 
		creating the Device Independent Bitmap.


	Input Arguments:
		LPPOINTERLIST lpPointerList 	pointer to structure that holds the handles
						of the elements that make up the DIB

	Output Arguments:
		None.

	Return Value: void
		None
*/

void PASCAL CleanupDIBPointers(LPPOINTERLIST lpPointerList )
{
 
    /* Clean up global memory usage */
	if ( lpPointerList->lpDIBheader )
	  {
	   GlobalUnlock(lpPointerList->hDIBheader);
	   lpPointerList->lpDIBheader = NULL;
	  }
	if ( lpPointerList->hDIBheader )
	  {
	   GlobalFree(lpPointerList->hDIBheader);
	   lpPointerList->hDIBheader = NULL;
	  }

	if ( lpPointerList->lpPal )
	  {
	    GlobalUnlock(lpPointerList->hPal);
	    lpPointerList->lpPal = NULL;
	  }
	if ( lpPointerList->hPal)
	  {
	    GlobalFree(lpPointerList->hPal);
	    lpPointerList->hPal = NULL;
	  }
	if ( lpPointerList->hpDIBdata )
	  {
	   GlobalUnlock(lpPointerList->hDIBdata);
	   lpPointerList->hpDIBdata = NULL;
	  }
	if ( lpPointerList->hDIBdata )
	  {
	   GlobalFree(lpPointerList->hDIBdata);
	   lpPointerList->hDIBdata = NULL;
	  }
	if ( lpPointerList->hGrayscalePal )
	  {
	   DeleteObject((HGDIOBJ)lpPointerList->hGrayscalePal);
	   lpPointerList->hGrayscalePal = NULL;
	  }
}


// Fill out the DIB header information
void FillDIBheader(BITMAPINFO * lpDIBheader, unsigned long ulHeight, unsigned long ulWidth, unsigned long ulPixelDepth, unsigned long ulBufferSize)
{
	int i;

	if(lpDIBheader == NULL)
		return;

    // Fill in the static parts of the DIB header
	lpDIBheader->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	lpDIBheader->bmiHeader.biWidth = ulWidth;
	lpDIBheader->bmiHeader.biHeight = ulHeight;
	lpDIBheader->bmiHeader.biPlanes = 1;

	// this value must be 1, 4, 8 or 24 so ulPixelDepth can only be a 1 or 3; 
	if(BitonalFlag==FALSE)
		lpDIBheader->bmiHeader.biBitCount = (WORD)(8 * ulPixelDepth);
	else
		lpDIBheader->bmiHeader.biBitCount = (WORD)1;

	lpDIBheader->bmiHeader.biCompression = BI_RGB; 
	lpDIBheader->bmiHeader.biSizeImage = ulBufferSize; 

	if(BitonalFlag==FALSE)
	{
		lpDIBheader->bmiHeader.biClrUsed = 256;
		lpDIBheader->bmiHeader.biClrImportant = 256; 
	}
	else
	{
		lpDIBheader->bmiHeader.biClrUsed = 2;
		lpDIBheader->bmiHeader.biClrImportant = 2; 
	}
	// we need to build color table for grayscale images
	switch(ColorMode)
	{
	case BITONAL:
		lpDIBheader->bmiColors[0].rgbRed =(BYTE)0;
		lpDIBheader->bmiColors[0].rgbGreen = (BYTE)0;
		lpDIBheader->bmiColors[0].rgbBlue = (BYTE)0;
		lpDIBheader->bmiColors[0].rgbReserved = 0;
		lpDIBheader->bmiColors[1].rgbRed =(BYTE)255;
		lpDIBheader->bmiColors[1].rgbGreen = (BYTE)255;
		lpDIBheader->bmiColors[1].rgbBlue = (BYTE)255;
		lpDIBheader->bmiColors[1].rgbReserved = 0;

		break;
	case GRAY_SCALE:
		// 128 grayscales are sufficient
		for (i = 0; i < nNumColors; i+=2)
		{
			lpDIBheader->bmiColors[i].rgbRed = lpDIBheader->bmiColors[i + 1].rgbRed = (BYTE) i;
			lpDIBheader->bmiColors[i].rgbGreen = lpDIBheader->bmiColors[i + 1].rgbGreen = (BYTE)i;
			lpDIBheader->bmiColors[i].rgbBlue = lpDIBheader->bmiColors[i + 1].rgbBlue = (BYTE)i;
			lpDIBheader->bmiColors[i].rgbReserved = 0;
		}
		break;
	case RGB_COLOR24:
		// a 24-bit DIB has no color table entries
		break;
	default:
		return;
	}

}


// Fill out the logical palette
void FillLogicalPalette(LPLOGPALETTE lpPal)
{
	int i;
	BYTE red,green,blue;

	if(lpPal == NULL)
		return;

	lpPal->palVersion = PALVERSION;
	if(ColorMode==BITONAL)
		lpPal->palNumEntries = 2;
	else
		lpPal->palNumEntries = MAXPALETTE;

	switch(ColorMode)
	{
	case BITONAL:
		for (i = 0; i < 2; i++)
		{
			lpPal->palPalEntry[i].peFlags =  0;
			lpPal->palPalEntry[i].peRed = (BYTE)(i*255);
			lpPal->palPalEntry[i].peGreen = (BYTE)(i*255);
			lpPal->palPalEntry[i].peBlue = (BYTE)(i*255);
		}
		break;
	case GRAY_SCALE:
		// Initialize the logical palettes to grayscale; 
		// 128 grayscales are sufficient
		for (i = 0; i < MAXPALETTE; i+=2)
		{
			lpPal->palPalEntry[i].peFlags =  0;
			lpPal->palPalEntry[i].peRed =  lpPal->palPalEntry[i + 1].peRed = (BYTE)i;
			lpPal->palPalEntry[i].peGreen = lpPal->palPalEntry[i + 1].peGreen = (BYTE)i;
			lpPal->palPalEntry[i].peBlue = lpPal->palPalEntry[i + 1].peBlue = (BYTE)i;
		}
		break;
	case RGB_COLOR24:

		// a 24-bit DIB has no color table entires, so set 
		// the number to the maximum value (MAXPALETTE)

		// generate the palette
		red=green=blue=0;
		for(i=0;i<lpPal->palNumEntries;i++)
		{
			lpPal->palPalEntry[i].peRed=red;
			lpPal->palPalEntry[i].peGreen=green;
			lpPal->palPalEntry[i].peBlue=blue;
			lpPal->palPalEntry[i].peFlags=(BYTE)0;
			
			if(!(red+=32))
				if(!(green+=32))
					blue+=64;
		}
		break;
	default:
		return;
	}

}


