// WaveletSettings.cpp : implementation file
//

#include "stdafx.h"
#include "Readfile.h"
#include "GAwavelet.h"
#include "WaveletSettings.h"
#include <math.h>
#include <time.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern int mrlevels;
extern int stepsize;
extern int threshold;
extern BOOL showresults;
extern CString wavelettype;
extern int waveletnumber;

extern int maxgen;
extern int popsize;
extern BOOL evolving;

int SUBIMAGE_WEIGHT =100;
int SUPERIMAGE_WEIGHT = 100;

	// Haar
	double haarh1[] = {0.70710678,0.70710678};
	double haarg1[] = {0.70710678,-0.70710678};
	int haarhlen = 2;
	double haarh2[] = {0.70710678,0.70710678};
	double haarg2[] = {-0.70710678,0.70710678};
	int haarglen = 2;

	// D4

	double daub4h1[] = {-0.129409523,0.224143868,0.836516304,0.482962913}; 
	double daub4g1[] = {-0.482962913,0.836516304,-0.224143868,-0.129409523}; 
	int daub4hlen = 4;
	double daub4h2[] = {0.482962913,0.836516304,0.224143868,-0.129409523}; 
	double daub4g2[] = {-0.129409523,-0.224143868,0.836516304,-0.482962913}; 
	int daub4glen = 4;

	// D6
	double daub6h1[] = {0.035213918,-0.085418499,-0.135057395,0.45990225,0.806950259,0.33262303}; 
	double daub6g1[] = {-0.33262303,0.806950259,-0.45990225,-0.135057395,0.085418499,0.035213918}; 
	int daub6hlen = 6;
	double daub6h2[] = {0.33262303,0.806950259,0.45990225,-0.135057395,-0.085418499,0.035213918}; 
	double daub6g2[] = {0.035213918,0.085418499,-0.135057395,-0.45990225,0.806950259,-0.33262303}; 
	int daub6glen = 6;

	// D8
	double daub8h1[] = {-0.010606602,0.032951176,0.030829856,-0.187100454,-0.028001429,0.63088067,0.714884956,0.230375389}; 
	double daub8g1[] = {-0.230375389,0.714884956,-0.63088067,-0.028001429,0.187100454,0.030829856,-0.032951176,-0.010606602}; 
	int daub8hlen = 8;
	double daub8h2[] = {0.230375389,0.714884956,0.63088067,-0.028001429,-0.187100454,0.030829856,0.032951176,-0.010606602}; 
	double daub8g2[] = {-0.010606602,-0.032951176,0.030829856,0.187100454,-0.028001429,-0.63088067,0.714884956,-0.230375389}; 
	int daub8glen = 8;

	// Two-six
	double tsh1[] = {0.70710678,0.70710678};
	double tsg1[] = {-0.088388,-0.088388,0.707107,-0.707107,0.088388,0.088388};
	int tshlen = 2;
	double tsh2[] = {-0.088388,0.088388,0.707107,0.707107,0.088388,-0.088388};
	double tsg2[] = {-0.70710678,0.70710678};
	int tsglen = 6;

	// five-third
	double fth1[] = {-0.176777,0.353553,1.060660,0.353553,-0.176777};
	double ftg1[] = {-0.353553,0.70710678,-0.353553};
	int fthlen = 5;
	double fth2[] = {0.353553,0.70710678,0.353553};
	double ftg2[] = {-0.176777,-0.353553,1.060660,-0.353553,-0.176777};
	int ftglen = 3;


void forward(double *X, int N, double *h, double *g, int hlen, int glen);
void inverse(double *X, int N, double *h, double *g, int hlen, int glen);
double GetMutationFactor();
double GetRandomNoiseFactor();

/////////////////////////////////////////////////////////////////////////////
// CWaveletSettings dialog

CWaveletSettings::CWaveletSettings(CWnd* pParent /*=NULL*/)
	: CDialog(CWaveletSettings::IDD, pParent)
{
	//{{AFX_DATA_INIT(CWaveletSettings)
	m_wavelettype = wavelettype;
	m_mrlevel = mrlevels;
	m_stepsize = stepsize;
	m_threshold = threshold;
	m_showresults = showresults;

	m_maxgen = maxgen;
	m_popsize = popsize;
	m_evolving = evolving;

	//}}AFX_DATA_INIT
	m_waveletnumber = waveletnumber;

	if(m_waveletnumber == 0)
		m_wavelettype = "Daubechies 4";
	else if(m_wavelettype == 1)
		m_wavelettype = "Haar Wavelet";
	else if(m_waveletnumber == 2)
		m_wavelettype = "2/6 Wavelet";
	else if(m_waveletnumber == 3)
		m_wavelettype = "5/3 Wavelet";
	else if(m_waveletnumber == 4)
		m_wavelettype = "Daubechies 6";
	else if(m_waveletnumber == 5)
		m_wavelettype = "Daubechies 8";
	else
		m_wavelettype = "Custom";
	CInitializeGH();
}


void CWaveletSettings::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CWaveletSettings)
	DDX_CBString(pDX, IDC_WAVELET_TYPE, m_wavelettype);
	DDX_CBIndex(pDX, IDC_MRLEVEL, m_mrlevel);
	DDX_CBIndex(pDX, IDC_STEPSIZE, m_stepsize);
	DDX_CBIndex(pDX, IDC_THRESHOLD, m_threshold);
	DDX_Check(pDX, IDC_SHOWRESULTS, m_showresults);

	DDX_CBIndex(pDX, IDC_MAXGEN, m_maxgen);
	DDX_CBIndex(pDX, IDC_POPSIZE, m_popsize);
	DDX_Check(pDX, IDC_EVOLVING, m_evolving);
	//}}AFX_DATA_MAP

	if(m_wavelettype == "Daubechies 4")
		m_waveletnumber = 0;
	else if(m_wavelettype == "Haar Wavelet")
		m_waveletnumber = 1;
	else if(m_wavelettype == "2/6 Wavelet")
		m_waveletnumber = 2;
	else if(m_wavelettype == "5/3 Wavelet")
		m_waveletnumber = 3;
	else if(m_wavelettype == "Daubechies 6")
		m_waveletnumber = 4;
	else if(m_wavelettype == "Daubechies 8")
		m_waveletnumber = 5;
	else
		m_waveletnumber = 6;
}

void CWaveletSettings::CWaveletTransform(double *Y, int width, int height)
{
	int x, y, k;
	double *temp;
	int lengthw, lengthh;

	temp = new double[height];

	lengthw = width;
	lengthh = height;
	for(k=0;k<mrlevels;k++)
	{
		if((lengthw < hlen)||(lengthh < hlen)||(lengthw < glen)||(lengthh < glen))
		{
			m_mrlevel = k;
			mrlevels = k;
			break;
		}

		// Horizontal
		for(y=0;y<lengthh;y++)
			forward(Y+y*width,lengthw,h1,g1,hlen,glen);
		// vertical
		for(x=0;x<lengthw;x++)
		{
			for(y=0;y<lengthh;y++)
				temp[y] = Y[x+y*width];
			forward(temp,lengthh,h1,g1,hlen,glen);
			for(y=0;y<lengthh;y++)
				Y[x+y*width] = temp[y];
		}
		lengthw = lengthw/2 + lengthw%2;
		lengthh = lengthh/2 + lengthh%2;
	}

	delete[] temp;
}

void CWaveletSettings::CWaveletInverseTransform (double *Y[], int width, int height, int imageCount)
{
	int x, y, k, j;
	double *temp;
	int lengthw, lengthh;

	temp = new double[height];
	mrlevels = m_mrlevel;

	for (j = 0; j < imageCount; j++)
		for(k=mrlevels-1;k>=0;k--)
		{
			lengthw = width>>k;
			lengthh = height>>k;
			if(width%((int)pow(2,k)) != 0)
				lengthw++;
			if(height%((int)pow(2,k)) != 0)
				lengthh++;
	
			// vertical
			for(x=0;x<lengthw;x++)
			{
				for(y=0;y<lengthh;y++)
					temp[y] = Y[j][x+y*width];
				inverse(temp,lengthh,h2,g2,hlen,glen);
				for(y=0;y<lengthh;y++)
					Y[j][x+y*width] = temp[y];
			}
			// horizontal
			for(y=0;y<lengthh;y++)
				inverse(Y[j]+y*width,lengthw,h2,g2,hlen,glen);
		}

	delete[] temp;
}

// CWaveletCoEvolvedInverseTransformGA
// This is a revised version of the (already revised) "standard" GA
// Revisions have been done to allow for the wavelet coefficients to
// try and maximize their MSE on the parent image while minimizing their
// MSE on the sub-image in order to (hopefully) "search" the image.

void CWaveletSettings::CWaveletCoEvolvedInverseTransformGA (double *Y[], double *originalY[], int width, int height, int imageCount, double *parentY, double *parentOriginalY, int parentWidth, int parentHeight)
{
	double *bestY[maxImageCount];

	//values are hard coded due to windows GUI programming madness

	popsize = 5000;
	maxgen = 2500;

	const int population_size = popsize;
	const int max_generations = maxgen;

	bool mutating = false;
	const bool crossing_over = true;
	bool new_best;
	int mutation_rate = 0; 
	int x, y, i, j, k, least;
	int G, M;
	double original_fitness;
	double best_h[40], best_g[40];
	double swap;
	double best_fitness;
	double **array_h, **array_g;
	double **new_array_h, **new_array_g;
	double *fitness;
	double *temp, *temp2;
	int lengthw, lengthh;
	double *transformedY[maxImageCount];
	double *g, *h;
	double *transformedParentY;
	int plengthw, plengthh;
	double maxParentMSE, minSubImageMSE/*, curParentMSEComparer, curSubImageMSEComparer*/;
	double subMSE, parentMSE;

	FILE *fh;

	// Seed the random-number generator with current time so that
	// the numbers will be different every time we run.

	srand( (unsigned)time( NULL ) );


	for (j = 0; j < imageCount; j++)
	{
		transformedY[j] = new double[width*height];
		for(i=0;i<width*height;i++)
			transformedY[j][i] = Y[j][i];
	}
	
	//initialize the transformed parent array
	transformedParentY = new double[parentWidth * parentHeight];
	for(i = 0; i < parentWidth * parentHeight; i++)
		transformedParentY[i] = parentY[i];

	//lots of array initialization...

	g = new double[glen];
	h = new double[hlen];
	array_h = new double*[popsize];
	array_g = new double*[popsize];
	new_array_h = new double*[popsize];
	new_array_g = new double*[popsize];
	for(i=0;i<popsize;i++)
	{
		array_h[i] = new double[hlen];
		array_g[i] = new double[glen];
		new_array_h[i] = new double[hlen];
		new_array_g[i] = new double[glen];
	}
	fitness = new double[popsize];

	for (j = 0; j < imageCount; j++)
		bestY[j] = new double[width*height];

	//set up the transforms to be evolved (allows for previously
	//evolved transforsm to be evolved further)

	memcpy(h,h2,hlen*sizeof(double));
	memcpy(g,g2,glen*sizeof(double));

	temp = new double[height];
	
	//apply the inverse transform using the original g, h
	//values to each image to determine a base fitness
	//note: this is just a cut-and-paste job from another 
	//method. some refactoring would be nice

	//here is the logic for the sub-images

	for (j = 0; j < imageCount; j++)
	{
		for(k=mrlevels-1;k>=0;k--)
		{
			lengthw = width>>k;
			lengthh = height>>k;
			if(width%((int)pow(2,k)) != 0)
				lengthw++;
			if(height%((int)pow(2,k)) != 0)
				lengthh++;

			// vertical
			for(x=0;x<lengthw;x++)
			{
				for(y=0;y<lengthh;y++)
					temp[y] = Y[j][x+y*width];
				inverse(temp,lengthh,h,g,hlen,glen);
				for(y=0;y<lengthh;y++)
					Y[j][x+y*width] = temp[y];
			}
			// horizontal
			for(y=0;y<lengthh;y++)
				inverse(Y[j]+y*width,lengthw,h,g,hlen,glen);
		}
	}

	//here is the logic for the parent image

	temp2 = new double[parentHeight];

	for(k=mrlevels-1;k>=0;k--)
	{
		plengthw = parentWidth>>k;
		plengthh = parentHeight>>k;
		if(parentWidth%((int)pow(2,k)) != 0)
			plengthw++;
		if(parentHeight%((int)pow(2,k)) != 0)
			plengthh++;

		// vertical
		for(x=0;x<plengthw;x++)
		{
			for(y=0;y<plengthh;y++)
				temp2[y] = parentY[x+y*parentWidth];
			inverse(temp2,plengthh,h,g,hlen,glen);
			for(y=0;y<plengthh;y++)
				parentY[x+y*parentWidth] = temp2[y];
		}
		// horizontal
		for(y=0;y<plengthh;y++)
			inverse(&parentY[y*parentWidth],plengthw,h,g,hlen,glen);
	}


	// Compute Error for restored image Y

	// Here the logic differs from the original GA.
	// We attempt to probe the pareto-optimal curve
	// and find g, h values which perform optimally on
	// the sub-image and very poorly on the super-image.

	minSubImageMSE = 0.0;

	// For each image, add up the squares of the difference in Y values for each pixel
	// and add to the total fitness. Lower is better.

	for (j = 0; j < imageCount; j++)
		for (x = 0; x < width; x++)
				for (y = 0; y < height; y++)
					minSubImageMSE += (Y[j][x+y*width] - originalY[j][x+y*width]) 
										* (Y[j][x+y*width] - originalY[j][x+y*width]);
//	curSubImageMSEComparer = minSubImageMSE;

	// Here we determine the parent MSE

	maxParentMSE = 0.0;
	for(x = 0; x < parentWidth * parentHeight; x++)
		maxParentMSE += (double)imageCount * (parentY[x] - parentOriginalY[x]) * (parentY[x] - parentOriginalY[x]);

//	curParentMSEComparer = maxParentMSE;

	// initialize original fitness
	
	original_fitness = SUBIMAGE_WEIGHT + SUPERIMAGE_WEIGHT;

	// Initialize "best seen so far" statistics using original h, g values
	
	// This uses hard-coded maximum length for best_g, best_h - will likely
	// want to generalize for later improvements (eg variable-length transforms)

	best_fitness = original_fitness;
	for (x = 0; x < hlen; x++)
		best_h[x] = h[x];
	for (x = 0; x < glen; x++)
		best_g[x] = g[x];

	// this chunk is likely superfluous now that evolution and application have
	// been separated... 

	for (j = 0; j < imageCount; j++)
		for (x = 0; x < width; x++)
			for (y = 0; y < height; y++)
				bestY[j][x+y*width] = Y[j][x+y*width];

	// transform evolution and application have been separated
	// and the arrays have been checked for NULL already
	// so this check is unnecessary.

	if ((evolving == 1) && (originalY != NULL))
	{
		// Introduce small perturbations to the coefficients of array_h and array_g.
		// (NOTE: the original h, g vectors are merely copied into array position 0.)

		for (k = 0; k < population_size; k++)
			for (x = 0; x < hlen; x++)
				if (k == 0)
					array_h[k][x] = h[x];
				else
					array_h[k][x] = h[x] * GetRandomNoiseFactor();

		for (k = 0; k < population_size; k++)
			for (x = 0; x < glen; x++)
				if (k == 0)
					array_g[k][x] = g[x];
				else
					array_g[k][x] = g[x] * GetRandomNoiseFactor();

		// Run a genetic algorithm for max_generations.
		for (G = 0; G < max_generations; G++)
		{
			// output progress to a file since adding a progress bar is an
			// "unnecessary" time investment
			if(G % 5 == 0)
			{
				fh = fopen("progress.txt", "w");
				fprintf(fh, "Beginning generation %d", G);
				fclose(fh);
			}
			new_best = false;
			
			// if we have new best values, update min, max variables
//			if(maxParentMSE > curParentMSEComparer)
//				curParentMSEComparer = maxParentMSE;
//			if(minSubImageMSE < curSubImageMSEComparer)
//				curSubImageMSEComparer = minSubImageMSE;

			// Evaluate the fitness of each of the M wavelets by performing an inverse transform
			// and calculating the error in the rebuilt images (relative to the original images).

			for (M = 0; M < population_size; M++)
			{
				for (j = 0; j < imageCount; j++)
				{
					// copy transformed Y vector to Y

					for (x = 0; x < width; x++)
						for (y = 0; y < height; y++)
							Y[j][x+y*width] = transformedY[j][x+y*width];


					// apply the current inverse transform to the current subimage

					for(k = mrlevels-1; k >= 0; k--)
					{
						lengthw = width>>k;
						lengthh = height>>k;
						if(width%((int)pow(2,k)) != 0)
							lengthw++;
						if(height%((int)pow(2,k)) != 0)
							lengthh++;
	
						// vertical
						for(x=0;x<lengthw;x++)
						{
							for(y=0;y<lengthh;y++)
								temp[y] = Y[j][x+y*width];
							inverse(temp,lengthh,array_h[M],array_g[M],hlen,glen);
							for(y=0;y<lengthh;y++)
								Y[j][x+y*width] = temp[y];
						}

						// horizontal
						for(y=0;y<lengthh;y++)
							inverse(Y[j]+y*width,lengthw,array_h[M],array_g[M],hlen,glen);
					}
				}

				// restore parentY values from transformedY

				for(x = 0; x < parentWidth * parentHeight; x++)
					parentY[x] = transformedParentY[x];

				// apply the current inverse transform to the parent image

				for(k=mrlevels-1;k>=0;k--)
				{
					plengthw = parentWidth>>k;
					plengthh = parentHeight>>k;
					if(parentWidth%((int)pow(2,k)) != 0)
						plengthw++;
					if(parentHeight%((int)pow(2,k)) != 0)
						plengthh++;

					// vertical
					for(x=0;x<plengthw;x++)
					{
						for(y=0;y<plengthh;y++)
							temp2[y] = parentY[x+y*parentWidth];
						inverse(temp2,plengthh,array_h[M],array_g[M],hlen,glen);
						for(y=0;y<plengthh;y++)
							parentY[x+y*parentWidth] = temp2[y];
					}
					// horizontal
					for(y=0;y<plengthh;y++)
						inverse(&parentY[y*parentWidth],plengthw,array_h[M],array_g[M],hlen,glen);
				}

				// measure fitness by comparing the inverse-transformed Y to the originalY
				// fitness measure should have been encapsulated for ease of change

				fitness[M] = 0.0;
				subMSE = 0.0;
				for (j = 0; j < imageCount; j++)
					for (x = 0; x < width; x++)
						for (y = 0; y < height; y++)
							subMSE += (Y[j][x+y*width] - originalY[j][x+y*width])
										  * (Y[j][x+y*width] - originalY[j][x+y*width]);
//				if(subMSE < minSubImageMSE)
//					minSubImageMSE = subMSE;
				
				// see original note on pareto fitness
				parentMSE = 0.0;
				for(x = 0; x < parentWidth * parentHeight; x++)
					parentMSE += (double)imageCount * (parentY[x] - parentOriginalY[x]) * (parentY[x] - parentOriginalY[x]);

//				if(parentMSE > maxParentMSE)
//					maxParentMSE = parentMSE;

//				fitness[M] = (subMSE / curSubImageMSEComparer) * 100.0 + (100.0 - (parentMSE / curParentMSEComparer)) * 100.0;
				fitness[M] = (subMSE / minSubImageMSE) * SUBIMAGE_WEIGHT + (maxParentMSE / parentMSE) * SUPERIMAGE_WEIGHT;

				// check for a new globally optimal fitness values
				if (fitness[M] < best_fitness)
				{
					// we have a new best-of-run individual
					new_best = true;

					// update "best ever fitness"
					best_fitness = fitness[M];

					// copy g and h values that produced "best ever fitness"
					for (x = 0; x < glen; x++)
						best_g[x] = array_g[M][x];
					for (x = 0; x < hlen; x++)
						best_h[x] = array_h[M][x];

					// save Y vector with "best ever fitness"
					for (j = 0; j < imageCount; j++)
						for (x = 0; x < width; x++)
							for (y = 0; y < height; y++)
								bestY[j][x+y*width] = Y[j][x+y*width];
				}
			}

			// Copy the best-of-generation individual into position 0 of the new generation.
			for (x = 0; x < glen; x++)
				new_array_g[0][x] = best_g[x];
			for (x = 0; x < hlen; x++)
				new_array_h[0][x] = best_h[x];

			// Perform tournament selection to create the rest of the new generation.
			for (M = 1; M < population_size; M++)
			{
				// Use a tournament of three individuals
				i = rand() % population_size;
				j = rand() % population_size;
				k = rand() % population_size;

				// identify the individual with the best (smallest) fitness value
				if (fitness[i] < fitness[j])
					if (fitness[i] < fitness[k])
						least = i;
					else
						least = k;
				else if (fitness[j] < fitness[k])
						least = j;
				else
						least = k;

				// copy the best individual from the tournament into the next generation
				for (i = 0; i < glen; i++)
					new_array_g[M][i] = array_g[least][i];
				for (i = 0; i < hlen; i++)
					new_array_h[M][i] = array_h[least][i];
			}

			if (crossing_over == true)
			{
				// Perform crossover on all but the first element of the next generation.
				for (M = 1; M < population_size; M++)
				{
					// select a random member k of the new population (k != 0)
					k = 0;
					while (k == 0)
						k = rand() % population_size;
	
					// select a crossover point between 0 and glen-1
					j = rand() % glen;
		
					// For the Mth and kth population members,
					// swap the g array values from positions j through glen-1 
					for (i = j; i < glen; i++)
					{
						swap = new_array_g[M][i];
						new_array_g[M][i] = new_array_g[k][i];
						new_array_g[k][i] = swap;
					}
	
					// select a crossover point between 0 and hlen-1
					j = rand() % hlen;

					// For the Mth and kth population members,
					// swap the h array values from positions j through hlen-1
					for (i = j; i < hlen; i++)
					{
						swap = new_array_h[M][i];
						new_array_h[M][i] = new_array_h[k][i];
						new_array_h[k][i] = swap;
					}
				}
			}

			if (new_best == false)
			{
				mutating = true;
				if (mutation_rate < 10)
					mutation_rate += 1;
			}
			else
			{
				mutating = false;
				mutation_rate = 0;
			}

			if (mutating == true)
				// Perform mutation on elements of the new generation.
				for (M = 1; M < population_size; M++)
				{
					for (i = 0; i < glen; i++)
						if (rand() % 100 < mutation_rate)
							new_array_g[M][i] = new_array_g[M][i] * GetMutationFactor();
					for (i = 0; i < hlen; i++)
						if (rand() % 100 < mutation_rate)
							new_array_h[M][i] = new_array_h[M][i] * GetMutationFactor();
				}

			// Replace the old generation with the new generation.
			for (M = 0; M < population_size; M++)
			{
				for (i = 0; i < glen; i++)
					array_g[M][i] = new_array_g[M][i];
				for (i = 0; i < hlen; i++)
					array_h[M][i] = new_array_h[M][i];
			}
		}

		// reload g and h values that produced "best ever fitness"
		for (x = 0; x < glen; x++)
			g[x] = best_g[x];
		for (x = 0; x < hlen; x++)
			h[x] = best_h[x];
	
		// recreate Y vectors from stored "best ever fitness" vectors
		for (j = 0; j < imageCount; j++)
			for (x = 0; x < width; x++)
				for (y = 0; y < height; y++)
					Y[j][x+y*width] = bestY[j][x+y*width];

		memcpy(h2,best_h,hlen*sizeof(double));
		memcpy(g2,best_g,glen*sizeof(double));
	}

	for(i=0;i<popsize;i++)
	{
		delete[] array_h[i];
		delete[] array_g[i];
		delete[] new_array_h[i];
		delete[] new_array_g[i];
	}

	delete[] array_h;
	delete[] array_g;
	delete[] new_array_h;
	delete[] new_array_g;
	delete[] fitness;
	delete[] temp;
	delete[] g;
	delete[] h;

	for (j = 0; j < imageCount; j++)
	{
		delete[] bestY[j];
		delete[] transformedY[j];
	}
	delete[] transformedParentY;
	delete[] temp2;
}


void CWaveletSettings::CWaveletInverseTransformGA (double *Y[], double *originalY[],
												  int width, int height, int imageCount)
{
	double *bestY[maxImageCount];

	//values are hard coded due to windows GUI programming madness

	popsize = 5000;
	maxgen = 2500;

	const int population_size = popsize;
	const int max_generations = maxgen;

	bool mutating = false;
	const bool crossing_over = true;
	bool new_best;
	int mutation_rate = 0; 
	int x, y, i, j, k, least;
	int G, M;
	double original_fitness;
	double best_h[40], best_g[40];
	double swap;
	double best_fitness;
	double **array_h, **array_g;
	double **new_array_h, **new_array_g;
	double *fitness;
	double *temp;
	int lengthw, lengthh;
	double *transformedY[maxImageCount];
	double *g, *h;

	// Seed the random-number generator with current time so that
	// the numbers will be different every time we run.

	srand( (unsigned)time( NULL ) );


	for (j = 0; j < imageCount; j++)
	{
		transformedY[j] = new double[width*height];
		for(i=0;i<width*height;i++)
			transformedY[j][i] = Y[j][i];
	}

	g = new double[glen];
	h = new double[hlen];
	array_h = new double*[popsize];
	array_g = new double*[popsize];
	new_array_h = new double*[popsize];
	new_array_g = new double*[popsize];
	for(i=0;i<popsize;i++)
	{
		array_h[i] = new double[hlen];
		array_g[i] = new double[glen];
		new_array_h[i] = new double[hlen];
		new_array_g[i] = new double[glen];
	}
	fitness = new double[popsize];

	for (j = 0; j < imageCount; j++)
		bestY[j] = new double[width*height];

	memcpy(h,h2,hlen*sizeof(double));
	memcpy(g,g2,glen*sizeof(double));

	temp = new double[height];
	
	for (j = 0; j < imageCount; j++)
	{
		for(k=mrlevels-1;k>=0;k--)
		{
			lengthw = width>>k;
			lengthh = height>>k;
			if(width%((int)pow(2,k)) != 0)
				lengthw++;
			if(height%((int)pow(2,k)) != 0)
				lengthh++;

			// vertical
			for(x=0;x<lengthw;x++)
			{
				for(y=0;y<lengthh;y++)
					temp[y] = Y[j][x+y*width];
				inverse(temp,lengthh,h,g,hlen,glen);
				for(y=0;y<lengthh;y++)
					Y[j][x+y*width] = temp[y];
			}
			// horizontal
			for(y=0;y<lengthh;y++)
				inverse(Y[j]+y*width,lengthw,h,g,hlen,glen);
		}
	}

	// Compute Error for restored image Y

	original_fitness = 0.0;
	for (j = 0; j < imageCount; j++)
		for (x = 0; x < width; x++)
				for (y = 0; y < height; y++)
					original_fitness += (Y[j][x+y*width] - originalY[j][x+y*width]) 
										* (Y[j][x+y*width] - originalY[j][x+y*width]);

	// Initialize "best seen so far" statistics using original h, g values
	best_fitness = original_fitness;
	for (x = 0; x < hlen; x++)
		best_h[x] = h[x];
	for (x = 0; x < glen; x++)
		best_g[x] = g[x];

	for (j = 0; j < imageCount; j++)
		for (x = 0; x < width; x++)
			for (y = 0; y < height; y++)
				bestY[j][x+y*width] = Y[j][x+y*width];

	if ((evolving == 1) && (originalY != NULL))
	{
		// Introduce small perturbations to the coefficients of array_h and array_g.
		// (NOTE: the original h, g vectors are merely copied into array position 0.)

		for (k = 0; k < population_size; k++)
			for (x = 0; x < hlen; x++)
				if (k == 0)
					array_h[k][x] = h[x];
				else
					array_h[k][x] = h[x] * GetRandomNoiseFactor();

		for (k = 0; k < population_size; k++)
			for (x = 0; x < glen; x++)
				if (k == 0)
					array_g[k][x] = g[x];
				else
					array_g[k][x] = g[x] * GetRandomNoiseFactor();

		// Run a genetic algorithm for max_generations.
		for (G = 0; G < max_generations; G++)
		{
			new_best = false;

			// Evaluate the fitness of each of the M wavelets by performing an inverse transform
			// and calculating the error in the rebuilt images (relative to the original images).

			for (M = 0; M < population_size; M++)
			{	
				// copy transformed Y vector to Y
				for (j = 0; j < imageCount; j++)
				{
					for (x = 0; x < width; x++)
						for (y = 0; y < height; y++)
							Y[j][x+y*width] = transformedY[j][x+y*width];

					for(k = mrlevels-1; k >= 0; k--)
					{
						lengthw = width>>k;
						lengthh = height>>k;
						if(width%((int)pow(2,k)) != 0)
							lengthw++;
						if(height%((int)pow(2,k)) != 0)
							lengthh++;
	
						// vertical
						for(x=0;x<lengthw;x++)
						{
							for(y=0;y<lengthh;y++)
								temp[y] = Y[j][x+y*width];
							inverse(temp,lengthh,array_h[M],array_g[M],hlen,glen);
							for(y=0;y<lengthh;y++)
								Y[j][x+y*width] = temp[y];
						}

						// horizontal
						for(y=0;y<lengthh;y++)
							inverse(Y[j]+y*width,lengthw,array_h[M],array_g[M],hlen,glen);
					}
				}

				// measure fitness by comparing the inverse-transformed Y to the originalY
				fitness[M] = 0.0;
				for (j = 0; j < imageCount; j++)
					for (x = 0; x < width; x++)
						for (y = 0; y < height; y++)
							fitness[M] += (Y[j][x+y*width] - originalY[j][x+y*width])
										  * (Y[j][x+y*width] - originalY[j][x+y*width]);

				// check for a new globally optimal fitness values
				if (fitness[M] < best_fitness)
				{
					// we have a new best-of-run individual
					new_best = true;

					// update "best ever fitness"
					best_fitness = fitness[M];

					// copy g and h values that produced "best ever fitness"
					for (x = 0; x < glen; x++)
						best_g[x] = array_g[M][x];
					for (x = 0; x < hlen; x++)
						best_h[x] = array_h[M][x];

					// save Y vector with "best ever fitness"
					for (j = 0; j < imageCount; j++)
						for (x = 0; x < width; x++)
							for (y = 0; y < height; y++)
								bestY[j][x+y*width] = Y[j][x+y*width];
				}
			}

			// Copy the best-of-generation individual into position 0 of the new generation.
			for (x = 0; x < glen; x++)
				new_array_g[0][x] = best_g[x];
			for (x = 0; x < hlen; x++)
				new_array_h[0][x] = best_h[x];


			// Perform tournament selection to create the rest of the new generation.
			for (M = 1; M < population_size; M++)
			{
				// Use a tournament of three individuals
				i = rand() % population_size;
				j = rand() % population_size;
				k = rand() % population_size;

				// identify the individual with the best (smallest) fitness value
				if (fitness[i] < fitness[j])
					if (fitness[i] < fitness[k])
						least = i;
					else
						least = k;
				else if (fitness[j] < fitness[k])
						least = j;
				else
						least = k;

				// copy the best individual from the tournament into the next generation
				for (i = 0; i < glen; i++)
					new_array_g[M][i] = array_g[least][i];
				for (i = 0; i < hlen; i++)
					new_array_h[M][i] = array_h[least][i];
			}

			if (crossing_over == true)
			{
				// Perform crossover on all but the first element of the next generation.
				for (M = 1; M < population_size; M++)
				{
					// select a random member k of the new population (k != 0)
					k = 0;
					while (k == 0)
						k = rand() % population_size;
	
					// select a crossover point between 0 and glen-1
					j = rand() % glen;
		
					// For the Mth and kth population members,
					// swap the g array values from positions j through glen-1 
					for (i = j; i < glen; i++)
					{
						swap = new_array_g[M][i];
						new_array_g[M][i] = new_array_g[k][i];
						new_array_g[k][i] = swap;
					}
	
					// select a crossover point between 0 and hlen-1
					j = rand() % hlen;

					// For the Mth and kth population members,
					// swap the h array values from positions j through hlen-1
					for (i = j; i < hlen; i++)
					{
						swap = new_array_h[M][i];
						new_array_h[M][i] = new_array_h[k][i];
						new_array_h[k][i] = swap;
					}
				}
			}

			if (new_best == false)
			{
				mutating = true;
				if (mutation_rate < 10)
					mutation_rate += 1;
			}
			else
			{
				mutating = false;
				mutation_rate = 0;
			}

			if (mutating == true)
				// Perform mutation on elements of the new generation.
				for (M = 1; M < population_size; M++)
				{
					for (i = 0; i < glen; i++)
						if (rand() % 100 < mutation_rate)
							new_array_g[M][i] = new_array_g[M][i] * GetMutationFactor();
					for (i = 0; i < hlen; i++)
						if (rand() % 100 < mutation_rate)
							new_array_h[M][i] = new_array_h[M][i] * GetMutationFactor();
				}

			// Replace the old generation with the new generation.
			for (M = 0; M < population_size; M++)
			{
				for (i = 0; i < glen; i++)
					array_g[M][i] = new_array_g[M][i];
				for (i = 0; i < hlen; i++)
					array_h[M][i] = new_array_h[M][i];
			}

			//fprintf(fh, "Checkpoint 3f\n");
		}

		// reload g and h values that produced "best ever fitness"
		for (x = 0; x < glen; x++)
			g[x] = best_g[x];
		for (x = 0; x < hlen; x++)
			h[x] = best_h[x];
	
		// recreate Y vectors from stored "best ever fitness" vectors
		for (j = 0; j < imageCount; j++)
			for (x = 0; x < width; x++)
				for (y = 0; y < height; y++)
					Y[j][x+y*width] = bestY[j][x+y*width];

		memcpy(h2,best_h,hlen*sizeof(double));
		memcpy(g2,best_g,glen*sizeof(double));
	}

	for(i=0;i<popsize;i++)
	{
		delete[] array_h[i];
		delete[] array_g[i];
		delete[] new_array_h[i];
		delete[] new_array_g[i];
	}

	delete[] array_h;
	delete[] array_g;
	delete[] new_array_h;
	delete[] new_array_g;
	delete[] fitness;
	delete[] temp;
	delete[] g;
	delete[] h;

	for (j = 0; j < imageCount; j++)
	{
		delete[] bestY[j];
		delete[] transformedY[j];
	}
}

BEGIN_MESSAGE_MAP(CWaveletSettings, CDialog)
	//{{AFX_MSG_MAP(CWaveletSettings)
		// NOTE: the ClassWizard will add message map macros here
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CWaveletSettings message handlers

void forward(double *X, int N, double *h, double *g, int hlen, int glen)
{
	int i, m, len;
	double *H, *G;
	int odd, index;
	double oddvalue;
	int hoffset, goffset;

	odd = N%2;
	if(odd)
	{
		len = N-1;
		oddvalue = X[N-1];
	}
	else 
		len = N;

	hoffset = (hlen>>1)-1;
	goffset = (glen-1)>>1;

	H = new double[len];
	G = new double[len];

	for(i=0;i<len;i++)
	{
		H[i] = 0;
		G[i] = 0;
		for(m=0;m<hlen;m++)
		{
			index = i+m-hoffset;
			if(index < 0)
				index = index + len;
			else if(index > len-1)
				index = index - len;
			H[i] = H[i]+h[m]*X[index];
		}
		for(m=0;m<glen;m++)
		{
			index = i+m-goffset;
			if(index < 0)
				index = index + len;
			else if(index > len-1)
				index = index - len;
			G[i] = G[i]+g[m]*X[index];
		}
	}
	for(i=0;i<len/2;i++)
	{
		X[i] = H[i*2];
		X[i+len/2+odd] = G[i*2];
	}
	if(odd) X[len/2] = oddvalue;

	delete[] H;
	delete[] G;
}

void inverse(double *X, int N, double *h, double *g, int hlen, int glen)
{
	int i, m, len;
	double *H, *G;
	double *upH, *upG;
	int odd, index;
	double oddvalue;
	int hoffset, goffset;

	odd = N%2;
	if(odd)
	{
		len = N-1;
		oddvalue = X[len/2];
	}
	else 
		len = N;

	hoffset = (hlen+1)>>1;
	goffset = glen>>1;

	H = new double[len];
	G = new double[len];
	upH = new double[len];
	upG = new double[len];

	// Upsample both high and low components;
	for(i=0;i<len;i+=2)
	{
		upH[i] = X[i/2];
		upG[i] = X[i/2+odd+len/2];
		upH[i+1] = 0;
		upG[i+1] = 0;
	}

	for(i=0;i<len;i++)
	{
		H[i] = 0;
		G[i] = 0;
		for(m=0;m<hlen;m++)
		{
			index = i+m-hoffset;
			if(index < 0)
				index = index + len;
			else if(index > len-1)
				index = index - len;
			H[i] = H[i]+h[m]*upH[index];
		}
		for(m=0;m<glen;m++)
		{
			index = i+m-goffset;
			if(index < 0)
				index = index + len;
			else if(index > len-1)
				index = index - len;
			G[i] = G[i]+g[m]*upG[index];
		}
	}
	for(i=0;i<len;i++)
	{
		X[i] = H[i]+G[i];
	}
	if(odd) X[N-1] = oddvalue;

	delete[] H;
	delete[] G;
	delete[] upH;
	delete[] upG;
}

double GetMutationFactor()
{
	double noise;
	noise = rand() % 100;
	if (noise < 30)
		return 0.99;
	else if (noise < 60)
		return 1.01;
	else if (noise < 75)
		return 0.98;
	else if (noise < 90)
		return 1.02;
	else if (noise < 93)
		return 0.95;
	else if (noise < 96)
		return 1.05;
	else if (noise < 97)
		return 0.90;
	else if (noise < 98)
		return 1.10;
	else
		return -1.0;
}

double GetRandomNoiseFactor()
{
	double noise;
	noise = rand() % 100;
	if (noise < 35)
		return 0.99;
	else if (noise < 70)
		return 1.01;
	else if (noise < 80)
		return 0.97;
	else if (noise < 90)
		return 1.03;
	else if (noise < 94)
		return 0.95;
	else if (noise < 98)
		return 1.05;
	else
		return -1.0;
}

void CWaveletSettings::CQuantizeCoefficients(double *Y, int width, int height, int stepsize, int threshold)
{
	int i;

	for(i=0;i<width*height;i++)
	{
		if((Y[i] < threshold)&&(Y[i] > 0))
			Y[i] = 0;
		else if((-Y[i] < threshold)&&(Y[i] < 0))
			Y[i] = 0;
		else if(stepsize > 0)
			Y[i] = (double)((int)(Y[i] / stepsize));
		else
			Y[i] = (double)((int)(Y[i]));
	}
}

void CWaveletSettings::CDequantizeCoefficients(double *Y, int width, int height, int stepsize)
{
	int i;

	for(i=0;i<width*height;i++)
	{
		if(stepsize > 0)
			Y[i] = Y[i] * stepsize;
		if(Y[i] > 0)
			Y[i] += (double)stepsize/2;
		if(Y[i] < 0)
			Y[i] -= (double)stepsize/2;
	}
}

//initializes g, h to pre-existing wavelet types

void CWaveletSettings::CInitializeGH()
{
	delete[] g1;
	delete[] h1;
	delete[] g2;
	delete[] h2;

	if(m_waveletnumber == 0)
	{
		hlen = daub4hlen;
		glen = daub4glen;
		g1 = new double[glen];
		g2 = new double[glen];
		h1 = new double[hlen];
		h2 = new double[hlen];
		memcpy(h1,daub4h1,daub4hlen*sizeof(double));
		memcpy(g1,daub4g1,daub4glen*sizeof(double));
		memcpy(h2,daub4h2,daub4hlen*sizeof(double));
		memcpy(g2,daub4g2,daub4glen*sizeof(double));
	}
	else if(m_waveletnumber == 1)
	{
		hlen = haarhlen;
		glen = haarglen;
		g1 = new double[glen];
		g2 = new double[glen];
		h1 = new double[hlen];
		h2 = new double[hlen];
		memcpy(h1,haarh1,haarhlen*sizeof(double));
		memcpy(g1,haarg1,haarglen*sizeof(double));
		memcpy(h2,haarh2,haarhlen*sizeof(double));
		memcpy(g2,haarg2,haarglen*sizeof(double));
	}
	else if(m_waveletnumber == 2)
	{
		hlen = tshlen;
		glen = tsglen;
		g1 = new double[glen];
		g2 = new double[glen];
		h1 = new double[hlen];
		h2 = new double[hlen];
		memcpy(h1,tsh1,tshlen*sizeof(double));
		memcpy(g1,tsg1,tsglen*sizeof(double));
		memcpy(h2,tsh2,tshlen*sizeof(double));
		memcpy(g2,tsg2,tsglen*sizeof(double));
	}
	else if(m_waveletnumber == 3)
	{
		hlen = fthlen;
		glen = ftglen;	
		g1 = new double[glen];
		g2 = new double[glen];
		h1 = new double[hlen];
		h2 = new double[hlen];
		memcpy(h1,fth1,fthlen*sizeof(double));
		memcpy(g1,ftg1,ftglen*sizeof(double));
		memcpy(h2,fth2,fthlen*sizeof(double));
		memcpy(g2,ftg2,ftglen*sizeof(double));
	}
	else if(m_waveletnumber == 4)
	{
		hlen = daub6hlen;
		glen = daub6glen;
		g1 = new double[glen];
		g2 = new double[glen];
		h1 = new double[hlen];
		h2 = new double[hlen];
		memcpy(h1,daub6h1,daub6hlen*sizeof(double));
		memcpy(g1,daub6g1,daub6glen*sizeof(double));
		memcpy(h2,daub6h2,daub6hlen*sizeof(double));
		memcpy(g2,daub6g2,daub6glen*sizeof(double));
	}

	else if(m_waveletnumber == 5)
	{
		hlen = daub8hlen;
		glen = daub8glen;
		g1 = new double[glen];
		g2 = new double[glen];
		h1 = new double[hlen];
		h2 = new double[hlen];
		memcpy(h1,daub8h1,daub8hlen*sizeof(double));
		memcpy(g1,daub8g1,daub8glen*sizeof(double));
		memcpy(h2,daub8h2,daub8hlen*sizeof(double));
		memcpy(g2,daub8g2,daub8glen*sizeof(double));
	}
}

//Don's code, modified a smidge (to reflect change to 'permanent' coefficients - ie to use h2, g2):
//Inverse transform for a single image
void CWaveletSettings::CWaveletInverseTransformTarget (double *Y[], int width, int height, int imageCount)
{
	int x, y, k, j;
	double *temp;
	int lengthw, lengthh;

	temp = new double[height];
	mrlevels = m_mrlevel;

	j = imageCount - 1;
	for(k=mrlevels-1;k>=0;k--)
	{
		lengthw = width>>k;
		lengthh = height>>k;
		if(width%((int)pow(2,k)) != 0)
			lengthw++;
		if(height%((int)pow(2,k)) != 0)
			lengthh++;

		// vertical
		for(x=0;x<lengthw;x++)
		{
			for(y=0;y<lengthh;y++)
				temp[y] = Y[j][x+y*width];
			inverse(temp,lengthh,h2,g2,hlen,glen);
			for(y=0;y<lengthh;y++)
				Y[j][x+y*width] = temp[y];
		}
		// horizontal
		for(y=0;y<lengthh;y++)
			inverse(Y[j]+y*width,lengthw,h2,g2,hlen,glen);
	}

	delete[] temp;
}
