/*
Cell class
This class contains the agents of our MultiCellular simulation. They perform actions based on
rules that depend on the cell's current state and/or the state or position of the cell's
neighbors.
This class was written by Nick Armstrong and modified by Heather Koyuk.
Spring 2005
*/

package client;

import vtk.*;
import java.util.*;
import vtk.vtkObject.*;
import java.lang.Math.*;

public class Cell {

	public static final float RADIUS = .5f;
	// HK moved XYZTree to neighbors class
	//private static  XYZTree theTree;
	private Neighbors neighbors;
	// HK moved child vector to be local to cell class (rather than in Visualization)
	private Vector cellChildren = new Vector();
	private Point3D center;
	private State state;
	private vtkActor vtkSphere;
	// HK for different sized radii
	private float radius;
	// HK for removal from the XYZTree
	private boolean dead = false;

	// HK make private later
	// Array to hold links to x,y, and z left and right children for XYZTree
	// this is XYZ children; name should be changed so as not to confuse with the cell's budding children
	public Cell[] children = new Cell[6];
	public int[] minmax = new int[6];
	public Cell next = null;

	// (a variety of) constructors
	// HK added constructors allowing different radii to be specified, along with the default
	// HK cleaned up constructor code
	public Cell(float x, float y, float z) {this.init(new Point3D(x,y,z), new State(), RADIUS);}
	public Cell(float x, float y, float z, float radius) {this.init(new Point3D(x,y,z), new State(), radius);}
	public Cell(Point3D center) {this.init(center, new State(), RADIUS);}
	public Cell(Point3D center, float radius) {this.init(center, new State(), radius);}
	public Cell(float x, float y, float z, State state) {this.init(new Point3D(x,y,z), state, RADIUS);}
	public Cell(float x, float y, float z, State state, float radius) {this.init(new Point3D(x,y,z), state, radius);}
	public Cell(Point3D center, State state) {this.init(center, state, RADIUS);}
	public Cell(Point3D center, State state, float radius) {this.init(center, state, radius);}

	// HK initialize the initial cell values (moved from constructors)
	private void init(Point3D center, State state, float radius){
		this.center = center;
		this.state = state;
		this.radius = radius;
		this.vtkSphere = makeSphere();
		this.compute();
		neighbors = new Neighbors(this);
	}
	private vtkActor makeSphere() {

		vtkSphereSource sphere = new vtkSphereSource();
		sphere.SetRadius(radius);
		sphere.SetCenter(center.x(), center.y(), center.z());
		sphere.SetThetaResolution(18);
		sphere.SetPhiResolution(18);

		// map to graphics objects
		vtkPolyDataMapper map = new vtkPolyDataMapper();
		map.SetInput(sphere.GetOutput());

		// actor coordinates geometry, properties, transformation
		vtkActor aSphere = new vtkActor();
		aSphere.SetMapper(map);
		aSphere.GetProperty().SetColor(0,0,1); // color blue
		return aSphere;

	}
	// HK actions will be called from here; the Visualization class will then check for
	// children, whether or not the cell is dead, etc.
	public void doAction(){
		Action action = Rules.getAction(this);
		perform(action);
	}
	// HK moved the following code from Visualization to here, then moved again
	/*	growChildren();
	//begin cell division code
 		destroy(); // remove from tree first
 		divide();
	// end cell division code
	}
	*/
	// HK cell performs the specified action
	public void perform(Action a){
		if(a instanceof Bud){
			Bud b = (Bud)a;
			for(int i = 0; i<b.children.size(); i++){
				ChildBud gb = (ChildBud)b.children.elementAt(i);
				GrowthVector tau = new GrowthVector(gb.direction, gb.magnitude);
				Point3D childcenter = Rules.calculateChildCenter(this.center, tau);
				if(childcenter!=null){
					Cell child = new Cell(childcenter, new State(tau));
					cellChildren.addElement(child);
				}
			}
		}
		else if(a instanceof Divide) growChildren();
	}
	// methodology by Armstrong, modified somewhat by HK
	// divides into exactly two cells, each of half the radius, occupying the space the parent used to occupy (each is scooched over +/- one fourth of a radius)
	public void divide() {
		this.destroy();
		//Vector children = new Vector();
		Cell child = null;
		State newChildState = null;
		Point3D newChildCenter = null;
		GrowthVector tau = null;
		for(int i = 0; i < state.getGrowthVectors().size(); i++) { // actually, cell has only one growth vector, indicating one of the two opposite directions in which the cell will divide
			tau = (GrowthVector) state.getGrowthVector(i);
			newChildState = Rules.calculateDividedChildState(state, tau);
			newChildCenter = Rules.calculateChildCenter(center, tau, radius/4.0f); // half of child's radius, which is 1/2 this radius: 1/2*1/2=1/4
			if(newChildState != null && newChildCenter != null) cellChildren.add(new Cell(newChildCenter, newChildState, radius/2.0f));
			tau = tau.invert(); // make other cell in opposite direction
			newChildState = Rules.calculateDividedChildState(state, tau);
			newChildCenter = Rules.calculateChildCenter(center, tau, radius/4.0f);
			if(newChildState != null && newChildCenter != null) cellChildren.add(new Cell(newChildCenter, newChildState, radius/2.0f));
		}
		//return children;
	}
	public void growChildren() {
		for(int i = 0; i < state.size(); i++) {
			// HK put all logic for creating child cell into Rules class
			Cell child = Rules.growChild(this, state.getGrowthVector(i));
			if(child != null) {
				// HK children are now local to cells
				cellChildren.add(child);
				child.insert();
			}
		}
	}

	public String toString() { return center.toString() + " " + state.toString();}
	public String centerToString(){ return center.toString(); }

	public vtkActor getActor() {return vtkSphere;}
	public Point3D getCenter() {return center;}
	public State getState() {return state;}
	public float x() {return center.x();}
	public float y() {return center.y();}
	public float z() {return center.z();}
	public float radius() {return radius;}
	// HK
	public Vector getChildren(){ return cellChildren; }
	// HK
	public boolean isDead(){ return dead;}
	// HK added magnitude variable to state class
	public float magnitude(){ return state.magnitude(); }
	// HK
	public void setMagnitude(float f){ state.setMagnitude(f); }

	// HK Computes min and max X, Y, and Z ranges for insertion into XYZTree
	private void compute(){
		for(int i = 0; i<6; i++) children[i] = null;
		minmax[0] = (int)(Math.floor(this.x()-this.radius));
		minmax[1] = (int)(Math.ceil(this.x()+this.radius));
		minmax[2] = (int)(Math.floor(this.y()-this.radius));
		minmax[3] = (int)(Math.ceil(this.y()+this.radius));
		minmax[4] = (int)(Math.floor(this.z()-this.radius));
		minmax[5] = (int)(Math.ceil(this.z()+this.radius));
	}
	// HK prints the X, Y, and Z ranges that the cell occupies (was used for debugging)
	public String printRange(){
		String s = "";
		for(int i = 0; i<6; i++) s+=("{"+minmax[i]+","+minmax[++i]+"}");
		return s;
	}
	// HK Moved all location-specific stuff (including XYZTree) except center to Neighbors class
	// HK: all neighbor and XYZTree functions
	public void insert(){ neighbors.insert(); }
	public void addNeighbor(Cell cell){ neighbors.addNeighbor(cell); }
	public void deleteNeighbor(Cell cell){ neighbors.deleteNeighbor(cell);}
	public float centerDistance(Cell neighbor){ return neighbors.centerDistance(neighbor);}
	public float centerDistance(float ex, float ey, float ez){ return neighbors.centerDistance(ex, ey, ez);}
	public float distance(Cell neighbor){ return neighbors.centerDistance(neighbor); }
	public float distance(Point3D centerr, float rad){return neighbors.distance(centerr, rad);}
	public boolean overlaps(Cell neighbor){ return neighbors.overlaps(neighbor);}
	public boolean overlaps(Point3D centerr, float rad){ return neighbors.overlaps(centerr, rad);}
	public void makeTree(){ neighbors.makeTree(this); }
	public Cell closest(){ return neighbors.closest(); }
	static public void clear(){ Neighbors.clear(); }
	public void destroy(){
		this.dead = true;
		neighbors.destroy();
	}
}