/*
Visualization class
Nick Armstrong
MultiCellular project
Spring 2005

This class is the main window for the simulation.
*/

package client;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.io.File;
import vtk.*;

public class Visualization {

	private vtkWindowToImageFilter imageFilter;
	private vtkJPEGWriter jpegWriter;
	private vtkRenderWindow renderWindow;
	private vtkRenderer renderer;
	private VisualizationPanel visPanel;
	private String startState;
	private Vector cells;
	private Vector activeCells;
	private Vector deadCells;
	// HK user-defined rules window
	private RuleWindow theRules = new RuleWindow();


	/** Load all the necessary VTK libraries **/
	static {
		System.loadLibrary("vtkCommonJava");
		System.loadLibrary("vtkFilteringJava");
		System.loadLibrary("vtkIOJava");
		System.loadLibrary("vtkImagingJava");
		System.loadLibrary("vtkGraphicsJava");
		System.loadLibrary("vtkRenderingJava");
	}

	/**
	 *      Constructor for Visualization object.
	 *
	 * @param      vPanel    VisualizationPanel the Visualization will be placed in.
	 */
	public Visualization(VisualizationPanel vPanel) {

		visPanel = vPanel;
		renderWindow = visPanel.GetRenderWindow();
		renderer = visPanel.GetRenderer();
		activeCells = new Vector();
		deadCells = new Vector();
		cells = new Vector();

		init();

		/** Capture the startState to be used for resetting **/
		startState = visPanel.captureCurrentState();
		changeBackgroundColor(0,0,0);

	}

	private void init() {
		Vector seedCells = Rules.seedCells();
		cells = seedCells;
		activeCells.addAll(seedCells);
		Cell seedCell = null;
		for(int i = 0; i < seedCells.size(); i++) {
			seedCell = (Cell) seedCells.get(i);
			renderer.AddActor(seedCell.getActor());
		}
		// HK starts tree with first seed cell as the root
		((Cell) (seedCells.firstElement())).makeTree();
	}

	public void timestep() {
		synchronized(activeCells) {
			Vector newActiveCells = new Vector();
			Vector children = null;
			Cell cell = null, childCell = null;
			while(!activeCells.isEmpty()) {	// foreach active cell
				cell = (Cell) activeCells.get(0);	// get the first cell in the list
					// HK moved actions to Cell class
				cell.doAction();
					// HK get new children
				children = cell.getChildren();
				cells.addAll(children);
					// remove current cell from visualization if it is dead
				if(cell.isDead()){
					renderer.RemoveActor(cell.getActor()); // remove actor from visualization
				}
				newActiveCells.addAll(children);

				for(int i = 0; i < children.size(); i++) {
					//float rad = 1.0F;
					Cell current = ((Cell)children.get(i));
					renderer.AddActor(((Cell)children.get(i)).getActor());
				}
				deadCells.add(cell);		// after a cell spawns (in each possible direction), it dies
				activeCells.remove(0);

			}

			activeCells = newActiveCells;

		}
	}

	public void repaintPanel() {
		synchronized(cells) {
			visPanel.resetCameraToRender();
//			visPanel.Render();
			visPanel.repaint();
		}
	}

	/**
	 *      Gets the startState of the Visualization instance
	 *
	 * @return        The startState of the Visualization
	 */
	public String getStartState() {
		return startState;
	}

	/**
	 *		Changes the background color of the Visualization with RGB values.
	 *
	 * @param	R	Red value of color, between 0 and 1
	 * @param	G	Green value of color, between 0 and 1
	 * @param	B	Blue value of color, between 0 and 1
	 */
	public void changeBackgroundColor( double R, double G, double B) {
		/**
		*	requires: R, G, and B to be valid double values between 0.0 and 1.0 inclusive.
		*	effects: changes the background color of the visualization given valid R, G, and B values. Repaints the visualization.
		*/

		/** Make sure RGB values are valid, if not, set to black. */
		if( !((R >= 0 && R <= 1) && (G >= 0 && G <= 1) && (B >= 0 && B <= 1)) ) {
			R = 0.0;
			G = 0.0;
			B = 0.0;
		}
		renderer.SetBackground(R, G, B);
		visPanel.repaint();
	}

	/**
	 * 		Takes a shap shot of the current Visualization, saves it as a .jpg file.
	 *
	 * @return		Returns path and filename of .jpg image created.
	 */
	public String takeSnapShot() {
		/**
		*	requires: filename is a valid path to the file to save to. sendToWhiteboard must be true if the image is being sent
		*			  to the whiteboard, false otherwise. A visualization must be currently loaded.
		*	effects: Creates a JPEG image of the visualization that is sent to the white board if sendToWhiteboard is true.
		*			 Otherwise, only saved on the hard drive.
		*	modifies: The image will occupy hard drive space. Starts and stops the CVEIDesktop progress bar.
		*	raises: An error could occur while writing the image to the hard drive.
		*/

		final String filename;
		final JFileChooser fc = new JFileChooser("../.."); // start looking one directory above the base
		int result = fc.showSaveDialog(visPanel);
		if(result == JFileChooser.APPROVE_OPTION) filename = fc.getSelectedFile().getAbsolutePath();
		else return "";

		/** Create a thread to take the snapshot - must play nicely with the Swing thread **/
		Runnable snapShot = new Runnable() {
		     public void run() {
		        imageFilter = new vtkWindowToImageFilter();
				jpegWriter = new vtkJPEGWriter();

				imageFilter.SetInput(renderWindow);
				jpegWriter.SetInput(imageFilter.GetOutput());

				jpegWriter.SetFileName(filename);
				jpegWriter.Write();

		     }
		 };

		 /** Asks the Swing thread to execute our thread when it feels like it.
		 	 This will really be invoked almost immediately. **/
		 SwingUtilities.invokeLater(snapShot);
		 return filename;

	}

	public void resetToStartState() {

		double[] cameraPosition = new double[3];
		double[] cameraFocalPoint = new double[3];
		double[] lightPosition = new double[3];
		double[] viewUp = new double[3];
		double cameraDistance = 0;
		double[] clippingRange = new double[2];

		StringTokenizer commandLine = new StringTokenizer(startState);
		String token;

		try {

			/** Get the camera position from the command string **/
			for (int i = 0; i < cameraPosition.length; i++) {
				token = commandLine.nextToken();
				cameraPosition[i] = Double.valueOf(token).doubleValue();
			}
			/** Get the camera focal point from the command string **/
			for (int i = 0; i < cameraFocalPoint.length; i++) {
				token = commandLine.nextToken();
				cameraFocalPoint[i] = Double.valueOf(token).doubleValue();
			}
			/** Get the light position from the command string **/
			for (int i = 0; i < lightPosition.length; i++) {
				token = commandLine.nextToken();
				lightPosition[i] = Double.valueOf(token).doubleValue();
			}

			/** Get the camera distance from the command string **/
			token = commandLine.nextToken();
			cameraDistance = Double.valueOf(token).doubleValue();

			/** Get the view up from the command string **/
			for (int i = 0; i < viewUp.length; i++) {
				token = commandLine.nextToken();
				viewUp[i] = Double.valueOf(token).doubleValue();
			}
			/** Get the clipping range from the command string **/
			for (int i = 0; i < clippingRange.length; i++) {
				token = commandLine.nextToken();
				clippingRange[i] = Double.valueOf(token).doubleValue();
			}
		}
		/** If a NoSuchElementException occurs from the tokenization of the commandLine, this exception is caught below. **/
		catch (NoSuchElementException e) {
			System.out.println("Invalid saved startState.");
		}

		/** If a NumberFormatException occurs from the conversion of a string to a double, this exception is caught below. **/
		catch (NumberFormatException e) {
			System.out.println("Invalid saved startState.");
		}
		// HK cleanup
		Cell.clear();
		cells.clear();
		activeCells.clear();
		deadCells.clear();
		visPanel.clearPanel();

		init();

		vtkCamera camera = visPanel.getCamera();
		vtkLight light = visPanel.getLight();

		/** Set all the values just captured **/
		camera.SetPosition(cameraPosition);
		camera.SetFocalPoint(cameraFocalPoint);
		light.SetPosition(lightPosition);
		camera.SetDistance(cameraDistance);
		camera.SetViewUp(viewUp);
		camera.SetClippingRange(clippingRange);

		/** Panel must be rendered and repainted before changes will be shown **/
		visPanel.Render();
		visPanel.repaint();

	}
	// HK retrieves and sets user-defined rules
  	public void setRules(){
		theRules.show();
	}

}
