/*
VisualizationFrame class
Nick Armstrong
MultiCellular project
Spring 2005

This class implements the content for the user interface.
Modified by Heather Koyuk to include a Rules button and associated functionality
*/

package client;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.io.File;
import vtk.*;

/**
 *      VisualizationFrame object that will hold the Visualization
 *		and control buttons.
 *
 * @author         SVA Group
 * @created        February 13, 2003
 */
public class VisualizationFrame extends JFrame {

	public static final int TIMESTEPS = 50;

	private JPanel moviePanel, contentPanel, buttonPanel, topButtonPanel,
					viewButtonPanel, interactionButtonPanel, colorButtonPanel;
	private JButton snapShotButton, resetButton, playButton, stepForwardButton, stopButton, fastForwardButton, pauseButton;
	// HK rules button
	private JButton setRulesButton;
	private JRadioButton zoomButton, rotateButton, translateButton,
				   	wireframeButton, surfaceButton, blackButton, whiteButton;
	private ButtonGroup interactionGroup, viewGroup, colorGroup;
	private VisualizationPanel visPanel;
	private Visualization visualization;
	private vtkRadioListener vtkListener;
	private Thread movieThread;
	private volatile boolean runMovie;
	private int currentTimestep;

	/** Load all the necessary VTK libraries **/
	static {
		System.loadLibrary("vtkCommonJava");
		System.loadLibrary("vtkFilteringJava");
		System.loadLibrary("vtkIOJava");
		System.loadLibrary("vtkImagingJava");
		System.loadLibrary("vtkGraphicsJava");
		System.loadLibrary("vtkRenderingJava");
	}

	/**
	 *      Constructor for VisualizationFrame object.
	 *
	 * @param      name       String for the title bar.
	 * @param      resize     Whether the frame can be resized.
	 * @param      close      Whether the frame can be closed.
	 * @param      maximize   Whether the frame can be maximized.
	 * @param      minimize   Whether the frame can be minimized.
	 * @param      width      Initial width of the frame.
	 * @param      height     Initial height of the frame.
	 * @param      xPosition  Initial x position of the frame.
	 * @param      yPosition  Initial y position of the frame.
	 * @param      visible    Whether the frame is visible.
	 */
	public VisualizationFrame(String name, boolean resize,
			boolean close, boolean maximize, boolean minimize,
			int width, int height, int xPosition, int yPosition,
			boolean visible) {

		super(name);
		super.setSize(width, height);
		super.setLocation(xPosition, yPosition);
		super.setVisible(visible);

		visPanel = new VisualizationPanel();

		contentPanel = new JPanel(new BorderLayout());
		buttonPanel = new JPanel();
//		buttonPanel.setLayout(new GridLayout(4, 1, 3, 3));
		buttonPanel.setLayout(new GridLayout(3, 1, 3, 3));
		buttonPanel.setBorder(BorderFactory.createEtchedBorder());

		vtkListener = new vtkRadioListener();

		makeMoviePanel();
//		makeTopButtonPanel();
		makeInteractionButtonPanel();
		makeViewButtonPanel();
		makeColorButtonPanel();

//		buttonPanel.add(topButtonPanel);
		buttonPanel.add(interactionButtonPanel);
		buttonPanel.add(viewButtonPanel);
		buttonPanel.add(colorButtonPanel);
		contentPanel.add(moviePanel, BorderLayout.NORTH);
		contentPanel.add(visPanel, BorderLayout.CENTER);
		contentPanel.add(buttonPanel, BorderLayout.EAST);
		getContentPane().add(contentPanel);
		show();

	}

	/**
	 *      Begins a Visualization using the data file given.
	 *
	 * @param	    filename  Data file (extension .3ds) to load in Visualization.
	 * @return		Returns 1 if loading of data file was successful, -1 otherwise.
	 */
	public int makeVisualization() {

		visPanel.clearPanel();
		visPanel.InteractionModeRotate();
		visPanel.viewAsSurface();
		rotateButton.setSelected(true);
		surfaceButton.setSelected(true);
		blackButton.setSelected(true);

		visualization = new Visualization(visPanel);
		visualization.changeBackgroundColor(0.0, 0.0, 0.0);

		snapShotButton.setEnabled(true);
		resetButton.setEnabled(true);

		return 1;

	}

	private void makeMoviePanel() {

		moviePanel = new JPanel(new GridLayout(1, 7, 3, 3));


		runMovie = false;
		currentTimestep = 0;

		playButton = new JButton(new ImageIcon("client/images/Play24.gif"));
		playButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runMovie = true;
					movieThread = 	new Thread() {
										public void run() {
											for(int i = currentTimestep; i < TIMESTEPS; i++) {
												if(runMovie == true) { // the new way to do threading, since Thread.stop() is deprecated
													visualization.timestep();
													visualization.repaintPanel();
													currentTimestep++;
												} else break;
												try {sleep(500);}
												catch(InterruptedException e) {e.printStackTrace();}
											}
										}
									};
					movieThread.start();
				}
			});
		moviePanel.add(playButton);

		stepForwardButton = new JButton(new ImageIcon("client/images/StepForward24.gif"));
		stepForwardButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runMovie = false;
					visualization.timestep();
					visualization.repaintPanel();
					currentTimestep++;
				}
			});
		moviePanel.add(stepForwardButton);

		fastForwardButton = new JButton(new ImageIcon("client/images/FastForward24.gif"));
//		moviePanel.add(fastForwardButton);

		pauseButton = new JButton(new ImageIcon("client/images/Pause24.gif"));
		pauseButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runMovie = false;
				}
			});
		moviePanel.add(pauseButton);

		stopButton = new JButton(new ImageIcon("client/images/Stop24.gif"));
		stopButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runMovie = false;
					visualization.resetToStartState();
					currentTimestep = 0;
				}
			});
		moviePanel.add(stopButton);

		resetButton = new JButton(new ImageIcon("client/images/Reset24.gif"));
		resetButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runMovie = false;
					visualization.resetToStartState();
				}
			});
		moviePanel.add(resetButton);

		snapShotButton = new JButton(new ImageIcon("client/images/Camera40.gif"));
		snapShotButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runMovie = false;
					visualization.takeSnapShot();
				}
			});
		moviePanel.add(snapShotButton);

		// HK rules button
		setRulesButton = new JButton("Rules");
		setRulesButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runMovie = false;
					visualization.setRules();
				}
			});
		moviePanel.add(setRulesButton);

	}

	/** Setup button panel for interaction radio buttons. **/
	private void makeInteractionButtonPanel() {
		rotateButton = new JRadioButton("Rotate");
		rotateButton.setActionCommand("rotate");

		/** The extra spaces make everything line up nicely. **/
		translateButton = new JRadioButton("Translate  ");
		translateButton.setActionCommand("translate");

		zoomButton = new JRadioButton("Zoom");
		zoomButton.setActionCommand("zoom");

		rotateButton.addActionListener(vtkListener);
		translateButton.addActionListener(vtkListener);
		zoomButton.addActionListener(vtkListener);

		rotateButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if(e.getSource() != rotateButton) return;
					visPanel.InteractionModeRotate();
				}
			});

		translateButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if(e.getSource() != translateButton) return;
					visPanel.InteractionModeTranslate();
				}
			});

		zoomButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if(e.getSource() != zoomButton) return;
					visPanel.InteractionModeZoom();
				}
			});

		interactionGroup = new ButtonGroup();
		interactionGroup.add(rotateButton);
		interactionGroup.add(translateButton);
		interactionGroup.add(zoomButton);
		rotateButton.setSelected(true);

		interactionButtonPanel = new JPanel();
		interactionButtonPanel.setLayout(new GridLayout(3, 1, 3, 3));
		interactionButtonPanel.add(rotateButton);
		interactionButtonPanel.add(translateButton);
		interactionButtonPanel.add(zoomButton);
		interactionButtonPanel.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createTitledBorder("Interaction"),
				BorderFactory.createEmptyBorder(0, 0, 0, 0)));
	}

	/** Setup button panel for view radio buttons. **/
	private void makeViewButtonPanel() {
		wireframeButton = new JRadioButton("Wireframe");
		wireframeButton.setActionCommand("wireframe");

		surfaceButton = new JRadioButton("Surface");
		surfaceButton.setActionCommand("surface");

		wireframeButton.addActionListener(vtkListener);
		surfaceButton.addActionListener(vtkListener);

		viewGroup = new ButtonGroup();
		viewGroup.add(surfaceButton);
		viewGroup.add(wireframeButton);
		surfaceButton.setSelected(true);

		viewButtonPanel = new JPanel();
		viewButtonPanel.setLayout(new GridLayout(2, 1, 3, 3));
		viewButtonPanel.add(surfaceButton);
		viewButtonPanel.add(wireframeButton);
		viewButtonPanel.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createTitledBorder("View"),
				BorderFactory.createEmptyBorder(0, 0, 0, 0)));
	}

	/** Setup button panel for color radio buttons. **/
	private void makeColorButtonPanel() {
		/** The extra spaces make everything line up nicely. **/
		blackButton = new JRadioButton("Black          ");
		blackButton.setActionCommand("black");

		whiteButton = new JRadioButton("White");
		whiteButton.setActionCommand("white");

		blackButton.addActionListener(vtkListener);
		whiteButton.addActionListener(vtkListener);

		colorGroup = new ButtonGroup();
		colorGroup.add(blackButton);
		colorGroup.add(whiteButton);
		blackButton.setSelected(true);

		colorButtonPanel = new JPanel();
		colorButtonPanel.setLayout(new GridLayout(2, 1, 3, 3));
		colorButtonPanel.add(blackButton);
		colorButtonPanel.add(whiteButton);
		colorButtonPanel.setBorder(BorderFactory.createCompoundBorder(
						BorderFactory.createTitledBorder("Background"),
						BorderFactory.createEmptyBorder(0, 0, 0, 0)));
	}

	private class vtkRadioListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			if(e.getSource() == rotateButton) visPanel.InteractionModeRotate();
			else if(e.getSource() == translateButton) visPanel.InteractionModeTranslate();
			else if(e.getSource() == zoomButton) visPanel.InteractionModeZoom();
			else if(e.getSource() == surfaceButton) visPanel.viewAsSurface();
			else if(e.getSource() == wireframeButton) visPanel.viewAsWireframe();
			else if(e.getSource() == blackButton) visualization.changeBackgroundColor(0.0, 0.0, 0.0);
			else if(e.getSource() == whiteButton) visualization.changeBackgroundColor(1.0, 1.0, 1.0);
//			visPanel.requestFocusInWindow();
		}
	}

	public static void main(String args[]) {
		VisualizationFrame app = new VisualizationFrame("visframe", true, true, true, true, 800, 800, 0, 0, true);
		app.makeVisualization();
		app.addWindowListener(
			new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					System.exit(0);
				}
			}
		);
	}

}