//Base code provided by Beginning 3D Game Programming by Tom Miller

using System;
using System.Drawing;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace SimonBlocks
{
    /// <summary>
    /// Color possibilities of the blocks
    /// </summary>
    public enum BlockColor : byte
    {
        Red = 0,
        Blue = 1,
        Green = 2,
        Yellow = 3,
        SandyBrown = 4,
        WhiteSmoke = 5,
        DarkGoldenRod = 6,
        CornflowerBlue = 7,
		DarkViolet = 8
    }

    public class Block
    {
        public const float MaximumVelocity = 75.0f;
        private const float MaxBoxScaleSpeed = 15.0f;
        private const float MaxBoxScale = 0.01f;
        private const float BoxSize = 3.0f; // World size of box

        private static readonly Color[] BoxColors = {Color.Red , Color.Blue, Color.Green, 
                                Color.Yellow, Color.SandyBrown, Color.WhiteSmoke,
                                Color.DarkGoldenrod, Color.CornflowerBlue, Color.Pink};

        // List of each possible material object for the box
        private static Material[] boxMaterials;
        private static Mesh boxMesh = null;

        private Vector3 pos; // Current position of the box in the level
        private Material material; // Current material color of the box

        // Possible colors of the blocks
        private BlockColor[] colors;
		private int blockType =0;
        private int colorIndex;
        private bool canWrap;
        private float time = 0.0f;
        private bool shouldPulse = false;
        private Vector3 velocity;

        public Block(Device dev, BlockColor[] list, int index, bool wrap, int inType)
        {
            if ((list == null) || (list.Length == 0))
                throw new ArgumentNullException("list", "You must pass in a valid list");

			blockType = inType;

            // Initialize the static data if needed
            if (boxMesh == null)
            {
                InitializeStaticData(dev);
            }

            // Set the default position
            pos = new Vector3(0,-1.0f, 0);

            // Store the information
            colors = list;
            colorIndex = index;
            canWrap = wrap;

            // Set the default material
            SetBlockColor(colors[colorIndex]);
        }

        private void InitializeStaticData(Device dev)
        {
             //Create the mesh for each box that will be drawn
			if(blockType == 0)
				boxMesh = Mesh.Box(dev, BoxSize, BoxSize, BoxSize);
			if(blockType == 1)
				boxMesh = Mesh.Sphere(dev, BoxSize/2, (int)BoxSize, (int)BoxSize);
            // Create the material list based on the colors of the boxes
            boxMaterials = new Material[BoxColors.Length];
            for (int i = 0; i < boxMaterials.Length; i++)
            {
                boxMaterials[i].Diffuse = boxMaterials[i].Ambient = BoxColors[i];
            }
        }

        private void SetBlockColor(BlockColor color)
        {
            material = boxMaterials[(byte)color];
        }

        public BlockColor BlockColor
        {
            get { return colors[colorIndex]; }
        }

        public void SetBlockPulse(bool pulse)
        {
            shouldPulse = pulse;
        }

        public void SetBlockTime(float totalTime)
        {
            time = totalTime;
        }

        public void SetBlockVelocity(Vector3 vel)
        {
            velocity = vel;
        }
        public void UpdateBlock(float elapsed)
        {
            // Update the blocks position based on velocity
            Vector3 velocityFrame = velocity;
            velocityFrame.Normalize();
            velocityFrame.Scale(velocity.Length() * elapsed);

            // Move the block
            pos += velocityFrame;
        }

        public void SelectNextColor()
        {
            // Are we at the end, and can wrap?  If so, reset the color index
            if ((colorIndex == (colors.Length - 1)) && (canWrap))
            {
                // Reset color index
                colorIndex = 0;
            }
            // Otherwise, if we're not at the end, increment the color index
            else if (colorIndex != (colors.Length - 1))
            {
                // Increment color index since we haven't gotten to the last one yet
                colorIndex++;
            }

            // Update material
            SetBlockColor(colors[colorIndex]);
        }

        public Vector3 Position
        {
            get { return pos; }
            set { pos = value; }
        }

        public static void Cleanup()
        {
            if (boxMesh != null)
            {
                boxMesh.Dispose();
            }
            boxMesh = null;
            boxMaterials = null;
        }

        public void Draw(Device dev)
        {
            // Set the device's material to the color of this box
            dev.Material = material;
            // Move the box into position
            if (shouldPulse)
            {
                float scaling = (float)Math.Abs(Math.Sin(time * MaxBoxScaleSpeed));
                scaling *= MaxBoxScale;
                float scaleFactor = 1.0f + scaling;
                dev.Transform.World = Matrix.Translation(pos) * Matrix.Scaling(
                    scaleFactor, 1.0f, scaleFactor);
            }
            else
            {
                // Move the box into position
                dev.Transform.World = Matrix.Translation(pos);
            }
            // Turn off any texture
            dev.SetTexture(0, null);
            // Draw the box
            boxMesh.DrawSubset(0);
        }
    }
}