using System;
using System.Configuration;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using Microsoft.Samples.DirectX.UtilityToolkit;

namespace SimonBlocks
{
	public class GameMain : IDeviceCreation, IFrameworkCallback
	{
		public const string GameName = "Simon Blocks";
		public static readonly string MediaPath = ConfigurationSettings.AppSettings.Get("MediaPath");
		
		private const float MaximumSpeed = 30.0f;
		private static readonly Vector3 CameraDefaultLocation = new Vector3(-2.5f, 15.0f, -25.0f);
		private static readonly Vector3 CameraDefaultLookAtLocation = new Vector3(-2.5f, 0.0f, -2.0f);
		private static readonly Vector3 CameraUp = new Vector3(0,1,0);
		private static readonly Vector3 LevelLight = new Vector3(0, -0.6f, 0.4f);

		private Framework sampleFramework = null;
		private Font statsFont = null; // Font for drawing text
		private Sprite textSprite = null; // Sprite for batching text calls
		private Effect effect = null; // D3DX Effect Interface
		private ModelViewerCamera camera = new ModelViewerCamera(); // A model viewing camera
		private bool isHelpShowing = true; // If true, renders the UI help text
		private Dialog hud = null; // dialog for standard controls
		private Dialog sampleUi = null; // dialog for sample specific controls
        private FrameworkMesh mesh = null; // Background mesh
		private Matrix viewMatrix;
		private Font gameFont = null;

		private Mesh levelMesh = null;
		private Texture[] levelTextures = null;
		private Texture buttonTexture = null;

		private MainUIScreen mainScreen = null;
		private bool mainScreenActive = false;
		private bool isLoadingLevel = false; //default off
		private bool instructionShow = true;

		private PlayerUIScreen playerScreen = null;
		private bool playerScreenActive = true;

		private Block[] lvlOneOrder;
		private Block[] lvlTwoOrder;

		private Player playerOne = null;
		private Player playerTwo = null;
		private bool hasPlayerTwo = false; //default only one player
		private Level currentLevel = null;
		private Level levelOne = null;
		private Level levelTwo = null;
		private int levelNumber = 1;
		private int numTotalLevels = 2;

		private int blockToShow;
		private float blockInstructionTime = 0.0f;
		private float winningLevelTime = 0.0f;
		private float ShowWinningTime = 2.5f;

		private Vector3 iBlockPosition = new Vector3(-2.5f, 0.0f, 2.0f);
		private Vector3 currentCameraPosition;

		private bool currentLevelPlayerOneCalc = false;
		private bool currentLevelPlayerTwoCalc = false;

		public GameMain(Framework f)
		{
			sampleFramework = f;
			hud = new Dialog(sampleFramework); 
			sampleUi = new Dialog(sampleFramework); 
		}
		/// <summary>
		/// Called during device initialization, this code checks the device for some 
		/// minimum set of capabilities, and rejects those that don't pass by returning false.
		/// </summary>
		public bool IsDeviceAcceptable(Caps caps, Format adapterFormat, Format backBufferFormat, bool windowed)
		{
			// Skip back buffer formats that don't support alpha blending
			if (!Manager.CheckDeviceFormat(caps.AdapterOrdinal, caps.DeviceType, adapterFormat, 
				Usage.QueryPostPixelShaderBlending, ResourceType.Textures, backBufferFormat))
				return false;

			return true;
		}

		/// <summary>
		/// This callback function is called immediately before a device is created to allow the 
		/// application to modify the device settings. The supplied settings parameter 
		/// contains the settings that the framework has selected for the new device, and the 
		/// application can make any desired changes directly to this structure.  Note however that 
		/// the sample framework will not correct invalid device settings so care must be taken 
		/// to return valid device settings, otherwise creating the Device will fail.  
		/// </summary>
		public void ModifyDeviceSettings(DeviceSettings settings, Caps caps)
		{
			// If device doesn't support HW T&L or doesn't support 1.1 vertex shaders in HW 
			// then switch to SWVP.
			if ( (!caps.DeviceCaps.SupportsHardwareTransformAndLight) ||
				(caps.VertexShaderVersion < new Version(1,1)) )
			{
				settings.BehaviorFlags = CreateFlags.SoftwareVertexProcessing;
			}
			else
			{
				settings.BehaviorFlags = CreateFlags.HardwareVertexProcessing;
			}

			// This application is designed to work on a pure device by not using 
			// any get methods, so create a pure device if supported and using HWVP.
			if ( (caps.DeviceCaps.SupportsPureDevice) && 
				((settings.BehaviorFlags & CreateFlags.HardwareVertexProcessing) != 0 ) )
				settings.BehaviorFlags |= CreateFlags.PureDevice;

			// Debugging vertex shaders requires either REF or software vertex processing 
			// and debugging pixel shaders requires REF.  
#if(DEBUG_VS)
            if (settings.DeviceType != DeviceType.Reference )
            {
                settings.BehaviorFlags &= ~CreateFlags.HardwareVertexProcessing;
                settings.BehaviorFlags |= CreateFlags.SoftwareVertexProcessing;
            }
#endif
#if(DEBUG_PS)
            settings.DeviceType = DeviceType.Reference;
#endif
		}


		/// <summary>
		/// This event will be fired immediately after the Direct3D device has been 
		/// created, which will happen during application initialization and windowed/full screen 
		/// toggles. This is the best location to create Pool.Managed resources since these 
		/// resources need to be reloaded whenever the device is destroyed. Resources created  
		/// here should be released in the Disposing event. 
		/// </summary>
	
		private void OnCreateDevice(object sender, DeviceEventArgs e)
		{
			// Initialize the stats font
			statsFont = ResourceCache.GetGlobalInstance().CreateFont(e.Device, 15, 0, FontWeight.Bold, 1, false, CharacterSet.Default,
				Precision.Default, FontQuality.Default, PitchAndFamily.FamilyDoNotCare | PitchAndFamily.DefaultPitch, "Arial");
			SurfaceDescription desc = e.BackBufferDescription;
			// Define DEBUG_VS and/or DEBUG_PS to debug vertex and/or pixel shaders with the 
			// shader debugger. Debugging vertex shaders requires either REF or software vertex 
			// processing, and debugging pixel shaders requires REF.  The 
			// ShaderFlags.Force*SoftwareNoOptimizations flag improves the debug experience in the 
			// shader debugger.  It enables source level debugging, prevents instruction 
			// reordering, prevents dead code elimination, and forces the compiler to compile 
			// against the next higher available software target, which ensures that the 
			// unoptimized shaders do not exceed the shader model limitations.  Setting these 
			// flags will cause slower rendering since the shaders will be unoptimized and 
			// forced into software.  See the DirectX documentation for more information about 
			// using the shader debugger.
			//ShaderFlags shaderFlags = ShaderFlags.None;
#if(DEBUG_VS)
            shaderFlags |= ShaderFlags.ForceVertexShaderSoftwareNoOptimizations;
#endif
#if(DEBUG_PS)
            shaderFlags |= ShaderFlags.ForcePixelShaderSoftwareNoOptimizations;
#endif

			gameFont = new Font(e.Device, new System.Drawing.Font("Arial", 18.0f));
			buttonTexture = TextureLoader.FromFile(e.Device, GameMain.MediaPath + "mainbuttons.png");
			ExtendedMaterial[] mtrls;
			levelMesh = Mesh.FromFile(MediaPath + "level.x", MeshFlags.Managed, e.Device, out mtrls);

			// Store the materials for later use and create textures
			if((mtrls != null) && (mtrls.Length >0))
			{
				levelTextures = new Texture[mtrls.Length];

				for(int i=0; i < mtrls.Length; i++)
				{
					levelTextures[i] = TextureLoader.FromFile(e.Device, MediaPath + mtrls[i].TextureFilename);
				}
			}

			mainScreen = new MainUIScreen(e.Device, desc.Width, desc.Height);
			mainScreen.NewGame += new EventHandler(OnNewGame);
			mainScreen.Quit += new EventHandler(OnMainQuit);

			playerScreen = new PlayerUIScreen(e.Device, desc.Width, desc.Height);
			playerScreen.OnePlayer += new EventHandler(OnOnePlayer);
			playerScreen.TwoPlayer += new EventHandler(OnTwoPlayer);

		}
		
		/// <summary>
		/// This event will be fired immediately after the Direct3D device has been 
		/// reset, which will happen after a lost device scenario. This is the best location to 
		/// create Pool.Default resources since these resources need to be reloaded whenever 
		/// the device is lost. Resources created here should be released in the OnLostDevice 
		/// event. 
		/// </summary>
		private void OnResetDevice(object sender, DeviceEventArgs e)
		{
			SurfaceDescription desc = e.BackBufferDescription;
			textSprite = new Sprite(e.Device);
            
			// Set the transformation matrices
			e.Device.Transform.Projection = Matrix.PerspectiveFovLH((float)Math.PI / 4, (float)desc.Width / (float)desc.Height, 0.1f, 1000000.0f);
			e.Device.Transform.View = Matrix.LookAtLH(new Vector3(0, 0, -54), new Vector3(), new Vector3(0,1,0));

			// Setup the camera's projection parameters
			float aspectRatio = (float)desc.Width / (float)desc.Height;
			camera.SetProjectionParameters((float)Math.PI / 4, aspectRatio, 0.1f, 1000.0f);
			camera.SetWindow(desc.Width, desc.Height);

			// Setup UI locations
			hud.SetLocation(desc.Width-170, 0);
			hud.SetSize(170,170);
			sampleUi.SetLocation(desc.Width - 170, desc.Height - 350);
			sampleUi.SetSize(170,300);

			//setup Lights
			e.Device.Lights[0].DiffuseColor = new ColorValue(1.0f, 1.0f, 1.0f, 1.0f);
			e.Device.Lights[0].Direction = LevelLight;
			e.Device.Lights[0].Type = LightType.Directional;
			sampleFramework.Device.Lights[0].Enabled = true;

			// Store the caps to check for texture filtering
			Caps caps = sampleFramework.DeviceCaps;

			// Set up the filtering mode for the texture
			if (caps.TextureFilterCaps.SupportsMinifyAnisotropic)
			{
				e.Device.SamplerState[0].MinFilter = TextureFilter.Anisotropic;
			}
			else if (caps.TextureFilterCaps.SupportsMinifyLinear)
			{
				e.Device.SamplerState[0].MinFilter = TextureFilter.Linear;
			}
			if (caps.TextureFilterCaps.SupportsMagnifyAnisotropic)
			{
				e.Device.SamplerState[0].MagFilter = TextureFilter.Anisotropic;
			}
			else if (caps.TextureFilterCaps.SupportsMagnifyLinear)
			{
				e.Device.SamplerState[0].MagFilter = TextureFilter.Linear;
			}
			if (caps.TextureFilterCaps.SupportsMipMapLinear)
			{
				e.Device.SamplerState[0].MipFilter = TextureFilter.Linear;
			}

			//gameFont.OnResetDevice();
		}
		
		/// <summary>
		/// This event function will be called fired after the Direct3D device has 
		/// entered a lost state and before Device.Reset() is called. Resources created
		/// in the OnResetDevice callback should be released here, which generally includes all 
		/// Pool.Default resources. See the "Lost Devices" section of the documentation for 
		/// information about lost devices.
		/// </summary>
		private void OnLostDevice(object sender, EventArgs e)
		{
			if(textSprite != null)
			{
				textSprite.Dispose();
				textSprite = null;
			}
			if(gameFont != null)
			{
				gameFont.Dispose();
				gameFont = null;
			}
		}

		/// <summary>
		/// This callback function will be called immediately after the Direct3D device has 
		/// been destroyed, which generally happens as a result of application termination or 
		/// windowed/full screen toggles. Resources created in the OnCreateDevice callback 
		/// should be released here, which generally includes all Pool.Managed resources. 
		/// </summary>
		private void OnDestroyDevice(object sender, EventArgs e)
		{
			// Clean up the textures used for the mesh level
			if ((levelTextures != null) && (levelTextures.Length > 0))
			{
				for(int i = 0; i < levelTextures.Length; i++)
				{
					levelTextures[i].Dispose();
				}
				// Clean up the mesh level itself
				levelMesh.Dispose();
			}
			
			//cleanup UIs and Objects
			if(mainScreen != null)
				mainScreen.Dispose();
			if(playerScreen != null)
				playerScreen.Dispose();
			if(playerOne != null)
				playerOne.Dispose();
			if(playerTwo != null)
				playerTwo.Dispose();
			if(buttonTexture != null)
				buttonTexture.Dispose();
			if(gameFont != null)
				gameFont.Dispose();

			Block.Cleanup();
		}

		///<summary>
		///Hook the Mouse events
		///</summary>
		private void OnMouseEvent(bool leftDown, bool rightDown, bool middleDown, bool side1Down, bool side2Down, int wheel, int x, int y)
		{
			if (!leftDown)
			{
				if(mainScreenActive)
					mainScreen.OnMouseMove(x,y);
				if(playerScreenActive)
					playerScreen.OnMouseMove(x,y);
			}
			else if (leftDown)
			{
				if(mainScreenActive)
					mainScreen.OnMouseClick(x,y);
				if(playerScreenActive)
					playerScreen.OnMouseClick(x,y);
			}
		}

		/// <summary>
		/// As a convenience, the sample framework inspects the incoming windows messages for
		/// keystroke messages and decodes the message parameters to pass relevant keyboard
		/// messages to the application.  The framework does not remove the underlying keystroke 
		/// messages, which are still passed to the application's MsgProc callback.
		/// </summary>
		private void OnKeyEvent(System.Windows.Forms.Keys key, bool isKeyDown, bool isKeyUp)
		{
			// Only do this when it's down
			if (isKeyDown)
			{
				if (mainScreenActive)
				{
					mainScreen.OnKeyPress(key);
				}
				else if(playerScreenActive)
					playerScreen.OnKeyPress(key);
				else
				{
					// Ignore any keystrokes while the fly through is displayed
					if (!isLoadingLevel)
					{
						// Otherwise, let the level handle the key strokes
						currentLevel.OnKeyPress(key);
					}
				}
			}
		}

		/// <summary>
		/// This callback function will be called once at the beginning of every frame. This is the
		/// best location for your application to handle updates to the scene, but is not 
		/// intended to contain actual rendering calls, which should instead be placed in the 
		/// OnFrameRender callback.  
		/// </summary>
		public void OnFrameMove(Device device, double appTime, float elapsedTime)
		{
			if (FrameworkTimer.IsStopped)
				return; // Nothing to do

			if (mainScreenActive || playerScreenActive)
			{
				return; // Nothing to do 
			}

			// Update the player
			playerOne.Update(elapsedTime, (float)appTime);

			if(playerTwo != null) //if there is a player two
				playerTwo.Update(elapsedTime, (float)appTime);

			PresentSequence(elapsedTime);
			// Make sure the level isn't started until the camera moves to position
			UpdateCamera(elapsedTime);

			//if level won and past the time marked to show win screen
			//load next level potentially; for now we quit
			if(playerTwo != null)
			{
				if (currentLevel.isOneGameWon && currentLevel.isTwoGameWon && (winningLevelTime > ShowWinningTime))
				{
					if(levelNumber < numTotalLevels)
					{
						levelNumber++;
						if(levelNumber == 2)
							currentLevel = levelTwo;
						blockToShow=0;
						instructionShow = true;
						currentLevelPlayerOneCalc = false;
						currentLevelPlayerTwoCalc = false;
					}
					else
						sampleFramework.CloseWindow();
				}
				if (currentLevel.isOneGameWon && currentLevel.isTwoGameWon)
				{
					winningLevelTime += elapsedTime;
				}
			}
			else
			{
				if ((currentLevel.isOneGameWon) && (winningLevelTime > ShowWinningTime))
				{
					if(levelNumber < numTotalLevels)
					{
						levelNumber++;
						if(levelNumber == 2)
							currentLevel = levelTwo;
						blockToShow=0;
						instructionShow = true;
						currentLevelPlayerOneCalc = false;
					}
					else
						sampleFramework.CloseWindow();
				}
				// If the level is won, then increment our timing for showing the screen
				if (currentLevel.isOneGameWon)
				{
					winningLevelTime += elapsedTime;
				}
			}
			// Update the current level
			currentLevel.Update(elapsedTime);
		}

		/// <summary>
		/// This callback function will be called at the end of every frame to perform all the 
		/// rendering calls for the scene, and it will also be called if the window needs to be 
		/// repainted. After this function has returned, the sample framework will call 
		/// Device.Present to display the contents of the next buffer in the swap chain
		/// </summary>
		public void OnFrameRender(Device device, double appTime, float elapsedTime)
		{
			bool beginSceneCalled = false;

			// Clear the render target and the zbuffer page 60
			device.Clear(ClearFlags.ZBuffer | ClearFlags.Target, 0, 1.0f, 0); //0x002D32AA for blue
			try
			{
				device.BeginScene();
				beginSceneCalled = true;
				
				if(mainScreenActive)
				{
					mainScreen.Draw();
				}
				if(playerScreenActive)
					playerScreen.Draw();
				if(!playerScreenActive && !mainScreenActive)
				{
					RenderGameScene(device, appTime);
				}
			}
			finally
			{
				if (beginSceneCalled)
					device.EndScene();
			}
		}

		private void RenderGameScene(Device device, double appTime)
		{
			// First render the level mesh, but before that is done, you will need
			// to turn off the zbuffer.  This isn't needed for this drawing
			device.RenderState.ZBufferEnable = false;
			device.RenderState.ZBufferWriteEnable = false;

			device.Transform.World = Matrix.Scaling(15,15,15) * Matrix.Translation(0,22,0);
			device.RenderState.Lighting = false;

			for(int i = 0; i < levelTextures.Length; i++)
			{
				device.SetTexture(0, levelTextures[i]);
				levelMesh.DrawSubset(i);
			}
			
			// Turn the zbuffer back on
			device.RenderState.ZBufferEnable = true;
			device.RenderState.ZBufferWriteEnable = true;
			
			device.RenderState.Lighting = true;

			if(instructionShow)
			{
				for(int i =0; i<currentLevel.BlockGrid.Length; i++)
				{
					if(currentLevel.WinOrder[blockToShow].BlockColor == currentLevel.BlockGrid[i].BlockColor)
						currentLevel.WinOrder[blockToShow].Position = currentLevel.BlockGrid[i].Position;
				}
				currentLevel.WinOrder[blockToShow].Draw(device);
			}
			else
			{
				playerOne.Draw(device, (float)appTime);

				if(playerTwo != null)
					playerTwo.Draw(device, (float)appTime);

				currentLevel.Draw(device);
			}
			int width = sampleFramework.DeviceSettings.presentParams.BackBufferWidth;
			int height = sampleFramework.DeviceSettings.presentParams.BackBufferHeight;

			if (isLoadingLevel)
			{
				string loadingText = string.Format(
					"Try Level {0}\r\nAre you ready?", levelNumber);

					gameFont.DrawText(null, loadingText, new System.Drawing.Rectangle(System.Drawing.Point.Empty, new System.Drawing.Size(width,height)),
						DrawTextFormat.Center | DrawTextFormat.VerticalCenter | DrawTextFormat.NoClip, System.Drawing.Color.WhiteSmoke);
			}
			else
			{
				// if there is a level happening, update some stats
				if (currentLevel != null && !instructionShow)
				{
					if(playerTwo != null)
					{
						// Draw current state
						string blockInfo = string.Format("Player One: Blocks Remaining {0} - Correct {1} \n Player Two: Blocks Remaining {2} - Correct {3}", 
							currentLevel.NumberBlocks - currentLevel.OneNumberCorrectBlocks, currentLevel.OneNumberCorrectBlocks,
							currentLevel.NumberBlocks - currentLevel.TwoNumberCorrectBlocks, currentLevel.TwoNumberCorrectBlocks);

						System.Drawing.Color statColor = System.Drawing.Color.White;
						// Change the color once you've gotten more than halfway complete
						if (((currentLevel.NumberBlocks - currentLevel.OneNumberCorrectBlocks) > currentLevel.OneNumberCorrectBlocks) || 
							((currentLevel.NumberBlocks - currentLevel.TwoNumberCorrectBlocks) > currentLevel.TwoNumberCorrectBlocks))
						{
							statColor = System.Drawing.Color.Turquoise;
						}

						gameFont.DrawText(null, blockInfo, new System.Drawing.Rectangle(0, 
							height - 60, width, height), 
							DrawTextFormat.Center | DrawTextFormat.NoClip, 
							statColor);

						if ((!currentLevel.isOneGameOver && !currentLevel.isTwoGameOver)) //no player won
						{
							string gameState = string.Format("Time Remaining: {0} \n Player One Score:{1} \t Player Two Score:{2}",
															currentLevel.TimeRemaining, playerOne.PlayerScore, playerTwo.PlayerScore);

							gameFont.DrawText(null, gameState, new System.Drawing.Rectangle(0, 
								30, width, height), DrawTextFormat.Center | DrawTextFormat.NoClip, 
								System.Drawing.Color.Yellow);
						}
						else if((currentLevel.isOneGameOver && !currentLevel.isTwoGameOver)) //1 finished
						{
							if(!currentLevelPlayerOneCalc)
							{
								playerOne.AddToScore(currentLevel.TimeLeft * currentLevel.TimeBonus);
								currentLevelPlayerOneCalc = true;
							}
							string gameState = string.Format("Time Remaining: {0} \n Player One Score:{1} \t Player Two Score:{2} \n Player One Has Finished!!! ", 
															currentLevel.TimeRemaining, playerOne.PlayerScore, playerTwo.PlayerScore);

							gameFont.DrawText(null, gameState, new System.Drawing.Rectangle(0, 
								30, width, height), DrawTextFormat.Center | DrawTextFormat.NoClip, 
								System.Drawing.Color.Yellow);
						}
						else if((!currentLevel.isOneGameOver && currentLevel.isTwoGameOver)) //2 finished
						{
							if(!currentLevelPlayerTwoCalc)
							{
								playerTwo.AddToScore(currentLevel.TimeLeft * currentLevel.TimeBonus);
								currentLevelPlayerTwoCalc = true;
							}
							string gameState = string.Format("Time Remaining: {0} \n Player One Score:{1} \t Player Two Score:{2} \n Player Two Has Finished!!!", 
								currentLevel.TimeRemaining, playerOne.PlayerScore, playerTwo.PlayerScore);

							gameFont.DrawText(null, gameState, new System.Drawing.Rectangle(0, 
								30, width, height), DrawTextFormat.Center | DrawTextFormat.NoClip, 
								System.Drawing.Color.Yellow);
						}
						else
						{
							string gameState;
							System.Drawing.Color c;
							if (currentLevel.isOneGameWonFirst) 
							{
								//need to calc scores
								if(!currentLevelPlayerTwoCalc)
								{
									playerTwo.AddToScore(currentLevel.TimeLeft * currentLevel.TimeBonus);
									currentLevelPlayerTwoCalc = true;
								}
								gameState = string.Format("Ha HA! Player One won this level! \n Player One Score:{0} \t Player Two Score:{1} ",
									playerOne.PlayerScore, playerTwo.PlayerScore);
								c = System.Drawing.Color.White;
							}
							else if (currentLevel.isTwoGameWonFirst)
							{
								//need to calc scores
								if(!currentLevelPlayerOneCalc)
								{
									playerOne.AddToScore(currentLevel.TimeLeft * currentLevel.TimeBonus);
									currentLevelPlayerOneCalc = true;
								}
								gameState = string.Format("Ha HA! Player Two won this level! \n Player One Score:{0} \t Player Two Score:{1} ",
									playerOne.PlayerScore, playerTwo.PlayerScore);
								c = System.Drawing.Color.White;
							}
							else if ((currentLevel.isOneGameOver && currentLevel.isTwoGameOver))
							{
								//need to calc score
								string winMsg = "";
									if(playerOne.PlayerScore > playerTwo.PlayerScore)
										winMsg="Player One Won!";
									if(playerOne.PlayerScore < playerTwo.PlayerScore)
										winMsg="Player Two Won!";
									if(playerOne.PlayerScore == playerTwo.PlayerScore)
										winMsg="Its a TIE!!!";
								gameState = string.Format(winMsg + " \n Player One Score:{0} \t Player Two Score:{1} ",
									playerOne.PlayerScore, playerTwo.PlayerScore);
								c = System.Drawing.Color.White;
							}
							else
							{
								gameState = string.Format("Game Over!!");
								c = System.Drawing.Color.Red;
							}
							gameFont.DrawText(null, gameState, new System.Drawing.Rectangle(0, 
								30, width, height), DrawTextFormat.Center | DrawTextFormat.NoClip, c);
						}
					}
					else
					{
						// Draw current state
						string blockInfo = string.Format("Blocks Remaining {0} - Correct {1}", 
							currentLevel.NumberBlocks - currentLevel.OneNumberCorrectBlocks, 
							currentLevel.OneNumberCorrectBlocks);

						System.Drawing.Color statColor = System.Drawing.Color.White;
						// Change the color once you've gotten more than halfway complete
						if ((currentLevel.NumberBlocks - currentLevel.OneNumberCorrectBlocks)
							> currentLevel.OneNumberCorrectBlocks)
						{
							statColor = System.Drawing.Color.Turquoise;
						}

						gameFont.DrawText(null, blockInfo, new System.Drawing.Rectangle(0, 
							height - 60, width, height), 
							DrawTextFormat.Center | DrawTextFormat.NoClip, 
							statColor);

						if (!currentLevel.isOneGameOver)
						{
							string gameState = string.Format("Time Remaining: {0} \n Player One Score:{1}",
								currentLevel.TimeRemaining, playerOne.PlayerScore);

							gameFont.DrawText(null, gameState, new System.Drawing.Rectangle(0, 
								30, width, height), DrawTextFormat.Center | DrawTextFormat.NoClip, 
								System.Drawing.Color.Yellow);
						}
						else
						{
							string gameState;
							System.Drawing.Color c;
							if (currentLevel.isOneGameWon)
							{
								if(!currentLevelPlayerOneCalc)
								{
									playerOne.AddToScore(currentLevel.TimeLeft * currentLevel.TimeBonus);
									currentLevelPlayerOneCalc = true;
								}
								gameState = string.Format("Congratulations! You win this level! \n Player One Score:{0}",
									playerOne.PlayerScore);
								c = System.Drawing.Color.White;
							}
							else
							{
								if(!currentLevelPlayerOneCalc)
								{
									playerOne.AddToScore(currentLevel.TimeLeft * currentLevel.TimeBonus);
									currentLevelPlayerOneCalc = true;
								}
								gameState = string.Format("Game Over!! \n Player One Score:{0}", playerOne.PlayerScore);
								c = System.Drawing.Color.Red;
							}
							gameFont.DrawText(null, gameState, new System.Drawing.Rectangle(0, 
								30, width, height), DrawTextFormat.Center | DrawTextFormat.NoClip, c);
						}
					}
				}
			}
		}


		private void OnNewGame(object sender, EventArgs e)
		{
			mainScreenActive = false;
			playerScreenActive = true;
			//CreateLevelObjects();
		}

		private void OnMainQuit(object sender, EventArgs e)
		{
			sampleFramework.CloseWindow();
		}

		private void OnOnePlayer(object sender, EventArgs e)
		{
			playerScreenActive = false;
			hasPlayerTwo = false;
			CreateLevelObjects();
		}

		private void OnTwoPlayer(object sender, EventArgs e)
		{
			playerScreenActive = false;
			hasPlayerTwo = true;
			CreateLevelObjects();
		}

		private void PresentSequence(float elapsedTime)
		{
			if(blockInstructionTime < currentLevel.InstructionTime && blockToShow < currentLevel.WinOrder.Length)
				blockInstructionTime += elapsedTime;
			else
			{
				blockInstructionTime = 0.0f;
				blockToShow++;
			}
			if(blockToShow == currentLevel.WinOrder.Length)
				instructionShow=false;
		}

		private void UpdateCamera(float elapsedTime)
		{
			if (currentCameraPosition != CameraDefaultLocation)
			{
				Vector3 diff = CameraDefaultLocation - currentCameraPosition;
				// Are we close enough to just move there?
				if (diff.Length() > (MaximumSpeed * elapsedTime))
				{
					// No we're not, move slowly there
					diff.Normalize();
					diff.Scale(MaximumSpeed * elapsedTime);
					currentCameraPosition += diff;
				}
				else
				{
					// Indeed we are, just move there
					currentCameraPosition = CameraDefaultLocation;
				}

				// Set the view transform now
				sampleFramework.Device.Transform.View = Matrix.LookAtLH(currentCameraPosition, 
					CameraDefaultLookAtLocation, CameraUp);
			}
			else
			{
				isLoadingLevel = false;
			}
		}


		private void CreateLevelObjects()
		{
			//create Player
			playerOne = new Player(sampleFramework.Device, 5.6f, 6.9f, 10, 10, 1);
			if(hasPlayerTwo)
				playerTwo = new Player(sampleFramework.Device, 5.6f, 6.9f, 10, 10, 2);

			//create Level
			BlockColor[] colors = new BlockColor[9];
			for (int i = 0; i < colors.Length; i++)
			{
				colors[i] = (BlockColor)i;
			}
			BlockColor[] red = new BlockColor[1];
			red[0] = (BlockColor)0;

			lvlOneOrder = new Block[5];
			lvlOneOrder[0] = new Block(sampleFramework.Device, colors, 5, false, 0);
			lvlOneOrder[1] = new Block(sampleFramework.Device, colors, 1, false, 0);
			lvlOneOrder[2] = new Block(sampleFramework.Device, colors, 3, false, 0);
			lvlOneOrder[3] = new Block(sampleFramework.Device, colors, 7, false, 0);
			lvlOneOrder[4] = new Block(sampleFramework.Device, colors, 2, false, 0);

			levelOne = new Level(sampleFramework.Device, 1, 3, playerOne, playerTwo, lvlOneOrder, 60.0f, 15, 2.5f);
			
			currentLevel = levelOne;

			lvlTwoOrder = new Block[7];
			lvlTwoOrder[0] = new Block(sampleFramework.Device, colors, 0, false, 0);
			lvlTwoOrder[1] = new Block(sampleFramework.Device, colors, 8, false, 0);
			lvlTwoOrder[2] = new Block(sampleFramework.Device, colors, 3, false, 0);
			lvlTwoOrder[3] = new Block(sampleFramework.Device, colors, 2, false, 0);
			lvlTwoOrder[4] = new Block(sampleFramework.Device, colors, 6, false, 0);
			lvlTwoOrder[5] = new Block(sampleFramework.Device, colors, 1, false, 0);
			lvlTwoOrder[6] = new Block(sampleFramework.Device, colors, 7, false, 0);

			levelTwo = new Level(sampleFramework.Device, 2, 3, playerOne, playerTwo, lvlTwoOrder, 50.0f, 25, 2.0f);
		}

		static int Main()
		{
			using(Framework sampleFramework = new Framework())
			{
				GameMain SimonMain = new GameMain(sampleFramework);
				// Set the callback functions. These functions allow the SimonMain framework to notify
				// the application about device changes, user input, and windows messages.  The 
				// callbacks are optional so you need only set callbacks for events you're interested 
				// in. However, if you don't handle the device reset/lost callbacks then the SimonMain 
				// framework won't be able to reset your device since the application must first 
				// release all device resources before resetting.  Likewise, if you don't handle the 
				// device created/destroyed callbacks then the SimonMain framework won't be able to 
				// recreate your device resources.
				sampleFramework.Disposing += new EventHandler(SimonMain.OnDestroyDevice);
				sampleFramework.DeviceLost += new EventHandler(SimonMain.OnLostDevice);
				sampleFramework.DeviceCreated += new DeviceEventHandler(SimonMain.OnCreateDevice);
				sampleFramework.DeviceReset += new DeviceEventHandler(SimonMain.OnResetDevice);

				sampleFramework.SetKeyboardCallback(new KeyboardCallback(SimonMain.OnKeyEvent));
				//sampleFramework.SetWndProcCallback(new WndProcCallback(SimonMain.OnMsgProc));
				sampleFramework.IsNotifiedOnMouseMove = true;
				sampleFramework.SetMouseCallback(new MouseCallback(SimonMain.OnMouseEvent));

				sampleFramework.SetCallbackInterface(SimonMain);
				try
				{

					// Show the cursor and clip it when in full screen
					sampleFramework.SetCursorSettings(true, true);

					// Initialize
					//SimonMain.InitializeApplication();

					// Initialize the SimonMain framework and create the desired window and Direct3D 
					// device for the application. Calling each of these functions is optional, but they
					// allow you to set several options which control the behavior of the sampleFramework.
					sampleFramework.Initialize( false, false, true ); // Parse the command line, handle the default hotkeys, and show msgboxes true, false, true 
					sampleFramework.CreateWindow("Simon Blocks");
					sampleFramework.CreateDevice( 0, true, Framework.DefaultSizeWidth, Framework.DefaultSizeHeight, SimonMain);

					// Pass control to the SimonMain framework for handling the message pump and 
					// dispatching render calls. The SimonMain framework will call your FrameMove 
					// and FrameRender callback when there is idle time between handling window messages.
					sampleFramework.MainLoop();

				}
#if(DEBUG)
				catch (Exception e)
				{
					// In debug mode show this error (maybe - depending on settings)
					sampleFramework.DisplayErrorMessage(e);
#else
            catch
            {
                // In release mode fail silently
#endif
					// Ignore any exceptions here, they would have been handled by other areas
					return (sampleFramework.ExitCode == 0) ? 1 : sampleFramework.ExitCode; // Return an error code here
				}

				// Perform any application-level cleanup here. Direct3D device resources are released within the
				// appropriate callback functions and therefore don't require any cleanup code here.
				return sampleFramework.ExitCode;
			}
		}
	}
}
