//Base code provided by Beginning 3D Game Programming by Tom Miller

using System;
using System.Configuration;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace SimonBlocks
{
	/// <summary>
	/// Summary description for Level.
	/// </summary>
	public class Level
	{

		private const float blockSpacing = 5.0f;

		private Block[] blockGrid = null;
		private Block chosenOneBlock = null;
		private Block chosenTwoBlock = null;
		private Block[] winOrder = null;
		private int gridSize = 3; //default 3X3 level grid
		private float startingLocation;

		private Player myPlayerOne = null;
		private Player myPlayerTwo = null;

		private int playerOneIndex;
		private int playerTwoIndex;
		private int totalOneMoves = 0;
		private int totalTwoMoves = 0;

		private int maxMoves = 0;
		private int levelNumber;

		private float maxTime = 0.0f;
		private float elapsedTime = 0.0f;
		private int tLeft= 0;
		private TimeSpan timeLeft;
		private int timeBonus; // bonus bonus awarded per sec left upon level completion

		private bool oneGameOver = false;
		private bool twoGameOver = false;
		private bool oneGameWon = false;
		private bool twoGameWon = false;
		private bool isFirstGameOver = true;
		private bool evalBoardOne = false;
		private bool evalBoardTwo = false;
		private bool oneGameWonFirst = false;
		private bool twoGameWonFirst = false;

		// Number of blockGrid (total)
		private int numberBlocks = 0;
		private int oneNumberCorrectBlocks = 0;
		private int twoNumberCorrectBlocks = 0;
		private float instructionTime;

		public Level(Device device, int inLevelNum, int inLevelSize, Player inPlayerOne, Player inPlayerTwo,
			Block[] winSet, float inTimer, int inTimeBonus, float inInstructionTime)
		{
			levelNumber = inLevelNum;
			myPlayerOne = inPlayerOne;
			myPlayerTwo = inPlayerTwo;
			winOrder = winSet;
			maxTime = inTimer;
			gridSize = inLevelSize;
			startingLocation = (blockSpacing * gridSize) / 2.0f;
			int blockIndex = 0;
			timeBonus = inTimeBonus;
			instructionTime = inInstructionTime;
			
			numberBlocks = winOrder.Length;
			blockGrid = new Block[gridSize * gridSize];
			
			// Create the block of colors now
			BlockColor[] colors = new BlockColor[9];
			for (int i = 0; i < colors.Length; i++)
			{
				colors[i] = (BlockColor)i;
			}


			for (int j = 0; j < gridSize; j++)
			{
				for (int i = 0; i < gridSize; i++)
				{
					Block b = new Block(device, colors, blockIndex, false, 1);
					b.Position = new Vector3(-startingLocation + (blockSpacing * i), -1.5f, 
						-startingLocation + (blockSpacing * j));
					blockGrid[blockIndex++] = b;
				}
			}
			playerOneIndex = 4; //player starts default in middle of the grid
			playerTwoIndex = 4; //player two starts default in middle left of the grid
		}

		private bool UpdatePlayerOne(bool force, int newPlayerIndex)
		{
			// Get the block the current player is over
			Block b = blockGrid[newPlayerIndex] as Block;
            
			if (force)
			{
				// Move the player to that position
				myPlayerOne.Position = new Vector3(b.Position.X, 0, b.Position.Z);
			}
			else
			{
				// Move the player to that position
				if (!myPlayerOne.MoveTo(new Vector3(b.Position.X, 0, b.Position.Z)))
					return false; // The player can't move here yet, he's still moving, just exit this method
			}
			return true;
		}

		private bool UpdatePlayerTwo(bool force, int newPlayerIndex)
		{
			// Get the block the current player is over
			Block b = blockGrid[newPlayerIndex] as Block;
            
			if (force)
			{
				// Move the player to that position
				myPlayerTwo.Position = new Vector3(b.Position.X, 0, b.Position.Z);
			}
			else
			{
				// Move the player to that position
				if (!myPlayerTwo.MoveTo(new Vector3(b.Position.X, 0, b.Position.Z)))
					return false; // The player can't move here yet, he's still moving, just exit this method
			}
			return true;
		}

		public void OnKeyPress(System.Windows.Forms.Keys key)
		{
			// Is the player trying to move?
				switch(key)
				{
					case System.Windows.Forms.Keys.Right:
						MoveOnePlayerX(true);
						break;
					case System.Windows.Forms.Keys.Left:
						MoveOnePlayerX(false);
						break;
					case System.Windows.Forms.Keys.Up:
						MoveOnePlayerZ(false);
						break;
					case System.Windows.Forms.Keys.Down:
						MoveOnePlayerZ(true);
						break;
					case System.Windows.Forms.Keys.Return:
						RecordPlayerOneChoice();
						break;

					case System.Windows.Forms.Keys.D:
						if(myPlayerTwo != null)
							MoveTwoPlayerX(true);
						break;
					case System.Windows.Forms.Keys.A:
						if(myPlayerTwo != null)
							MoveTwoPlayerX(false);
						break;
					case System.Windows.Forms.Keys.W:
						if(myPlayerTwo != null)
							MoveTwoPlayerZ(false);
						break;
					case System.Windows.Forms.Keys.S:
						if(myPlayerTwo != null)
							MoveTwoPlayerZ(true);
						break;
					case System.Windows.Forms.Keys.Tab:
						if(myPlayerTwo != null)
							RecordPlayerTwoChoice();
						break;
				}
		}
		private bool CanPlayerOneMoveHorizontal(bool right)
		{
			if ( ((playerOneIndex % gridSize) == 0) && (!right))
			{
				return false;
			}
			else if ( ((playerOneIndex % gridSize) == (gridSize - 1)) && (right))
			{
				return false;
			}

			// Check to see if there is a piece in the new position
			return (blockGrid[playerOneIndex + ((right) ? 1 : -1)] != null);
		}

		private bool CanPlayerOneMoveVertical(bool pos)
		{
			if ( ((playerOneIndex / gridSize) == 0) && (pos))
			{
				return false;
			}
			else if ( ((playerOneIndex / gridSize) == (gridSize - 1)) && (!pos))
			{
				return false;
			}

			return (blockGrid[playerOneIndex + ((pos) ? -gridSize : gridSize)] != null);
		}

		private bool CanPlayerTwoMoveHorizontal(bool right)
		{
			if ( ((playerTwoIndex % gridSize) == 0) && (!right))
			{
				return false;
			}
			else if ( ((playerTwoIndex % gridSize) == (gridSize - 1)) && (right))
			{
				return false;
			}

			// Check to see if there is a piece in the new position
			return (blockGrid[playerTwoIndex + ((right) ? 1 : -1)] != null);
		}

		private bool CanPlayerTwoMoveVertical(bool pos)
		{
			if ( ((playerTwoIndex / gridSize) == 0) && (pos))
			{
				return false;
			}
			else if ( ((playerTwoIndex / gridSize) == (gridSize - 1)) && (!pos))
			{
				return false;
			}

			return (blockGrid[playerTwoIndex + ((pos) ? -gridSize : gridSize)] != null);
		}

		/// Is the player moving in the positive z direction
		/// returns true if the move was made; false otherwise
		public bool MoveOnePlayerZ(bool pos)
		{
			if (isOneGameOver)
				return false; // Nothing to do if the game is over

			myPlayerOne.SetDirection(pos ? PlayerDirection.Down : PlayerDirection.Up);
			if (CanPlayerOneMoveVertical(pos))
			{
				// Update the block here
				if (UpdatePlayerOne(false, playerOneIndex + ((pos) ? -gridSize : gridSize)))
				{
					// Update the player index
					playerOneIndex += (pos) ? -gridSize : gridSize;

					// Update the number of moves
					totalOneMoves++;
					return true;
				}
			}

			return false;
		}


		/// Is the player moving right
		/// true if the move was made; false otherwise
		public bool MoveOnePlayerX(bool right)
		{
			if (isOneGameOver)
				return false; // Nothing to do if the game is over

			myPlayerOne.SetDirection(right ? PlayerDirection.Right : PlayerDirection.Left);
			if (CanPlayerOneMoveHorizontal(right))
			{
				// Update the block here
				if (UpdatePlayerOne(false, playerOneIndex + ((right) ? 1 : -1)))
				{

					// Update the player index
					playerOneIndex += (right) ? 1 : -1;

					// Update the number of moves
					totalOneMoves++;
					return true;
				}
			}

			return false;
		}

		/// Is the player moving in the positive z direction
		/// returns true if the move was made; false otherwise
		public bool MoveTwoPlayerZ(bool pos)
		{
			if (twoGameOver)
				return false; // Nothing to do if the game is over

			myPlayerTwo.SetDirection(pos ? PlayerDirection.Down : PlayerDirection.Up);
			if (CanPlayerTwoMoveVertical(pos))
			{
				// Update the block here
				if (UpdatePlayerTwo(false, playerTwoIndex + ((pos) ? -gridSize : gridSize)))
				{
					// Update the player index
					playerTwoIndex += (pos) ? -gridSize : gridSize;

					// Update the number of moves
					totalTwoMoves++;
					return true;
				}
			}

			return false;
		}


		/// Is the player moving right
		/// true if the move was made; false otherwise
		public bool MoveTwoPlayerX(bool right)
		{
			if (twoGameOver)
				return false; // Nothing to do if the game is over

			myPlayerTwo.SetDirection(right ? PlayerDirection.Right : PlayerDirection.Left);
			if (CanPlayerTwoMoveHorizontal(right))
			{
				// Update the block here
				if (UpdatePlayerTwo(false, playerTwoIndex + ((right) ? 1 : -1)))
				{

					// Update the player index
					playerTwoIndex += (right) ? 1 : -1;

					// Update the number of moves
					totalTwoMoves++;
					return true;
				}
			}

			return false;
		}

		public void Update(float time)
		{
			// Update the currently running time
			elapsedTime += time;
			tLeft = (int)(maxTime - elapsedTime);

			// Calculate the time remaining
			if (maxTime > elapsedTime)
			{
				timeLeft = new TimeSpan(0,0, tLeft);
			}

			// Has player one won the game?
			if (GameOneWon(elapsedTime))
			{
				// Set the variables and return
				oneGameOver = true;
				oneGameWon = true;
				if(!twoGameWonFirst)
					oneGameWonFirst=true;
			}
			else if(myPlayerTwo != null)
			{
				// Has player two won the game?
				if (GameTwoWon(elapsedTime))
				{
					// Set the variables and return
					twoGameOver = true;
					twoGameWon = true;
					if(!oneGameWonFirst)
						twoGameWonFirst=true;
				}
			}
			else if (elapsedTime >= maxTime)
			{
				// Set the variables and quit
				oneGameOver = true;
				twoGameOver = true;
				oneGameWon = false;
			}

			// Update the blockGrid if the game is over
			if (isOneGameOver && !isOneGameWon && isTwoGameOver && !isTwoGameWon)
			{
				// If this is the first time the game is over,
				// randomly assign a velocity to each block
				if (isFirstGameOver)
				{
					isFirstGameOver = false;
					Random r = new Random();
					int index = 0;
					foreach(Block b in blockGrid)
					{
						if(myPlayerTwo != null)
						{
							if (((b != null) && (playerOneIndex != index)) || ((b != null) && (playerTwoIndex != index)))
							{
								// Assign a random velocity
								float x, y, z = 0.0f;
								x = (float)r.NextDouble() * Block.MaximumVelocity;
								y = (float)r.NextDouble() * (Block.MaximumVelocity * 2);
								z = (float)r.NextDouble() * Block.MaximumVelocity;
								if (r.Next(50) > 25) { x *= -1; }
								if (r.Next(50) > 25) { y *= -1; }
								if (r.Next(50) > 25) { z *= -1; }
								b.SetBlockVelocity(new Vector3(x, y, z));
							}
						}
						else
						{
							if (((b != null) && (playerOneIndex != index)))
							{
								// Assign a random velocity
								float x, y, z = 0.0f;
								x = (float)r.NextDouble() * Block.MaximumVelocity;
								y = (float)r.NextDouble() * (Block.MaximumVelocity * 2);
								z = (float)r.NextDouble() * Block.MaximumVelocity;
								if (r.Next(50) > 25) { x *= -1; }
								if (r.Next(50) > 25) { y *= -1; }
								if (r.Next(50) > 25) { z *= -1; }
								b.SetBlockVelocity(new Vector3(x, y, z));
							}
						}
						index++;
					}
				}
				foreach(Block b in blockGrid)
				{
					if (b != null)
					{
						b.UpdateBlock(time);
					}
				}
			}

		}

		private void RecordPlayerOneChoice()
		{
			Block b = blockGrid[playerOneIndex] as Block;
			chosenOneBlock = b;
			evalBoardOne = true;
		}

		private void RecordPlayerTwoChoice()
		{
			Block b = blockGrid[playerTwoIndex] as Block;
			chosenTwoBlock = b;
			evalBoardTwo = true;
		}

		private bool GameOneWon(float totalTime)
		{
			bool won = false;
			if(chosenOneBlock != null && evalBoardOne)
			{
				if(oneNumberCorrectBlocks == winOrder.Length)
					won = true;
				else
				{
					if(winOrder[oneNumberCorrectBlocks].BlockColor == chosenOneBlock.BlockColor)
					{
						oneNumberCorrectBlocks++;
						if(oneNumberCorrectBlocks == winOrder.Length)
							won = true;
					}
					else
						oneNumberCorrectBlocks=0;
				}
				evalBoardOne = false;
			}
			return won;
		}

		private bool GameTwoWon(float totalTime)
		{
			bool won = false;
			if(chosenTwoBlock != null && evalBoardTwo)
			{
				if(twoNumberCorrectBlocks == winOrder.Length)
					won = true;
				else
				{
					if(winOrder[twoNumberCorrectBlocks].BlockColor == chosenTwoBlock.BlockColor)
					{
						twoNumberCorrectBlocks++;
						if(twoNumberCorrectBlocks == winOrder.Length)
							won = true;
					}
					else
						twoNumberCorrectBlocks=0;
				}
				evalBoardTwo = false;
			}
			return won;
		}

		public void Draw(Device device)
		{
			// Render every block
			foreach(Block b in blockGrid)
			{
				// It's possible the block may be null
				if (b != null)
				{
					b.Draw(device);
				}
			}
		}

		public Block[] WinOrder
		{
			get {return winOrder;}
		}

		public Block[] BlockGrid
		{
			get {return blockGrid;}
		}

		public int NumberBlocks
		{
			get { return numberBlocks; }
		}

		public int OneNumberCorrectBlocks
		{
			get { return oneNumberCorrectBlocks; }
		}

		public int TwoNumberCorrectBlocks
		{
			get { return twoNumberCorrectBlocks; }
		}

		public string TimeRemaining
		{
			get { return timeLeft.ToString(); }
		}

		public int TotalOneMoves
		{
			get { return totalOneMoves; }
		}

		public int TotalTwoMoves
		{
			get { return totalTwoMoves; }
		}

		public int MaximumMoves
		{
			get { return maxMoves; }
		}

		public bool isOneGameOver
		{
			get { return oneGameOver; }
		}

		public bool isTwoGameOver
		{
			get { return twoGameOver; }
		}

		public bool isOneGameWon
		{
			get { return oneGameWon; }
		}

		public bool isTwoGameWon
		{
			get { return twoGameWon; }
		}

		public bool isOneGameWonFirst
		{
			get { return oneGameWonFirst; }
		}

		public bool isTwoGameWonFirst
		{
			get { return twoGameWonFirst; }
		}

		public int TimeBonus
		{
			get {return timeBonus;}
		}

		public int TimeLeft
		{
			get {return tLeft;}
		}

		public float InstructionTime
		{
			get {return instructionTime;}
		}
	}
}
