//Base code provided by Beginning 3D Game Programming by Tom Miller

using System;
using System.Configuration;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace SimonBlocks
{
	/// <summary>
	/// The main selection screen for the game
	/// </summary>
	public class MainUIScreen : UIScreen, IDisposable
	{
		private Texture buttonTextures = null;
		private Texture messageTexture = null;

		private UIButton newButton = null;
		private UIButton exitButton = null;

		// Events
		public event EventHandler NewGame;
		public event EventHandler Quit;

		public MainUIScreen(Device device, int width, int height) : base(device, width, height)
		{
			messageTexture = TextureLoader.FromFile(device, GameMain.MediaPath + "SimonMainMenu.png");
			// Mark the background texture as centered
			StoreTexture(messageTexture, width, height, false);

			buttonTextures = TextureLoader.FromFile(device, GameMain.MediaPath + "MainButtons.png");

			newButton = new UIButton(renderSprite, buttonTextures, buttonTextures,new Rectangle(0,LargeButtonHeight * 1,
				LargeButtonWidth, LargeButtonHeight), new Rectangle(0,0,LargeButtonWidth, LargeButtonHeight), 
				new Point((width - LargeButtonWidth) / 2,(height - (LargeButtonHeight * 4)) / 2));

			exitButton = new UIButton(renderSprite, buttonTextures, buttonTextures, new Rectangle(0,LargeButtonHeight * 3,
				LargeButtonWidth, LargeButtonHeight), new Rectangle(0,LargeButtonHeight * 2,LargeButtonWidth, LargeButtonHeight), 
				new Point((width - LargeButtonWidth) / 2,(height - (LargeButtonHeight * 2)) / 2));

			newButton.Click += new EventHandler(OnNewButton);
			exitButton.Click += new EventHandler(OnExitButton);
		}

		public override void Draw()
		{
			// Start rendering sprites
			base.BeginSprite();
			// Draw the background
			base.Draw();

			// Now the buttons
			newButton.Draw();
			exitButton.Draw();

			// You're done rendering sprites
			base.EndSprite();
		}

		private void OnNewButton(object sender, EventArgs e)
		{
			if (NewGame != null)
				NewGame(this, e);
		}

		private void OnExitButton(object sender, EventArgs e)
		{
			if (Quit != null)
				Quit(this, e);
		}

		public void OnMouseMove(int x, int y)
		{
			newButton.OnMouseMove(x, y);
			exitButton.OnMouseMove(x, y);
		}

		public void OnMouseClick(int x, int y)
		{
			newButton.OnMouseClick(x, y);
			exitButton.OnMouseClick(x, y);
		}

		public void OnKeyPress(System.Windows.Forms.Keys key)
		{
			switch(key)
			{
				case Keys.Escape: //exit game
					OnExitButton(this, EventArgs.Empty);
					break;
				case Keys.Enter: //start new game
					OnNewButton(this, EventArgs.Empty);
					break;
			}
		}

		#region IDisposable Members
		/// <summary>
		/// Cleanup in case dispose isn't called
		/// </summary>
		~MainUIScreen()
		{
			Dispose();
		}
		/// <summary>
		/// Cleanup any resources
		/// </summary>
		public override void Dispose()
		{
			GC.SuppressFinalize(this);
			if (messageTexture != null)
			{
				messageTexture.Dispose();
			}
			if (buttonTextures != null)
			{
				buttonTextures.Dispose();
			}
			buttonTextures = null;
			messageTexture = null;
			base.Dispose();
		}

		#endregion
	}
}
