//Base code provided by Beginning 3D Game Programming by Tom Miller

using System;
using System.Configuration;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace SimonBlocks
{
    public enum PlayerColor
    {
        Blue,
        Green,
        Purple,
        Red,
        Gray,
        Yellow,
        Min = Blue,
        Max = Yellow,
    }

    public enum PlayerDirection
    {
        Down,
        Up,
        Left,
        Right
    }

    public class Player : IDisposable
    {
        // Rendering constants
        private const float ScaleConstant = 0.1f;
        private static readonly Matrix[] RotationMatrices = new Matrix[] {
            Matrix.RotationY((float)Math.PI * 3.0f / 2.0f),
            Matrix.RotationY((float)Math.PI / 2.0f),
            Matrix.RotationY((float)Math.PI * 2),
            Matrix.RotationY((float)Math.PI) };

        private static readonly Matrix ScalingMatrix = Matrix.Scaling(
            ScaleConstant, ScaleConstant, ScaleConstant);

        // Maximum speed the player will move per second
        private const float MaxMovement = 30.0f;
        // Maximum 'bounce' speed
        private const float BounceSpeed = 10.0f;
        private const float MaxBounceHeight = 0.4f;

		private float innerRadius, outerRadius;
		int sides, rings;
		private static Mesh playerMesh = null; // Mesh for the player
        
		private static Material playerMaterial; // Material
        private Matrix rotation = RotationMatrices[0];

        private Vector3 pos; // The player's real position
        private Vector3 moveToPos; // Where the player is moving to
        private bool isMoving = false;
        private float playerHeight = 0.0f;
		private int playerScore;
		private static int playerNum;

        public Player(Device dev, float inInnerRadius, float inOuterRadius, int inSides, int inRings, int inPlayerNum)
        {
            innerRadius = inInnerRadius;
			outerRadius = inOuterRadius;
			sides = inSides;
			rings = inRings;
			playerScore = 0; //init player score to 0
			playerNum = inPlayerNum;
			
			// Initialize the static data if needed
			if (playerMesh == null)
			{
				InitializeStaticData(dev);
			}
        }

		private void InitializeStaticData(Device dev)
		{
			playerMesh = Mesh.Torus(dev, innerRadius, outerRadius, sides, rings);
			// Create the material list based on the colors of the boxes
			playerMaterial = new Material();
			playerMaterial.Diffuse = playerMaterial.Ambient = System.Drawing.Color.DarkBlue;
		}

        public Vector3 Position
        {
            get { return pos; }
            set { pos = moveToPos = value;}
        }

        public void Draw(Device device, float appTime)
        {
            // Set the world transform
            device.Transform.World = ScalingMatrix * rotation *
                Matrix.Translation(pos.X, playerHeight, pos.Z);

			if(playerNum==2)
				playerMaterial.Diffuse = playerMaterial.Ambient = System.Drawing.Color.Red;
            // Set the model's material
            device.Material = playerMaterial;
			device.SetTexture(0, null);

            // Render our model
            playerMesh.DrawSubset(0);
        }

        public void SetDirection(PlayerDirection dir)
        {
            rotation = RotationMatrices[(byte)dir];
        }

		public void AddToScore(int toAdd)
		{
			playerScore += toAdd;
		}

		public int PlayerScore
		{
			get {return playerScore;}
		}

        public bool MoveTo(Vector3 newPosition)
        {
            if (!isMoving)
            {
                isMoving = true;
                // Just store the new position, it will be used during update
                moveToPos = newPosition;
                return true;
            }
            else
            {
                return false;
            }
        }

        public void Update(float elapsed, float total)
        {
            // Calculate the new player position if needed
            playerHeight = (float)(Math.Abs(Math.Sin(total * BounceSpeed))) * MaxBounceHeight;

            if (pos != moveToPos)
            {
                Vector3 diff = moveToPos - pos;
                // Are we close enough to just move there?
                if (diff.LengthSq() > (MaxMovement * elapsed))
                {
                    // No we're not, move slowly there
                    diff.Normalize();
                    diff.Scale(MaxMovement * elapsed);
                    pos += diff;
                }
                else
                {
                    isMoving = false;
                    // Indeed we are, just move there
                    pos = moveToPos;
                }
            }
            else
            {
                isMoving = false;
            }
        }

        /// <summary>
        /// Clean up any resources in case you forgot to dispose this object
        /// </summary>
        ~Player()
        {
            Dispose();
        }

        public void Dispose()
        {
            // Suppress finalization
            GC.SuppressFinalize(this);

            if (playerMesh != null)
            {
                playerMesh.Dispose();
            }
            playerMesh = null;
        }
    }
}