//Code based on code provided by Beginning 3D Game Programming by Tom Miller

using System;
using System.IO;
using System.Runtime.InteropServices;

namespace SimonBlocks
{
	public class Timer
	{
		[System.Security.SuppressUnmanagedCodeSecurity]
		[DllImport("kernel32")]
		private static extern bool QueryPerformanceFrequency(ref long PerformanceFrequency);
		[System.Security.SuppressUnmanagedCodeSecurity]
		[DllImport("kernel32")]
		private static extern bool QueryPerformanceCounter(ref long PerformanceCount);

		private long numberTicksPerSec = 0;
		private long startTime = 0;
		private long elapsedTime = 0;
		private long lastTime = 0;

		private bool isTimerRunning = false;

		public Timer()
		{
			// Use QueryPerformanceFrequency() to get frequency of timer.  
			long ticksPerSec = 0;
			if (QueryPerformanceFrequency(ref ticksPerSec))
			{
				numberTicksPerSec = ticksPerSec;
			}
			else
			{
				// Unfortunately we can't use QPF, throw an error for this case.
				throw new InvalidOperationException("Your system does not support a high resolution timer.");
			}
		}

		public void Start()
		{
			if (!QueryPerformanceCounter(ref startTime))
			{
				throw new InvalidOperationException("Failed to get the latest performance counter");
			}
			lastTime = startTime;
			isTimerRunning = true;
		}

		public void Stop()
		{
			Update();
			isTimerRunning = false;
		}

		public void Update()
		{
			if (!isTimerRunning)
				return; // Nothing to do

			// Store the last elapsed time
			elapsedTime = lastTime;

			if (!QueryPerformanceCounter(ref lastTime))
			{
				throw new InvalidOperationException("Failed to get the latest performance counter");
			}
		}

		public bool IsRunning
		{
			get { return isTimerRunning; }
		}

		public float ElapsedTime
		{
			get
			{
				return ((float)(lastTime - elapsedTime) / (float)numberTicksPerSec);
			}
		}

		public float TotalTime
		{
			get
			{
				return ((float)(lastTime - startTime) / (float)numberTicksPerSec);
			}
		}
	}
}
