//Code based on code provided by Beginning 3D Game Programming by Tom Miller

using System;
using System.Configuration;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace SimonBlocks
{
	public abstract class UIScreen : IDisposable
	{
		// Constants
		public static readonly Vector3 ObjectCenter = new Vector3();
		public static readonly int SpriteColor = unchecked((int)0xffffffff);

		protected const int SmallButtonWidth = 128;
		protected const int SmallButtonHeight = 64;
		protected const int LargeButtonWidth = 256;
		protected const int LargeButtonHeight = 64;

		protected Sprite renderSprite = null;
		protected Texture backgroundTexture = null;
		protected Rectangle backgroundSource;

		// Screen size
		protected int screenWidth = 0;
		protected int screenHeight = 0;

		// Should the object be centered?
		private bool isCentered = false;
		protected Vector3 centerUpper;

		public UIScreen(Device device, int width, int height)
		{
			// Create the sprite object
			renderSprite = new Sprite(device);

			// Hook the device events to 'fix' the sprite if needed
			device.DeviceLost += new EventHandler(OnDeviceLost);
			device.DeviceReset += new EventHandler(OnDeviceReset);

			StoreTexture(null, width, height, false);
		}

		protected void StoreTexture(Texture background, int width, int height,
			bool centerTexture)
		{
			// Store the background texture
			backgroundTexture = background;

			if (backgroundTexture != null)
			{
				// Get the background texture
				using (Surface s = backgroundTexture.GetSurfaceLevel(0))
				{
					SurfaceDescription desc = s.Description;
					backgroundSource = new Rectangle(0, 0,
						desc.Width, desc.Height);
				}
			}

			// Store the width/height
			screenWidth = width;
			screenHeight = height;

			// Store the centered texture
			isCentered = centerTexture;

			if (isCentered)
			{
				centerUpper = new Vector3((float)(width - backgroundSource.Width) / 2.0f,
					(float)(height - backgroundSource.Height) / 2.0f, 0.0f);
			}
		}

		private void OnDeviceLost(object sender, EventArgs e)
		{
			// The device has been lost, make sure the sprite cleans itself up
			if (renderSprite != null)
				renderSprite.OnLostDevice();
		}

		private void OnDeviceReset(object sender, EventArgs e)
		{
			// The device has been reset, make sure the sprite fixes itself up
			if (renderSprite != null)
				renderSprite.OnResetDevice();
		}


		protected void BeginSprite()
		{
			renderSprite.Begin(SpriteFlags.AlphaBlend);
		}

		protected void EndSprite()
		{
			renderSprite.End();
		}

		public virtual void Draw()
		{
			// Render the background if it exists
			if (backgroundTexture != null)
			{
				if (isCentered)
				{
					// Render to the screen centered
					renderSprite.Draw(backgroundTexture, backgroundSource, 
						ObjectCenter, centerUpper, SpriteColor);
				}
				else
				{
					// Scale the background to the right size
					renderSprite.Transform = Matrix.Scaling(
						(float)screenWidth / (float)backgroundSource.Width,
						(float)screenHeight / (float)backgroundSource.Height,
						0);

					// Render it to the screen
					renderSprite.Draw(backgroundTexture, backgroundSource, 
						ObjectCenter, ObjectCenter, SpriteColor);

					// Reset the transform
					renderSprite.Transform = Matrix.Identity;
				}
			}
		}

		~UIScreen()
		{
			Dispose();
		}
		/// <summary>
		/// Cleanup any resources
		/// </summary>
		public virtual void Dispose()
		{
			GC.SuppressFinalize(this);
			// Dispose the sprite
			if (renderSprite != null)
			{
				renderSprite.Dispose();
			}
			renderSprite = null;
		}
	}
}
