//{{{ GPL Notice
/*
 *  BugItemComposer.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.ProColPlugin;
import com.enderak.procol.client.model.BugItem;
import com.enderak.procol.common.net.*;
import java.awt.*;
import java.text.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import org.gjt.sp.jedit.jEdit;
//}}}

/**
 *  Dialog to create or edit a bug list item
 *
 *@author    Justin Dieters
 */
public class BugItemComposer extends JFrame {
	private BugItemComposerPanel panel;

	//{{{ Constructor
	/**
	 *  Constructor for the BugItemComposer for creating a new bug item
	 *
	 *@param  userListIn  The list of available users to assign the bug item to
	 */
	public BugItemComposer(Vector userListIn) {
		Toolkit kit           = Toolkit.getDefaultToolkit();
		Dimension screenSize  = kit.getScreenSize();

		this.setTitle(jEdit.getProperty("procol.client.messagecomposer.buglist.title"));
		this.getContentPane().add(panel = new BugItemComposerPanel(this, userListIn));
		this.setSize(425, 350);
		this.setLocation((screenSize.width / 2) - (this.getWidth() / 2), (screenSize.height / 2) - (this.getHeight() / 2));
		this.setVisible(true);
	}


	/**
	 *  Constructor for the BugItemCompuser for editing an existing bug item
	 *
	 *@param  bugItem     The bug item to edit
	 *@param  userListIn  The list of available users to assign the bug item to
	 */
	public BugItemComposer(BugItem bugItem, Vector userListIn) {
		Toolkit kit           = Toolkit.getDefaultToolkit();
		Dimension screenSize  = kit.getScreenSize();

		this.setTitle(jEdit.getProperty("procol.client.messagecomposer.buglist.title"));
		this.getContentPane().add(panel = new BugItemComposerPanel(this, bugItem, userListIn));
		this.setSize(400, 300);
		this.setLocation((screenSize.width / 2) - (this.getWidth() / 2), (screenSize.height / 2) - (this.getHeight() / 2));
		this.setVisible(true);
	}
	//}}}

	private class BugItemComposerPanel extends JPanel {
		//{{{ Data Members
		private JTextArea descriptionInput;
		private JScrollPane scrollPane;
		private JTextField subjectInput;
		private JList userList;
		private JSpinner dateSpinner;
		private JCheckBox dateCheckbox;
		private JSlider prioritySlider, completeSlider;
		private JButton sendButton, cancelButton;
		//}}}

		//{{{ Constructors
		/**
		 *  Constructor for the BugItemComposerPanel object
		 *
		 *@param  parentFrame  The parent frame
		 *@param  userListIn   The user list
		 */
		public BugItemComposerPanel(final BugItemComposer parentFrame, final Vector userListIn) {
			initGUI(userListIn);

			sendButton.addActionListener(
				new java.awt.event.ActionListener() {
					public void actionPerformed(java.awt.event.ActionEvent evt) {
						String date  = "null";
						if (dateCheckbox.isSelected()) {
							date = "None";
						} else {
							date = new SimpleDateFormat("MM/dd/yy HH:mm:ss").format(((Date)(dateSpinner.getValue())));
						}
						ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.BUG_ITEM_ADD, subjectInput.getText() + "\n" + prioritySlider.getValue() + "\n" + userList.getSelectedValue() + "\n" + date + "\n" + completeSlider.getValue() + "\n" + descriptionInput.getText(), PacketFactory.NORMAL_PRIORITY);
						parentFrame.dispose();
					}
				});

			cancelButton.addActionListener(
				new java.awt.event.ActionListener() {
					public void actionPerformed(java.awt.event.ActionEvent evt) {
						parentFrame.dispose();
					}
				});

			ProColPlugin.getClient().getProjectInfo();
		}


		/**
		 *  Constructor for the BugItemComposerPanel object
		 *
		 *@param  parentFrame  The parent frame
		 *@param  bugItem      The bug item to edit
		 *@param  userListIn   The user list
		 */
		public BugItemComposerPanel(final BugItemComposer parentFrame, final BugItem bugItem, final Vector userListIn) {
			initGUI(userListIn);
			System.out.println("index=" + bugItem.index);
			System.out.println("dueDate=" + bugItem.dueDate);
			System.out.println("subject=" + bugItem.subject);
			System.out.println("priority=" + bugItem.priority);
			System.out.println("complete=" + bugItem.complete);
			System.out.println("assignedTo=" + bugItem.assignedTo);
			System.out.println("description=" + bugItem.description);
			prioritySlider.setValue(Integer.parseInt(bugItem.priority));
			completeSlider.setValue(Integer.parseInt(bugItem.complete));
			if (bugItem.dueDate.equals("null") || bugItem.dueDate.equals("None")) {
				dateCheckbox.setSelected(true);
			} else {
				try {
					dateSpinner.getModel().setValue(new SimpleDateFormat("MM/dd/yy HH:mm:ss").parse(bugItem.dueDate));
				} catch (ParseException pe) {
					System.err.println(pe);
				}
			}
			userList.setSelectedValue(bugItem.assignedTo, true);
			subjectInput.setText(bugItem.subject);
			descriptionInput.setText(bugItem.description);

			sendButton.addActionListener(
				new java.awt.event.ActionListener() {
					public void actionPerformed(java.awt.event.ActionEvent evt) {
						String date  = "null";
						if (dateCheckbox.isSelected()) {
							date = "None";
						} else {
							date = new SimpleDateFormat("MM/dd/yy HH:mm:ss").format(((Date)(dateSpinner.getValue())));
						}
						ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.BUG_ITEM_UPDATE, bugItem.index + "\n" + subjectInput.getText() + "\n" + prioritySlider.getValue() + "\n" + userList.getSelectedValue() + "\n" + date + "\n" + completeSlider.getValue() + "\n" + descriptionInput.getText(), PacketFactory.NORMAL_PRIORITY);
						parentFrame.dispose();
					}
				});

			cancelButton.addActionListener(
				new java.awt.event.ActionListener() {
					public void actionPerformed(java.awt.event.ActionEvent evt) {
						parentFrame.dispose();
					}
				});

			ProColPlugin.getClient().getProjectInfo();
		}
		//}}}

		//{{{ initGUI
		private void initGUI(Vector userListIn) {
			subjectInput = new JTextField();
			subjectInput.selectAll();
			descriptionInput = new JTextArea();
			descriptionInput.setLineWrap(true);
			descriptionInput.setWrapStyleWord(true);
			userList = new JList(userListIn);
			scrollPane = new JScrollPane(descriptionInput);

			this.setLayout(new BorderLayout());

			JPanel topInputPanel                  = new JPanel();
			topInputPanel.setLayout(new BorderLayout());

			JPanel subjectInputPanel              = new JPanel();
			subjectInputPanel.setLayout(new BorderLayout());
			subjectInputPanel.add(subjectInput, BorderLayout.CENTER);
			subjectInputPanel.setBorder(new TitledBorder(jEdit.getProperty("procol.client.messagecomposer.subject")));
			topInputPanel.add(subjectInputPanel, BorderLayout.NORTH);

			JPanel dateInputPanel                 = new JPanel();
			dateInputPanel.setLayout(new BorderLayout());
			dateSpinner = new JSpinner(new SpinnerDateModel());
			dateCheckbox = new JCheckBox(jEdit.getProperty("procol.client.messagecomposer.noduedate"));
			dateInputPanel.add(dateSpinner, BorderLayout.CENTER);
			dateInputPanel.add(dateCheckbox, BorderLayout.EAST);
			dateInputPanel.setBorder(new TitledBorder(jEdit.getProperty("procol.client.messagecomposer.duedate")));
			topInputPanel.add(dateInputPanel, BorderLayout.CENTER);

			JPanel priorityAndCompleteInputPanel  = new JPanel();
			priorityAndCompleteInputPanel.setLayout(new BorderLayout());
			prioritySlider = new JSlider(0, 10, 5);
			prioritySlider.setLabelTable(prioritySlider.createStandardLabels(5));
			prioritySlider.setMinorTickSpacing(1);
			prioritySlider.setSnapToTicks(true);
			prioritySlider.setPaintTicks(true);
			prioritySlider.setPaintLabels(true);
			completeSlider = new JSlider(0, 100, 0);
			completeSlider.setLabelTable(completeSlider.createStandardLabels(50));
			completeSlider.setMinorTickSpacing(10);
			completeSlider.setSnapToTicks(true);
			completeSlider.setPaintTicks(true);
			completeSlider.setPaintLabels(true);

			JPanel priorityPanel                  = new JPanel();
			priorityPanel.setLayout(new BorderLayout());
			priorityPanel.add(prioritySlider, BorderLayout.CENTER);
			priorityPanel.setBorder(new TitledBorder(jEdit.getProperty("procol.client.messagecomposer.priority")));

			JPanel completePanel                  = new JPanel();
			completePanel.setLayout(new BorderLayout());
			completePanel.add(completeSlider, BorderLayout.CENTER);
			completePanel.setBorder(new TitledBorder(jEdit.getProperty("procol.client.messagecomposer.complete")));

			priorityAndCompleteInputPanel.add(priorityPanel, BorderLayout.WEST);
			priorityAndCompleteInputPanel.add(completePanel, BorderLayout.EAST);
			//priorityAndCompleteInputPanel.add(new JPanel(), BorderLayout.CENTER);

			topInputPanel.add(priorityAndCompleteInputPanel, BorderLayout.SOUTH);

			this.add(topInputPanel, BorderLayout.NORTH);

			JPanel descriptionInputPanel          = new JPanel();
			descriptionInputPanel.setLayout(new BorderLayout());
			descriptionInputPanel.add(scrollPane, BorderLayout.CENTER);
			descriptionInputPanel.setBorder(new TitledBorder(jEdit.getProperty("procol.client.messagecomposer.description")));
			this.add(descriptionInputPanel, BorderLayout.CENTER);

			JPanel userListPanel                  = new JPanel();
			userListPanel.setLayout(new BorderLayout());
			userListPanel.add(userList, BorderLayout.CENTER);
			userListPanel.setBorder(new TitledBorder(jEdit.getProperty("procol.client.messagecomposer.assign")));
			this.add(userListPanel, BorderLayout.WEST);
			// Send Button
			sendButton = new JButton(jEdit.getProperty("procol.label.send"));

			// Cancel Button
			cancelButton = new JButton(jEdit.getProperty("procol.label.cancel"));

			JPanel buttonPanel                    = new JPanel();
			buttonPanel.add(sendButton);
			buttonPanel.add(cancelButton);
			this.add(buttonPanel, BorderLayout.SOUTH);
		}
		//}}}
	}
}
