//{{{ GPL Notice
/*
 *  BugListPanel.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.*;
import com.enderak.procol.client.model.*;
import com.enderak.procol.client.net.*;
import com.enderak.procol.common.gui.*;
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.gui.*;
//}}}

/**
 *  Displays a list of current bug items
 *
 *@author    Justin Dieters
 */
public class BugListPanel extends GenericListPanel {
	//{{{ Data Members
	protected static DefaultListModel listModel  = new DefaultListModel();
	private RolloverButton newButton, modifyButton, deleteButton;
	private Vector allUsers;
	//}}}

	//{{{ Constructors
	/**
	 *  Constructor for the BugListPanel object
	 *
	 *@param  view      jEdit View, specified by jEdit
	 *@param  position  position of dockable, specified by jEdit
	 */
	public BugListPanel(View view, String position) {
		super(view, position, "buglist");
		listModel.clear();
		itemList.setModel(listModel);
		MouseListener mouseListener  =
			new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					int index  = itemList.locationToIndex(e.getPoint());
					itemList.setSelectedIndex(index);
					if (e.getButton() == MouseEvent.BUTTON3) {
						new BugListPopupMenu((BugItem)(itemList.getModel().getElementAt(index))).show((JComponent)e.getSource(), e.getX(), e.getY());
					}
				}
			};
		itemList.addMouseListener(mouseListener);

		ButtonPanel buttonPanel      = new ButtonPanel(isVerticalAlign);
		//{{{ New Item Button
		newButton = buttonPanel.createRolloverButton("bugitem.new");
		newButton.addActionListener(this);
		//}}}
		//{{{ Reply Item Button
		modifyButton = buttonPanel.createRolloverButton("bugitem.modify");
		modifyButton.addActionListener(this);
		//}}}
		buttonPanel.createRigidSpace();
		//{{{ Delete Item Button
		deleteButton = buttonPanel.createRolloverButton("bugitem.delete");
		deleteButton.addActionListener(this);
		//}}}

		if (isVerticalAlign) {
			add(buttonPanel, java.awt.BorderLayout.NORTH);
		} else {
			add(buttonPanel, java.awt.BorderLayout.WEST);
		}

		initNetworkCommunication();
	}
	//}}}

	//{{{ initNetworkCommunication
	/**
	 *  Initialize network communication for this panel. Should not be called until
	 *  client is connected to server
	 */
	public final void initNetworkCommunication() {
		if (ProColPlugin.getClient() != null && ProColPlugin.getClient().isConnected()) {
			ProColPlugin.getClient().getIMH().addObserver(this);
			listModel.addElement(jEdit.getProperty("procol.client.buglist.gettinglistinfo"));
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_BUG_ITEM_LIST, PacketFactory.MAX_PRIORITY);
			newButton.setEnabled(true);
			allUsers = ProColPlugin.getClient().getUserList();
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_PROJECT_USERS, PacketFactory.MAX_PRIORITY);
		}
	}
	//}}}

	//{{{ stopNetworkCommunication
	/**
	 *  Stops network communication for this panel. Should only be called when
	 *  client disconnects
	 */
	public void stopNetworkCommunication() {
		if (ProColPlugin.getClient() != null && !ProColPlugin.getClient().isConnected()) {
			ProColPlugin.getClient().getIMH().deleteObserver(this);
			listModel.clear();
			newButton.setEnabled(false);
		}
	}
	//}}}

	//{{{ Action/Observer Listeners
	//{{{ actionPerformed
	/**
	 *@param  evt  The action event
	 */
	public void actionPerformed(java.awt.event.ActionEvent evt) {
		String actionCommand  = evt.getActionCommand();
		selectedItems = itemList.getSelectedValues();
		if (actionCommand.equals("bugitem.new")) {
			new BugItemComposer(allUsers);
		} else if (actionCommand.equals("bugitem.modify")) {
			new BugItemComposer((BugItem)(selectedItems[0]), allUsers);
		} else if (actionCommand.equals("bugitem.delete")) {
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.BUG_ITEM_DELETE, ((BugItem)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
		}
	}
	//}}}

	//{{{ valueChanged(ListSelectionEvent e)
	/**
	 *@param  e  The ListSelection event
	 */
	public void valueChanged(ListSelectionEvent e) {
		if (!e.getValueIsAdjusting()) {
			selectedItems = itemList.getSelectedValues();
			if (selectedItems.length == 1) {
				ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_BUG_ITEM, ((BugItem)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
				modifyButton.setEnabled(true);
				deleteButton.setEnabled(true);
			} else if (selectedItems.length == 0) {
				modifyButton.setEnabled(false);
				deleteButton.setEnabled(false);
			}
		}
	}
	//}}}

	//{{{ update
	/**
	 *@param  o    Observable that called the update
	 *@param  arg  Object argument
	 */
	public void update(Observable o, Object arg) {
		if (arg instanceof ProColIncomingMessage) {
			ProColIncomingMessage message  = (ProColIncomingMessage)arg;
			switch (message.requestCode) {
				case RequestType.NEW_BUG_ITEM:
					//{{{
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_BUG_ITEM_LIST, PacketFactory.MAX_PRIORITY);
					break;
				//}}}
				case RequestType.BUG_ITEM_LIST:
					//{{{
					Properties itemProps           = new Properties();
					try {
						itemProps.load(new ByteArrayInputStream(message.data));
					} catch (IOException ioe) {
						System.out.println(ioe);
					}

					String current                 = itemProps.getProperty("current", "0");
					listModel.clear();
					while (!current.equals("0")) {
						listModel.addElement(
								new BugItem(current,
								itemProps.getProperty(current + ".subject", "(No Subject)"),
								itemProps.getProperty(current + ".duedate", "None"),
								itemProps.getProperty(current + ".priority", "5"),
								itemProps.getProperty(current + ".complete", "0"),
								itemProps.getProperty(current + ".assigned", "null"))
								);

						current = itemProps.getProperty(current + ".previous", "0");
					}
					if (listModel.isEmpty()) {
						messageDisplayEditorPane.setText(jEdit.getProperty("procol.client.buglist.nomessages"));
					}
					break;
				//}}}
				case RequestType.BUG_ITEM:
					//{{{
					if (selectedItems.length == 1) {
						BugItem tempItem  = ((BugItem)(selectedItems[0]));
						tempItem.description = new String(message.data);
						messageDisplayEditorPane.setText("<font size=\"+1\"><b>" + tempItem.subject + "</b></font><br><b>Assigned to: " + tempItem.assignedTo + "<br>Priority: " + tempItem.priority + "<br>Complete: " + tempItem.complete + "%<br>Due: " + tempItem.dueDate + "</b><hr>" + tempItem.description);
					}
					break;
				//}}}
				case RequestType.BUG_ITEM_ADD_FAILED:
					//{{{
					break;
				//}}}
				case RequestType.BUG_ITEM_DELETE_FAILED:
					//{{{
					break;
				//}}}
				case RequestType.GET_PROJECT_USERS:
					//{{{
					allUsers = ProColUtils.dataStringToVector(message.data);
					break;
				//}}}
				default:
			}
		}
	}
	//}}}

	//{{{ finalize
	/**
	 *  Overrides finalize to delete this Observer from IncomingMessageHandler
	 *
	 *@exception  Throwable  Throws Throwable exceptions
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		ProColPlugin.getClient().getIMH().deleteObserver(this);
	}
	//}}}
	//}}}

	private class BugListPopupMenu extends JPopupMenu {
		//{{{ Data members
		private MenuHandler menuHandler  = new MenuHandler();
		private BugItem itemSelected;
		//}}}

		//{{{ Constructors
		/**
		 *  Constructor for the BugListPopupMenu object
		 *
		 *@param  itemSelectedIn  The currently selected Bug Item
		 */
		public BugListPopupMenu(BugItem itemSelectedIn) {
			itemSelected = itemSelectedIn;
			add(createMenuItem("new"));
			add(createMenuItem("modify"));
			add(createMenuItem("delete"));
		}
		//}}}

		//{{{ Misc. Methods
		//{{{ createMenuItem
		private JMenuItem createMenuItem(String name) {
			JMenuItem tempMenuItem  = new JMenuItem(jEdit.getProperty("procol.client." + name + ".label"));
			tempMenuItem.setActionCommand(name);
			tempMenuItem.addActionListener(menuHandler);
			return tempMenuItem;
		}
		//}}}
		//}}}

		//{{{ ActionHandler class
		private class MenuHandler implements ActionListener {
			//{{{ actionPerformed
			/**
			 *  Handles menu actions
			 *
			 *@param  evt  event to handle
			 */
			public void actionPerformed(ActionEvent evt) {
				String actionCommand  = evt.getActionCommand();
				if (actionCommand.equals("new")) {
					new BugItemComposer(allUsers);
				} else if (actionCommand.equals("modify")) {
					new BugItemComposer(((BugItem)(selectedItems[0])), allUsers);
				} else if (actionCommand.equals("delete")) {
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.BUG_ITEM_DELETE, ((BugItem)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
				}
			} //}}}
		}
		//}}}
	}
}

