//{{{ GPL Notice
/*
 *  CheckInPanel.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.ProColPlugin;
import com.enderak.procol.client.model.*;
import com.enderak.procol.client.net.*;
import com.enderak.procol.common.gui.*;
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.*;
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.gui.*;
//}}}

/**
 *  Displays a list of files waiting to be checked in
 *
 *@author    Justin Dieters
 */
public class CheckInPanel extends GenericUploadPanel {
//{{{ Data Members
	protected static Hashtable waitingFiles      = new Hashtable();
	protected static DefaultListModel listModel  = new DefaultListModel();
	private RolloverButton checkinButton;
//}}}

//{{{ Constructors
	/**
	 *  Constructor for the CheckInPanel object
	 *
	 *@param  view      jEdit View, specified by jEdit
	 *@param  position  position of dockable, specified by jEdit
	 */
	public CheckInPanel(View view, String position) {
		super(view, position, "checkin");
		fileList.setModel(listModel);

		ButtonPanel buttonPanel  = new ButtonPanel(isVerticalAlign);
		//{{{ Check In Button
		checkinButton = buttonPanel.createRolloverButton("checkin.file");
		checkinButton.addActionListener(this);
		//}}}

		if (isVerticalAlign) {
			add(buttonPanel, java.awt.BorderLayout.NORTH);
		} else {
			add(buttonPanel, java.awt.BorderLayout.WEST);
		}
	}

//{{{ Misc methods
	//{{{ addFile
	/**
	 *  Adds a file to the file list
	 *
	 *@param  fileIn  the file
	 *@param  destIn  the destination path
	 */
	public void addFile(ProColFile fileIn, URI destIn) {
		listModel.addElement(new DownloadFile(fileIn, destIn));
		fileList.setModel(listModel);
		fileList.setSelectedIndex(0);
	}
	//}}}
	//{{{ removeFile
	/**
	 *  Removes a file from the file list
	 *
	 *@param  fileIn  The file to remove
	 */
	public void removeFile(DownloadFile fileIn) {
		listModel.removeElement(fileIn);
		if (listModel.isEmpty()) {
			ProColPlugin.getClient().getIMH().deleteObserver(this);
			jEdit.getActiveView().getDockableWindowManager().hideDockableWindow("procol.client.dockable.checkin");
		}
	}
	//}}}
//}}}

//{{{ Action/Observer Listeners
	//{{{ actionPerformed
	/**
	 *@param  evt  The action event
	 */
	public void actionPerformed(java.awt.event.ActionEvent evt) {
		String actionCommand  = evt.getActionCommand();
		if (actionCommand.equals("checkin.file")) {
			newMajor = ((Integer)major.getValue()).intValue();
			newMinor = ((Integer)minor.getValue()).intValue();
			newRevision = ((Integer)revision.getValue()).intValue();
			changeLog = changeLogTextArea.getText();
			ProColPlugin.getClient().getIMH().addObserver(this);
			for (int i = 0; i < selectedFiles.length; i++) {
				DownloadFile downloadFile  = (DownloadFile)selectedFiles[i];
				ProColFile theFile         = downloadFile.file;
				int[] fileVersion          = theFile.getVersionAsArray();
				URI filePath               = downloadFile.destURI;
				System.out.println("||||| " + filePath);
				waitingFiles.put(filePath, downloadFile);
				if (incrementRadioButton.isSelected()) {
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.CHECK_IN_FILE_INFO, filePath + "\n" + (newMajor + fileVersion[0]) + "." + (newMinor + fileVersion[1]) + "." + (newRevision + fileVersion[2]) + "\n" + changeLog, PacketFactory.NORMAL_PRIORITY);
				} else if (absoluteRadioButton.isSelected()) {
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.CHECK_IN_FILE_INFO, filePath + "\n" + newMajor + "." + newMinor + "." + newRevision + "\n" + changeLog, PacketFactory.NORMAL_PRIORITY);
				} else if (noChangeRadioButton.isSelected()) {
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.CHECK_IN_FILE_INFO, filePath + "\n" + fileVersion[0] + "." + fileVersion[1] + "." + fileVersion[2] + "\n" + changeLog, PacketFactory.NORMAL_PRIORITY);
				}
			}
		}
		validateVersionSpinners();
	}
	//}}}

	//{{{ valueChanged(ListSelectionEvent e)
	/**
	 *@param  e  The selection event
	 */
	public void valueChanged(ListSelectionEvent e) {
		selectedFiles = fileList.getSelectedValues();
		if (selectedFiles.length == 1) {
			checkinButton.setEnabled(true);
			newInfoPanel.setBorder(new TitledBorder(((DownloadFile)selectedFiles[0]).file.getName() + " v." + ((DownloadFile)selectedFiles[0]).file.getVersion()));
		} else if (selectedFiles.length > 1) {
			checkinButton.setEnabled(true);
			newInfoPanel.setBorder(new TitledBorder("(" + selectedFiles.length + " " + jEdit.getProperty("procol.label.multiplefilesselected")));
		} else {
			checkinButton.setEnabled(false);
			newInfoPanel.setBorder(new TitledBorder("(" + jEdit.getProperty("procol.label.nofilesselected") + ")"));
		}

		validateVersionSpinners();
	}
	//}}}

	//{{{ update
	/**
	 *@param  o    Observable that called the update
	 *@param  arg  Object argument
	 */
	public void update(Observable o, Object arg) {
		if (arg instanceof ProColIncomingMessage) {
			ProColIncomingMessage message  = (ProColIncomingMessage)arg;
			URI filePath;
			DownloadFile tempFile;
			switch (message.requestCode) {
				case RequestType.CHECK_IN_FILE_INFO_OK:
					//{{{
					filePath = URI.create(new String(message.data));
					// System.out.println("Server responded info OK, server is now waiting for file: " + filePath);
					tempFile = (DownloadFile)waitingFiles.get(filePath);
					// System.out.println("Found file: " + tempFile.toURI());
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.CHECK_IN_FILE_WRITE, filePath, tempFile.file, PacketFactory.MIN_PRIORITY);
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_WRITE_OK:
					//{{{
					filePath = URI.create(new String(message.data));
					tempFile = (DownloadFile)waitingFiles.get(filePath);
					waitingFiles.remove(tempFile); // remove after confirmation
					removeFile(tempFile);
					break;
				//}}}
				case RequestType.CHECK_IN_ALREADY_CHECKED_IN:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.checkin.title"), jEdit.getProperty("procol.client.error.alreadycheckedin"));
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_NAME_FAILED:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.checkin.title"), jEdit.getProperty("procol.client.error.filename"));
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_VERSION_NO_OVERWRITE:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.checkin.title"), jEdit.getProperty("procol.client.error.versionoverwrite"));
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_VERSION_OLD:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.checkin.title"), jEdit.getProperty("procol.client.error.versionold"));
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_VERSION_IO_ERROR:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.checkin.title"), jEdit.getProperty("procol.client.error.versionioerror"));
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_NOT_FOUND:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.checkin.title"), jEdit.getProperty("procol.client.error.filenotfound"));
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_IO_ERROR:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.upload.title"), jEdit.getProperty("procol.client.error.fileioerror"));
					break;
				//}}}
				case RequestType.CHECK_IN_FILE_UNKNOWN_ERROR:
					//{{{
					ProColPlugin.getClient().getIMH().deleteObserver(this);
					ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.checkin.title"), jEdit.getProperty("procol.client.error.undefined"));
					break;
				//}}}
				default:
			}
		}
	}
	//}}}

	/**
	 *@exception  Throwable  Description of the Exception
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		ProColPlugin.getClient().getIMH().deleteObserver(this);
	}
	//}}}
//}}}
}

