//{{{ GPL Notice
/*
 *  FileInfoFrame.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.ProColPlugin;
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import java.awt.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.*;
import org.gjt.sp.jedit.jEdit;
//}}}

/**
 *  Displays information for a file
 *
 *@author    Justin Dieters
 */
public class FileInfoFrame extends JFrame {
	private FileInfoPanel panel;

	//{{{ Constructor
	/**
	 *  Constructor for the FileInfoFrame object
	 *
	 *@param  fileIn  Description of the Parameter
	 */
	public FileInfoFrame(ProColFile fileIn) {
		Toolkit kit           = Toolkit.getDefaultToolkit();
		Dimension screenSize  = kit.getScreenSize();

		this.setTitle(jEdit.getProperty("procol.client.fileinfo.title"));
		this.getContentPane().add(panel = new FileInfoPanel(this, fileIn));
		this.setSize(500, 400);
		this.setLocation((screenSize.width / 2) - (this.getWidth() / 2), (screenSize.height / 2) - (this.getHeight() / 2));
		this.setVisible(true);
	}
	//}}}

	//{{{ dispose
	/**  Dispose of the frame */
	public void dispose() {
		super.dispose();
		ProColPlugin.getClient().getIMH().deleteObserver(panel);
	}
	//}}}

	private class FileInfoPanel extends JPanel implements Observer {

		//{{{ Data Members
		private JLabel checkedOutInfoLabel;
		private JPanel fileInfoPanel, leftPanel, rightPanel;
		private JEditorPane historyEditPane;
		private JScrollPane historyScrollPane;
		private ProColFile file;
		private URI projectFilesURI            = ProColPlugin.getClient().getProject().getProjectFilesURI();
		//}}}

		//{{{ Constructors
		/**
		 *  Constructor for the FileInfoPanel object
		 *
		 *@param  parentFrame  Description of the Parameter
		 *@param  fileIn       Description of the Parameter
		 */
		public FileInfoPanel(final FileInfoFrame parentFrame, ProColFile fileIn) {
			ProColPlugin.getClient().getIMH().addObserver(this);
			this.file = fileIn;
			this.setBorder(new TitledBorder(null, file.getName() + " v." + file.getVersion(), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new Font("Dialog", Font.BOLD, 16)));
			this.setLayout(new BorderLayout());

			//{{{ Button Panel
			JButton okButton    = new JButton(jEdit.getProperty("procol.label.ok"));
			okButton.addActionListener(
				new java.awt.event.ActionListener() {
					public void actionPerformed(java.awt.event.ActionEvent evt) {
						parentFrame.dispose();
					}
				});
			JPanel buttonPanel  = new JPanel();
			buttonPanel.add(okButton, BorderLayout.CENTER);
			//}}}

			//{{{ top file info panel
			leftPanel = new JPanel();
			leftPanel.setLayout(new BoxLayout(leftPanel, BoxLayout.Y_AXIS));
			leftPanel.add(new JLabel(jEdit.getProperty("procol.client.fileinfo.checkedout")));
			leftPanel.add(new JLabel(jEdit.getProperty("procol.client.fileinfo.changelog")));

			rightPanel = new JPanel();
			rightPanel.setLayout(new javax.swing.BoxLayout(rightPanel, BoxLayout.Y_AXIS));
			checkedOutInfoLabel = new JLabel(file.getOwner());
			rightPanel.add(checkedOutInfoLabel);
			rightPanel.add(new JLabel(""));

			fileInfoPanel = new JPanel();
			fileInfoPanel.setLayout(new javax.swing.BoxLayout(fileInfoPanel, BoxLayout.X_AXIS));
			fileInfoPanel.add(leftPanel);
			fileInfoPanel.add(Box.createHorizontalStrut(10));
			fileInfoPanel.add(rightPanel);
			//}}}

			historyScrollPane = new JScrollPane();
			historyEditPane = new JEditorPane();
			historyEditPane.setContentType("text/html");
			historyEditPane.setText(jEdit.getProperty("procol.client.fileinfo.gettinginfo"));
			historyEditPane.setEditable(false);
			historyScrollPane.setViewportView(historyEditPane);

			this.add(fileInfoPanel, java.awt.BorderLayout.NORTH);
			this.add(historyScrollPane, java.awt.BorderLayout.CENTER);
			this.add(buttonPanel, java.awt.BorderLayout.SOUTH);

			ProColPlugin.getClient().getFileHistory(projectFilesURI.relativize(projectFilesURI.resolve(file.toURI())).toString());
		}
		//}}}

		//{{{ update
		/**
		 *@param  o    Description of the Parameter
		 *@param  arg  Description of the Parameter
		 */
		public void update(Observable o, Object arg) {
			if (arg instanceof ProColIncomingMessage) {
				ProColIncomingMessage message  = (ProColIncomingMessage)arg;
				if (message.requestCode == RequestType.FILE_HISTORY) {
					Properties fileProps        = new Properties();
					try {
						fileProps.load(new ByteArrayInputStream(message.data));
					} catch (IOException e) {
						ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.fileinfo.title"), jEdit.getProperty("procol.client.error.fileinfo.propertyload"));
					}
					//{{{ generate history list
					StringBuffer historyBuffer  = new StringBuffer(message.data.length);
					String currentVersion       = fileProps.getProperty("current");
					while (!currentVersion.equals("null")) {
						historyBuffer.append("<font size=\"+1\"><b>").append(currentVersion).append("</b></font><br>");
						historyBuffer.append("<b>").append(fileProps.getProperty(currentVersion + ".date")).append("</b>");
						if (!fileProps.getProperty(currentVersion + ".user").equals("")) {
							historyBuffer.append(" by: <b>").append(fileProps.getProperty(currentVersion + ".user")).append("</b>");
						}
						historyBuffer.append("<br>").append(fileProps.getProperty(currentVersion + ".changelog")).append("<br><br><br>");
						currentVersion = fileProps.getProperty(currentVersion + ".previous");
					}
					historyEditPane.setText(historyBuffer.toString());
					//}}}
				}
			}
		}
		//}}}
	}
}
