//{{{ GPL Notice
/*
 *  PrivateMessagePanel.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.*;
import com.enderak.procol.client.model.*;
import com.enderak.procol.client.net.*;
import com.enderak.procol.common.gui.*;
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.gui.*;
//}}}

/**
 *  Displays a dialog with a list of messages and panel to display messages
 *
 *@author    Justin Dieters
 */
public class PrivateMessagePanel extends GenericListPanel {
	//{{{ Data Members
	protected static DefaultListModel listModel  = new DefaultListModel();
	private RolloverButton newButton, replyButton, deleteButton;
	private Vector allUsers;
	//}}}

	//{{{ Constructors
	/**
	 *  Constructor for the CheckInPanel object
	 *
	 *@param  view      View associated with panel, as specified from jEdit
	 *@param  position  Position of panel, as specified from jEdit
	 */
	// public PrivateMessagePanel(PrivateMessageFrame parent) {
	public PrivateMessagePanel(View view, String position) {
		super(view, position, "privatemessages");
		listModel.clear();
		itemList.setModel(listModel);

		MouseListener mouseListener  =
			new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					int index  = itemList.locationToIndex(e.getPoint());
					itemList.setSelectedIndex(index);
					if (e.getButton() == MouseEvent.BUTTON3) {
						new PrivateMessagePopupMenu((PrivateMessage)(itemList.getModel().getElementAt(index))).show((JComponent)e.getSource(), e.getX(), e.getY());
					}
				}
			};
		itemList.addMouseListener(mouseListener);

		ButtonPanel buttonPanel      = new ButtonPanel(isVerticalAlign);
		//{{{ New Message Button
		newButton = buttonPanel.createRolloverButton("privatemessages.new");
		newButton.addActionListener(this);
		//}}}
		//{{{ Reply Message Button
		replyButton = buttonPanel.createRolloverButton("privatemessages.reply");
		replyButton.addActionListener(this);
		//}}}
		buttonPanel.createRigidSpace();
		//{{{ Delete Message Button
		deleteButton = buttonPanel.createRolloverButton("privatemessages.delete");
		deleteButton.addActionListener(this);
		//}}}

		if (isVerticalAlign) {
			add(buttonPanel, java.awt.BorderLayout.NORTH);
		} else {
			add(buttonPanel, java.awt.BorderLayout.WEST);
		}

		initNetworkCommunication();
	}

	//{{{ initNetworkCommunication
	/**  Enables features that require a network connection */
	public final void initNetworkCommunication() {
		if (ProColPlugin.getClient() != null && ProColPlugin.getClient().isConnected()) {
			ProColPlugin.getClient().getIMH().addObserver(this);
			listModel.addElement(jEdit.getProperty("procol.client.privatemessages.gettinglistinfo"));
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_PRIVATE_MESSAGE_LIST, PacketFactory.MAX_PRIORITY);
			newButton.setEnabled(true);
			allUsers = ProColPlugin.getClient().getUserList();
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_PROJECT_USERS, PacketFactory.MAX_PRIORITY);
		}
	}
	//}}}

	//{{{ stopNetworkCommunication
	/**  Disables features that require a network connection */
	public void stopNetworkCommunication() {
		if (ProColPlugin.getClient() != null && !ProColPlugin.getClient().isConnected()) {
			ProColPlugin.getClient().getIMH().deleteObserver(this);
			listModel.clear();
			newButton.setEnabled(false);
		}
	}
	//}}}

	//{{{ Action/Observer Listeners
	//{{{ actionPerformed
	/**
	 *@param  evt  The ActionEvent event
	 */
	public void actionPerformed(java.awt.event.ActionEvent evt) {
		String actionCommand  = evt.getActionCommand();
		if (actionCommand.equals("privatemessages.new")) {
			new PrivateMessageComposer("null", "null", "(No Subject)", allUsers);
		} else if (actionCommand.equals("privatemessages.reply")) {
			new PrivateMessageComposer(((PrivateMessage)(selectedItems[0])).index, ((PrivateMessage)(selectedItems[0])).sender, "Re: " + ((PrivateMessage)(selectedItems[0])).subject, allUsers);
		} else if (actionCommand.equals("privatemessages.delete")) {
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.PRIVATE_MESSAGE_DELETE, ((PrivateMessage)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
		}
	}
	//}}}

	//{{{ valueChanged(ListSelectionEvent e)
	/**
	 *@param  e  The ListSelectionEvent event
	 */
	public void valueChanged(ListSelectionEvent e) {
		if (!e.getValueIsAdjusting()) {
			selectedItems = itemList.getSelectedValues();
			if (selectedItems.length == 1) {
				ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_PRIVATE_MESSAGE, ((PrivateMessage)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
				replyButton.setEnabled(true);
				deleteButton.setEnabled(true);
			} else if (selectedItems.length == 0) {
				replyButton.setEnabled(false);
				deleteButton.setEnabled(false);
			}
		}
	}
	//}}}

	//{{{ update
	/**
	 *@param  o    Observable that called the update
	 *@param  arg  Object argument
	 */
	public void update(Observable o, Object arg) {
		if (arg instanceof ProColIncomingMessage) {
			ProColIncomingMessage message  = (ProColIncomingMessage)arg;
			switch (message.requestCode) {
				case RequestType.NEW_PRIVATE_MESSAGE:
					//{{{
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_PRIVATE_MESSAGE_LIST, PacketFactory.MAX_PRIORITY);
					break;
				//}}}
				case RequestType.PRIVATE_MESSAGE_LIST:
					//{{{
					Properties privateMessageProps = new Properties();
					try {
						privateMessageProps.load(new ByteArrayInputStream(message.data));
					} catch (IOException ioe) {
						System.out.println(ioe);
					}

					String current                 = privateMessageProps.getProperty("current", "0");
					listModel.clear();
					while (!current.equals("0")) {
						listModel.addElement(new PrivateMessage(current, privateMessageProps.getProperty(current + ".date", "(No Date)"), privateMessageProps.getProperty(current + ".subject", "(No Subject)"), privateMessageProps.getProperty(current + ".sender", "(No Sender)")));
						current = privateMessageProps.getProperty(current + ".previous", "0");
					}
					if (listModel.isEmpty()) {
						messageDisplayEditorPane.setText(jEdit.getProperty("procol.client.privatemessages.nomessages"));
					}
					break;
				//}}}
				case RequestType.PRIVATE_MESSAGE:
					//{{{
					if (selectedItems.length == 1) {
						PrivateMessage tempMessage  = ((PrivateMessage)(selectedItems[0]));
						messageDisplayEditorPane.setText("<font size=\"+1\"><b>" + tempMessage.subject + "</b></font><br><b>From: " + tempMessage.sender + "<br>" + tempMessage.date + "</b><hr>" + new String(message.data));
					}
					break;
				//}}}
				case RequestType.PRIVATE_MESSAGE_USER_NOT_EXIST:
					//{{{
					break;
				//}}}
				case RequestType.PRIVATE_MESSAGE_ADD_FAILED:
					//{{{
					break;
				//}}}
				case RequestType.PRIVATE_MESSAGE_DELETE_FAILED:
					//{{{
					break;
				//}}}
				case RequestType.GET_PROJECT_USERS:
					//{{{
					allUsers = ProColUtils.dataStringToVector(message.data);
					break;
				//}}}
				default:
			}
		}
	}
	//}}}

	/**
	 *@exception  Throwable  Description of the Exception
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		ProColPlugin.getClient().getIMH().deleteObserver(this);
	}
	//}}}
	//}}}

	/**
	 *  Generates a pop-up menu for the message list
	 *
	 *@author    Justin Dieters
	 */
	private class PrivateMessagePopupMenu extends JPopupMenu {
//{{{ Data members
		private MenuHandler menuHandler      = new MenuHandler();
		private PrivateMessage itemSelected;
//}}}

//{{{ Constructors
		/**
		 *  Constructor for the PrivateMessagePopupMenu object
		 *
		 *@param  itemSelectedIn  The currently selected message
		 */
		public PrivateMessagePopupMenu(PrivateMessage itemSelectedIn) {
			itemSelected = itemSelectedIn;
			add(createMenuItem("new"));
			add(createMenuItem("reply"));
			add(createMenuItem("delete"));
		}
//}}}

//{{{ Misc. Methods
		//{{{ createMenuItem
		private JMenuItem createMenuItem(String name) {
			JMenuItem tempMenuItem  = new JMenuItem(jEdit.getProperty("procol.client." + name + ".label"));
			tempMenuItem.setActionCommand(name);
			tempMenuItem.addActionListener(menuHandler);
			return tempMenuItem;
		}
		//}}}
//}}}

//{{{ ActionHandler class
		private class MenuHandler implements ActionListener {
			//{{{ actionPerformed
			/**
			 *@param  evt  Description of the Parameter
			 */
			public void actionPerformed(ActionEvent evt) {
				String actionCommand  = evt.getActionCommand();
				if (actionCommand.equals("new")) {
					new PrivateMessageComposer("null", "null", "(No Subject)", allUsers);
				} else if (actionCommand.equals("reply")) {
					new PrivateMessageComposer(((PrivateMessage)(selectedItems[0])).index, ((PrivateMessage)(selectedItems[0])).sender, "Re: " + ((PrivateMessage)(selectedItems[0])).subject, allUsers);
				} else if (actionCommand.equals("delete")) {
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.PRIVATE_MESSAGE_DELETE, ((PrivateMessage)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
				}
			}
			//}}}
		}
//}}}
	}
}

