//{{{ GPL Notice
/*
 *  ProColClientDockale.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.*;
import com.enderak.procol.client.model.ProColClientProject;
import com.enderak.procol.common.model.ProColFile;
import com.enderak.procol.common.net.*;
import com.enderak.procol.server.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.URI;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.gui.*;
import org.gjt.sp.jedit.io.*;
import org.gjt.sp.jedit.msg.*;
//}}}

/**
 *  Main ProCol user interface class. Responsible for setting up plugin
 *  interface, handling EBMessages, and displaying error and informational
 *  messages.
 *
 *@author    Justin Dieters
 */
public class ProColClientDockable extends JPanel implements EBComponent {
//{{{ Data Members
	/**  The panel that contains the graphical interface items */
	public static ProColPanel proColPanel;
	private View view;
	private String position, currentlySavingBuffer;
	private boolean floating;
	private boolean isVerticalAlign;
	//}}}

//{{{ Constructors
	/**
	 *  Constructor for the ProCol object, called by jEdit using the command
	 *  definition in dockables.xml
	 *
	 *@param  view      Automatically passed by jEdit when dockable plugin window
	 *      is created
	 *@param  position  Automatically passed by jEdit when dockable plugin window
	 *      is created; this contains the docking position of this panel. This is
	 *      used by this class to determine whether to set up interface with a
	 *      horizonal or vertical layout.
	 */
	public ProColClientDockable(View view, String position) {
		super();

		this.view = view;
		this.position = position;
		this.floating = position.equals(DockableWindowManager.FLOATING);
		this.setLayout(new BorderLayout());

		// define size, if floating
		if (floating) {
			this.setPreferredSize(new Dimension(250, 500));
		}

		// determine if panels should have a vertical or horizontal layout
		if (position.equals(DockableWindowManager.TOP) || position.equals(DockableWindowManager.BOTTOM)) {
			this.isVerticalAlign = false;
		} else {
			this.isVerticalAlign = true;
		}

		proColPanel = new ProColPanel(isVerticalAlign);
		this.add(proColPanel, java.awt.BorderLayout.CENTER);
	}
//}}}

//{{{ Message display methods
	//{{{ displayError
	/**
	 *  Displays an error message box
	 *
	 *@param  titleIn  Title of error box
	 *@param  errorIn  Error message to display
	 */
	public static void displayError(String titleIn, String errorIn) {
		JOptionPane.showMessageDialog(null, errorIn, titleIn, JOptionPane.ERROR_MESSAGE);
	}
	//}}}

	//{{{ displayInfo
	/**
	 *  Displays an informational message box
	 *
	 *@param  titleIn    Title of message box
	 *@param  messageIn  Informational message to display
	 */
	public static void displayInfo(String titleIn, String messageIn) {
		JOptionPane.showMessageDialog(null, messageIn, titleIn, JOptionPane.INFORMATION_MESSAGE);
	}
	//}}}
//}}}

//{{{ EditBus methods
	//{{{ addNotify
	/**
	 *  Adds this dockable to the EditBus notification list <br>
	 *  <br>
	 *  Also adds the ProColPanel to the ProColClient Observers list. This is
	 *  required because jEdit recreates the plugin's panel whenver it is reloaded
	 *  or the Docking posistion changes.
	 */
	public void addNotify() {
		super.addNotify();
		EditBus.addToBus(this);
		ProColPlugin.getClient().addObserver(proColPanel);
		ProColPlugin.getClient().getMessageFactory().addObserver(proColPanel);
		ProColPlugin.getClient().forceNotify();
	}
	//}}}

	//{{{ removeNotify
	/**
	 *  Removes this dockable from the EditBus notification list <br>
	 *  <br>
	 *  Also removes the ProColPanel from the ProColClient Observers list. This is
	 *  required because jEdit recreates the plugin's panel whenver it is reloaded
	 *  or the Docking posistion changes.
	 */
	public void removeNotify() {
		super.removeNotify();
		EditBus.removeFromBus(this);
		ProColPlugin.getClient().deleteObserver(proColPanel);
		ProColPlugin.getClient().getMessageFactory().deleteObserver(proColPanel);
		ProColPlugin.getClient().forceNotify();
	}
	//}}}

	//{{{ handleMessage
	/**
	 *  Handles EBMessages (EditBus Messages) from jEdit
	 *
	 *@param  message  Message from jEdit's EditBus to handle
	 */
	public void handleMessage(EBMessage message) {
		// gets all EBMessages from jEdit and filters them to the appropriate parts of ProCol when applicable
		// System.out.println("got EBMessage in ProColClientDockable " + message.toString());
		Buffer theBuffer;
		if (ProColPlugin.getClient().isConnected() && message instanceof BufferUpdate && ProColPlugin.getClient().getProject().isOpenBuffer(theBuffer = ((BufferUpdate)message).getBuffer())) {
			// if message is BufferUpdate message and Buffer was opened via ProCol
			Object what  = ((BufferUpdate)message).getWhat();
			if (what == BufferUpdate.CLOSED) {
				handleClosedBuffer(((BufferUpdate)message).getBuffer());
			} else if (what == BufferUpdate.SAVING) {
				currentlySavingBuffer = ((BufferUpdate)message).getBuffer().getPath();
			} else if (what == BufferUpdate.SAVED) {
				if (theBuffer.getPath().equals(currentlySavingBuffer)) {
					handleSavedBuffer(((BufferUpdate)message).getBuffer());
				} else {
					displayInfo("File saved to alternate location", new File(currentlySavingBuffer).getName() + " was saved to alternate location,\nChecking original back with server.");
				}
			}
		}
	}
	//}}}
//}}}

//{{{ Misc methods
	//{{{ openProject
	/**
	 *@param  projectName  Sets the project name in the ProCol window
	 */
	public void openProject(String projectName) {
		proColPanel.projectPanel.setBorder(new TitledBorder(projectName));
	}
	//}}}

	//{{{ handleClosedBuffer
	private void handleClosedBuffer(Buffer buffer) {
		ProColClientProject project  = ProColPlugin.getClient().getProject();
		ProColFile bufferFile        = project.getFileForBuffer(buffer);
		URI projectFilesURI          = project.getProjectFilesURI();
		String checkInOnClose        = jEdit.getProperty("options.procol.client.checkinonclose", "modified");
		if (checkInOnClose.equals("yes")) {
			DockableWindowManager dockableWindowManager  = jEdit.getActiveView().getDockableWindowManager();
			dockableWindowManager.showDockableWindow("procol.client.dockable.checkin");
			((CheckInPanel)(dockableWindowManager.getDockableWindow("procol.client.dockable.checkin"))).addFile(bufferFile, projectFilesURI.relativize(projectFilesURI.resolve(bufferFile.toURI())));
		} else if (checkInOnClose.equals("modified") && project.isModifiedBuffer(buffer)) {
			DockableWindowManager dockableWindowManager  = jEdit.getActiveView().getDockableWindowManager();
			dockableWindowManager.showDockableWindow("procol.client.dockable.checkin");
			((CheckInPanel)(dockableWindowManager.getDockableWindow("procol.client.dockable.checkin"))).addFile(bufferFile, projectFilesURI.relativize(projectFilesURI.resolve(bufferFile.toURI())));
		} else if (checkInOnClose.equals("modified") && !project.isModifiedBuffer(buffer)) {
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.CHECK_IN_FILE, projectFilesURI.relativize(projectFilesURI.resolve(bufferFile.toURI())).toString(), PacketFactory.NORMAL_PRIORITY);
		} else if (checkInOnClose.equals("ask")) {
			if (JOptionPane.showConfirmDialog(null, "Check in " + buffer.getName() + "?", "Buffer Closed", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE) == JOptionPane.YES_OPTION) {
				DockableWindowManager dockableWindowManager  = jEdit.getActiveView().getDockableWindowManager();
				dockableWindowManager.showDockableWindow("procol.client.dockable.checkin");
				((CheckInPanel)(dockableWindowManager.getDockableWindow("procol.client.dockable.checkin"))).addFile(bufferFile, projectFilesURI.relativize(projectFilesURI.resolve(bufferFile.toURI())));
			}
		}
		project.removeBuffer(buffer);
	}
	//}}}

	//{{{ handleSavedBuffer
	private void handleSavedBuffer(Buffer buffer) {
		ProColClientProject project  = ProColPlugin.getClient().getProject();
		URI projectFilesURI          = project.getProjectFilesURI();
		project.addModifiedBuffer(buffer);
		String updateOnSave          = jEdit.getProperty("options.procol.client.updateonsave", "no");
		String checkInOnSave         = jEdit.getProperty("options.procol.client.updateonsave", "no");
		ProColFile bufferFile        = project.getFileForBuffer(buffer);
		if (updateOnSave.equals("yes")) {
			ProColPlugin.getClient().updateFile(bufferFile);
		} else if (updateOnSave.equals("ask")) {
			if (JOptionPane.showConfirmDialog(null, "Upload changes to " + buffer.getName() + "?", "Buffer Saved", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE) == JOptionPane.YES_OPTION) {
				ProColPlugin.getClient().updateFile(bufferFile);
			}
		}
		if (checkInOnSave.equals("yes")) {
			DockableWindowManager dockableWindowManager  = jEdit.getActiveView().getDockableWindowManager();
			dockableWindowManager.showDockableWindow("procol.client.dockable.checkin");
			((CheckInPanel)(dockableWindowManager.getDockableWindow("procol.client.dockable.checkin"))).addFile(bufferFile, projectFilesURI.relativize(projectFilesURI.resolve(bufferFile.toURI())));
		} else if (checkInOnSave.equals("ask")) {
			if (JOptionPane.showConfirmDialog(null, "Check in " + buffer.getName() + "?", "Buffer Saved", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE) == JOptionPane.YES_OPTION) {
				DockableWindowManager dockableWindowManager  = jEdit.getActiveView().getDockableWindowManager();
				dockableWindowManager.showDockableWindow("procol.client.dockable.checkin");
				((CheckInPanel)(dockableWindowManager.getDockableWindow("procol.client.dockable.checkin"))).addFile(bufferFile, projectFilesURI.relativize(projectFilesURI.resolve(bufferFile.toURI())));
			}
		}
	}
	//}}}
//}}}
}
