//{{{ GPL Notice
/*
 *  ProColPanel.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
// from ProCol
import com.enderak.procol.*;
import com.enderak.procol.common.gui.*;
import com.enderak.procol.common.net.*;

// from Java:
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;

// from Swing:
import javax.swing.*;
import javax.swing.event.*;

// from jEdit:
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.gui.*;
import org.gjt.sp.jedit.io.*;
//}}}

/**
 *  The main graphical interface panel
 *
 *@author    Justin Dieters
 */
public class ProColPanel extends JPanel implements Observer { // progress bars
	private final static int NUM_BARS           = 2;
	private final static int PRIMARY            = 0;
	private final static int SECONDARY          = 1;
	//{{{ Data members
	// panels
	/**  The panel which contains the file tree and associated elements */
	public ProjectPanel projectPanel;
	/**  The panel which contains the user list and associated elements */
	public UserPanel userPanel;
	private ButtonPanel topButtonPanel;
	private JSplitPane projectSplitPane;
	private ProColProgressBar[] progressBar     = new ProColProgressBar[NUM_BARS];
	private JPanel progressPanel                = new JPanel();
	private int progressID, totalListenFor;
	private RolloverOnOffButton connectButton;
	private RolloverButton changeProjectButton, optionsButton, helpButton, projectInfoButton;
	//}}}

	//{{{ Constructor
	/**
	 *  Constructor for the ProColPanel
	 *
	 *@param  isVerticalAlign  True if vertically aligned (floating or docked
	 *      left/right), false otherwise
	 */
	public ProColPanel(boolean isVerticalAlign) {
		setLayout(new java.awt.BorderLayout());
		this.setAlignmentX(Component.CENTER_ALIGNMENT);

		topButtonPanel = new ButtonPanel(isVerticalAlign);
		projectPanel = new ProjectPanel(isVerticalAlign);
		userPanel = new UserPanel(isVerticalAlign);

		for (int i = 0; i < progressBar.length; i++) {
			progressBar[i] = new ProColProgressBar();
		}

		if (isVerticalAlign) {
			projectSplitPane = new JSplitPane(javax.swing.JSplitPane.VERTICAL_SPLIT, true, projectPanel, userPanel);
			add(topButtonPanel, java.awt.BorderLayout.NORTH);
			progressBar[PRIMARY].setOrientation(JProgressBar.HORIZONTAL);
			progressBar[SECONDARY].setOrientation(JProgressBar.HORIZONTAL);
			progressPanel.setLayout(new java.awt.BorderLayout());
			progressPanel.add(progressBar[PRIMARY], java.awt.BorderLayout.NORTH);
			progressPanel.add(progressBar[SECONDARY], java.awt.BorderLayout.SOUTH);
			progressBar[SECONDARY].setVisible(false);
			add(progressPanel, java.awt.BorderLayout.SOUTH);
		} else {
			projectSplitPane = new JSplitPane(javax.swing.JSplitPane.HORIZONTAL_SPLIT, true, projectPanel, userPanel);
			add(topButtonPanel, java.awt.BorderLayout.WEST);
			progressBar[PRIMARY].setOrientation(JProgressBar.VERTICAL);
			progressBar[SECONDARY].setOrientation(JProgressBar.VERTICAL);
			progressPanel.setLayout(new java.awt.BorderLayout());
			progressPanel.add(progressBar[PRIMARY], java.awt.BorderLayout.EAST);
			progressPanel.add(progressBar[SECONDARY], java.awt.BorderLayout.WEST);
			progressBar[SECONDARY].setVisible(false);
			add(progressPanel, java.awt.BorderLayout.EAST);
		}

		projectSplitPane.setOneTouchExpandable(true);
		projectSplitPane.setResizeWeight(0.5);
		projectSplitPane.setDividerLocation(0.75);
		add(projectSplitPane, java.awt.BorderLayout.CENTER);

		//{{{ Create buttons and define button actions
		// Connect / Disconnect
		connectButton = topButtonPanel.createRolloverOnOffButton("connect");
		connectButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.connectToServer();
				}
			});
		connectButton.setEnabled(true);

		topButtonPanel.createRigidSpace();

		// Project Info
		projectInfoButton = topButtonPanel.createButton("projectinfo");
		projectInfoButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.getProjectInfo();
				}
			});

		// Change Project
		changeProjectButton = topButtonPanel.createButton("changeproject");
		changeProjectButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.changeProject();
				}
			});

		topButtonPanel.createRigidSpace();

		// Options
		optionsButton = topButtonPanel.createButton("options");
		optionsButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					new org.gjt.sp.jedit.options.PluginOptions(
							JOptionPane.getFrameForComponent(ProColPanel.this),
							jEdit.getProperty("options.procol.client.buttonopen"));
				}
			});
		optionsButton.setEnabled(true);

		// Help
		helpButton = topButtonPanel.createButton("help");
		helpButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					new org.gjt.sp.jedit.help.HelpViewer(
							this.getClass().getResource(
							jEdit.getProperty("plugin.com.enderak.procol.ProColPlugin.docs")));
				}
			});
		helpButton.setEnabled(true);
	}
	//}}}

	//{{{ Progress Bar Methods
	/**  Start the default progress bar */
	public void startProgress() {
		progressBar[PRIMARY].startProgress();
	}


	/**
	 *  Start the default progress bar with a message
	 *
	 *@param  message  The message to display
	 */
	public void startProgress(String message) {
		progressBar[PRIMARY].startProgress(message);
	}


	/**  Stop the default progress bar */
	public void stopProgress() {
		progressBar[PRIMARY].stopProgress();
	}
	//}}}

	//{{{ Observer methods

	/**
	 *  Invoked from parent ProColClientDockable class.
	 *
	 *@param  o    the observable object
	 *@param  arg  an argument passed to the notifyObservers method<br>
	 *
	 *      <ul>
	 *        <li> null - updates everything</li>
	 *        <li> 'CONNECTION' - updates the connection status</li>
	 *        <li> 'USERLIST' - Updates the user list</li>
	 *        <li> 'PROJECT' - Updates the project list</li>
	 *        <li> ProColPacket - Increments the progress bar based on packet
	 *        coming in</li>
	 *        <li> ProColIncomingMessage - resets progress bar based on the new
	 *        message</li>
	 *      </ul>
	 *
	 */
	public void update(Observable o, Object arg) {
		if (arg == null || ((arg instanceof String) && ((String)arg).equals("CONNECTION"))) {
			this.setConnected(ProColPlugin.getClient().isConnected() && ProColPlugin.getClient().isAuthenticated());
			projectPanel.setConnected(ProColPlugin.getClient().isConnected() && ProColPlugin.getClient().isAuthenticated());
			userPanel.setConnected(ProColPlugin.getClient().isConnected() && ProColPlugin.getClient().isAuthenticated());
		}

		if (arg == null || ((arg instanceof String) && ((String)arg).equals("USERLIST"))) {
			// System.out.println("userList update");
			userPanel.setUserList(ProColPlugin.getClient().getUserList());
		}

		if (arg == null || ((arg instanceof String) && ((String)arg).equals("PROJECT"))) {
			// System.out.println("project update");
			projectPanel.setProject(ProColPlugin.getClient().getProject());
		}
		if (arg instanceof ProColPacket) {
			// System.out.println((ProColPacket)arg);
			updateProgress(PRIMARY, (ProColPacket)arg);
		}
		if (arg instanceof ProColIncomingMessage) {
			ProColIncomingMessage message  = (ProColIncomingMessage)arg;
			if (message.requestCode == totalListenFor) {
				progressBar[SECONDARY].incrementProgress();
				return;
			}
			switch (message.requestCode) {
				case RequestType.DOWNLOAD_FILE:
					//{{{
					progressBar[PRIMARY].setProgress(0, "");
					break;
				//}}}
				case RequestType.CHECK_OUT_FILE:
					//{{{
					progressBar[PRIMARY].setProgress(0, "");
					break;
				//}}}
				default:
			}
		}
	}
	//}}}

	//{{{ Modifier methods
	/**
	 *  Toggles button states based on if client is connected or not
	 *
	 *@param  isConnected  The new connected value; true if connected, false if not
	 *      connected
	 */
	public void setConnected(boolean isConnected) {
		connectButton.setOnOff(isConnected);
		changeProjectButton.setEnabled(isConnected);
		projectInfoButton.setEnabled(isConnected);
	}


	private void updateProgress(int barIndex, ProColPacket packetIn) {
		if (packetIn.dataRemaining == packetIn.dataSize) {
			// if packet is last packet
			if (packetIn.messageID == progressID) {
				progressBar[barIndex].stopProgress();
			}
		} else {
			// if packet is not the last packet
			if (packetIn.messageID != progressID) {
				// if progress not already started, start new progress
				progressID = packetIn.messageID;
				progressBar[barIndex].startProgress(0, packetIn.dataRemaining);
			}
			progressBar[barIndex].incrementProgress(packetIn.dataSize);
		}
	}


	private void startDownloadProgress(int numFiles, int downloadType, String progressBarText) {
		progressBar[SECONDARY].setVisible(true);
		progressBar[SECONDARY].startProgress(0, numFiles, progressBarText);
		totalListenFor = downloadType;
	}


	private void stopDownloadProgress() {
		progressBar[PRIMARY].stopProgress();
		progressBar[SECONDARY].setVisible(false);
		progressBar[SECONDARY].stopProgress();
		totalListenFor = RequestType.NULL;
	}
	//}}}
}

