//{{{ GPL Notice
/*
 *  ProjectPanel.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.ProColPlugin;
import com.enderak.procol.client.*;
import com.enderak.procol.client.gui.*;
import com.enderak.procol.client.model.*;
import com.enderak.procol.common.gui.*;
import com.enderak.procol.common.model.*;
import java.awt.BorderLayout;
import java.awt.event.*;
import java.net.URI;
import java.util.*;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.*;
import javax.swing.tree.*;
import org.gjt.sp.jedit.gui.RolloverButton;
import org.gjt.sp.jedit.jEdit;
//}}}

/**
 *  The panel which displays the file tree and associated elements
 *
 *@author    Justin Dieters
 */
public class ProjectPanel extends JPanel {
	//{{{ Data members
	private ButtonPanel projectButtonPanel;
	private JScrollPane projectScrollPane;
	private ProColTree projectTree;
	private DefaultMutableTreeNode projectFilesRootNode;
	private DefaultTreeModel treeModel;
	private RolloverButton snapshotButton, newFileButton, newDirButton, uploadFileButton, uploadDirButton;
	//}}}

	//{{{ Constructors
	/**
	 *  Constructor for the ProjectPanel object
	 *
	 *@param  isVerticalAlign  True if vertically aligned (floating or docked
	 *      left/right), false otherwise
	 */
	public ProjectPanel(boolean isVerticalAlign) {
		super();

		this.setLayout(new BorderLayout());
		this.setBorder(new TitledBorder(jEdit.getProperty("procol.client.project.toplabel")));

		projectButtonPanel = new ButtonPanel(isVerticalAlign);
		projectScrollPane = new JScrollPane();
		projectTree = new ProColTree();
		projectTree.setRootVisible(false);
		projectTree.setShowsRootHandles(true);
		projectTree.setDragEnabled(true);

		MouseListener mouseListener  =
			new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					if (ProColPlugin.getClient().isConnected() && ProColPlugin.getClient().isAuthenticated()) {
						TreePath selectedPath  = projectTree.getPathForLocation(e.getX(), e.getY());

						if (e.getButton() == MouseEvent.BUTTON3) {
							if (!projectTree.isPathSelected(selectedPath) && selectedPath != null) {
								// if path selected not currently selected, change selection to that path
								projectTree.setSelectionPath(selectedPath);
							}
							if (selectedPath != null) {
								new ProjectPopupMenu(projectTree.getSelectionPaths()).show((JComponent)e.getSource(), e.getX(), e.getY());
							} else {
								new ProjectPopupMenu(null).show((JComponent)e.getSource(), e.getX(), e.getY());
							}
						} else if (e.getButton() == MouseEvent.BUTTON2) {
							projectTree.setSelectionPath(selectedPath);
							ProColClientDockable.displayInfo("File Info", selectedPath.toString());
						} else if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2 && ((ProColFile)((DefaultMutableTreeNode)(projectTree.getSelectionPath().getLastPathComponent())).getUserObject()).isFile()) {
							// if double clicked on a file
							ProColPlugin.getClient().checkOutFile(projectTree.getSelectionPaths());
						}
					}
				}
			};
		projectTree.addMouseListener(mouseListener);

		projectScrollPane.setViewportView(projectTree);

		this.add(projectScrollPane, java.awt.BorderLayout.CENTER);

		//{{{ Button creation
		//{{{ Get snapshot
		snapshotButton = projectButtonPanel.createButton("snapshot");
		snapshotButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.getClient().downloadSnapshot();
				}
			});
		//}}}

		//{{{ New File
		newFileButton = projectButtonPanel.createButton("create.file");
		newFileButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.getClient().newFile(projectTree.getSelectionPaths());
				}
			});
		//}}}

		//{{{ New Directory
		newDirButton = projectButtonPanel.createButton("create.dir");
		newDirButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.getClient().newDir(projectTree.getSelectionPaths());
				}
			});
		//}}}

		//{{{ Upload File
		uploadFileButton = projectButtonPanel.createButton("upload.file");
		uploadFileButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.getClient().uploadFiles();
				}
			});
		//}}}

		//{{{ Upload Directory
		uploadDirButton = projectButtonPanel.createButton("upload.dir");
		uploadDirButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					ProColPlugin.getClient().uploadDirs();
				}
			});
		//}}}
		//}}}

		if (isVerticalAlign) {
			this.add(projectButtonPanel, java.awt.BorderLayout.SOUTH);
		} else {
			this.add(projectButtonPanel, java.awt.BorderLayout.EAST);
		}
	}
	//}}}

	//{{{ Modifier methods
	//{{{ setConnected
	/**
	 *  Sets the connected status of the ProjectPanel
	 *
	 *@param  isConnected  The new connected value
	 */
	public void setConnected(boolean isConnected) {
		// System.out.println("in projectPanel.setConnected " + isConnected);
		snapshotButton.setEnabled(isConnected);
		newFileButton.setEnabled(isConnected);
		newDirButton.setEnabled(isConnected);
		uploadFileButton.setEnabled(isConnected);
		uploadDirButton.setEnabled(isConnected);
		//if (!isConnected) {
		//setProject(null);
		//} else {
		setProject(ProColPlugin.getClient().getProject());
		//}
		this.projectTree.setModel(treeModel);
	}
	//}}}

	//{{{ setProject
	/**
	 *  Sets the currently opened project within the panel
	 *
	 *@param  projectIn  The project
	 */
	public void setProject(ProColClientProject projectIn) {
		if (projectIn != null) {
			this.setBorder(new TitledBorder(projectIn.getName()));
			this.projectFilesRootNode = projectIn.getProjectFilesRootNode();
			this.treeModel = new DefaultTreeModel(projectFilesRootNode);
		} else {
			this.setBorder(new TitledBorder(jEdit.getProperty("procol.client.project.toplabel")));
			this.projectFilesRootNode = null;
			this.treeModel = null;
		}
		this.projectTree.setModel(treeModel);
	}
	//}}}

	//{{{ nodeChanged
	/**
	 *  Called whenever a node in the file tree has changed, so it can be redrawn
	 *
	 *@param  nodeIn  Description of the Parameter
	 */
	public void nodeChanged(TreeNode nodeIn) {
		treeModel.nodeChanged(nodeIn);
		treeModel.reload(); // not a very good solution, need to fix!
	}
	//}}}

	//{{{ nodesWereInserted
	/**
	 *  Called whenever a node has been inserted in the file tree, so it can be
	 *  redrawn
	 *
	 *@param  node          The node inserted
	 *@param  childIndices  chilren indicies
	 */
	public void nodesWereInserted(TreeNode node, int[] childIndices) {
		treeModel.nodesWereInserted(node, childIndices);
		treeModel.reload(); // not a very good solution, need to fix!
	}
	//}}}
//}}}

	//{{{ Accessor methods
	//{{{ getCurrentDirectoryURI
	/**
	 *  Gets the URI associated with the currently selected directory
	 *
	 *@return    the URI
	 */
	public URI getCurrentDirectoryURI() {
		DefaultMutableTreeNode tempNode  = projectFilesRootNode;
		TreePath treePath                = projectTree.getSelectionPath();
		if (treePath != null) {
			tempNode = (DefaultMutableTreeNode)(treePath.getLastPathComponent());
		}
		ProColFile tempFile              = (ProColFile)tempNode.getUserObject();
		while (!tempFile.isDirectory()) {
			tempNode = (DefaultMutableTreeNode)(tempNode.getParent());
			tempFile = (ProColFile)tempNode.getUserObject();
		}
		// return tempFile.getProColPath();
		return ((ProColFile)(projectFilesRootNode.getUserObject())).toURI().relativize(tempFile.toURI());
	}
	//}}}
//}}}

	/**
	 *  Generates a popup menu based on selected TreePath's
	 *
	 *@author    Justin Dieters
	 */
	private class ProjectPopupMenu extends JPopupMenu {
		//{{{ Data members
		private MenuHandler menuHandler          = new MenuHandler();
		private TreePath[] selectedPaths;
		private DefaultMutableTreeNode tempNode;
		private ProColFile tempFile;
//}}}

		//{{{ Constructors
		/**
		 *  Constructor for the ProjectPopupMenu object
		 *
		 *@param  selectedPathsIn  Array of selected TreePath's
		 */
		public ProjectPopupMenu(TreePath[] selectedPathsIn) {
			try {
				boolean singleFileSelected  = false;
				boolean ableToCheckIn       = true;
				boolean ableToCheckOut      = true;
				selectedPaths = selectedPathsIn;
				if (selectedPaths != null) {
					singleFileSelected = (selectedPaths.length == 1);
					// check ableToCheckOut/In to break early if no need to check all files
					for (int i = 0; i < selectedPaths.length && (ableToCheckOut || ableToCheckIn); i++) {
						tempNode = (DefaultMutableTreeNode)selectedPaths[i].getLastPathComponent();
						tempFile = (ProColFile)tempNode.getUserObject();
						if (tempFile.isCheckedOut()) {
							ableToCheckOut = false;
						}
						if (!tempFile.isCheckedOutBy(ProColPlugin.getClient().getUser().getName())) {
							ableToCheckIn = false;
						}
					}
					if (ableToCheckOut) {
						add(createMenuItem("checkout"));
					} else if (ableToCheckIn) {
						add(createMenuItem("checkin"));
					}
					add(createMenuItem("download"));
					if (singleFileSelected && ableToCheckOut) {
						add(createMenuItem("rename"));
					}
					if (ableToCheckOut) {
						add(createMenuItem("delete"));
					}
					addSeparator();
				}

				add(createMenuItem("create.file"));
				add(createMenuItem("create.dir"));
				add(createMenuItem("upload.file"));
				add(createMenuItem("upload.dir"));

				if (selectedPaths != null) {
					if (singleFileSelected) {
						addSeparator();
						add(createMenuItem("fileinfo"));
					}
				}
			} catch (NullPointerException e) {
				return;
			}
		}
//}}}

		//{{{ Misc. Methods
		//{{{ createMenuItem
		private JMenuItem createMenuItem(String name) {
			JMenuItem tempMenuItem  = new JMenuItem(jEdit.getProperty("procol.client." + name + ".label"));
			tempMenuItem.setActionCommand(name);
			tempMenuItem.addActionListener(menuHandler);
			return tempMenuItem;
		}
		//}}}
//}}}

		//{{{ ActionHandler class
		private class MenuHandler implements ActionListener {
			//{{{ actionPerformed
			/**
			 *@param  evt  Description of the Parameter
			 */
			public void actionPerformed(ActionEvent evt) {
				String actionCommand  = evt.getActionCommand();

				if (actionCommand.equals("checkout")) {
					ProColPlugin.getClient().checkOutFile(selectedPaths);
				} else if (actionCommand.equals("download")) {
					ProColPlugin.getClient().downloadFile(selectedPaths);
				} else if (actionCommand.equals("rename")) {
					ProColPlugin.getClient().renameFile(selectedPaths);
				} else if (actionCommand.equals("delete")) {
					ProColPlugin.getClient().deleteFile(selectedPaths);
				} else if (actionCommand.equals("upload.file")) {
					ProColPlugin.getClient().uploadFiles();
				} else if (actionCommand.equals("upload.dir")) {
					ProColPlugin.getClient().uploadDirs();
				} else if (actionCommand.equals("create.file")) {
					ProColPlugin.getClient().newFile(selectedPaths);
				} else if (actionCommand.equals("create.dir")) {
					ProColPlugin.getClient().newDir(selectedPaths);
				} else if (actionCommand.equals("fileinfo")) {
					tempNode = (DefaultMutableTreeNode)selectedPaths[0].getLastPathComponent();
					tempFile = (ProColFile)tempNode.getUserObject();
					new FileInfoFrame(tempFile);
				}
			} //}}}
		}
//}}}
	}
}
