//{{{ GPL Notice
/*
 *  TodoListPanel.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.gui;

//{{{ Imports
import com.enderak.procol.*;
import com.enderak.procol.client.model.*;
import com.enderak.procol.client.net.*;
import com.enderak.procol.common.gui.*;
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.gui.*;
//}}}

/**
 *  Displays a list of current todo items
 *
 *@author    Justin Dieters
 */
public class TodoListPanel extends GenericListPanel {
	//{{{ Data Members
	protected static DefaultListModel listModel  = new DefaultListModel();
	private RolloverButton newButton, modifyButton, deleteButton;
	private Vector allUsers;
	//}}}

	//{{{ Constructors
	/**
	 *  Constructor for the CheckInPanel object
	 *
	 *@param  view      jEdit View, specified by jEdit
	 *@param  position  position of dockable, specified by jEdit
	 */
	public TodoListPanel(View view, String position) {
		super(view, position, "todolist");
		listModel.clear();
		itemList.setModel(listModel);
		MouseListener mouseListener  =
			new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					int index  = itemList.locationToIndex(e.getPoint());
					itemList.setSelectedIndex(index);
					if (e.getButton() == MouseEvent.BUTTON3) {
						new TodoListPopupMenu((TodoItem)(itemList.getModel().getElementAt(index))).show((JComponent)e.getSource(), e.getX(), e.getY());
					}
				}
			};
		itemList.addMouseListener(mouseListener);

		ButtonPanel buttonPanel      = new ButtonPanel(isVerticalAlign);
		//{{{ New Item Button
		newButton = buttonPanel.createRolloverButton("todoitem.new");
		newButton.addActionListener(this);
		//}}}
		//{{{ Reply Item Button
		modifyButton = buttonPanel.createRolloverButton("todoitem.modify");
		modifyButton.addActionListener(this);
		//}}}
		buttonPanel.createRigidSpace();
		//{{{ Delete Item Button
		deleteButton = buttonPanel.createRolloverButton("todoitem.delete");
		deleteButton.addActionListener(this);
		//}}}

		if (isVerticalAlign) {
			add(buttonPanel, java.awt.BorderLayout.NORTH);
		} else {
			add(buttonPanel, java.awt.BorderLayout.WEST);
		}

		initNetworkCommunication();
	}
	//}}}

	//{{{ initNetworkCommunication
	/**
	 *  Initialize network communication for this panel. Should not be called until
	 *  client is connected to server
	 */
	public final void initNetworkCommunication() {
		if (ProColPlugin.getClient() != null && ProColPlugin.getClient().isConnected()) {
			ProColPlugin.getClient().getIMH().addObserver(this);
			listModel.addElement(jEdit.getProperty("procol.client.todolist.gettinglistinfo"));
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_TODO_ITEM_LIST, PacketFactory.MAX_PRIORITY);
			newButton.setEnabled(true);
			allUsers = ProColPlugin.getClient().getUserList();
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_PROJECT_USERS, PacketFactory.MAX_PRIORITY);
		}
	}
	//}}}

	//{{{ stopNetworkCommunication
	/**
	 *  Stops network communication for this panel. Should only be called when
	 *  client disconnects
	 */
	public void stopNetworkCommunication() {
		if (ProColPlugin.getClient() != null && !ProColPlugin.getClient().isConnected()) {
			ProColPlugin.getClient().getIMH().deleteObserver(this);
			listModel.clear();
			newButton.setEnabled(false);
		}
	}
	//}}}

	//{{{ Action/Observer Listeners
	//{{{ actionPerformed
	/**
	 *@param  evt  The action event
	 */
	public void actionPerformed(java.awt.event.ActionEvent evt) {
		String actionCommand  = evt.getActionCommand();
		selectedItems = itemList.getSelectedValues();
		if (actionCommand.equals("todoitem.new")) {
			new TodoItemComposer(allUsers);
		} else if (actionCommand.equals("todoitem.modify")) {
			new TodoItemComposer((TodoItem)(selectedItems[0]), allUsers);
		} else if (actionCommand.equals("todoitem.delete")) {
			ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.TODO_ITEM_DELETE, ((TodoItem)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
		}
	}
	//}}}

	//{{{ valueChanged(ListSelectionEvent e)
	/**
	 *@param  e  The ListSelection event
	 */
	public void valueChanged(ListSelectionEvent e) {
		if (!e.getValueIsAdjusting()) {
			selectedItems = itemList.getSelectedValues();
			if (selectedItems.length == 1) {
				ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_TODO_ITEM, ((TodoItem)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
				modifyButton.setEnabled(true);
				deleteButton.setEnabled(true);
			} else if (selectedItems.length == 0) {
				modifyButton.setEnabled(false);
				deleteButton.setEnabled(false);
			}
		}
	}
	//}}}

	//{{{ update
	/**
	 *@param  o    Observable that called the update
	 *@param  arg  Object argument
	 */
	public void update(Observable o, Object arg) {
		if (arg instanceof ProColIncomingMessage) {
			ProColIncomingMessage message  = (ProColIncomingMessage)arg;
			switch (message.requestCode) {
				case RequestType.NEW_TODO_ITEM:
					//{{{
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.GET_TODO_ITEM_LIST, PacketFactory.MAX_PRIORITY);
					break;
				//}}}
				case RequestType.TODO_ITEM_LIST:
					//{{{
					Properties itemProps           = new Properties();
					try {
						itemProps.load(new ByteArrayInputStream(message.data));
					} catch (IOException ioe) {
						System.out.println(ioe);
					}

					String current                 = itemProps.getProperty("current", "0");
					listModel.clear();
					while (!current.equals("0")) {
						listModel.addElement(
								new TodoItem(current,
								itemProps.getProperty(current + ".subject", "(No Subject)"),
								itemProps.getProperty(current + ".duedate", "None"),
								itemProps.getProperty(current + ".priority", "5"),
								itemProps.getProperty(current + ".complete", "0"),
								itemProps.getProperty(current + ".assigned", "null"))
								);

						current = itemProps.getProperty(current + ".previous", "0");
					}
					if (listModel.isEmpty()) {
						messageDisplayEditorPane.setText(jEdit.getProperty("procol.client.todolist.nomessages"));
					}
					break;
				//}}}
				case RequestType.TODO_ITEM:
					//{{{
					if (selectedItems.length == 1) {
						TodoItem tempItem  = ((TodoItem)(selectedItems[0]));
						tempItem.description = new String(message.data);
						messageDisplayEditorPane.setText("<font size=\"+1\"><b>" + tempItem.subject + "</b></font><br><b>Assigned to: " + tempItem.assignedTo + "<br>Priority: " + tempItem.priority + "<br>Complete: " + tempItem.complete + "%<br>Due: " + tempItem.dueDate + "</b><hr>" + tempItem.description);
					}
					break;
				//}}}
				case RequestType.TODO_ITEM_ADD_FAILED:
					//{{{
					break;
				//}}}
				case RequestType.TODO_ITEM_DELETE_FAILED:
					//{{{
					break;
				//}}}
				case RequestType.GET_PROJECT_USERS:
					//{{{
					allUsers = ProColUtils.dataStringToVector(message.data);
					break;
				//}}}
				default:
			}
		}
	}
	//}}}

	//{{{ finalize
	/**
	 *@exception  Throwable  Description of the Exception
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		ProColPlugin.getClient().getIMH().deleteObserver(this);
	}
	//}}}
	//}}}

	private class TodoListPopupMenu extends JPopupMenu {
//{{{ Data members
		private MenuHandler menuHandler  = new MenuHandler();
		private TodoItem itemSelected;
//}}}

//{{{ Constructors
		/**
		 *  Constructor for the TodoListPopupMenu object
		 *
		 *@param  itemSelectedIn  The currently selected Todo Item
		 */
		public TodoListPopupMenu(TodoItem itemSelectedIn) {
			itemSelected = itemSelectedIn;
			add(createMenuItem("new"));
			add(createMenuItem("modify"));
			add(createMenuItem("delete"));

		}
//}}}

//{{{ Misc. Methods
		//{{{ createMenuItem
		private JMenuItem createMenuItem(String name) {
			JMenuItem tempMenuItem  = new JMenuItem(jEdit.getProperty("procol.client." + name + ".label"));
			tempMenuItem.setActionCommand(name);
			tempMenuItem.addActionListener(menuHandler);
			return tempMenuItem;
		}
		//}}}
//}}}

//{{{ ActionHandler class
		private class MenuHandler implements ActionListener {
			//{{{ actionPerformed
			/**
			 *@param  evt  Description of the Parameter
			 */
			public void actionPerformed(ActionEvent evt) {
				String actionCommand  = evt.getActionCommand();
				if (actionCommand.equals("new")) {
					new TodoItemComposer(allUsers);
				} else if (actionCommand.equals("modify")) {
					new TodoItemComposer(((TodoItem)(selectedItems[0])), allUsers);
				} else if (actionCommand.equals("delete")) {
					ProColPlugin.getClient().getPacketFactory().addToQueue(RequestType.TODO_ITEM_DELETE, ((TodoItem)(selectedItems[0])).index, PacketFactory.MAX_PRIORITY);
				}
			} //}}}
		}
//}}}
	}
}

