//{{{ GPL Notice
/*
 *  ProColClientProject.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.model;

//{{{ Imports
import com.enderak.procol.ProColPlugin;
import com.enderak.procol.client.gui.*;
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.util.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.SwingUtilities;
import javax.swing.tree.*;
import org.gjt.sp.jedit.*;
//}}}

/**
 *  The project representation on the client side, namely, it keeps track of
 *  files and buffers that are currently open by the client
 *
 *@author    Justin Dieters
 */
public class ProColClientProject extends ProColProject {
//{{{ Data members
	private LinkedList openBuffers      = new LinkedList();
	private LinkedList modifiedBuffers  = new LinkedList();
	private LinkedList waitingFiles     = new LinkedList();
//}}}

//{{{ Constructors
	/**
	 *  Create a new project
	 *
	 *@param  projectURIIn  The base URI for the project
	 *@param  name          The name of the project
	 */
	public ProColClientProject(URI projectURIIn, String name) {
		super(projectURIIn, name);

		projectDir = new ProColFile(projectURIIn);
		if (!projectDir.exists()) {
			// create project directory if none exists
			projectDir.mkdirs();
			System.out.println("Empty project directory created for project " + name + ": " + projectDir.getPath());
		}
		projectFilesDir = new ProColFile(this.projectURI.getPath(), ProColPlugin.getClient().getProperty("options.procol.client.files.dir", "files"));
		if (!projectFilesDir.exists()) {
			// create project files directory if none exists
			projectFilesDir.mkdirs();
			System.out.println("Empty project files directory created for project " + name + ": " + projectFilesDir.getPath());
		}
		projectFilesURI = projectFilesDir.toURI();
		this.projectFilesRootNode = new DefaultMutableTreeNode(projectFilesDir);
	}

//}}}

//{{{ Accessor Methods
	//{{{ isOpenBuffer
	/**
	 *  Checks if a buffer that is checked out is currently opened by the client
	 *
	 *@param  bufferIn  The buffer to check
	 *@return           true if buffer is open, false otherwise
	 */
	public boolean isOpenBuffer(Buffer bufferIn) {
		return openBuffers.contains(bufferIn);
	}
	//}}}

	//{{{ isModifiedBuffer
	/**
	 *  Checks if a buffer that is checked out has been modified by the client
	 *
	 *@param  bufferIn  The buffer to check
	 *@return           true if buffer has been modified, false otherwise
	 */
	public boolean isModifiedBuffer(Buffer bufferIn) {
		return modifiedBuffers.contains(bufferIn);
	}
	//}}}
//}}}

//{{{ Modifier methods
	//{{{ addTree
	/**
	 *  Creates a complete tree based on data from the server
	 *
	 *@param  dataIn  The tree information
	 */
	public void addTree(byte[] dataIn) {
		projectFilesRootNode.removeAllChildren();
		StringTokenizer tokenizer  = new StringTokenizer(new String(dataIn), "\n");

		while (tokenizer.hasMoreTokens()) {
			addToTree(tokenizer.nextToken(), tokenizer.nextToken(), tokenizer.nextToken(), tokenizer.nextToken());
		}
	}
	//}}}

	//{{{ addBuffer
	/**
	 *  Adds a buffer to the open buffers list
	 *
	 *@param  bufferIn  The buffer
	 */
	public void addBuffer(Buffer bufferIn) {
		openBuffers.add(bufferIn);
	}
	//}}}

	//{{{ addModifedBuffer
	/**
	 *  Adds a buffer to the modified buffers list
	 *
	 *@param  bufferIn  The buffer
	 */
	public void addModifiedBuffer(Buffer bufferIn) {
		modifiedBuffers.add(bufferIn);
	}
	//}}}

	//{{{ updateFileInfo
	/**
	 *  Updates the info for a file
	 *
	 *@param  dataIn  the updated data from the server
	 */
	public void updateFileInfo(byte[] dataIn) {
		StringTokenizer tokenizer        = new StringTokenizer(new String(dataIn), "\n");
		URI path                         = URI.create(tokenizer.nextToken());
		String owner                     = tokenizer.nextToken();
		String version                   = tokenizer.nextToken();
		DefaultMutableTreeNode fileNode  = getNodeForFile(path);
		if (fileNode != null) {
			ProColFile file  = ((ProColFile)fileNode.getUserObject());
			file.setOwner(owner);
			file.setVersion(version);
			ProColClientDockable.proColPanel.projectPanel.nodeChanged(fileNode);
		} else {
			addToTree(dataIn);
		}
	}
	//}}}

	//{{{ closeProject

	/**  Closes the project - closes all open buffers, initiating check-in of files */
	public void close() {
		synchronized (openBuffers) {
			ListIterator iterator  = openBuffers.listIterator(0);
			while (iterator.hasNext()) {
				Buffer tempBuffer  = (Buffer)iterator.next();
				jEdit.closeBuffer(jEdit.getActiveView(), tempBuffer);
			}
		}
	}
	//}}}

	//{{{ getFileForBuffer(Buffer)
	/**
	 *  Returns the ProColFile associated with a buffer
	 *
	 *@param  bufferIn  The buffer
	 *@return           The File associated with the buffer
	 */
	public ProColFile getFileForBuffer(Buffer bufferIn) {
		URI bufferURI  = projectFilesURI.relativize(projectFilesURI.resolve(new File(bufferIn.getPath()).toURI()));
		return getFileInTree(bufferURI.getPath());
	}
	//}}}

	//{{{ removeBuffer
	/**
	 *  Removes a buffer from both the open buffers and modified buffers lists
	 *
	 *@param  bufferIn  Description of the Parameter
	 */
	public void removeBuffer(Buffer bufferIn) {
		openBuffers.remove(bufferIn);
		modifiedBuffers.remove(bufferIn);
	}
	//}}}
//}}}

//{{{ I/O Methods
	//{{{ readyForDownload
	/**
	 *  Gets a file ready to be downloaded by the client
	 *
	 *@param  fileURI  The path of the file within the ProCol Tree
	 *@param  destURI  The path to store the file on the hard drive
	 */
	public void readyForDownload(URI fileURI, URI destURI) {
		System.out.println("READYING " + fileURI + " FOR DOWNLOAD TO " + destURI);
		downloadingFiles.put(fileURI, new DownloadFile(destURI));
	}
	//}}}

	//{{{ writeAndOpenFile
	/**
	 *  Writes a file to the disk and opens it in jEdit
	 *
	 *@param  dataIn    The file data
	 *@param  filePath  the file path to write the data to
	 */
	public void writeAndOpenFile(String filePath, byte[] dataIn) {
		URI tempURI  = URI.create(filePath);
		filePath = tempURI.toString();
		writeFile(filePath, dataIn);

		synchronized (waitingFiles) {
			waitingFiles.add(getFileInTree(tempURI.getPath()));
		}
		SwingUtilities.invokeLater(
			new Runnable() {
				public void run() {
					synchronized (waitingFiles) {
						while (!waitingFiles.isEmpty()) {
							openBuffers.add(jEdit.openFile(jEdit.getActiveView(), ((ProColFile)(waitingFiles.removeFirst())).getPath()));
						}
					}
				}
			});
		System.out.println("*** Opened buffer for " + filePath);
	}
	//}}}

	//{{{ addToTree(byte[])
	/**
	 *  Adds a file or directory to the tree
	 *
	 *@param  dataIn
	 */
	public void addToTree(byte[] dataIn) {
		StringTokenizer tokenizer  = new StringTokenizer(new String(dataIn), "\n");
		addToTree(tokenizer.nextToken(), tokenizer.nextToken(), tokenizer.nextToken(), tokenizer.nextToken());
	}
	//}}}

	//{{{ addToTree(String, String, String, String)
	private void addToTree(String filePath, String owner, String version, String isDirectory) {
		ProColFile theFile                  = new ProColFile(this.projectFilesURI.resolve(filePath));
		ProColFile childFile;
		theFile.setOwner(owner);
		theFile.setVersion(version);
		if (isDirectory.equals("true")) {
			if (!theFile.exists()) {
				theFile.mkdirs();
			}
		}
		File currentParent                  = theFile.getParentFile();
		Stack nodeStack                     = new Stack();
		DefaultMutableTreeNode currentNode  = this.projectFilesRootNode;
		DefaultMutableTreeNode childNode;

		while (currentParent != null && !currentParent.toURI().equals(this.projectFilesURI)) {
			nodeStack.push(currentParent);
			currentParent = currentParent.getParentFile();
		}
		while (!nodeStack.isEmpty()) {
			boolean foundChild  = false;
			currentParent = (File)nodeStack.pop();
			for (int i = currentNode.getChildCount() - 1; i >= 0; i--) {
				childNode = (DefaultMutableTreeNode)currentNode.getChildAt(i);
				childFile = (ProColFile)childNode.getUserObject();
				if (childFile.toURI().equals(currentParent.toURI())) {
					foundChild = true;
					currentNode = (DefaultMutableTreeNode)currentNode.getChildAt(i);
					break;
				}
			}
			if (!foundChild) {
				DefaultMutableTreeNode newParentNode  = new DefaultMutableTreeNode(new ProColFile(currentParent.toURI()));
				currentNode.add(newParentNode);
				currentNode = newParentNode;
			}
		}
		currentNode.add(new DefaultMutableTreeNode(theFile, theFile.isFile()));
	}
	//}}}
//}}}
}
