//{{{ GPL Notice
/*
 *  IncomingMessageHandler.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.net;

//{{{ Imports
import com.enderak.procol.client.gui.*;
import com.enderak.procol.client.model.*;
import com.enderak.procol.client.net.*;
import com.enderak.procol.client.util.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import java.io.*;
import java.net.URI;
import java.util.*;
//}}}

/**
 *  Handles messages coming in from the server
 *
 *@author    Justin Dieters
 */
public class IncomingMessageHandler extends EnhancedObservable implements Runnable {
//{{{ Data Members
	private boolean isRunning                = true;
	private ProColClient client;
	private MessageFactory messageFactory;
	private String filePath;
	private int clientMajorVersion, clientMinorVersion, serverMajorVersion, serverMinorVersion           = 0;
	private ClientErrorHandler errorHandler;
	//}}}

//{{{ Constructor
	/**
	 *  Constructor for the IncomingMessageHandler object
	 *
	 *@param  clientIn  The client this handler is associated with
	 */
	public IncomingMessageHandler(ProColClient clientIn) {
		client = clientIn;
		messageFactory = clientIn.getMessageFactory();
		errorHandler = new ClientErrorHandler(client);
		new Thread(errorHandler, "ClientErrorHandler").start();
	}
	//}}}

//{{{ Thread Methods
	//{{{ run
	/**
	 *  Main processing method for the IncomingMessageHandler object. Waits for new
	 *  messages and handles them when they arrive
	 */
	public void run() {
		this.isRunning = true;
		while (this.isRunning) {
			synchronized (messageFactory) {
				while (!messageFactory.hasNext() && this.isRunning) {
					try {
						messageFactory.wait();
					} catch (InterruptedException ie) {
					}
				}
				this.handleMessage((ProColIncomingMessage)(messageFactory.getNext()));
			}
		}
		System.out.println("Ending IMH run");
	}
	//}}}

	//{{{ close()
	/**  Closes the message handler */
	public void close() {
		isRunning = false;
		client.getUser().setAuthenticated(false);
	}
	//}}}
	//}}}

//{{{ handleMessage
	private void handleMessage(ProColIncomingMessage message) {
		if (message == null) {
			return;
		}

		System.out.println("Handling message: " + message.toString() + " " + message.isError());

		if (message.isError()) {
			errorHandler.addErrorMessage(message);
			return;
		}

		switch (message.requestCode) {
			case RequestType.NO_PROJECTS:
				//{{{
				ProColClientDockable.displayError("No Projects", client.getUser().getName() + " is not authorized to access any projects on this server.");
				client.close();
				break;
			//}}}
			case RequestType.USER_LIST:
				//{{{
				client.setUserList(ProColUtils.dataStringToVector(message.data));
				break;
			//}}}
			case RequestType.LOAD_TREE:
				//{{{
				//System.out.println(new String(message.data));
				client.addTree(message.data);
				break;
			//}}}
			case RequestType.NEW_PRIVATE_MESSAGE:
				//{{{
				// turn on private message button
				ProColClientDockable.proColPanel.userPanel.newPrivateMessage();
				// notify in case private message window is open
				this.forceNotify(message);
				break;
			//}}}
			case RequestType.NEW_PUBLIC_MESSAGE:
				//{{{
				// turn on private message button
				ProColClientDockable.proColPanel.userPanel.newPublicMessage();
				this.forceNotify(message);
				break;
			//}}}
			case RequestType.NEW_TODO_ITEM:
				//{{{
				// turn on private message button
				ProColClientDockable.proColPanel.userPanel.newTodoListItem();
				this.forceNotify(message);
				break;
			//}}}
			case RequestType.NEW_BUG_ITEM:
				//{{{
				// turn on private message button
				ProColClientDockable.proColPanel.userPanel.newBugListItem();
				this.forceNotify(message);
				break;
			//}}}
			// case RequestType.NEW_CALENDAR_ITEM:
			//{{{
			// turn on private message button
			// ProColClientDockable.proColPanel.userPanel.newCalendarItem();
			// this.forceNotify(message);
			// break;
			//}}}
			case RequestType.USER_INFO:
				//{{{
				ProColClientDockable.displayInfo("User Info", new String(message.data));
				break;
			//}}}
			case RequestType.GOODBYE:
				//{{{
				client.close();
				break;
			//}}}
			case RequestType.HELLO:
				//{{{
				new com.enderak.procol.client.gui.LoginFrame(new String(message.data));
				break;
			//}}}
			case RequestType.PROTOCOL_OK:
				//{{{
				getProtocolVersions(message);
				break;
			//}}}
			case RequestType.PROTOCOL_WARN:
				//{{{
				ProColClientDockable.displayError("Connection Warning", "The server communication protocol does not match your client.\nThe session will continue, however it is recommended that you install the newest version,\nor contact the server administrator to upgrade to the newest version.\nServer protocol: " + new String(message.data) + "\nClient protocol: " + RequestType.PROTOCOL_VERSION);
				getProtocolVersions(message);
				break;
			//}}}
			case RequestType.DOWNLOAD_FILE:
			{ //{{{

				StringTokenizer tokenizer  = new StringTokenizer(new String(message.data), "\n");
				filePath = tokenizer.nextToken();
				client.getProject().createFile(filePath);
				client.getProject().writeFile(filePath, message.data);
				break;
			}
			//}}}
			case RequestType.DOWNLOAD_SNAPSHOT_FILE:
			{ //{{{

				StringTokenizer tokenizer  = new StringTokenizer(new String(message.data), "\n");
				filePath = tokenizer.nextToken();
				client.getProject().readyForDownload(URI.create(filePath), client.getSnapshotURI().resolve(filePath));
				client.getProject().createFile(filePath);
				client.getProject().writeFile(filePath, message.data);
				break;
			}
			//}}}
			case RequestType.CHECK_OUT_FILE:
				//{{{
				StringTokenizer tokenizer = new StringTokenizer(new String(message.data), "\n");
				filePath = tokenizer.nextToken();
				System.out.println("FILEPATH: " + filePath);
				client.getProject().createFile(filePath);
				client.getProject().writeAndOpenFile(filePath, message.data);
				break;
			//}}}
			case RequestType.UPDATE_FILE_INFO:
				//{{{
				client.getProject().updateFileInfo(message.data);
				break;
			//}}}
			default:
				//{{{
				// If IMH doesn't handle message, notify any potential custom handlers that may be waiting
				this.forceNotify(message);
			//}}}
		}
	}
	//}}}

//{{{ Misc Methods
	//{{{ getProtocolVersions
	private void getProtocolVersions(ProColIncomingMessage message) {
		StringTokenizer tokenizer;

		//break up client version
		tokenizer = new StringTokenizer(RequestType.PROTOCOL_VERSION, ".");
		clientMajorVersion = Integer.parseInt(tokenizer.nextToken());
		clientMinorVersion = Integer.parseInt(tokenizer.nextToken());

		//break up server version
		tokenizer = new StringTokenizer(new String(message.data), ".");
		serverMajorVersion = Integer.parseInt(tokenizer.nextToken());
		serverMinorVersion = Integer.parseInt(tokenizer.nextToken());
	}
	//}}}
//}}}
}
