//{{{ GPL Notice
/*
 *  ProColClient.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.client.net;

//{{{ Imports
import com.enderak.procol.client.*;
import com.enderak.procol.client.gui.*;
import com.enderak.procol.client.model.*;
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.swing.*;
import javax.swing.tree.*;
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.browser.VFSBrowser;
//}}}

/**
 *  Main client class for ProCol - responsible for receiving all network
 *  communication from the server and starting handler threads, as well as
 *  notifying Observers of any network status changes
 *
 *@author    Justin Dieters
 */
public class ProColClient extends EnhancedObservable implements Runnable {
//{{{ Data Members
	private final static String DEFAULT_PROCOL_DIR_NAME    = ".procol" + System.getProperty("file.separator") + "client";
	private final static String DEFAULT_PROJECTS_DIR_NAME  = "projects";
	private final static String FILE_SEPARATOR             = System.getProperty("file.separator");
	private final static int MAX_FILES_TO_DISPLAY          = 10;
	protected boolean isRunning                            = false;
	protected Vector userList                              = new Vector();
	private ProColClientUser user;
	private String host;
	private int port;
	private SSLSocket server;
	private PacketFactory packetFactory                    = new PacketFactory();
	private MessageFactory messageFactory                  = new MessageFactory();
	private DataInputStream in;
	private IncomingMessageHandler imh;
	private OutgoingMessageHandler omh;
	private ProColClientProject project;
	private File projectDir, proColDir;
	private Thread thisThread;
	private URI snapshotURI                                = null;
	private URI downloadURI                                = null;
//}}}

//{{{ Constructors
	/**  Constructor for the ProColClient */
	public ProColClient() {
		loadProperties();
	}
	//}}}
//}}}

//{{{ Properties methods
	//{{{ loadProperties()
	/**  Description of the Method */
	private final void loadProperties() {
		proColDir = new ProColFile(getProperty("options.procol.client.directory", System.getProperty("user.home") + FILE_SEPARATOR + DEFAULT_PROCOL_DIR_NAME));
		if (!proColDir.exists()) {
			proColDir.mkdirs();
			System.out.println("Empty procol client directory created: " + proColDir.getPath());
		}

		projectDir = new ProColFile(getProperty("options.procol.client.directory.projects", proColDir.getPath() + FILE_SEPARATOR + DEFAULT_PROJECTS_DIR_NAME));
		if (!projectDir.exists()) {
			projectDir.mkdirs();
			System.out.println("Empty procol project directory created: " + projectDir.getPath());
		}
	}
	//}}}

//{{{ Thread methods
	//{{{ run
	/**
	 *  Main processing method for the ProColClient - receives and handles messages
	 *  from the server
	 */
	public void run() {
		this.isRunning = true;
		while (this.isRunning) {
			receive();
		}
		close();
	}
	//}}}
//}}}

//{{{ Communication methods
	//{{{ receive()
	/**  Description of the Method */
	public void receive() {
		try {
			int messageID      = in.readInt();
			int requestCode    = in.readInt();
			int dataRemaining  = in.readInt();
			int dataSize       = in.readInt();
			byte[] data        = new byte[dataSize];
			in.readFully(data);
			messageFactory.addPacket(new ProColPacket(messageID, requestCode, dataRemaining, dataSize, data));
		} catch (EOFException eofe) {
			if (isRunning) {
				ProColClientDockable.displayError(jEdit.getProperty("procol.client.error.connection.title"), jEdit.getProperty("procol.client.error.connection.unexpecteddisconnect"));
				this.isRunning = false;
			}
		} catch (IOException ioe) {
			System.err.println("IOException reading from server in ProColClient! " + ioe);
			this.isRunning = false;
		}
	}
	//}}}
//}}}

//{{{ Client control methods
	//{{{ connectToServer(String hostIn, int portIn)
	/**
	 *  Connects the client to the server, and authenticates the user
	 *
	 *@param  hostIn  IP or hostname of the server
	 *@param  portIn  Port to connect to on server
	 *@return         true if connection and authentication successful, false
	 *      otherwise
	 */
	public boolean connectToServer(String hostIn, int portIn) {
		ProColClientDockable.proColPanel.startProgress("Connecting to " + hostIn + ":" + portIn);
		host = hostIn;
		port = portIn;
		packetFactory.resetMessageNum();
		SSLSocketFactory sslsocketfactory  = (SSLSocketFactory)SSLSocketFactory.getDefault();
		try {
			server = (SSLSocket)sslsocketfactory.createSocket(host, port);
		} catch (IOException ioe) {
			ProColClientDockable.displayError("Network Error", "Unable to open network socket from server!");
			ProColClientDockable.proColPanel.stopProgress();
			return false;
		}
		SSLCommon.enableAnonymousCipherSuites(server);
		try {
			in = new DataInputStream(server.getInputStream());
		} catch (IOException ioexception1) {
			ProColClientDockable.displayError("Network Error", "Unable to open input stream from server!");
			ProColClientDockable.proColPanel.stopProgress();
			return false;
		}

		this.isRunning = true;
		this.forceNotify("CONNECTION");

		imh = new IncomingMessageHandler(this);
		new Thread(imh, "IMH").start();
		omh = new OutgoingMessageHandler(this.server, this.packetFactory);
		omh.start();
		thisThread = new Thread(this, "ProColClient");
		thisThread.start();

		sendHello();
		ProColClientDockable.proColPanel.stopProgress();
		return true;
	}
	//}}}
	//{{{ close()
	/**  Description of the Method */
	public void close() {
		System.out.println("closing client");
		try {
			this.isRunning = false;
			this.project = null;
			this.imh.close();
			this.omh.close();
			this.server.close();
			this.forceNotify("CONNECTION");
		} catch (IOException ioe) {
			ProColClientDockable.displayError("Unable to disconnect!", "An error occured while disconnecting from server!");
		}
		System.out.println("Done closing client");
	}
	//}}}
//}}}

//{{{ Outgoing messages
	//{{{ sendHello
	/** */
	public void sendHello() {
		packetFactory.addToQueue(RequestType.HELLO, RequestType.PROTOCOL_VERSION, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
	//{{{ authenticateUser
	/**
	 *  Description of the Method
	 *
	 *@param  userNameIn  Description of the Parameter
	 *@param  passwordIn  Description of the Parameter
	 */
	public void authenticateUser(String userNameIn, String passwordIn) {
		ProColClientDockable.proColPanel.startProgress("Authenticating");
		packetFactory.addToQueue(RequestType.AUTHENTICATE, "" + userNameIn + "\n" + passwordIn + "\n", PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
	//{{{ getProjectList
	/**  Gets the projectList attribute of the ProColClient object */
	public void getProjectList() {
		packetFactory.addToQueue(RequestType.GET_PROJECT_LIST, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
	//{{{ getProjectInfo
	/** */
	public void getProjectInfo() {
		getProjectInfo(project.getName());
	}
	//}}}
	//{{{ getProjectInfo
	/**
	 *@param  projectIn  Description of the Parameter
	 */
	public void getProjectInfo(String projectIn) {
		// System.out.println("getting info for: " + projectIn);
		if (projectIn != null) {
			packetFactory.addToQueue(RequestType.GET_PROJECT_INFO, projectIn, PacketFactory.NORMAL_PRIORITY);
		}
	}
	//}}}
	//{{{ getUserInfo
	/**
	 *@param  userName  Description of the Parameter
	 */
	public void getUserInfo(String userName) {
		packetFactory.addToQueue(RequestType.GET_USER_INFO, userName, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
	//{{{ getFileHistory
	/**
	 *@param  fileName  Description of the Parameter
	 */
	public void getFileHistory(String fileName) {
		packetFactory.addToQueue(RequestType.GET_FILE_HISTORY, fileName, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
	//{{{ downloadSnapshot
	/** */
	public void downloadSnapshot() {
		if (snapshotURI == null) {
			snapshotURI = project.getProjectFilesURI();
		}
		String[] snapshotDir;
		snapshotDir = GUIUtilities.showVFSFileDialog(jEdit.getActiveView(), snapshotURI.getPath(), VFSBrowser.CHOOSE_DIRECTORY_DIALOG, false);
		snapshotURI = new File(snapshotDir[0]).toURI();
		packetFactory.addToQueue(RequestType.DOWNLOAD_SNAPSHOT, PacketFactory.MIN_PRIORITY);
	}
	//}}}
	//{{{ checkOutFile
	/**
	 *@param  selectedPaths  Description of the Parameter
	 */
	public void checkOutFile(TreePath[] selectedPaths) {
		URI projectFilesURI              = project.getProjectFilesURI();
		DefaultMutableTreeNode tempNode;
		ProColFile tempFile;
		URI tempURI;
		for (int i = 0; i < selectedPaths.length; i++) {
			tempNode = (DefaultMutableTreeNode)selectedPaths[i].getLastPathComponent();
			tempFile = (ProColFile)tempNode.getUserObject();
			tempURI = projectFilesURI.relativize(tempFile.toURI());
			System.out.println("1: " + projectFilesURI);
			System.out.println("2: " + projectFilesURI.relativize(tempURI));
			System.out.println("3: " + new File(projectFilesURI.getPath() + File.separator + tempURI.getPath()).toURI());
			sendFileRequest(tempURI, new File(projectFilesURI.getPath() + File.separator + tempURI.getPath()).toURI(), RequestType.CHECK_OUT_FILE);
		}
	}
	//}}}
	//{{{ updateFile
	/**
	 *@param  file  Description of the Parameter
	 */
	public void updateFile(ProColFile file) {
		upload(file);
	}
	//}}}
	//{{{ downloadFile
	/**
	 *@param  selectedPaths  Description of the Parameter
	 */
	public void downloadFile(TreePath[] selectedPaths) {
		if (downloadURI == null) {
			downloadURI = project.getProjectFilesURI();
		}
		String[] saveFile;
		if (selectedPaths.length == 1) {
			DefaultMutableTreeNode tempNode  = (DefaultMutableTreeNode)selectedPaths[0].getLastPathComponent();
			ProColFile tempFile              = (ProColFile)tempNode.getUserObject();
			URI fileNameURI                  = tempFile.getParentFile().toURI().relativize(tempFile.toURI());
			if (tempFile.isDirectory()) {
				saveFile = GUIUtilities.showVFSFileDialog(jEdit.getActiveView(), downloadURI.getPath(), VFSBrowser.CHOOSE_DIRECTORY_DIALOG, false);
				downloadURI = new File(saveFile[0]).toURI();
			} else {
				saveFile = GUIUtilities.showVFSFileDialog(jEdit.getActiveView(), downloadURI.resolve(fileNameURI).getPath(), VFSBrowser.CHOOSE_DIRECTORY_DIALOG, false);
				downloadURI = new File(saveFile[0]).toURI();
			}
		} else {
			saveFile = GUIUtilities.showVFSFileDialog(jEdit.getActiveView(), downloadURI.getPath(), VFSBrowser.CHOOSE_DIRECTORY_DIALOG, false);
			downloadURI = new File(saveFile[0]).toURI();
		}

		URI projectFilesURI              = project.getProjectFilesURI();
		DefaultMutableTreeNode tempNode;
		ProColFile tempFile;
		URI tempURI;
		for (int i = 0; i < selectedPaths.length; i++) {
			tempNode = (DefaultMutableTreeNode)selectedPaths[i].getLastPathComponent();
			tempFile = (ProColFile)tempNode.getUserObject();
			tempURI = projectFilesURI.relativize(tempFile.toURI());
			sendFileRequest(tempURI, new File(downloadURI.getPath() + File.separator + tempFile.getName()).toURI(), RequestType.DOWNLOAD_FILE);
			// sendFileRequest(tempURI, destURI.relativize(tempURI), request);
		}
	}
	//}}}
	//{{{ uploadFiles
	/** */
	public void uploadFiles() {
		upload(GUIUtilities.showVFSFileDialog(jEdit.getActiveView(), null, VFSBrowser.OPEN_DIALOG, true));
	}
	//}}}
	//{{{ uploadDirs
	/** */
	public void uploadDirs() {
		upload(GUIUtilities.showVFSFileDialog(jEdit.getActiveView(), null, VFSBrowser.CHOOSE_DIRECTORY_DIALOG, true));
	}
	//}}}
	//{{{ deleteFile
	/**
	 *@param  selectedPaths  Description of the Parameter
	 */
	public void deleteFile(TreePath[] selectedPaths) {
		StringBuffer deletedFilesStringBuffer  = new StringBuffer();
		URI projectFilesURI                    = project.getProjectFilesURI();
		DefaultMutableTreeNode tempNode;
		ProColFile tempFile;
		for (int i = 0; (i < selectedPaths.length && i < MAX_FILES_TO_DISPLAY); i++) {
			tempNode = (DefaultMutableTreeNode)selectedPaths[i].getLastPathComponent();
			tempFile = (ProColFile)tempNode.getUserObject();
			deletedFilesStringBuffer.append("\n").append(projectFilesURI.relativize(tempFile.toURI()).getPath());
		}
		if (selectedPaths.length > MAX_FILES_TO_DISPLAY) {
			deletedFilesStringBuffer.append("\n... (").append(selectedPaths.length - MAX_FILES_TO_DISPLAY).append(" more files)");
		}
		if (JOptionPane.showConfirmDialog(null, "Are you sure you want to delete the following file(s)?\n" + deletedFilesStringBuffer.toString(), "Delete File Confirmation", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE) == JOptionPane.YES_OPTION) {

			for (int i = 0; i < selectedPaths.length; i++) {
				tempNode = (DefaultMutableTreeNode)selectedPaths[i].getLastPathComponent();
				tempFile = (ProColFile)tempNode.getUserObject();
				packetFactory.addToQueue(RequestType.DELETE_FILE, projectFilesURI.relativize(tempFile.toURI()), PacketFactory.NORMAL_PRIORITY);
			}

		}
	}
	//}}}
	//{{{ renameFile
	/**
	 *@param  selectedPaths  Description of the Parameter
	 */
	public void renameFile(TreePath[] selectedPaths) {
		if (selectedPaths.length != 1) {
			return;
		}
		String renameTo  = JOptionPane.showInputDialog(null, "Enter a new name for " + (((ProColFile)((DefaultMutableTreeNode)(selectedPaths[0].getLastPathComponent())).getUserObject())).getName(), "Rename File", JOptionPane.QUESTION_MESSAGE);
		if (renameTo != null && !renameTo.equals("")) {
			URI projectFilesURI                = project.getProjectFilesURI();
			DefaultMutableTreeNode sourceNode  = (DefaultMutableTreeNode)selectedPaths[0].getLastPathComponent();
			ProColFile sourceFile              = (ProColFile)sourceNode.getUserObject();
			String destParent;
			if (sourceFile.isDirectory()) {
				destParent = sourceFile.getPath();
			} else {
				destParent = sourceFile.getParent();
			}
			ProColFile destFile                = new ProColFile(destParent + File.separator + renameTo);
			URI sourceURI                      = projectFilesURI.relativize(sourceFile.toURI());
			URI destURI                        = projectFilesURI.relativize(projectFilesURI.resolve(destFile.toURI()));
			packetFactory.addToQueue(RequestType.RENAME_FILE, sourceURI.toString() + "\n" + destURI.toString(), PacketFactory.NORMAL_PRIORITY);
		}
	}
	//}}}
	//{{{ newFile
	/**
	 *@param  selectedPaths  Description of the Parameter
	 */
	public void newFile(TreePath[] selectedPaths) {
		String newFile  = JOptionPane.showInputDialog(null, "Enter name of file to create.", "Create File", JOptionPane.QUESTION_MESSAGE);
		if (newFile != null && !newFile.equals("")) {
			URI projectFilesURI                = project.getProjectFilesURI();
			DefaultMutableTreeNode sourceNode;
			if (selectedPaths != null) {
				sourceNode = (DefaultMutableTreeNode)selectedPaths[0].getLastPathComponent();
			} else {
				sourceNode = project.getProjectFilesRootNode();
			}
			ProColFile sourceFile              = (ProColFile)sourceNode.getUserObject();
			String newParent;
			if (sourceFile.isDirectory()) {
				newParent = sourceFile.getPath();
			} else {
				newParent = sourceFile.getParent();
			}
			ProColFile tempFile                = new ProColFile(newParent + File.separator + newFile);
			URI newURI                         = projectFilesURI.relativize(projectFilesURI.resolve(tempFile.toURI()));
			packetFactory.addToQueue(RequestType.NEW_FILE, newURI.toString(), PacketFactory.NORMAL_PRIORITY);
		}
	}
	//}}}
	//{{{ newDir
	/**
	 *@param  selectedPaths  Description of the Parameter
	 */
	public void newDir(TreePath[] selectedPaths) {
		String newFile  = JOptionPane.showInputDialog(null, "Enter name of directory to create.", "Create Directory", JOptionPane.QUESTION_MESSAGE);
		if (newFile != null && !newFile.equals("")) {
			URI projectFilesURI                = project.getProjectFilesURI();
			DefaultMutableTreeNode sourceNode;
			if (selectedPaths != null) {
				sourceNode = (DefaultMutableTreeNode)selectedPaths[0].getLastPathComponent();
			} else {
				sourceNode = project.getProjectFilesRootNode();
			}
			ProColFile sourceFile              = (ProColFile)sourceNode.getUserObject();
			String newParent;
			if (sourceFile.isDirectory()) {
				newParent = sourceFile.getPath();
			} else {
				newParent = sourceFile.getParent();
			}
			ProColFile tempFile                = new ProColFile(newParent + File.separator + newFile);
			URI newURI                         = projectFilesURI.relativize(projectFilesURI.resolve(tempFile.toURI()));
			packetFactory.addToQueue(RequestType.NEW_DIRECTORY, newURI.toString(), PacketFactory.NORMAL_PRIORITY);
		}
	}
	//}}}
	//{{{ closeProject
	/** */
	public void closeProject() {
		packetFactory.addToQueue(RequestType.CLOSE_PROJECT, PacketFactory.NORMAL_PRIORITY);
		this.project = null;
	}
	//}}}
	//{{{ joinProject
	/**
	 *  Description of the Method
	 *
	 *@param  projectIn  Description of the Parameter
	 */
	public void joinProject(String projectIn) {
		if (this.project != null) {
			closeProject();
		}
		File tempFile  = new File(projectDir.getPath() + FILE_SEPARATOR + projectIn);
		this.project = new ProColClientProject(tempFile.toURI(), projectIn);
		this.forceNotify("PROJECT");
		ProColClientDockable.proColPanel.startProgress("Opening Project");
		packetFactory.addToQueue(RequestType.OPEN_PROJECT, projectIn, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
	//{{{ addPrivateMessage
	/**
	 *@param  messageTo  Description of the Parameter
	 *@param  message    Description of the Parameter
	 *@param  subject
	 */
	public void addPrivateMessage(String messageTo, String subject, String message) {
		packetFactory.addToQueue(RequestType.PRIVATE_MESSAGE_ADD, messageTo + "\n" + subject + "\n" + message, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
	//{{{ disconnect
	/** */
	public void disconnect() {
		project.close();
		packetFactory.addToQueue(RequestType.DISCONNECT, PacketFactory.MAX_PRIORITY);
	}
	//}}}
	//{{{ killServerRemotely
	/** */
	public void killServerRemotely() {
		packetFactory.addToQueue(RequestType.KILL_SERVER, PacketFactory.MAX_PRIORITY);
	}
	//}}}
	//{{{ sendFileRequest
	/**
	 *@param  fileURI  Description of the Parameter
	 *@param  destURI  Description of the Parameter
	 *@param  request  Description of the Parameter
	 */
	public void sendFileRequest(URI fileURI, URI destURI, int request) {
		System.out.println("SENDING FILE REQUEST FOR " + fileURI + " TO " + destURI);
		DefaultMutableTreeNode rootNode;
		ProColFile currentFile;

		if (fileURI == null) {
			rootNode = project.getProjectFilesRootNode();
			currentFile = null;
		} else {
			rootNode = project.getNodeForFile(fileURI);
			currentFile = (ProColFile)rootNode.getUserObject();
		}

		if (!rootNode.isLeaf()) {
			Enumeration children                 = rootNode.children();
			DefaultMutableTreeNode currentChild;
			while (children.hasMoreElements()) {
				currentChild = (DefaultMutableTreeNode)(children.nextElement());
				currentFile = (ProColFile)currentChild.getUserObject();
				URI newDest  = new File(destURI.getPath() + File.separator + currentFile.getName()).toURI();
				sendFileRequest(project.getProjectFilesURI().relativize(currentFile.toURI()), newDest, request);
			}
		} else if (currentFile != null) {
			if (new File(destURI).isDirectory()) {
				destURI = new File(destURI.getPath() + File.separator + currentFile.getName()).toURI();
			}
			project.readyForDownload(fileURI, destURI);
			packetFactory.addToQueue(request, fileURI, PacketFactory.NORMAL_PRIORITY);
		}
	}
	//}}}
//}}}

//{{{ Incoming messages
	//{{{ setUserList
	/**
	 *@param  userListIn
	 */
	public void setUserList(Vector userListIn) {
		this.userList = userListIn;
		this.forceNotify("USERLIST");
	}
	//}}}
	//{{{ addTree
	/**
	 *@param  treeData
	 */
	public void addTree(byte[] treeData) {
		this.project.addTree(treeData);
		ProColClientDockable.proColPanel.stopProgress();
		this.forceNotify("CONNECTION");
	}
	//}}}
//}}}

//{{{ Accessor Methods
	//{{{ getProject
	/**
	 *@return    the project
	 */
	public ProColClientProject getProject() {
		return project;
	}
	//}}}
	//{{{ isConnected
	/**
	 *  Gets the isRunning attribute of the ProColClient object
	 *
	 *@return    The isRunning value
	 */
	public boolean isConnected() {
		return this.isRunning;
	}
	//}}}
	//{{{ isAuthenticated
	/**
	 *@return    true if user is authenticated
	 */
	public boolean isAuthenticated() {
		if (this.user == null) {
			return false;
		} else {
			return this.user.isAuthenticated();
		}
	}
	//}}}
	//{{{ getServer
	/**
	 *  Gets the currently isRunning server IP or hostname
	 *
	 *@return    The server IP or hostname
	 */
	public String getServer() {
		return host;
	}
	//}}}
	//{{{ getPort
	/**
	 *  Gets the currently isRunning port number
	 *
	 *@return    The port number
	 */
	public int getPort() {
		return port;
	}
	//}}}
	//{{{ getUser
	/**
	 *  Gets the user attribute of the ProColClient object
	 *
	 *@return    The user
	 */
	public ProColClientUser getUser() {
		return user;
	}
	//}}}
	//{{{ getUserList
	/**
	 *  Gets the userList attribute of the ProColClient object
	 *
	 *@return    The userList value
	 */
	public Vector getUserList() {
		return userList;
	}
	//}}}
	//{{{ getChannel
	/**
	 *  Gets the channel attribute of the ServerConnection object
	 *
	 *@return    The channel value
	 */
	public SSLSocket getServerSocket() {
		return this.server;
	}
	//}}}
	//{{{ getPacketFactory
	/**
	 *@return    the packet factory
	 */
	public PacketFactory getPacketFactory() {
		return packetFactory;
	}
	//}}}
	//{{{ getMessageFactory
	/**
	 *@return    the message factory
	 */
	public MessageFactory getMessageFactory() {
		return messageFactory;
	}
	//}}}
	//{{{ getIMH
	/**
	 *@return    the incoming message handler
	 */
	public IncomingMessageHandler getIMH() {
		return this.imh;
	}
	//}}}
	//{{{ getOMH
	/**
	 *@return    the outgoing message handler
	 */
	public OutgoingMessageHandler getOMH() {
		return this.omh;
	}
	//}}}
	//{{{ getSnapshotURI
	/**
	 *@return    the shapshot URI
	 */
	public URI getSnapshotURI() {
		return snapshotURI;
	}
	//}}}
//}}}

//{{{ Modifer methods
	//{{{ setUser
	/**
	 *@param  userIn
	 */
	public void setUser(ProColClientUser userIn) {
		this.user = userIn;
	}
	//}}}
	//{{{ getProperty(String propertyIn)
	/**
	 *  Gets the property attribute of the ProColServer class
	 *
	 *@param  propertyIn  Description of the Parameter
	 *@return             The property value
	 */
	public String getProperty(String propertyIn) {
		return jEdit.getProperty(propertyIn);
	}
	//}}}
	//{{{ getProperty(String propertyIn, String defaultIn)
	/**
	 *  Gets the property attribute of the ProColServer class
	 *
	 *@param  propertyIn  Description of the Parameter
	 *@param  defaultIn   Description of the Parameter
	 *@return             The property value
	 */
	public String getProperty(String propertyIn, String defaultIn) {
		return jEdit.getProperty(propertyIn, defaultIn);
	}
	//}}}
	//{{{ upload(String[])
	private void upload(String[] filePaths) {
		for (int i = 0; i < filePaths.length; i++) {
			upload(new ProColFile(filePaths[i]));
		}
	}
	//}}}
	//{{{ upload(ProColFile)
	private void upload(ProColFile fileIn) {
		if (fileIn == null) {
			System.err.println("ERROR! fileIn is null in ProColClient.upload(ProColFile)");
			return;
		}
		if (fileIn.isDirectory()) {
			ProColFile[] subfiles  = fileIn.listProColFiles();
			for (int i = 0; i < subfiles.length; i++) {
				upload(subfiles[i]);
			}
		} else if (fileIn.isFile()) {
			jEdit.getActiveView().getDockableWindowManager().showDockableWindow("procol.client.dockable.upload");
			URI fileNameURI  = fileIn.getParentFile().toURI().relativize(fileIn.toURI());
			((UploadPanel)(jEdit.getActiveView().getDockableWindowManager().getDockableWindow("procol.client.dockable.upload")))
					.addFile(fileIn, fileNameURI);
		}
	}
	//}}}
//}}}
}
