/*
 *  ButtonPanel.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package com.enderak.procol.common.gui;
import java.awt.Dimension;

//{{{ Imports
import java.net.URL;
import javax.swing.*;
import org.gjt.sp.jedit.*;
import org.gjt.sp.jedit.gui.RolloverButton;
//}}}

/**
 *  Creates a panel to display RolloverButtons within the ProCol plugin
 *  interface
 *
 *@author    Justin Dieters
 */
public class ButtonPanel extends JPanel {
	//{{{ Constructors
	/**  Creates an empty ButtonPanel in a horizontal orientation */
	public ButtonPanel() {
		super();
		this.setLayout(new javax.swing.BoxLayout(this, javax.swing.BoxLayout.X_AXIS));
	}


	/**
	 *  Creates an empty ButtonPanel in a particular orientation
	 *
	 *@param  axisIn  The orientation of the ButtonPanel. Valid values are
	 *      javax.swing.BoxLayout.X_AXIS and javax.swing.BoxLayout.Y_AXIS
	 */
	public ButtonPanel(int axisIn) {
		super();
		this.setLayout(new javax.swing.BoxLayout(this, axisIn));
	}


	/**
	 *  Creates an empty ButtonPanel, oriented properly for either a Vertically
	 *  aligned plugin (floating or docked to the left or right) or a horizontally
	 *  alighed plugin (docked to the top or bottom)
	 *
	 *@param  isVerticalAlign  true if aligned vertically, false if aligned
	 *      horizontally
	 */
	public ButtonPanel(boolean isVerticalAlign) {
		super();
		if (isVerticalAlign) {
			this.setLayout(new javax.swing.BoxLayout(this, javax.swing.BoxLayout.X_AXIS));
		} else {
			this.setLayout(new javax.swing.BoxLayout(this, javax.swing.BoxLayout.Y_AXIS));
		}
	}
	//}}}

	//{{{ Creation methods
	//{{{ RolloverButton creation methods
	/**
	 *  Creates a RolloverButton using jEdit's property list. Properties used with
	 *  this function are defined in the ProCol.props file
	 *
	 *@param  buttonPrefix  The prefix for the button properties
	 *@return               The button created
	 */
	public RolloverButton createButton(String buttonPrefix) {
		RolloverButton tempButton  = this.createRolloverButton(
				jEdit.getProperty("procol.graphics_dir") + jEdit.getProperty("procol.button." + buttonPrefix + ".icon"),
				jEdit.getProperty("procol.button." + buttonPrefix + ".text"),
				jEdit.getProperty("procol.button." + buttonPrefix + ".text"));
		tempButton.setActionCommand(buttonPrefix);
		this.add(tempButton);
		return tempButton;
	}


	/**
	 *  Creates a RolloverButton with explicit settings passed in and adds it to
	 *  the ButtonPanel
	 *
	 *@param  buttonImageIn  Image to display with the button
	 *@param  noImageTextIn  Text to display with the button, if there is no image
	 *      to display
	 *@param  toolTipTextIn  Tooltip to display with the button
	 *@return                The button created
	 */
	public RolloverButton createButton(String buttonImageIn, String noImageTextIn, String toolTipTextIn) {
		RolloverButton tempButton  = this.createRolloverButton(buttonImageIn, noImageTextIn, toolTipTextIn);
		this.add(tempButton);
		return tempButton;
	}


	/**
	 *  Creates a RolloverButton using jEdit's property list. Properties used with
	 *  this function are defined in the ProCol.props file
	 *
	 *@param  buttonPrefix  The prefix for the button properties
	 *@return               The button created
	 */
	public RolloverButton createRolloverButton(String buttonPrefix) {
		RolloverButton tempButton  = this.createRolloverButton(
				jEdit.getProperty("procol.graphics_dir") + jEdit.getProperty("procol.button." + buttonPrefix + ".icon"),
				jEdit.getProperty("procol.button." + buttonPrefix + ".text"),
				jEdit.getProperty("procol.button." + buttonPrefix + ".text"));
		tempButton.setActionCommand(buttonPrefix);
		this.add(tempButton);
		return tempButton;
	}


	/**
	 *  Creates a RolloverButton with explicit settings passed in and adds it to
	 *  the ButtonPanel
	 *
	 *@param  buttonImageIn  Image to display with the button
	 *@param  noImageTextIn  Text to display with the button, if there is no image
	 *      to display
	 *@param  toolTipTextIn  Tooltip to display with the button
	 *@return                The button created
	 */
	public RolloverButton createRolloverButton(String buttonImageIn, String noImageTextIn, String toolTipTextIn) {
		RolloverButton tempButton  = new RolloverButton();
		URL imageURL               = this.getClass().getResource(buttonImageIn);
		if (imageURL == null) {
			tempButton.setText(noImageTextIn);
		} else {
			tempButton.setIcon(new ImageIcon(imageURL));
			tempButton.setBorderPainted(false);
		}
		tempButton.setToolTipText(toolTipTextIn);
		tempButton.setEnabled(false);
		this.add(tempButton);
		return tempButton;
	}
	//}}}

	//{{{ RolloverOnOffButton creation methods
	/**
	 *  Creates a RolloverOnOffButton with explicit settings passed in and adds it
	 *  to the ButtonPanel
	 *
	 *@param  buttonImageOnIn   Image to display when the button is in the on state
	 *@param  buttonImageOffIn  Image to display when the button is in the off
	 *      state
	 *@param  noImageTextOnIn   Text to display when the button is in the on state,
	 *      if there is no image to display
	 *@param  noImageTextOffIn  Text to display when the button is in the off
	 *      state, if there is no image to display
	 *@param  toolTipTextOnIn   Tooltip to display when the button is in the on
	 *      state
	 *@param  toolTipTextOffIn  Tooltip to display when the button is in the off
	 *      state
	 *@return                   The button created
	 */
	public RolloverOnOffButton createRolloverOnOffButton(String buttonImageOnIn, String buttonImageOffIn, String noImageTextOnIn, String noImageTextOffIn, String toolTipTextOnIn, String toolTipTextOffIn) {
		RolloverOnOffButton tempButton  = new RolloverOnOffButton(buttonImageOnIn, buttonImageOffIn, noImageTextOnIn, noImageTextOffIn, toolTipTextOnIn, toolTipTextOffIn);
		tempButton.setEnabled(false);
		this.add(tempButton);
		return tempButton;
	}


	/**
	 *  Creates a RolloverOnOffButton using jEdit's property list. Properties used
	 *  with this function are defined in the ProCol.props file
	 *
	 *@param  buttonPrefix  The prefix for the button properties
	 *@return               The button created
	 */
	public RolloverOnOffButton createRolloverOnOffButton(String buttonPrefix) {
		RolloverOnOffButton tempButton  = new RolloverOnOffButton("procol.button." + buttonPrefix);
		tempButton.setActionCommand(buttonPrefix);
		tempButton.setEnabled(false);
		this.add(tempButton);
		return tempButton;
	}
	//}}}

	//{{{ Misc creation methods
	/**  Creates a rigid spacer of the default size */
	public void createRigidSpace() {
		this.createRigidSpace(8);
	}


	/**
	 *  Creates a rigid spacer of a specified size
	 *
	 *@param  space  The number of pixels to make the rigid space
	 */
	public void createRigidSpace(int space) {
		this.add(Box.createRigidArea(new Dimension(space, space)));
	}


	/**  Creates an expandable spacer of the default size */
	public void createExpandableSpace() {
		this.createExpandableSpace(8);
	}


	/**
	 *  Creates an expandable spacer of a specified size
	 *
	 *@param  space  The number of pixels to make the expandable space
	 */
	public void createExpandableSpace(int space) {
		this.add(Box.createHorizontalStrut(space));
		this.add(Box.createVerticalStrut(space));
	}


	/**  Creates a section which expands to fill available space */
	public void createExpandableSection() {
		this.add(Box.createHorizontalGlue());
		this.add(Box.createVerticalGlue());
	}
	//}}}
	//}}}
}
