//{{{ GPL Notice
/*
 *  ProColFile.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.common.model;

//{{{ Imports
import java.io.*;
import java.net.*;
import java.util.*;
//}}}

/**
 *  A file for use with ProCol
 *
 *@author    Justin Dieters
 */
public class ProColFile extends File {
	//{{{ Data members
	private final static int NUM_VERSIONS  = 3;
	private Properties versions            = new Properties();
	private String owner;
	private File versionsFile;
	//}}}

	/**
	 *  Creates a new ProColFile with the specified pathname
	 *
	 *@param  pathname  The pathname to the file on disk
	 */
	public ProColFile(String pathname) {
		super(pathname);
	}


	/**
	 *  Creates a new ProColFile with the specified pathname and filename
	 *
	 *@param  pathname  The pathname to the file on disk
	 *@param  filename  The filename of the file on disk
	 */
	public ProColFile(String pathname, String filename) {
		super(pathname, filename);
	}


	/**
	 *  Creates a new ProColFile with the specified URI
	 *
	 *@param  uri  URI to the file
	 */
	public ProColFile(URI uri) {
		super(uri);
	}


	/**
	 *  Sets the file where version information is kept
	 *
	 *@param  versionsURI  The URI to the versions file
	 */
	public void setVersionsFile(URI versionsURI) {
		if (!this.isDirectory()) {
			versionsFile = new File(versionsURI);
			try {
				versions.load(new FileInputStream(versionsFile));
			} catch (FileNotFoundException fnfe) {
				try {
					versionsFile.getParentFile().mkdirs();
					versionsFile.createNewFile();
					createNewVersion("0.0.0", "Versions file created by ProCol server", null);
					System.out.println("Versions file created: " + versionsFile.getPath());
				} catch (IOException ioe) {
					System.err.println("IOException creating new versions file: " + versionsFile.getPath());
				}
			} catch (IOException ioe) {
				System.err.println("IOException loading server properties: " + ioe);
			}
		}
	}


	/**
	 *  Gets the versions URI
	 *
	 *@return    the versions URI
	 */
	public URI getVersionsURI() {
		if (versionsFile == null) {
			return null;
		} else {
			return versionsFile.toURI();
		}
	}


	/**
	 *  Gets the versions File
	 *
	 *@return    the version File
	 */
	public File getVersionsFile() {
		if (versionsFile == null) {
			return null;
		} else {
			return versionsFile;
		}
	}


	/**
	 *  Checks out the file
	 *
	 *@param  userIn  The user checking out the file
	 *@return         True if checkout was successful, false if not
	 */
	public boolean checkOut(String userIn) {
		if (owner != null) {
			return false;
		} else {
			setOwner(userIn);
			return true;
		}
	}


	/**
	 *  Checks in a file
	 *
	 *@param  userIn  The user checking in the file
	 *@return         True if checkin was successful, false if not
	 */
	public boolean checkIn(String userIn) {
		// System.out.println("CHECKING IN " + getName() + " FROM " + userIn + "::" + owner);
		if (userIn.equals(owner)) {
			return setOwner(null);
		} else {
			return false;
		}
	}


	/**
	 *  Gets the current owner of the file
	 *
	 *@return    The name of the current file, or null if no owner
	 */
	public String getOwner() {
		return owner;
	}


	/**
	 *  Gets the current version of the file as a String
	 *
	 *@return    The version string, or 'null' if no version
	 */
	public String getVersion() {
		return versions.getProperty("current", "null");
	}


	/**
	 *  Gets the current version of the file as an int array
	 *
	 *@return    the int array, or [0,0,0] if no version
	 */
	public int[] getVersionAsArray() {
		int[] versionArray  = new int[NUM_VERSIONS];
		if (getVersion().equals("null")) {
			for (int i = 0; i < NUM_VERSIONS; i++) {
				versionArray[i] = 0;
			}
			return versionArray;
		} else {
			StringTokenizer tokenizer  = new StringTokenizer(getVersion(), ".");
			for (int i = 0; i < NUM_VERSIONS; i++) {
				versionArray[i] = new Integer(tokenizer.nextToken()).intValue();
			}
			return versionArray;
		}
	}


	/**
	 *  Sets the current version of the file
	 *
	 *@param  versionIn  The new version string
	 */
	public void setVersion(String versionIn) {
		versions.setProperty("current", versionIn);
	}


	/**
	 *  Creates a new version of the file
	 *
	 *@param  newVersion    The new version string
	 *@param  newChangeLog  The new changelog
	 *@param  userNameIn    The updating user
	 *@return               True if new version creation was successful, false
	 *      otherwise
	 */
	public boolean createNewVersion(String newVersion, String newChangeLog, String userNameIn) {
		// System.out.println(newVersion + " " + getVersion());
		versions.setProperty(newVersion + ".previous", getVersion());
		// System.out.println(newChangeLog);
		if (newChangeLog == null) {
			versions.setProperty(newVersion + ".changelog", "");
		} else {
			versions.setProperty(newVersion + ".changelog", newChangeLog);
		}
		// System.out.println(userNameIn);
		if (userNameIn == null) {
			versions.setProperty(newVersion + ".user", "");
		} else {
			versions.setProperty(newVersion + ".user", userNameIn);
		}
		versions.setProperty(newVersion + ".date", new Date().toString());
		versions.setProperty("current", newVersion);
		// System.out.println(versionsFile.toURI());
		try {
			versions.store(new FileOutputStream(versionsFile), "Version information for " + this.getPath());
			// System.out.println("!");
		} catch (IOException ioe) {
			System.err.println("IOException versions file: " + versionsFile.getPath());
			return false;
		}
		return true;
	}


	/**
	 *  Sets the owner of this file
	 *
	 *@param  ownerIn
	 *@return          True if was able to set owner, false otherwise (i.e. already
	 *      has an owner)
	 */
	public boolean setOwner(String ownerIn) {
		if (ownerIn == null || ownerIn.equals("null")) {
			owner = null;
		} else {
			if (owner == null) {
				owner = ownerIn;
			} else {
				return false;
			}
		}
		return true;
	}


	/**
	 *  Get the checked-out status of the file
	 *
	 *@return    True if checked out, false if not
	 */
	public boolean isCheckedOut() {
		return (owner != null);
	}


	/**
	 *  Check if the file is checked out by a specific user
	 *
	 *@param  userName  the user's name
	 *@return           True if checked out by that user. False if checked out by
	 *      someone else or not checked out
	 */
	public boolean isCheckedOutBy(String userName) {
		return (owner != null && owner.equals(userName));
	}


	/**
	 *  Get this file as a printable String
	 *
	 *@return    The name of the file
	 */
	public String toString() {
		return getName();
	}


	/**
	 *  Returns this file info
	 *
	 *@param  relativeRoot  The root of this file to reletiveize from
	 *@return               The file info in String format
	 */
	public String getFileInfo(URI relativeRoot) {
		return relativeRoot.relativize(this.toURI()) + "\n"
				 + getOwner() + "\n"
				 + getVersion() + "\n"
				 + isDirectory() + "\n";
	}


	/**
	 *  Gets all child files as a ProColFile[] array
	 *
	 *@return    All children as ProColFile'ss.
	 */
	public ProColFile[] listProColFiles() {
		File[] temp  = ((File)this).listFiles();
		if (temp == null) {
			return null;
		} else {
			ProColFile[] proColTemp  = new ProColFile[temp.length];
			for (int i = 0; i < temp.length; i++) {
				proColTemp[i] = new ProColFile(temp[i].getPath());
			}
			return proColTemp;
		}
	}
}

