//{{{ GPL Notice
/*
 *  ProColProject.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.common.model;

//{{{ Imports
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.tree.*;
//}}}

/**
 *  Maintains information about a project within ProCol
 *
 *@author    Justin Dieters
 */
public class ProColProject extends EnhancedObservable {
//{{{ Data members
	protected final static int NUM_VERSIONS                = 3;
	protected String name                                  = "";
	protected DefaultMutableTreeNode projectFilesRootNode;
	protected Hashtable downloadingFiles                   = new Hashtable();
	protected URI projectURI, projectFilesURI;
	protected ProColFile projectDir, projectFilesDir;
	protected int j                                        = 0;
//}}}

//{{{ Constructors
	/**
	 *  Constructor for the ProColProject object
	 *
	 *@param  projectURIIn  Description of the Parameter
	 *@param  nameIn        Description of the Parameter
	 */
	public ProColProject(URI projectURIIn, String nameIn) {
		this.projectURI = projectURIIn;
		this.name = nameIn;
	}

//}}}

//{{{ Accessor Methods
	//{{{ getName
	/**
	 *@return    the project name
	 */
	public String getName() {
		return this.name;
	}
	//}}}

	//{{{ getProjectURI
	/**
	 *@return    the project URI
	 */
	public URI getProjectURI() {
		return this.projectURI;
	}
	//}}}

	//{{{ getProjectFilesURI
	/**
	 *@return    the project files URI
	 */
	public URI getProjectFilesURI() {
		return this.projectFilesURI;
	}
	//}}}

	//{{{ getProjectFilesRootNode
	/**
	 *@return    the project files root node
	 */
	public DefaultMutableTreeNode getProjectFilesRootNode() {
		return this.projectFilesRootNode;
	}
	//}}}

	//{{{ getNodeForFile(URI)
	/**
	 *  Gets a node associated with the file
	 *
	 *@param  fileURI  The file URI
	 *@return          the Node
	 */
	public DefaultMutableTreeNode getNodeForFile(URI fileURI) {
		System.out.println("URI");
		return getNodeForFile(fileURI.getPath());
	}
	//}}}

	//{{{ getNodeForFile(String)
	/**
	 *@param  filePath  Path to the file from the project files root directory.
	 *      String can be formed with URI.getPath() (i.e. spaces should not be
	 *      escaped)
	 *@return           the Node
	 */
	public DefaultMutableTreeNode getNodeForFile(String filePath) {
		System.out.println("String");
		System.out.println("Looking for file: '" + filePath + "'");
		if (filePath.equals("")) {
			return projectFilesRootNode;
		}
		StringTokenizer tokenizer            = new StringTokenizer(filePath, "/");
		String currentFileName               = tokenizer.nextToken();
		DefaultMutableTreeNode currentNode   = getProjectFilesRootNode();
		//URI currentDirURI = ((ProColFile)currentNode.getUserObject()).toURI();
		Enumeration children                 = currentNode.children();
		DefaultMutableTreeNode currentChild;
		ProColFile tempFile;
		while (children.hasMoreElements()) {
			currentChild = (DefaultMutableTreeNode)(children.nextElement());
			//System.out.println("***" + ((ProColFile)(((DefaultMutableTreeNode)currentChild).getUserObject())).toURI().toString() + " =? " + currentDirURI.resolve(currentFileName));
			// if (((ProColFile)(((DefaultMutableTreeNode)currentChild).getUserObject())).toURI().equals(currentDirURI.resolve(currentFileName))) {
			tempFile = (ProColFile)((DefaultMutableTreeNode)currentChild).getUserObject();
			if (tempFile.getName().equals(currentFileName)) {
				if (tokenizer.hasMoreTokens()) {
					// found child, but not at file we are looking for yet
					currentNode = currentChild;
					// currentDirURI = ((ProColFile)currentNode.getUserObject()).toURI();
					children = currentNode.children();
					currentFileName = tokenizer.nextToken();
					//currentFileName += URI.create(tokenizer.nextToken()+ "/");
				} else {
					// found child, and it's file we are looking for
					// System.out.println("File found, returning node.");
					return (DefaultMutableTreeNode)currentChild;
				}
			}
		}
		// System.out.println("File not found: " + filePath);
		return null;
	}
	//}}}

	//{{{ getFileInTree(URI)
	/**
	 *@param  fileURI  URI of the file
	 *@return          The File, or null if file not in tree
	 */
	public ProColFile getFileInTree(URI fileURI) {
		DefaultMutableTreeNode tempNode  = getNodeForFile(fileURI);
		if (tempNode == null) {
			return null;
		}
		return (ProColFile)tempNode.getUserObject();
	}
	//}}}

	//{{{ getFileInTree(String)
	/**
	 *@param  fileString  The path to the file
	 *@return             the file, or null if not in tree
	 */
	public ProColFile getFileInTree(String fileString) {
		DefaultMutableTreeNode tempNode  = getNodeForFile(fileString);
		if (tempNode == null) {
			return null;
		}
		return (ProColFile)tempNode.getUserObject();
	}
	//}}}
//}}}

//{{{ File I/O Methods
	//{{{ createFile(URI)
	/**
	 *@param  fileURI  Description of the Parameter
	 *@return          Description of the Return Value
	 */
	public int createFile(URI fileURI) {
		// *** does not make backups ***
		DownloadFile tempDownloadFile        = (DownloadFile)downloadingFiles.get(fileURI);
		// System.out.println(tempDownloadFile);
		ProColFile currentlyDownloadingFile  = new ProColFile(tempDownloadFile.destURI.getPath());
		System.out.println("createFile: " + currentlyDownloadingFile);
		tempDownloadFile.file = currentlyDownloadingFile;
		// System.out.println(tempDownloadFile.file);
		if (currentlyDownloadingFile == null) {
			currentlyDownloadingFile = new ProColFile(projectFilesURI.resolve(fileURI));
		}
		try {
			currentlyDownloadingFile.getParentFile().mkdirs();
			currentlyDownloadingFile.createNewFile();
		} catch (IOException ioe) {
			System.err.println(ioe);
			return RequestType.FILE_NAME_FAILED;
		}
		return RequestType.FILE_INFO_OK;
	}
	//}}}

	//{{{ createFile(String)
	/**
	 *@param  fileString  Description of the Parameter
	 *@return             Description of the Return Value
	 */
	public int createFile(String fileString) {
		return createFile(URI.create(fileString));
	}
	//}}}

	//{{{ writeFile
	/**
	 *@param  dataIn    Description of the Parameter
	 *@param  filePath  Description of the Parameter
	 *@return           Description of the Return Value
	 */
	public int writeFile(String filePath, byte[] dataIn) {
		URI tempURI                          = URI.create(filePath);
		filePath = tempURI.getPath();
		System.out.println("WRITING FILE: " + filePath);
		DownloadFile tempDownloadFile        = (DownloadFile)downloadingFiles.remove(tempURI);
		// System.out.println("hashtable size: " + downloadingFiles.size());
		ProColFile currentlyDownloadingFile  = tempDownloadFile.file;
		// System.out.println(currentlyDownloadingFile);
		if (currentlyDownloadingFile != null) {
			// System.out.println(currentlyDownloadingFile.toURI());
			try {
				FileOutputStream fos  = new FileOutputStream(currentlyDownloadingFile);
				// System.out.println("SKIPPING " + (filePath.getBytes().length + 1) + " BYTES, writing to " + (dataIn.length - filePath.getBytes().length - 1) + " bytes of " + dataIn.length);
				fos.write(dataIn, filePath.getBytes().length + 1, dataIn.length - filePath.getBytes().length - 1); // write everything after the file path
				fos.close();
				// System.out.println("WROTE OK!!");
				return RequestType.FILE_WRITE_OK;
			} catch (FileNotFoundException fnfe) {
				System.err.println(fnfe);
				return RequestType.FILE_NOT_FOUND;
			} catch (IOException ioe) {
				System.err.println(ioe);
				return RequestType.FILE_IO_ERROR;
			}
		}
		return RequestType.FILE_NOT_EXPECTED;
	}
	//}}}
//}}}

//{{{ Misc Methods
	//{{{ printTree
	/**
	 *@param  node  Description of the Parameter
	 */
	protected void printTree(DefaultMutableTreeNode node) {
		String tempString  = "";
		for (int j = 0; j < node.getLevel(); j++) {
			tempString = tempString + "--";
		}
		System.out.println(tempString + (((ProColFile)(node.getUserObject())).getName()));

		for (int i = 0; i < node.getChildCount(); i++) {
			printTree((DefaultMutableTreeNode)(node.getChildAt(i)));
		}
	}
	//}}}
//}}}
}
