//{{{ GPL Notice
/*
 *  OutgoingMessageHandler.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.common.net;

//{{{ Imports
import com.enderak.procol.client.gui.*;
import com.enderak.procol.client.model.*;
import com.enderak.procol.client.net.*;
import com.enderak.procol.common.*;
import com.enderak.procol.common.net.*;
import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import javax.net.ssl.SSLSocket;
//}}}

/**
 *  Handles the sending of outgoing messages
 *
 *@author    Justin Dieters
 */
public class OutgoingMessageHandler extends Thread {
	private SSLSocket client;
	private boolean isRunning            = false;
	private DataOutputStream out;
	private PacketFactory packetFactory;


	/**
	 *  Constructor for the OutgoingMessageHandler object
	 *
	 *@param  clientIn         The socket to send to
	 *@param  packetFactoryIn  The factory to pull packets from
	 */
	public OutgoingMessageHandler(SSLSocket clientIn, PacketFactory packetFactoryIn) {
		client = clientIn;
		packetFactory = packetFactoryIn;
	}


	/**  Main processing method for the OutgoingMessageHandler object */
	public void run() {
		try {
			out = new DataOutputStream(client.getOutputStream());
		} catch (IOException ioexception1) {
			System.err.println("getOutputStream failed");
			return;
		}

		this.isRunning = true;
		while (this.isRunning) {
			synchronized (packetFactory) {
				while (!packetFactory.hasNext() && this.isRunning) {
					try {
						packetFactory.wait();
					} catch (InterruptedException ie) {
						System.out.println(ie);
					}
				}
				if (packetFactory.hasNext()) {
					this.send(packetFactory.getNext());
				}

				synchronized (this) {
					this.notifyAll();
				}
			}
		}
		System.out.println("Ending OMH run");
	}
//{{{ Communication methods
	//{{{ close()
	/**  close this handler */
	public void close() {
		this.isRunning = false;
		synchronized (this.packetFactory) {
			System.out.println("CLOSING OMH!");
			this.packetFactory.notifyAll();
		}
	}
	//}}}

	//{{{ send()
	/**
	 *  Send a packet
	 *
	 *@param  tempPacket  The packet to send
	 */
	public void send(ProColPacket tempPacket) {
		if (tempPacket == null) {
			return;
		}
		synchronized (out) {
			try {
				out.writeInt(tempPacket.messageID);
				out.writeInt(tempPacket.requestCode);
				out.writeInt(tempPacket.dataRemaining);
				out.writeInt(tempPacket.dataSize);
				if (tempPacket.dataSize > 0) {
					out.write(tempPacket.data, 0, tempPacket.dataSize);
				}
				out.flush();
			} catch (IOException ioe) {
				System.err.println("IOException in OMH: " + ioe);
			}
		}
	}
	//}}}
//}}}
}

