/*
 *  PacketFactory.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package com.enderak.procol.common.net;

import com.enderak.procol.common.util.*;

//{{{ Imports
import java.io.*;
import java.net.*;
import java.util.*;
//}}}

/**
 *  Generates packets to send
 *
 *@author    Justin Dieters
 */
public class PacketFactory {
	/**  The maximum priority */
	public final static int MAX_PRIORITY      = 2;
	/**  The minimum priority */
	public final static int MIN_PRIORITY      = 0;
	/**  The normal priority */
	public final static int NORMAL_PRIORITY   = 1;
	//{{{ Data Members
	private final static int MAX_PACKET_SIZE  = 2048;
	private final static int HEADER_SIZE      = 16;
	private final static int MAX_DATA_SIZE    = MAX_PACKET_SIZE - HEADER_SIZE;
	private int messageID, dataRemaining, dataSize, read                     = 0;
	private boolean messageInProgress         = false;
	private DataInputStream dataStream;
	private ProColOutgoingMessage message;
	private LinkedList[] messageQueue         = new LinkedList[MAX_PRIORITY + 1];
	//}}}


	/**  Constructor for the PacketFactory object */
	public PacketFactory() {
		super();
		for (int i = MIN_PRIORITY; i <= MAX_PRIORITY; i++) {
			messageQueue[i] = new LinkedList();
		}
	}


	/**
	 *  Adds a message to the queue with no data section
	 *
	 *@param  requestIn  The request code
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, int priority) {
		synchronized (this) {
			messageQueue[priority].add(new ProColOutgoingMessage(messageID++, requestIn, new DataInputStream(new ByteArrayInputStream("".getBytes()))));
			this.notifyAll();
		}
	}


	/**
	 *  Adds a message to the queue, using a String for the data section
	 *
	 *@param  dataIn     The data
	 *@param  requestIn  The request code
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, String dataIn, int priority) {
		synchronized (this) {
			messageQueue[priority].add(new ProColOutgoingMessage(messageID++, requestIn, new DataInputStream(new ByteArrayInputStream(dataIn.getBytes()))));
			this.notifyAll();
		}
	}


	/**
	 *  Adds a message to the queue, using a URI for the data section
	 *
	 *@param  requestIn  The request code
	 *@param  uriIn      the URI
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, URI uriIn, int priority) {
		addToQueue(requestIn, uriIn.toString(), priority);
	}


	/**
	 *  Adds a message to the queue, using a byte array for the data section
	 *
	 *@param  dataIn     The data
	 *@param  requestIn  The request code
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, byte[] dataIn, int priority) {
		synchronized (this) {
			messageQueue[priority].add(new ProColOutgoingMessage(messageID++, requestIn, new DataInputStream(new ByteArrayInputStream(dataIn))));
			this.notifyAll();
		}
	}


	/**
	 *  Adds a message to the queue, using an int for the data section
	 *
	 *@param  dataIn     The data
	 *@param  requestIn  The request code
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, int dataIn, int priority) {
		synchronized (this) {
			messageQueue[priority].add(new ProColOutgoingMessage(messageID++, requestIn, new DataInputStream(new ByteArrayInputStream(ProColUtils.intToByteArray(dataIn)))));
			this.notifyAll();
		}
	}


	/**
	 *  Adds a message to the queue, using an File for the data section
	 *
	 *@param  requestIn  The request code
	 *@param  fileIn     The file
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, File fileIn, int priority) {
		synchronized (this) {
			//try {
			messageQueue[priority].add(new ProColOutgoingMessage(messageID++, requestIn, fileIn));
			//} catch (FileNotFoundException e) {
			//System.err.println("FileNotFound in PacketFactory: " + e);
			//}
			this.notifyAll();
		}
	}


	/**
	 *  Adds a message to the queue, using an File and the File's URI for the data
	 *  section
	 *
	 *@param  requestIn  The request code
	 *@param  fileIn     The file
	 *@param  fileURIIn  The file URI
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, URI fileURIIn, File fileIn, int priority) {
		addToQueue(requestIn, fileURIIn.toString(), fileIn, priority);
	}


	/**
	 *  Adds a message to the queue, using an File and the File's path string for
	 *  the data section
	 *
	 *@param  requestIn  The request code
	 *@param  fileIn     The file
	 *@param  filePath   The file path
	 *@param  priority   The priority
	 */
	public void addToQueue(int requestIn, String filePath, File fileIn, int priority) {
		synchronized (this) {
			filePath += "\n";
			messageQueue[priority].add(new ProColOutgoingMessage(messageID++, requestIn, new DataInputStream(new ByteArrayInputStream(filePath.getBytes())), fileIn));
			this.notifyAll();
		}
	}


	/**
	 *  Gets the next Packet from the queue
	 *
	 *@return    The next packet
	 */
	public ProColPacket getNext() {
		if (messageInProgress) {
			dataSize = 0;
			byte[] data  = new byte[MAX_DATA_SIZE];
			try {
				do {
					// System.out.println(dataStream.available());
					read = dataStream.read(data, dataSize, data.length - dataSize);
					if (read > 0) {
						dataSize += read;
					}
				} while (read >= 0 && dataSize < MAX_DATA_SIZE);
				dataRemaining = dataStream.available() + dataSize;
			} catch (IOException e) {
				System.err.println("IOExeception reading message.dataStream in PacketFactory: " + e);
			}
			if (dataSize < MAX_DATA_SIZE) {
				try {
					dataStream.close();
					dataStream = null;
				} catch (IOException e) {
					System.err.println("IOExeception closing message.dataStream in PacketFactory: " + e);
				}
				messageInProgress = false;
			}
			return new ProColPacket(message.messageID, message.requestCode, dataRemaining, dataSize, data);
		} else if (hasNext()) {
			for (int i = MAX_PRIORITY; i >= MIN_PRIORITY; i--) {
				if (!messageQueue[i].isEmpty()) {
					// System.out.println("*** Getting message from queue " + i);
					message = (ProColOutgoingMessage)(messageQueue[i].removeFirst());
					break;
				}
			}
			dataStream = message.getDataInputStream();
			if (dataStream == null) {
				addToQueue(RequestType.FILE_IO_ERROR, NORMAL_PRIORITY);
				return null;
			}
			messageInProgress = true;
			return this.getNext();
		} else {
			return null;
		}
	}


	/**
	 *  Determines if there is another packet waiting
	 *
	 *@return    true if there is another packet, false if not
	 */
	public boolean hasNext() {
		if (messageInProgress) {
			return true;
		}

		for (int i = MIN_PRIORITY; i <= MAX_PRIORITY; i++) {
			if (!messageQueue[i].isEmpty()) {
				return true;
			}
		}

		return false;
	}


	/**  Resets the message number to zero */
	public void resetMessageNum() {
		this.messageID = 0;
	}
}

