//{{{ GPL Notice
/*
 *  ProColServerProject.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.server.model;

//{{{ Imports
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import com.enderak.procol.server.*;
import com.enderak.procol.server.net.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.tree.*;
//}}}

/**
 *  The project model for the ProColServer
 *
 *@author    Justin Dieters
 */
public class ProColServerProject extends ProColProject {
//{{{ Data members
	/**  the list of connections that have this project open */
	public Vector connectionList     = new Vector();
	private URI versionsURI, backupURI, privateMessagesURI, publicMessagesURI, todoListURI, bugListURI, calendarURI;
	private int numBackups;
	private boolean makeBackups, versionsAllowNoChange, allowAnonymousLogin;
	private String backupSuffix, versionsSuffix;
	private Properties projectProps  = new Properties();
	private Vector allowedUsers      = new Vector();
//}}}

//{{{ Constructors
	/**
	 *  Constructor for the ProColServerProject
	 *
	 *@param  projectRootIn  The root URI
	 *@param  loadFilesNow   true if files should be loaded now, or false if
	 *      loading should be deferred
	 *@param  name           The name of this project
	 */
	public ProColServerProject(URI projectRootIn, String name, boolean loadFilesNow) {
		super(projectRootIn, name);

		try {
			this.projectProps.load(new FileInputStream(new File(projectURI.getPath() + File.separator + name + ".props")));
		} catch (FileNotFoundException fnfe) {
			ProColServer.printErr("FileNotFoundException loading server properties: " + fnfe);
		} catch (IOException ioe) {
			ProColServer.printErr("IOException loading server properties: " + ioe);
		}

		initOptions();

		if (loadFilesNow) {
			this.loadFiles();
		}
	}
	//}}}

	//{{{ loadFiles()
	/**  Loads the files for this project */
	public final void loadFiles() {
		ProColServer.printInfo("Loading files for " + name + " project.");
		this.loadFiles(this.projectFilesURI);
	}
	//}}}

	//{{{ checkin
	/**
	 *  Checks in a file
	 *
	 *@param  fileURI  The URI for the file to check in
	 *@param  user     The user checking in the file
	 *@return          The result to send back to the client
	 */
	public int checkIn(URI fileURI, ProColUser user) {
		DefaultMutableTreeNode fileNode  = getNodeForFile(fileURI);
		if (fileNode == null) {
			return RequestType.CHECK_IN_FILE_NOT_FOUND;
		}
		ProColFile file                  = (ProColFile)fileNode.getUserObject();
		if (file.checkIn(user.getName())) {
			forceNotify(file);
			return RequestType.CHECK_IN_FILE_OK;
		} else {
			return RequestType.CHECK_IN_ALREADY_CHECKED_IN;
		}
	}
	//}}}

	//{{{ addConnection(ServerConnection connectionIn)
	/**
	 *  Adds a Connection to the project
	 *
	 *@param  connectionIn  The Connection to add
	 */
	public void addConnection(ServerConnection connectionIn) {
		connectionList.add(connectionIn);
		addObserver(connectionIn);
		if (this.projectFilesRootNode == null) {
			this.loadFiles();
		}
		forceNotify("CONNECTIONS");
	}
	//}}}

	//{{{ removeConnection(ServerConnection connectionIn)
	/**
	 *  Removes a Connection to the project
	 *
	 *@param  connectionIn  The Connection to remove
	 */
	public void removeConnection(ServerConnection connectionIn) {
		connectionList.remove(connectionIn);
		deleteObserver(connectionIn);
		connectionIn = null;
		ProColServer.printInfo("Removed connection from " + name + ", # connections = " + connectionList.size());
		if (this.connectionList.isEmpty()) {
			unloadFiles();
		}
		forceNotify("CONNECTIONS");
	}
	//}}}

	//{{{ closeAllConnections()
	/**  closes all connections associated with this project */
	public void closeAllConnections() {
		while (connectionList.size() > 0) {
			((ServerConnection)(connectionList.get(0))).close();
		}
	}
	//}}}
//}}}

//{{{ Accessor methods
	//{{{ getUserList()
	/**
	 *  Gets the userList attribute of the ProColServer class
	 *
	 *@return    The userList value
	 */
	public Vector getUserList() {
		Vector tempUserList              = new Vector();
		ServerConnection tempConnection;
		for (int i = 0; i < connectionList.size(); i++) {
			tempConnection = (ServerConnection)connectionList.elementAt(i);
			tempUserList.add(tempConnection.getUser().getName());
		}
		return tempUserList;
	}
	//}}}

	//{{{ getProjectDescription
	/**
	 *  Gets the description for this project
	 *
	 *@return    The description from the project properties
	 */
	public String getProjectDescription() {
		return projectProps.getProperty("description", "<none>");
	}
	//}}}

	//{{{ getOptimalStringBufferSize
	/**
	 *  Quickly gets the optimal Stringbuffer size for a tree when sending file
	 *  info <br>
	 *  Due to using leafs instead of children of rootNode, this only really works
	 *  right for a complete tree <br>
	 *  Need to fix to work with subtree
	 *
	 *@param  rootNode  Description of the Parameter
	 *@return           the size
	 */
	public int getOptimalStringBufferSize(DefaultMutableTreeNode rootNode) {

		int tempSize                        = 0;
		DefaultMutableTreeNode currentLeaf  = rootNode.getFirstLeaf();
		while (currentLeaf != null) {
			tempSize += ((ProColFile)currentLeaf.getUserObject()).getFileInfo(((ProColFile)rootNode.getUserObject()).toURI()).length();
			currentLeaf = currentLeaf.getNextLeaf();
		}
		return tempSize;
	}
	//}}}

	//{{{ getConnectionFor()
	/**
	 *  Returns the connection for a certain user
	 *
	 *@param  user  The user name
	 *@return       The ServerConnection associated with the user, or null of user
	 *      not connected
	 */
	public ServerConnection getConnectionFor(String user) {
		Enumeration connections  = connectionList.elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection.getUser().getName().equals(user)) {
				return currentConnection;
			}
		}
		return null;
	}
//}}}

	//{{{ getConnections
	/**
	 *  Returns the open connections associated with this project
	 *
	 *@return    The Vector of connections
	 */
	public Vector getConnections() {
		return connectionList;
	}
	//}}}

	//{{{ getAllowedUsers()
	/**
	 *  Gets the users allowed access to this project
	 *
	 *@return    Vector containing the user names
	 */
	public Vector getAllowedUsers() {
		return allowedUsers;
	}
	//}}}

	//{{{ getPrivateMessagesURI
	/**
	 *  Gets the URI for the private messages directory
	 *
	 *@return    the URI
	 */
	public URI getPrivateMessagesURI() {
		return privateMessagesURI;
	}
	//}}}

	//{{{ getPublicMessagesURI
	/**
	 *  Gets the URI for the public messages list
	 *
	 *@return    the URI
	 */
	public URI getPublicMessagesURI() {
		return publicMessagesURI;
	}
	//}}}

	//{{{ getTodoListURI
	/**
	 *  Gets the URI for the todo list
	 *
	 *@return    the URI
	 */
	public URI getTodoListURI() {
		return todoListURI;
	}
	//}}}

	//{{{ getBugListURI
	/**
	 *  Gets the URI for the bug list
	 *
	 *@return    the URI
	 */
	public URI getBugListURI() {
		return bugListURI;
	}
	//}}}

	//{{{ getCalendarURI
	/**
	 *  Gets the URI for the calendar list
	 *
	 *@return    the URI
	 */
	public URI getCalendarURI() {
		return calendarURI;
	}
	//}}}
	//}}}

//{{{ I/O Methods
	//{{{ addDownloadFile
	/**
	 *  Adds a file to be downloaded to the download list
	 *
	 *@param  infoIn      file info from the client - version, changelog, etc
	 *@param  userNameIn  user uploading the file
	 *@return             reply code to client
	 */
	public int addDownloadFile(byte[] infoIn, String userNameIn) {
		StringTokenizer tokenizer          = new StringTokenizer(new String(infoIn), "\n");
		String filePath                    = tokenizer.nextToken();
		URI fileURI                        = URI.create(filePath);
		System.out.println("1");
		ProColFile currentlyUploadingFile  = getFileInTree(fileURI);
		System.out.println("2");
		String newVersion                  = new String();
		StringBuffer newChangeLog          = new StringBuffer();

		if (currentlyUploadingFile == null) {
			currentlyUploadingFile = new ProColFile(projectFilesURI.resolve(fileURI));
			currentlyUploadingFile.setVersionsFile(versionsURI.resolve(projectFilesURI.relativize(currentlyUploadingFile.toURI()).toString() + versionsSuffix));
		}
		//{{{ get verson info, if available
		if (tokenizer.hasMoreTokens()) {
			newVersion = tokenizer.nextToken();
			while (tokenizer.hasMoreTokens()) {
				newChangeLog.append(tokenizer.nextToken());
				if (tokenizer.hasMoreTokens()) {
					newChangeLog.append("\n");
				}
			}

			tokenizer = new StringTokenizer(newVersion, ".");
			int[] newVersionArray  = new int[NUM_VERSIONS];
			for (int i = 0; i < NUM_VERSIONS; i++) {
				newVersionArray[i] = new Integer(tokenizer.nextToken()).intValue();
			}

			int[] oldVersionArray  = currentlyUploadingFile.getVersionAsArray();

			if (newVersion.equals(currentlyUploadingFile.getVersion()) && !versionsAllowNoChange) {
				currentlyUploadingFile = null;
				return RequestType.FILE_VERSION_NO_OVERWRITE;
			}
			for (int i = 0; i < NUM_VERSIONS; i++) {
				if (newVersionArray[i] > oldVersionArray[i]) {
					// as long as a higher precedence version has been increased, lower versions can be decreased
					break;
				} else if (newVersionArray[i] < oldVersionArray[i]) {
					currentlyUploadingFile = null;
					return RequestType.FILE_VERSION_OLD;
				}
			}
		}
		//}}}
		downloadingFiles.put(filePath, new DownloadFile(currentlyUploadingFile, newVersion, newChangeLog.toString())); // add file to Hashtable
		return RequestType.FILE_INFO_OK;
	}
	//}}}

	//{{{ writeFile
	/**
	 *  writes a file to the disk, overrides write(byte[]) from
	 *  com.enderak.procol.common.model.ProColProject to do backups, etc
	 *
	 *@param  dataIn    The file data
	 *@param  filePath  the file's path
	 *@param  userName  user uploading the file
	 *@return           reply code to client
	 */
	public int writeFile(String filePath, byte[] dataIn, String userName) {
		DownloadFile uploadFile  = (DownloadFile)downloadingFiles.remove(filePath);
		if (uploadFile != null) {
			ProColFile currentlyUploadingFile  = uploadFile.file;
			String newVersion                  = uploadFile.version;
			String newChangeLog                = uploadFile.changeLog;
			//{{{ make backups
			if (makeBackups) {
				URI backupFileURI  = backupURI.resolve(projectFilesURI.relativize(projectFilesURI.resolve(new File(currentlyUploadingFile.getPath() + backupSuffix).toURI())));
				File backupFile    = new File(backupFileURI.getPath() + "0");
				if (!backupFile.getParentFile().exists()) {
					backupFile.getParentFile().mkdirs();
				}
				File tempFile;
				File tempFile2;
				for (int i = (numBackups - 2); i >= 0; i--) {
					tempFile = new File(backupFileURI.getPath() + i);
					tempFile2 = new File(backupFileURI.getPath() + (i + 1));
					if (tempFile.exists()) {
						if (tempFile2.exists()) {
							tempFile2.delete();
						}
						if (!tempFile.renameTo(tempFile2)) {
							ProColServer.printErr("Error backing up local file " + tempFile.getPath());
						}
					}
				}
				if (backupFile.exists()) {
					backupFile.delete();
				}
				currentlyUploadingFile.renameTo(backupFile);
			}
			//}}}

			//{{{ create file
			try {
				currentlyUploadingFile.getParentFile().mkdirs();
				currentlyUploadingFile.createNewFile();
			} catch (IOException ioe) {
				ProColServer.printErr(ioe.toString());
				currentlyUploadingFile = null;
				return RequestType.FILE_NAME_FAILED;
			}
			//}}}

			//{{{ create new version
			if (newVersion != null && newChangeLog != null) {
				if (!currentlyUploadingFile.createNewVersion(newVersion, newChangeLog, currentlyUploadingFile.getOwner())) {
					currentlyUploadingFile = null;
					return RequestType.FILE_VERSION_IO_ERROR;
				}
			}
			//}}}

			//{{{ write file to disk
			try {
				FileOutputStream fos  = new FileOutputStream(currentlyUploadingFile);
				fos.write(dataIn, filePath.getBytes().length + 1, dataIn.length - filePath.getBytes().length - 1); // write everything after the file path
				fos.close();
			} catch (FileNotFoundException fnfe) {
				ProColServer.printErr(fnfe.toString());
				currentlyUploadingFile = null;
				return RequestType.FILE_NOT_FOUND;
			} catch (IOException ioe) {
				ProColServer.printErr(ioe.toString());
				currentlyUploadingFile = null;
				return RequestType.FILE_IO_ERROR;
			}
			if (getNodeForFile(projectFilesURI.relativize(projectFilesURI.resolve(currentlyUploadingFile.toURI()))) == null) {
				DefaultMutableTreeNode destParentNode  = getNodeForFile(projectFilesURI.relativize(projectFilesURI.resolve(currentlyUploadingFile.getParentFile().toURI())));
				destParentNode.add(new DefaultMutableTreeNode(currentlyUploadingFile));
			}
			currentlyUploadingFile.checkIn(userName);
			forceNotify(currentlyUploadingFile);
			forceNotify("FILE_LIST");
			return RequestType.FILE_WRITE_OK;
		} else {
			return RequestType.FILE_NOT_EXPECTED;
		}
		//}}}
	}
	//}}}

	//{{{ deleteFile
	/**
	 *  deletes a file from the disk
	 *
	 *@param  fileURI  the file to delete
	 *@return          reply code to client
	 */
	public int deleteFile(URI fileURI) {
		DefaultMutableTreeNode theNode  = getNodeForFile(fileURI);
		ProColFile theFile              = getFileInTree(fileURI);
		if (theFile.isCheckedOut()) {
			return RequestType.FILE_CHECKED_OUT;
		}
		// TODO: add check for user access rights
		boolean deleteResult            = theFile.delete();
		if (deleteResult) {
			theNode.removeFromParent();
			this.forceNotify("FILE_LIST");
			return RequestType.FILE_OK;
		} else {
			return RequestType.FILE_IO_ERROR;
		}
	}
	//}}}
	//{{{ moveFile
	/**
	 *  moves a file on the disk
	 *
	 *@param  sourceURI  the source file
	 *@param  destURI    the destination file
	 *@return            reply code to client
	 */
	public int moveFile(URI sourceURI, URI destURI) {
		DefaultMutableTreeNode sourceNode      = getNodeForFile(sourceURI);
		ProColFile sourceFile                  = getFileInTree(sourceURI);
		ProColFile destFile                    = new ProColFile(projectFilesURI.resolve(destURI));
		DefaultMutableTreeNode destParentNode  = getNodeForFile(projectFilesURI.relativize(projectFilesURI.resolve(destFile.getParentFile().toURI())));
		// TODO: add check for user access rights
		destFile.getParentFile().mkdirs();
		if (!destFile.exists()) {
			boolean moveResult  = sourceFile.renameTo(destFile);
			if (moveResult) {
				URI destVersionsURI  = versionsURI.resolve(projectFilesURI.relativize(new File(destFile.getPath() + versionsSuffix).toURI()));
				sourceFile.getVersionsFile().renameTo(new File(destVersionsURI));
				destFile.setVersionsFile(destVersionsURI);
				destParentNode.add(new DefaultMutableTreeNode(destFile));
				sourceNode.removeFromParent();
				this.forceNotify("FILE_LIST");
				return RequestType.FILE_OK;
			} else {
				return RequestType.FILE_IO_ERROR;
			}
		} else {
			return RequestType.FILE_EXISTS;
		}
	}
	//}}}
	//{{{ newFile
	/**
	 *  creates a new file or directory
	 *
	 *@param  fileURI      The file to create
	 *@param  isDirectory  True if file to create is a directory, false if a file
	 *@return              reply code to client
	 */
	public int newFile(URI fileURI, boolean isDirectory) {
		ProColFile newFile                    = new ProColFile(projectFilesURI.resolve(fileURI));
		DefaultMutableTreeNode newParentNode  = getNodeForFile(projectFilesURI.relativize(projectFilesURI.resolve(newFile.getParentFile().toURI())));
		if (!newFile.exists()) {
			boolean newResult  = false;
			if (isDirectory) {
				newResult = newFile.mkdirs();
			} else {
				try {
					newResult = newFile.createNewFile();
					if (newResult) {
						URI newVersionsURI  = versionsURI.resolve(projectFilesURI.relativize(new File(newFile.getPath() + versionsSuffix).toURI()));
						newFile.setVersionsFile(newVersionsURI);
					}
				} catch (IOException ioe) {
					return RequestType.FILE_IO_ERROR;
				}
			}
			if (!newResult) {
				return RequestType.FILE_IO_ERROR;
			} else {
				newParentNode.add(new DefaultMutableTreeNode(newFile));
				this.forceNotify("FILE_LIST");
				return RequestType.FILE_OK;
			}
		} else {
			return RequestType.FILE_EXISTS;
		}
	}
	//}}}

	//{{{ unloadFiles
	/**  unloads the files for this project */
	public void unloadFiles() {
		ProColServer.printInfo("Unloading files for " + name + " project.");
		this.projectFilesRootNode = null;
	}
	//}}}
//}}}

//{{{ Modifier methods
	//{{{ initOptions
	private void initOptions() {
		projectFilesURI = getRootURI("files.dir", "files", true);
		versionsURI = getRootURI("versions.dir", "versions", true);
		backupURI = getRootURI("backup.dir", "backup", true);
		privateMessagesURI = getRootURI("privatemessages.dir", "privmsg", true);
		publicMessagesURI = getRootURI("publicmessages.dir", "publicmsg", false);
		todoListURI = getRootURI("todolist.dir", "todo", false);
		bugListURI = getRootURI("buglist.dir", "bugs", false);
		calendarURI = getRootURI("calendar.dir", "calendar", false);

		versionsSuffix = projectProps.getProperty("versions.suffix", ".versions");
		versionsAllowNoChange = projectProps.getProperty("versions.allownochange", "false").equals("true");

		numBackups = Integer.parseInt(projectProps.getProperty("backup.number", "3"));
		backupSuffix = projectProps.getProperty("backup.suffix", "~");
		makeBackups = projectProps.getProperty("backup.enabled", "true").equals("true");

		allowAnonymousLogin = projectProps.getProperty("users.anonymous", "deny").equals("allow");

		String tempUsers           = projectProps.getProperty("users.allow", "");
		StringTokenizer tokenizer  = new StringTokenizer(tempUsers, ", ");
		while (tokenizer.hasMoreTokens()) {
			String temp  = tokenizer.nextToken().trim();
			allowedUsers.add(temp);
		}
	}
	//}}}

	//{{{ getRootURI
	private URI getRootURI(String name, String defaultName, boolean isDir) {
		File file  = new File(this.projectURI.getPath() + File.separator + projectProps.getProperty(name, defaultName));
		if (!file.exists() && isDir) {
			file.mkdirs();
			ProColServer.printInfo("Empty project directory created: " + file.getPath());
		}
		return file.toURI();
	}
	//}}}

	//{{{ loadFiles(URI projectRootIn)
	private void loadFiles(URI projectRootIn) {
		ProColFile tempFileRoot  = new ProColFile(projectRootIn);
		this.projectFilesRootNode = new DefaultMutableTreeNode(tempFileRoot);
		this.loadFile(projectFilesRootNode);
		this.forceNotify("FILE_LIST");
	}
	//}}}

	//{{{ loadFile
	private void loadFile(DefaultMutableTreeNode node) {
		File[] files  = ((File)(node.getUserObject())).listFiles();
		if (files != null) {
			Arrays.sort(files, new ProColFileComparator());
			for (int i = 0; i < files.length; i++) {
				ProColFile tempFile              = new ProColFile(files[i].getPath());
				tempFile.setVersionsFile(versionsURI.resolve(projectFilesURI.relativize(new File(tempFile.getPath() + versionsSuffix).toURI())));
				DefaultMutableTreeNode tempNode  = new DefaultMutableTreeNode(tempFile);
				node.add(tempNode);
				this.loadFile(tempNode);
			}
		}
	}
	//}}}
//}}}
}

