//{{{ GPL Notice
/*
 *  IncomingMessageHandler.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.server.net;

//{{{ Imports
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import com.enderak.procol.server.*;
import com.enderak.procol.server.model.*;
import java.io.*;
import java.net.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
//}}}

/**
 *  Maintains the server-side connection between the client and the server
 *
 *@author    Justin Dieters
 */
public class IncomingMessageHandler extends Thread {
//{{{ Data members
	protected ProColPacket tempPacket;
	protected boolean authenticated        = false;
	private ServerConnection connection;
	private PacketFactory packetFactory;
	private MessageFactory messageFactory;
	private boolean isRunning              = false;
	private URI tempURI;
	private StringTokenizer tokenizer;
	private String filePath;
	private URI fileURI;
	private int result;
	//}}}

//{{{ Constructors
	/**
	 *  Constructor for the IncomingMessageHandler object
	 *
	 *@param  connectionIn  The connection this handler is associated with
	 */
	public IncomingMessageHandler(ServerConnection connectionIn) {
		connection = connectionIn;
		packetFactory = connectionIn.getPacketFactory();
		messageFactory = connectionIn.getMessageFactory();
	}
	//}}}

//{{{ Thread methods
	//{{{ run
	/**
	 *  Main processing method for the IncomingMessageHandler object. Waits for new
	 *  messages and handles them when they arrive
	 */
	public void run() {
		this.isRunning = true;
		while (this.isRunning) {
			synchronized (messageFactory) {
				while (!messageFactory.hasNext() && this.isRunning) {
					try {
						messageFactory.wait();
					} catch (InterruptedException ie) {
					}
				}
				this.handleMessage((ProColIncomingMessage)(messageFactory.getNext()));
			}
		}
	}
	//}}}

	//{{{ close
	/**  Closes the message handler */
	public void close() {
		this.isRunning = false;
	}
	//}}}
//}}}

//{{{ handleMessage
	private void handleMessage(ProColIncomingMessage message) {
		if (message == null) {
			return;
		}
		ProColServer.printInfo("Handling message: " + message.toString());

		if (this.authenticated) {
			switch (message.requestCode) {
				case RequestType.CHECK_OUT_FILE:
				{ //{{{
					tokenizer = new StringTokenizer(new String(message.data), "\n");
					fileURI = URI.create(tokenizer.nextToken());
					connection.checkOutFiles(fileURI);
					break;
				} //}}}
				case RequestType.CLOSE_PROJECT:
				{ //{{{
					this.connection.closeProject();
					break;
				} //}}}
				case RequestType.DISCONNECT:
				{ //{{{
					this.authenticated = false;
					connection.close();
					break;
				} //}}}
				case RequestType.DOWNLOAD_FILE:
				{ //{{{
					tokenizer = new StringTokenizer(new String(message.data), "\n");
					fileURI = URI.create(tokenizer.nextToken());
					connection.sendFiles(fileURI, RequestType.DOWNLOAD_FILE);
					break;
				} //}}}
				case RequestType.DOWNLOAD_SNAPSHOT:
				{ //{{{
					connection.sendSnapshot();
					break;
				} //}}}
				case RequestType.OPEN_PROJECT:
				{ //{{{
					connection.openProject(new String(message.data));
					connection.sendFileTree();
					break;
				} //}}}
				case RequestType.GET_FILE_HISTORY:
				{ //{{{
					fileURI = URI.create(new String(message.data));
					connection.sendFileHistory(fileURI);
					break;
				} //}}}
				case RequestType.GET_FILE_TREE:
				{ //{{{
					connection.sendFileTree();
					break;
				} //}}}
				case RequestType.GET_PROJECT_INFO:
				{ //{{{
					packetFactory.addToQueue(RequestType.PROJECT_INFO, RunProColServer.getServer().getProject(new String(message.data)).getProjectDescription(), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_PROJECT_LIST:
				{ //{{{
					Vector tempList  = RunProColServer.getServer().getAllowedProjects(connection.getUser());
					if (tempList.isEmpty()) {
						packetFactory.addToQueue(RequestType.NO_PROJECTS, PacketFactory.NORMAL_PRIORITY);
					} else {
						packetFactory.addToQueue(RequestType.PROJECT_LIST, ProColUtils.vectorToString(tempList), PacketFactory.NORMAL_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.GET_USER_INFO:
				{ //{{{
					packetFactory.addToQueue(RequestType.USER_INFO, connection.getUser().getUserInfo(), PacketFactory.MAX_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_USER_LIST:
				{ //{{{
					packetFactory.addToQueue(RequestType.USER_LIST, ProColUtils.vectorToString(connection.getProject().getUserList()), PacketFactory.MAX_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_PROJECT_USERS:
				{ //{{{
					packetFactory.addToQueue(RequestType.GET_PROJECT_USERS, ProColUtils.vectorToString(connection.getProject().getAllowedUsers()), PacketFactory.MAX_PRIORITY);
					break;
				} //}}}
				case RequestType.KILL_SERVER:
				{ //{{{
					RunProColServer.stopServer();
					break;
				} //}}}
				case RequestType.GET_PRIVATE_MESSAGE_LIST:
				{ //{{{
					connection.getPrivateMessageList();
					break;
				} //}}}
				case RequestType.PRIVATE_MESSAGE_ADD:
				{ //{{{
					try {
						String privateMessage    = new String(message.data);
						tokenizer = new StringTokenizer(privateMessage, "\n");
						String recipient         = tokenizer.nextToken();
						String replyTo           = tokenizer.nextToken();
						String subject           = tokenizer.nextToken();
						StringBuffer theMessage  = new StringBuffer();
						while (tokenizer.hasMoreTokens()) {
							theMessage.append(tokenizer.nextToken());
							if (tokenizer.hasMoreTokens()) {
								theMessage.append("\n");
							}
						}
						packetFactory.addToQueue(connection.addPrivateMessage(connection.getUser().getName(), recipient, replyTo, subject, theMessage.toString()), recipient, PacketFactory.NORMAL_PRIORITY);
					} catch (NoSuchElementException nsee) {
						packetFactory.addToQueue(RequestType.PRIVATE_MESSAGE_ADD_FAILED, PacketFactory.MAX_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.PRIVATE_MESSAGE_DELETE:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(connection.deletePrivateMessage(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_PRIVATE_MESSAGE:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(RequestType.PRIVATE_MESSAGE, connection.getPrivateMessage(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_PUBLIC_MESSAGE_LIST:
				{ //{{{
					connection.getPublicMessageList();
					break;
				} //}}}
				case RequestType.PUBLIC_MESSAGE_ADD:
				{ //{{{
					try {
						String publicMessage     = new String(message.data);
						tokenizer = new StringTokenizer(publicMessage, "\n");
						String replyTo           = tokenizer.nextToken();
						String subject           = tokenizer.nextToken();
						StringBuffer theMessage  = new StringBuffer();
						while (tokenizer.hasMoreTokens()) {
							theMessage.append(tokenizer.nextToken());
							if (tokenizer.hasMoreTokens()) {
								theMessage.append("\n");
							}
						}
						packetFactory.addToQueue(connection.addPublicMessage(connection.getUser().getName(), replyTo, subject, theMessage.toString()), PacketFactory.NORMAL_PRIORITY);
					} catch (NoSuchElementException nsee) {
						packetFactory.addToQueue(RequestType.PUBLIC_MESSAGE_ADD_FAILED, PacketFactory.MAX_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.PUBLIC_MESSAGE_DELETE:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(connection.deletePublicMessage(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_PUBLIC_MESSAGE:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(RequestType.PUBLIC_MESSAGE, connection.getPublicMessage(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_TODO_ITEM_LIST:
				{ //{{{
					connection.getTodoItemList();
					break;
				} //}}}
				case RequestType.TODO_ITEM_ADD:
				{ //{{{
					try {
						String theMessage         = new String(message.data);
						tokenizer = new StringTokenizer(theMessage, "\n");
						String subject            = tokenizer.nextToken();
						String priority           = tokenizer.nextToken();
						String assignedTo         = tokenizer.nextToken();
						String dueDate            = tokenizer.nextToken();
						String complete           = tokenizer.nextToken();
						StringBuffer description  = new StringBuffer();
						while (tokenizer.hasMoreTokens()) {
							description.append(tokenizer.nextToken());
							if (tokenizer.hasMoreTokens()) {
								description.append("\n");
							}
						}
						packetFactory.addToQueue(connection.addTodoItem(connection.getUser().getName(), subject, priority, assignedTo, dueDate, complete, description.toString()), PacketFactory.NORMAL_PRIORITY);
					} catch (NoSuchElementException nsee) {
						packetFactory.addToQueue(RequestType.TODO_ITEM_ADD_FAILED, PacketFactory.MAX_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.TODO_ITEM_UPDATE:
				{ //{{{
					try {
						String theMessage         = new String(message.data);
						tokenizer = new StringTokenizer(theMessage, "\n");
						String itemNumber         = tokenizer.nextToken();
						String subject            = tokenizer.nextToken();
						String priority           = tokenizer.nextToken();
						String assignedTo         = tokenizer.nextToken();
						String dueDate            = tokenizer.nextToken();
						String complete           = tokenizer.nextToken();
						StringBuffer description  = new StringBuffer();
						while (tokenizer.hasMoreTokens()) {
							description.append(tokenizer.nextToken());
							if (tokenizer.hasMoreTokens()) {
								description.append("\n");
							}
						}
						packetFactory.addToQueue(connection.updateTodoItem(connection.getUser().getName(), itemNumber, subject, priority, assignedTo, dueDate, complete, description.toString()), PacketFactory.NORMAL_PRIORITY);
					} catch (NoSuchElementException nsee) {
						packetFactory.addToQueue(RequestType.TODO_ITEM_UPDATE_FAILED, PacketFactory.MAX_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.TODO_ITEM_DELETE:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(connection.deleteTodoItem(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_TODO_ITEM:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(RequestType.TODO_ITEM, connection.getTodoItem(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_BUG_ITEM_LIST:
				{ //{{{
					connection.getBugItemList();
					break;
				} //}}}
				case RequestType.BUG_ITEM_ADD:
				{ //{{{
					try {
						String theMessage         = new String(message.data);
						tokenizer = new StringTokenizer(theMessage, "\n");
						String subject            = tokenizer.nextToken();
						String priority           = tokenizer.nextToken();
						String assignedTo         = tokenizer.nextToken();
						String dueDate            = tokenizer.nextToken();
						String fixed              = tokenizer.nextToken();
						StringBuffer description  = new StringBuffer();
						while (tokenizer.hasMoreTokens()) {
							description.append(tokenizer.nextToken());
							if (tokenizer.hasMoreTokens()) {
								description.append("\n");
							}
						}
						packetFactory.addToQueue(connection.addBugItem(connection.getUser().getName(), subject, priority, assignedTo, dueDate, fixed, description.toString()), PacketFactory.NORMAL_PRIORITY);
					} catch (NoSuchElementException nsee) {
						packetFactory.addToQueue(RequestType.BUG_ITEM_ADD_FAILED, PacketFactory.MAX_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.BUG_ITEM_UPDATE:
				{ //{{{
					try {
						String theMessage         = new String(message.data);
						tokenizer = new StringTokenizer(theMessage, "\n");
						String itemNumber         = tokenizer.nextToken();
						String subject            = tokenizer.nextToken();
						String priority           = tokenizer.nextToken();
						String assignedTo         = tokenizer.nextToken();
						String dueDate            = tokenizer.nextToken();
						String complete           = tokenizer.nextToken();
						StringBuffer description  = new StringBuffer();
						while (tokenizer.hasMoreTokens()) {
							description.append(tokenizer.nextToken());
							if (tokenizer.hasMoreTokens()) {
								description.append("\n");
							}
						}
						packetFactory.addToQueue(connection.updateBugItem(connection.getUser().getName(), itemNumber, subject, priority, assignedTo, dueDate, complete, description.toString()), PacketFactory.NORMAL_PRIORITY);
					} catch (NoSuchElementException nsee) {
						packetFactory.addToQueue(RequestType.BUG_ITEM_UPDATE_FAILED, PacketFactory.MAX_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.BUG_ITEM_DELETE:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(connection.deleteBugItem(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.GET_BUG_ITEM:
				{ //{{{
					String messageID  = new String(message.data);
					packetFactory.addToQueue(RequestType.BUG_ITEM, connection.getBugItem(messageID), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.UPLOAD_FILE_INFO:
				{ //{{{
					String fileInfo  = new String(message.data);
					tokenizer = new StringTokenizer(fileInfo, "\n");
					filePath = tokenizer.nextToken();
					result = connection.getProject().addDownloadFile(message.data, connection.getUser().getName());
					sendUploadResult(result, filePath);
					break;
				} //}}}
				case RequestType.UPLOAD_FILE_WRITE:
				{ //{{{
					ProColServer.printInfo("got file data");
					tokenizer = new StringTokenizer(new String(message.data), "\n");
					filePath = tokenizer.nextToken();
					result = connection.getProject().writeFile(filePath, message.data, connection.getUser().getName());
					sendUploadResult(result, filePath);
					break;
				} //}}}
				case RequestType.CHECK_IN_FILE:
				{ //{{{
					tempURI = URI.create(new String(message.data));
					int checkInResponse  = connection.getProject().checkIn(tempURI, connection.getUser());
					packetFactory.addToQueue(checkInResponse, new String(message.data), PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				case RequestType.CHECK_IN_FILE_INFO:
				{ //{{{
					String fileInfo            = new String(message.data);
					StringTokenizer tokenizer  = new StringTokenizer(fileInfo, "\n");
					filePath = tokenizer.nextToken();
					result = connection.getProject().addDownloadFile(message.data, connection.getUser().getName());

					sendCheckInResult(result, filePath);
					break;
				} //}}}
				case RequestType.CHECK_IN_FILE_WRITE:
				{ //{{{
					ProColServer.printInfo("got file data");
					tokenizer = new StringTokenizer(new String(message.data), "\n");
					filePath = tokenizer.nextToken();
					result = connection.getProject().writeFile(filePath, message.data, connection.getUser().getName());
					sendCheckInResult(result, filePath);
					break;
				} //}}}
				case RequestType.DELETE_FILE:
				{ //{{{
					tempURI = URI.create(new String(message.data));
					result = connection.getProject().deleteFile(tempURI);
					sendDeleteResult(result, new String(message.data));
					break;
				} //}}}
				case RequestType.MOVE_FILE:
				{ //{{{
					tokenizer = new StringTokenizer(new String(message.data), "\n");
					String existPath  = tokenizer.nextToken();
					String newPath    = tokenizer.nextToken();
					URI sourceURI     = URI.create(existPath);
					URI destURI       = URI.create(newPath);
					result = connection.getProject().moveFile(sourceURI, destURI);
					sendMoveResult(result, existPath, newPath);
					break;
				} //}}}
				case RequestType.RENAME_FILE:
				{ //{{{
					tokenizer = new StringTokenizer(new String(message.data), "\n");
					String existPath  = tokenizer.nextToken();
					String newPath    = tokenizer.nextToken();
					URI sourceURI     = URI.create(existPath);
					URI destURI       = URI.create(newPath);
					result = connection.getProject().moveFile(sourceURI, destURI);
					sendRenameResult(result, existPath, newPath);
					break;
				} //}}}
				case RequestType.NEW_FILE:
				{ //{{{
					String newPath  = new String(message.data);
					URI newURI      = URI.create(newPath);
					int result      = connection.getProject().newFile(newURI, false);
					sendCreateResult(result, newPath);
					break;
				} //}}}
				case RequestType.NEW_DIRECTORY:
				{ //{{{
					String newPath  = new String(message.data);
					URI newURI      = URI.create(newPath);
					int result      = connection.getProject().newFile(newURI, true);
					sendCreateResult(result, newPath);
					break;
				} //}}}
				default:
				{ //{{{
					ProColServer.printErr("Unknown client request: " + message.toString());
					packetFactory.addToQueue(RequestType.INVALID, Integer.toHexString(message.requestCode), PacketFactory.NORMAL_PRIORITY);
				} //}}}
			}
		} else {
			switch (message.requestCode) {
				case RequestType.AUTHENTICATE:
				{ //{{{
					BufferedReader tempReader  = new BufferedReader(new StringReader(new String(message.data)));
					try {
						String username           = tempReader.readLine();
						String password           = tempReader.readLine();
						int authenticationResult  = RunProColServer.getServer().authenticateUser(username, password);
						if (authenticationResult == RequestType.AUTHENTICATION_OK) {
							this.authenticated = true;
							this.connection.setUser(new ProColUser(username, RunProColServer.getServer().getUserProperty(username + ".fullname")));
						}
						packetFactory.addToQueue(authenticationResult, username, PacketFactory.NORMAL_PRIORITY);
					} catch (IOException e) {
						packetFactory.addToQueue(RequestType.AUTHENTICATION_ERROR, PacketFactory.NORMAL_PRIORITY);
					}
					break;
				} //}}}
				case RequestType.HELLO:
				{ //{{{
					packetFactory.addToQueue(RequestType.HELLO, RunProColServer.getServer().getProperty("options.procol.server.hellomessage", "Authenticate"), PacketFactory.NORMAL_PRIORITY);
					packetFactory.addToQueue(this.connection.checkProtocol(message), RequestType.PROTOCOL_VERSION, PacketFactory.NORMAL_PRIORITY);
					break;
				} //}}}
				default:
				{ //{{{
					ProColServer.printErr("Unknown client request: " + message.toString());
					packetFactory.addToQueue(RequestType.INVALID, Integer.toHexString(message.requestCode), PacketFactory.NORMAL_PRIORITY);
				} //}}}
			}
		}
	}
	//}}}

//{{{ Send result methods
	//{{{ sendCheckInResult
	private void sendCheckInResult(int result, String filePath) {
		if (result == RequestType.FILE_OK) {
			result = RequestType.CHECK_IN_FILE_OK;
		} else if (result == RequestType.FILE_INFO_OK) {
			result = RequestType.CHECK_IN_FILE_INFO_OK;
		} else if (result == RequestType.FILE_WRITE) {
			result = RequestType.CHECK_IN_FILE_WRITE;
		} else if (result == RequestType.FILE_WRITE_OK) {
			result = RequestType.CHECK_IN_FILE_WRITE_OK;
		} else if (result == RequestType.FILE_NAME_FAILED) {
			result = RequestType.CHECK_IN_FILE_NAME_FAILED;
		} else if (result == RequestType.FILE_VERSION_NO_OVERWRITE) {
			result = RequestType.CHECK_IN_FILE_VERSION_NO_OVERWRITE;
		} else if (result == RequestType.FILE_VERSION_OLD) {
			result = RequestType.CHECK_IN_FILE_VERSION_OLD;
		} else if (result == RequestType.FILE_VERSION_IO_ERROR) {
			result = RequestType.CHECK_IN_FILE_VERSION_IO_ERROR;
		} else if (result == RequestType.FILE_NOT_FOUND) {
			result = RequestType.CHECK_IN_FILE_NOT_FOUND;
		} else if (result == RequestType.FILE_IO_ERROR) {
			result = RequestType.CHECK_IN_FILE_IO_ERROR;
		} else if (result == RequestType.FILE_NOT_EXPECTED) {
			result = RequestType.CHECK_IN_FILE_NOT_EXPECTED;
		} else if (result == RequestType.FILE_UNKNOWN_ERROR) {
			result = RequestType.CHECK_IN_FILE_UNKNOWN_ERROR;
		}
		packetFactory.addToQueue(result, filePath, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}

	//{{{ sendUploadResult
	private void sendUploadResult(int result, String filePath) {
		if (result == RequestType.FILE_OK) {
			result = RequestType.UPLOAD_FILE_OK;
		} else if (result == RequestType.FILE_INFO_OK) {
			result = RequestType.UPLOAD_FILE_INFO_OK;
		} else if (result == RequestType.FILE_WRITE) {
			result = RequestType.UPLOAD_FILE_WRITE;
		} else if (result == RequestType.FILE_WRITE_OK) {
			result = RequestType.UPLOAD_FILE_WRITE_OK;
		} else if (result == RequestType.FILE_NAME_FAILED) {
			result = RequestType.UPLOAD_FILE_NAME_FAILED;
		} else if (result == RequestType.FILE_VERSION_IO_ERROR) {
			result = RequestType.UPLOAD_FILE_VERSION_IO_ERROR;
		} else if (result == RequestType.FILE_IO_ERROR) {
			result = RequestType.UPLOAD_FILE_IO_ERROR;
		} else if (result == RequestType.FILE_NOT_EXPECTED) {
			result = RequestType.UPLOAD_FILE_NOT_EXPECTED;
		} else if (result == RequestType.FILE_UNKNOWN_ERROR) {
			result = RequestType.UPLOAD_FILE_UNKNOWN_ERROR;
		}
		packetFactory.addToQueue(result, filePath, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}

	//{{{ sendMoveResult
	private void sendMoveResult(int result, String existPath, String newPath) {
		if (result == RequestType.FILE_OK) {
			result = RequestType.MOVE_FILE_OK;
		} else if (result == RequestType.FILE_EXISTS) {
			result = RequestType.MOVE_FILE_EXISTS;
		} else if (result == RequestType.FILE_IO_ERROR) {
			result = RequestType.MOVE_FILE_IO_ERROR;
		} else if (result == RequestType.FILE_NOT_ALLOWED) {
			result = RequestType.MOVE_FILE_NOT_ALLOWED;
		} else if (result == RequestType.FILE_UNKNOWN_ERROR) {
			result = RequestType.MOVE_FILE_UNKNOWN_ERROR;
		}
		packetFactory.addToQueue(result, existPath + "\n" + newPath, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}

	//{{{ sendRenameResult
	private void sendRenameResult(int result, String existPath, String newPath) {
		if (result == RequestType.FILE_OK) {
			result = RequestType.RENAME_FILE_OK;
		} else if (result == RequestType.FILE_EXISTS) {
			result = RequestType.RENAME_FILE_EXISTS;
		} else if (result == RequestType.FILE_IO_ERROR) {
			result = RequestType.RENAME_FILE_IO_ERROR;
		} else if (result == RequestType.FILE_NOT_ALLOWED) {
			result = RequestType.RENAME_FILE_NOT_ALLOWED;
		} else if (result == RequestType.FILE_UNKNOWN_ERROR) {
			result = RequestType.RENAME_FILE_UNKNOWN_ERROR;
		}
		packetFactory.addToQueue(result, existPath + "\n" + newPath, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}

	//{{{ sendDeleteResult
	private void sendDeleteResult(int result, String filePath) {
		if (result == RequestType.FILE_OK) {
			result = RequestType.DELETE_FILE_OK;
		} else if (result == RequestType.FILE_IO_ERROR) {
			result = RequestType.DELETE_FILE_IO_ERROR;
		} else if (result == RequestType.FILE_NOT_ALLOWED) {
			result = RequestType.DELETE_FILE_NOT_ALLOWED;
		} else if (result == RequestType.FILE_UNKNOWN_ERROR) {
			result = RequestType.DELETE_FILE_UNKNOWN_ERROR;
		}
		packetFactory.addToQueue(result, filePath, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}

	//{{{ sendCreateResult
	private void sendCreateResult(int result, String filePath) {
		if (result == RequestType.FILE_OK) {
			result = RequestType.NEW_FILE_OK;
		} else if (result == RequestType.FILE_IO_ERROR) {
			result = RequestType.NEW_FILE_IO_ERROR;
		} else if (result == RequestType.FILE_NOT_ALLOWED) {
			result = RequestType.NEW_FILE_NOT_ALLOWED;
		} else if (result == RequestType.FILE_UNKNOWN_ERROR) {
			result = RequestType.NEW_FILE_UNKNOWN_ERROR;
		}
		packetFactory.addToQueue(result, filePath, PacketFactory.NORMAL_PRIORITY);
	}
	//}}}
//}}}
}
