//{{{ GPL Notice
/*
 *  ProColServerOptionPane.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.server.net;

//{{{ Imports
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;

import com.enderak.procol.common.util.*;
import com.enderak.procol.server.*;
import com.enderak.procol.server.gui.*;
import com.enderak.procol.server.model.*;
import com.enderak.procol.server.util.*;
import com.sun.net.ssl.internal.ssl.Provider;

import java.io.*;
import java.security.Security;
import java.text.*;
import java.util.*;
import javax.net.ssl.*;

import org.gjt.sp.jedit.jEdit;
//}}}

/**
 *  The main ProCol server class. Accepts connections and creates
 *  ServerConnections to maintain them. Keeps track of the project and
 *  connection lists, server properties, etc.
 *
 *@author    Justin Dieters
 */
public final class ProColServer extends EnhancedObservable implements Runnable {
//{{{ Data members
	private final static String DEFAULT_PROPS_FILE_NAME    = "/com/enderak/procol/server/server.props";
	private final static String DEFAULT_PROCOL_DIR_NAME    = ".procol" + System.getProperty("file.separator") + "server";
	private final static String DEFAULT_PROJECTS_DIR_NAME  = "projects";
	private final static String DEFAULT_USER_PROPS_NAME    = "users.props";
	private final static String DEFAULT_CUSTOM_PROPS_NAME  = "server.props";
	private final static String FILE_SEPARATOR             = System.getProperty("file.separator");
	private static SSLServerSocket serverSocket            = null;
	private static Vector connections                      = new Vector();
	private boolean serverRunning                          = false;
	private Properties serverProps, userProps;
	private boolean withinJEdit                            = false;
	private Vector projectList                             = new Vector();
	private File projectDir, proColDir, customServerPropsFile, userPropsFile;
	private File[] projectDirs;
//}}}

//{{{ Print methods
	//{{{ printErr(String errIn)
	/**
	 *  Prints an error message. If within jEdit, displays error in Server Control
	 *  Panel. If standalone, prints to System.err.
	 *
	 *@param  errIn  Description of the Parameter
	 */
	public static void printErr(String errIn) {
		String err  = "[" + new SimpleDateFormat(RunProColServer.getServer().getProperty("options.procol.server.logtimeformat", "MM/dd/yy HH:mm:ss")).format(new Date()) + "] ERROR: " + errIn;
		if (RunProColServer.getServer().isWithinJEdit()) {
			((ProColServerDockableLog)(jEdit.getLastView().getDockableWindowManager().getDockable("procol.server.dockable.log"))).printErr(err);
		} else {
			System.err.println(err);
		}
	}
	//}}}

	//{{{ printInfo(String infoIn)
	/**
	 *  Prints an informational message. If within jEdit, displays message in
	 *  Server Control Panel. If standalone, prints to System.out.
	 *
	 *@param  infoIn  Description of the Parameter
	 */
	public static void printInfo(String infoIn) {
		String info  = "[" + new SimpleDateFormat(RunProColServer.getServer().getProperty("options.procol.server.logtimeformat", "MM/dd/yy HH:mm:ss")).format(new Date()) + "] " + infoIn;
		if (RunProColServer.getServer().isWithinJEdit()) {
			((ProColServerDockableLog)(jEdit.getLastView().getDockableWindowManager().getDockable("procol.server.dockable.log"))).printInfo(info);
		} else {
			System.out.println(info);
		}
	}
	//}}}
//}}}

//{{{ Thread methods
	//{{{ run()
	/**  Main processing method for the ProColServer object */
	public void run() {

		loadProperties(); // must load properties before using printInfo or printErr...

		printInfo("Starting ProCol Server...");

		initProjects();

		// printInfo(jEdit.getEditServer().getPort() + "=?" + this.getPort());
		if (withinJEdit && jEdit.getEditServer().getPort() == this.getPort()) {
			this.setProperty("options.procol.server.port", "" + (this.getPort() + 1));
			ProColServerDockable.displayError("Port in Use", "jEdit has reserved port " + "jEdit.getEditServer().getPort()" + " for use with EditServer.\nThe ProCol server will use port " + this.getPort() + " instead.\nRestarting jEdit should allow the ProCol server to start normally.");
		}

		try {
			Security.addProvider(new Provider());
			SSLServerSocketFactory sslserversocketfactory  = (SSLServerSocketFactory)SSLServerSocketFactory.getDefault();
			serverSocket = (SSLServerSocket)sslserversocketfactory.createServerSocket(getPort());
			SSLCommon.enableAnonymousCipherSuites(serverSocket);
		} catch (IOException e) {
			ProColServer.printErr("IOException opening server socket channel!");
			return;
		}

		this.serverRunning = true;

		printInfo("ProCol Server running on port " + getPort());

		while (this.serverRunning) {
			try {
				ServerConnection connection  = new ServerConnection((SSLSocket)serverSocket.accept());
				connections.add(connection);
			} catch (IOException ioexception) {
				ProColServer.printErr("IOException opening socket! " + ioexception);
			}
		}

		ProColServer.printInfo("Stopping ProCol Server...");

		this.closeAllConnections();

		ProColServer.printInfo("ProCol Server Stopped");
	}
	//}}}
//}}}

//{{{ Server control methods
	//{{{ stopServer()
	/**  Description of the Method */
	public void stopServer() {
		this.serverRunning = false;
		try {
			serverSocket.close();
		} catch (IOException ioexception) {
			ProColServer.printErr("Could not close socket");
		}
		for (int i = 0; i < connections.size(); i++) {
			closeConnection(((ServerConnection)connections.elementAt(i)));
		}
		connections.removeAllElements();
	}
	//}}}

	/**
	 *  Description of the Method
	 *
	 *@param  connectionIn  Description of the Parameter
	 */
	public void closeConnection(ServerConnection connectionIn) {
		connectionIn.close();
	}


	/**  Description of the Method */
	public void closeAllConnections() {
		for (ListIterator i = projectList.listIterator(); i.hasNext(); ) {
			((ProColServerProject)i.next()).closeAllConnections();
			i.remove();
		}
	}

	//{{{ setProperty(String propertyIn)
	/**
	 *  Gets the property attribute of the ProColServer class
	 *
	 *@param  propertyIn  Description of the Parameter
	 *@param  keyIn
	 *@return             The property value
	 */
	public Object setProperty(String keyIn, String propertyIn) {
		return serverProps.setProperty(keyIn, propertyIn);
	}
	//}}}

	//{{{ getProperty(String propertyIn)
	/**
	 *  Gets the property attribute of the ProColServer class
	 *
	 *@param  propertyIn  Description of the Parameter
	 *@return             The property value
	 */
	public String getProperty(String propertyIn) {
		return serverProps.getProperty(propertyIn);
		//}
	}
	//}}}

	//{{{ getProperty(String propertyIn, String defaultIn)
	/**
	 *  Gets the property attribute of the ProColServer class
	 *
	 *@param  propertyIn  Description of the Parameter
	 *@param  defaultIn   Description of the Parameter
	 *@return             The property value
	 */
	public String getProperty(String propertyIn, String defaultIn) {
		return serverProps.getProperty(propertyIn, defaultIn);
		//}
	}
	//}}}

	//{{{ getUserProperty(String propertyIn)
	/**
	 *  Gets the userProperty attribute of the ProColServer class
	 *
	 *@param  propertyIn  Description of the Parameter
	 *@return             The userProperty value
	 */
	public String getUserProperty(String propertyIn) {
		return userProps.getProperty(propertyIn);
	}
	//}}}

	//{{{ getUserProperty(String propertyIn, String defaultIn)
	/**
	 *  Gets the userProperty attribute of the ProColServer class
	 *
	 *@param  propertyIn  Description of the Parameter
	 *@param  defaultIn   Description of the Parameter
	 *@return             The userProperty value
	 */
	public String getUserProperty(String propertyIn, String defaultIn) {
		return userProps.getProperty(propertyIn, defaultIn);
	}
	//}}}
//}}}

//{{{ Accessor methods
	//{{{ isWithinJEdit()
	/**
	 *  Gets the withinJEdit attribute of the ProColServer class
	 *
	 *@return    The withinJEdit value
	 */
	public boolean isWithinJEdit() {
		return withinJEdit;
	}
	//}}}

	//{{{ isRunning()
	/**
	 *  Gets the running attribute of the ProColServer class
	 *
	 *@return    The running value
	 */
	public boolean isRunning() {
		return serverRunning;
	}
	//}}}

	//{{{ getPort()
	/**
	 *  Gets the port attribute of the ProColServer class
	 *
	 *@return    The port value
	 */
	public int getPort() {
		return Integer.parseInt(this.getProperty("options.procol.server.port", "4937"));
	}
	//}}}

	//{{{ authenticateUser(String usernameIn, String passwordIn)
	/**
	 *  Description of the Method
	 *
	 *@param  usernameIn  Description of the Parameter
	 *@param  passwordIn  Description of the Parameter
	 *@return             Description of the Return Value
	 */
	public int authenticateUser(String usernameIn, String passwordIn) {
		if (isUserLoggedIn(usernameIn)) {
			return RequestType.USER_ALREADY_LOGGED_IN;
		}
		if (passwordIn.equals(getUserProperty(usernameIn + ".password"))) {
			return RequestType.AUTHENTICATION_OK;
		} else if (getUserProperty(usernameIn + ".password") == null) {
			return RequestType.USER_DOES_NOT_EXIST;
		} else {
			return RequestType.PASSWORD_FAILED;
		}
	}
	//}}}

	//{{{ isUserLoggedIn(String usernameIn)
	/**
	 *  Gets the userLoggedIn attribute of the ProColServer class
	 *
	 *@param  usernameIn  Description of the Parameter
	 *@return             The userLoggedIn value
	 */
	public boolean isUserLoggedIn(String usernameIn) {
		for (int i = 0; i < projectList.size(); i++) {
			Vector tempUserList  = ((ProColServerProject)projectList.elementAt(i)).getUserList();
			for (int j = 0; j < tempUserList.size(); j++) {
				if (usernameIn.equals(tempUserList.elementAt(j))) {
					return true;
				}
			}
		}

		return false;
	}
	//}}}

	//{{{ getProjectList()
	/**
	 *  Gets the projectList attribute of the ProColServer class
	 *
	 *@return    The projectList value
	 */
	public Vector getProjectList() {
		return projectList;
	}
	//}}}

	//{{{ getAllowedProjects(ProColUser userIn)
	/**
	 *  Gets the allowedProjects attribute of the ProColServer class
	 *
	 *@param  userIn  Description of the Parameter
	 *@return         The allowedProjects value
	 */
	public Vector getAllowedProjects(ProColUser userIn) {
		Vector tempProjectList  = new Vector();
		Enumeration projects    = projectList.elements();
		while (projects.hasMoreElements()) {
			ProColServerProject currentProject  = (ProColServerProject)projects.nextElement();
			if (currentProject.getAllowedUsers().contains(userIn.getName())) {
				tempProjectList.add(currentProject.getName());
			}
		}
		return tempProjectList;
	}
	//}}}

	//{{{ getProject(String projectName)
	/**
	 *  Gets the project attribute of the ProColServer object
	 *
	 *@param  projectName  Description of the Parameter
	 *@return              The project value
	 */
	public ProColServerProject getProject(String projectName) {
		for (int i = 0; i < projectList.size(); i++) {
			if (projectName.equals(((ProColServerProject)(projectList.elementAt(i))).getName())) {
				return ((ProColServerProject)(projectList.elementAt(i)));
			}
		}
		if (this.getProperty("options.procol.server.projects.dynamicload", "false").equals("true")) {
			reloadProjectDirs();
			for (int i = 0; i < projectDirs.length; i++) {
				if (projectDirs[i].getName().equals(projectName)) {
					ProColServerProject newProject  = new ProColServerProject(projectDirs[i].toURI(), projectName, true);
					projectList.add(newProject);
					return newProject;
				}
			}
		}
		return null;
	}
	//}}}
//}}}

//{{{ Modifier methods
	//{{{ setWithinJEdit(boolean withinJEditIn)
	/**
	 *  Sets the withinJEdit attribute of the ProColServer class
	 *
	 *@param  withinJEditIn  The new withinJEdit value
	 */
	public void setWithinJEdit(boolean withinJEditIn) {
		withinJEdit = withinJEditIn;
	}
	//}}}

	//{{{ finalize()
	/** */
	protected void finalize() {
		try {
			serverSocket.close();
		} catch (IOException ioexception) {
			ProColServer.printErr("Could not close socket");
		}
	}
//}}}
//}}}

//{{{ Properties methods
	//{{{ loadProperties()
	/**  Description of the Method */
	private void loadProperties() {
		// load defaults from jar file
		serverProps = new Properties();
		userProps = new Properties();
		try {
			// load default server properties
			serverProps.load(ProColServer.class.getResourceAsStream(DEFAULT_PROPS_FILE_NAME));
		} catch (IOException e) {
			ProColServer.printErr("Error loading default server properties: " + e);
		}

		proColDir = new File(System.getProperty("user.home") + FILE_SEPARATOR + DEFAULT_PROCOL_DIR_NAME);
		if (!proColDir.exists()) {
			proColDir.mkdirs();
			ProColServer.printInfo("Empty procol server directory created: " + proColDir.getPath());
		}

		customServerPropsFile = new File(proColDir.getPath() + FILE_SEPARATOR + getProperty("options.procol.server.customprops", DEFAULT_CUSTOM_PROPS_NAME));
		try {
			// load custom server properties
			serverProps.load(new FileInputStream(customServerPropsFile));
		} catch (FileNotFoundException fnfe) {
			try {
				customServerPropsFile.createNewFile();
				ProColServer.printInfo("Empty server properties file created: " + customServerPropsFile.getPath());
			} catch (IOException ioe) {
				ProColServer.printErr("IOException creating empty server properties file: " + customServerPropsFile.getPath());
			}

		} catch (IOException ioe) {
			ProColServer.printErr("IOException loading server properties: " + ioe);
		}

		userPropsFile = new File(proColDir.getPath() + FILE_SEPARATOR + getProperty("options.procol.server.users.list", DEFAULT_USER_PROPS_NAME));
		try {
			userProps.load(new FileInputStream(userPropsFile));
		} catch (FileNotFoundException fnfe) {
			try {
				userPropsFile.createNewFile();
				ProColServer.printInfo("Empty user properties file created: " + userPropsFile.getPath());
			} catch (IOException ioe) {
				ProColServer.printErr("IOException creating empty user properties file: " + userPropsFile.getPath());
			}
		} catch (IOException ioe) {
			ProColServer.printErr("IOException loading server properties: " + ioe);
		}

		// TODO: check command line arguments for overridden options
	}
	//}}}

	//{{{ initProjects()
	/**  Description of the Method */
	private void initProjects() {
		projectDir = new ProColFile(proColDir.getPath() + FILE_SEPARATOR + getProperty("options.procol.server.projects.dir", DEFAULT_PROJECTS_DIR_NAME));
		if (!projectDir.exists()) {
			// create projects directory if none exists
			projectDir.mkdirs();
			ProColServer.printInfo("Empty procol project directory created: " + projectDir.getPath());
		}

		reloadProjectDirs();
		projectList = new Vector();
		for (int i = 0; i < projectDirs.length; i++) {
			ProColServerProject newProject  = new ProColServerProject(projectDirs[i].toURI(), projectDirs[i].getName(), this.getProperty("options.procol.server.projects.dynamicload", "false").equals("false"));
			ProColServer.printInfo("Project loaded: " + newProject.getName());
			projectList.add(newProject);
		}
	}
	//}}}

	private void reloadProjectDirs() {
		projectDirs = projectDir.listFiles(new ProjectFileFilter());
	}
	//}}}
//}}}
}

