//{{{ GPL Notice
/*
 *  ServerConnection.java
 *  :tabSize=4:indentSize=4:noTabs=false:
 *  :folding=explicit:collapseFolds=1:
 *
 *  part of the ProCol plugin for the jEdit text editor
 *  Copyright (C) 2003-2004 Justin Dieters
 *  enderak@yahoo.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//}}}
package com.enderak.procol.server.net;

//{{{ Imports
import com.enderak.procol.common.model.*;
import com.enderak.procol.common.net.*;
import com.enderak.procol.common.util.*;
import com.enderak.procol.server.*;
import com.enderak.procol.server.model.*;
import com.enderak.procol.server.net.*;
import java.io.*;
import java.net.URI;
import java.text.*;
import java.util.*;
import javax.net.ssl.SSLSocket;
import javax.swing.tree.DefaultMutableTreeNode;
//}}}

/**
 *  Maintains the server-side connection between the client and the server
 *
 *@author    Justin Dieters
 */
public class ServerConnection extends Thread implements Observer {
//{{{ Data members
	private SSLSocket client;
	private IncomingMessageHandler imh;
	private OutgoingMessageHandler omh;
	private DataInputStream in;
	private PacketFactory packetFactory    = new PacketFactory();
	private MessageFactory messageFactory  = new MessageFactory();
	private ProColUser user;
	private ProColServerProject project;
	private boolean isRunning              = false;
	private int clientMajorVersion, clientMinorVersion, serverMajorVersion, serverMinorVersion         = 0;
//}}}

//{{{ Constructors
	//{{{ ServerConnection(SSLSocket sslsocket)
	/**
	 *  Constructor for the ServerConnection object
	 *
	 *@param  sslsocket  Description of the Parameter
	 */
	public ServerConnection(SSLSocket sslsocket) {

		if (sslsocket == null) {
			ProColServer.printErr("NULL SSLSocket in ServerConnection constructor");
			return;
		} else {
			this.client = sslsocket;
			SSLCommon.enableAnonymousCipherSuites(client);
			imh = new IncomingMessageHandler(this);
			imh.start();
			omh = new OutgoingMessageHandler(this.client, this.packetFactory);
			omh.start();
			this.start();
		}

	}
	//}}}
//}}}

//{{{ Thread methods
	//{{{ run
	/**  Main processing method for the ServerConnection object */
	public void run() {
		try {
			in = new DataInputStream(client.getInputStream());
		} catch (IOException ioexception) {
			ProColServer.printErr("getInputStream failed");
			return;
		}

		this.isRunning = true;

		while (this.isRunning) {
			this.receive();
		}

		close();
	}
	//}}}
//}}}

//{{{ Communication Methods
	//{{{ receive()
	/**  receives a packet from the network */
	public void receive() {
		try {
			int messageID      = in.readInt();
			int requestCode    = in.readInt();
			int dataRemaining  = in.readInt();
			int dataSize       = in.readInt();
			byte[] data        = new byte[dataSize];
			in.readFully(data);
			messageFactory.addPacket(new ProColPacket(messageID, requestCode, dataRemaining, dataSize, data));
		} catch (IOException ioe) {
			ProColServer.printErr("IOException reading from client in ServerConnection! " + ioe);
			this.isRunning = false;
		}
	}
	//}}}
//}}}

//{{{ Connection control methods
	//{{{ close()
	/**  closes this connection */
	public void close() {
		ProColServer.printInfo("ServerConnection closing!");
		try {
			packetFactory.addToQueue(RequestType.GOODBYE, PacketFactory.MAX_PRIORITY);
			this.user = null;
			this.closeProject();
			synchronized (omh) {
				try {
					omh.wait(5000); // wait 5 seconds for GOODBYE message to be sent
					in.close();
					imh.close();
					omh.close();
				} catch (InterruptedException ie) {
					ProColServer.printErr(ie.toString());
				}
			}
		} catch (IOException ioe) {
			ProColServer.printErr("Error in ServerConnection " + ioe);
		}
		ProColServer.printInfo("ServerConnection closed!");
	}
	//}}}

	//{{{ closeProject
	/**  closes the project associated with this connection */
	public void closeProject() {
		if (this.project != null) {
			this.project.removeConnection(this);
			this.project = null;
		}
	}
	//}}}
//}}}

//{{{ Sign-on methods
	//{{{ openProject(String projectNameIn)
	/**
	 *  Description of the Method
	 *
	 *@param  projectNameIn  Description of the Parameter
	 */
	public void openProject(String projectNameIn) {
		this.project = RunProColServer.getServer().getProject(projectNameIn);
		this.project.addConnection(this);
	}
	//}}}

	//{{{ checkProtocol
	/**
	 *  Checks the client protocol version compared to this protocol version
	 *
	 *@param  message  the message
	 *@return          the reply code to the client
	 */
	public int checkProtocol(ProColIncomingMessage message) {
		StringTokenizer tokenizer;

		//break up client version
		tokenizer = new StringTokenizer(new String(message.data), ".");
		clientMajorVersion = Integer.parseInt(tokenizer.nextToken());
		clientMinorVersion = Integer.parseInt(tokenizer.nextToken());

		//break up server version
		tokenizer = new StringTokenizer(new String(RequestType.PROTOCOL_VERSION), ".");
		serverMajorVersion = Integer.parseInt(tokenizer.nextToken());
		serverMinorVersion = Integer.parseInt(tokenizer.nextToken());

		if (clientMajorVersion == serverMajorVersion) {
			if (clientMinorVersion == serverMinorVersion) {
				return RequestType.PROTOCOL_OK;
			} else {
				return RequestType.PROTOCOL_WARN;
			}
		} else if (clientMajorVersion > serverMajorVersion) {
			return RequestType.PROTOCOL_TOO_NEW;
		} else {
			return RequestType.PROTOCOL_TOO_OLD;
		}
	}
	//}}}
//}}}

//{{{ Accessor methods
	//{{{ getClientSocket
	/**
	 *  Gets the client socket that is associated with this connection
	 *
	 *@return    The client socket
	 */
	public SSLSocket getClientSocket() {
		return this.client;
	}
	//}}}

	//{{{ getUser
	/**
	 *  Gets the user associated with this ServerConnection
	 *
	 *@return    The user
	 */
	public ProColUser getUser() {
		return this.user;
	}
	//}}}

	//{{{ setUser
	/**
	 *  Sets the user associated with this connection
	 *
	 *@param  userIn
	 */
	public void setUser(ProColUser userIn) {
		this.user = userIn;
	}
	//}}}

	//{{{ getProject
	/**
	 *  Gets the project that is associated with this connection
	 *
	 *@return    The project
	 */
	public ProColServerProject getProject() {
		return this.project;
	}
	//}}}

	//{{{ getPacketFactory
	/**
	 *  Gets the packet factory that is associated with this connection
	 *
	 *@return    The packet factory
	 */
	public PacketFactory getPacketFactory() {
		return packetFactory;
	}
	//}}}

	//{{{ getMessageFactory
	/**
	 *  Gets the message factory that is associated with this connection
	 *
	 *@return    The message factory
	 */
	public MessageFactory getMessageFactory() {
		return messageFactory;
	}
	//}}}
//}}}

//{{{ Observable Methods
	//{{{ update
	/**
	 *  Invoked whenever a class that is being observed changes.
	 *
	 *@param  o    the observable object
	 *@param  arg  an argument passed to the notifyObservers method<br>
	 *
	 */
	public void update(Observable o, Object arg) {
		if (arg == null || ((arg instanceof String) && ((String)arg).equals("CONNECTIONS"))) {
			this.packetFactory.addToQueue(RequestType.USER_LIST, ProColUtils.vectorToString(this.project.getUserList()), PacketFactory.MAX_PRIORITY);
		} else if (arg == null || ((arg instanceof String) && ((String)arg).equals("FILE_LIST"))) {
			sendFileTree();
		} else if (arg instanceof ProColFile) {
			// if arg is a file, update client with file properties
			packetFactory.addToQueue(RequestType.UPDATE_FILE_INFO, ((ProColFile)arg).getFileInfo(project.getProjectFilesURI()), PacketFactory.MAX_PRIORITY);
		}
	}
	//}}}
//}}}

//{{{ Outgoing messages
	//{{{ sendFileTree
	/**  Sends the file tree to the client */
	public void sendFileTree() {
		DefaultMutableTreeNode currentLeaf  = project.getProjectFilesRootNode().getFirstLeaf();
		StringBuffer tempBuffer             = new StringBuffer(project.getOptimalStringBufferSize(project.getProjectFilesRootNode()));
		while (currentLeaf != null) {
			ProColFile currentFile  = (ProColFile)currentLeaf.getUserObject();
			tempBuffer.append(currentFile.getFileInfo(project.getProjectFilesURI()));
			currentLeaf = currentLeaf.getNextLeaf();
		}
		packetFactory.addToQueue(RequestType.LOAD_TREE, tempBuffer.toString(), PacketFactory.NORMAL_PRIORITY);
	}
	//}}}

	//{{{ setFileHistory
	/**
	 *  Sends the file history to the client
	 *
	 *@param  filePath  the file path
	 */
	public void sendFileHistory(URI filePath) {
		DefaultMutableTreeNode rootNode  = project.getNodeForFile(filePath);
		ProColFile currentFile           = (ProColFile)rootNode.getUserObject();
		packetFactory.addToQueue(RequestType.FILE_HISTORY, currentFile.getVersionsFile(), PacketFactory.NORMAL_PRIORITY);
	}
	//}}}

	//{{{ sendFiles
	/**
	 *  Sends multiple files to the client
	 *
	 *@param  fileURI  the root URI or file URI
	 *@param  request  The request type
	 */
	public void sendFiles(URI fileURI, int request) {
		try {
			DefaultMutableTreeNode rootNode;
			ProColFile currentFile;

			if (fileURI == null) {
				rootNode = project.getProjectFilesRootNode();
				currentFile = null;
			} else {
				rootNode = project.getNodeForFile(fileURI);
				currentFile = (ProColFile)rootNode.getUserObject();
			}

			if (!rootNode.isLeaf()) {
				Enumeration children                 = rootNode.children();
				DefaultMutableTreeNode currentChild;
				while (children.hasMoreElements()) {
					currentChild = (DefaultMutableTreeNode)(children.nextElement());
					currentFile = (ProColFile)currentChild.getUserObject();
					this.sendFiles(project.getProjectFilesURI().relativize(currentFile.toURI()), request);
				}
			} else if (currentFile != null) {
				packetFactory.addToQueue(request, fileURI, currentFile, PacketFactory.MIN_PRIORITY);
			}
		} catch (NullPointerException npe) {
			packetFactory.addToQueue(RequestType.FILE_NOT_FOUND, fileURI, PacketFactory.NORMAL_PRIORITY);
		}
	}
	//}}}

	//{{{ sendFiles
	/**
	 *  Send all files to the client
	 *
	 *@param  request  The request type
	 */
	public void sendFiles(int request) {
		sendFiles(null, request);
	}
	//}}}

	//{{{ sendSnapshot
	/**  Send all files to the client, as a snapshot */
	public void sendSnapshot() {
		sendFiles(null, RequestType.DOWNLOAD_SNAPSHOT_FILE);
	}
	//}}}

	//{{{ checkOutFiles
	/**
	 *  Check out files
	 *
	 *@param  filePath  The root URI or the file URI
	 */
	public void checkOutFiles(URI filePath) {
		try {
			DefaultMutableTreeNode rootNode  = project.getNodeForFile(filePath);
			ProColFile currentFile           = (ProColFile)rootNode.getUserObject();

			if (!rootNode.isLeaf()) {
				Enumeration children                 = rootNode.children();
				DefaultMutableTreeNode currentChild;
				while (children.hasMoreElements()) {
					currentChild = (DefaultMutableTreeNode)(children.nextElement());
					currentFile = (ProColFile)currentChild.getUserObject();
					this.checkOutFiles(project.getProjectFilesURI().relativize(currentFile.toURI()));
				}
			} else {
				if (currentFile.checkOut(user.getName())) {
					project.forceNotify(currentFile);
					packetFactory.addToQueue(RequestType.CHECK_OUT_FILE, filePath, currentFile, PacketFactory.MIN_PRIORITY);
				} else {
					packetFactory.addToQueue(RequestType.CHECK_OUT_ALREADY_CHECKED_OUT, filePath, PacketFactory.NORMAL_PRIORITY);
				}
			}
		} catch (NullPointerException npe) {
			packetFactory.addToQueue(RequestType.CHECK_OUT_FILE_NOT_FOUND, filePath, PacketFactory.NORMAL_PRIORITY);
		}
	}
	//}}}

	//{{{ addPrivateMessage
	/**
	 *  Adds a private message to the recipient's list
	 *
	 *@param  sender     The Sender
	 *@param  recipient  The Recipient
	 *@param  replyTo    The previous message number, if any
	 *@param  subject    The subject
	 *@param  message    The message
	 *@return            Reply code to the client
	 */
	public int addPrivateMessage(String sender, String recipient, String replyTo, String subject, String message) {
		if (!project.getAllowedUsers().contains(recipient)) {
			return RequestType.PRIVATE_MESSAGE_USER_NOT_EXIST;
		} else {
			File privateMessageFile         = new File(project.getPrivateMessagesURI().getPath() + File.separator + recipient);
			Properties privateMessageProps  = new Properties();
			if (privateMessageFile.exists()) {
				try {
					privateMessageProps.load(new FileInputStream(privateMessageFile));
				} catch (IOException ioe) {
					ProColServer.printErr("IOException loading private message queue for " + recipient);
					return RequestType.PRIVATE_MESSAGE_ADD_FAILED;
				}
			}

			int currentID                   = Integer.parseInt(privateMessageProps.getProperty("current", "0"));
			currentID++;
			privateMessageProps.setProperty("current", "" + currentID);
			privateMessageProps.setProperty(currentID + ".sender", sender);
			privateMessageProps.setProperty(currentID + ".date", "" + new SimpleDateFormat(RunProColServer.getServer().getProperty("options.procol.server.logtimeformat", "MM/dd/yy HH:mm:ss")).format(new Date()));
			privateMessageProps.setProperty(currentID + ".previous", "" + (currentID - 1));
			privateMessageProps.setProperty((currentID - 1) + ".next", "" + currentID);
			privateMessageProps.setProperty(currentID + ".subject", subject);
			privateMessageProps.setProperty(currentID + ".message", message);
			privateMessageProps.setProperty(currentID + ".replyto", replyTo);

			try {
				privateMessageProps.store(new FileOutputStream(privateMessageFile), "Private Messages for " + recipient);
			} catch (IOException ioe) {
				ProColServer.printErr("IOException storing private message queue for " + recipient);
				return RequestType.PRIVATE_MESSAGE_ADD_FAILED;
			}
		}

		ServerConnection recipientConnection  = project.getConnectionFor(recipient);
		if (recipientConnection != null) {
			recipientConnection.getPacketFactory().addToQueue(RequestType.NEW_PRIVATE_MESSAGE, PacketFactory.MAX_PRIORITY);
		}

		return RequestType.PRIVATE_MESSAGE_OK;
	}
	//}}}

	//{{{ getPrivateMessageList
	/**  Send private message list to the client */
	public void getPrivateMessageList() {
		StringBuffer tempBuffer         = new StringBuffer();
		File privateMessageFile         = new File(project.getPrivateMessagesURI().getPath() + File.separator + user.getName());
		Properties privateMessageProps  = new Properties();
		if (privateMessageFile.exists()) {
			try {
				privateMessageProps.load(new FileInputStream(privateMessageFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return;
			}
		}

		String current                  = privateMessageProps.getProperty("current", "0");
		tempBuffer.append("current=" + current);
		while (!current.equals("0")) {
			tempBuffer.append("\n" + current + ".date=" + privateMessageProps.getProperty(current + ".date"));
			tempBuffer.append("\n" + current + ".subject=" + privateMessageProps.getProperty(current + ".subject"));
			tempBuffer.append("\n" + current + ".sender=" + privateMessageProps.getProperty(current + ".sender"));
			tempBuffer.append("\n" + current + ".previous=" + privateMessageProps.getProperty(current + ".previous"));
			tempBuffer.append("\n" + current + ".replyto=" + privateMessageProps.getProperty(current + ".replyto"));
			current = privateMessageProps.getProperty(current + ".previous", "0");
		}

		packetFactory.addToQueue(RequestType.PRIVATE_MESSAGE_LIST, tempBuffer.toString(), PacketFactory.MAX_PRIORITY);
	}
	//}}}

	//{{{ getPrivateMessage
	/**
	 *  Return contents of a message to the client
	 *
	 *@param  messageID  The message to get
	 *@return            The message
	 */
	public String getPrivateMessage(String messageID) {
		Properties privateMessageProps  = new Properties();
		File privateMessageFile         = new File(project.getPrivateMessagesURI().getPath() + File.separator + user.getName());
		if (privateMessageFile.exists()) {
			try {
				privateMessageProps.load(new FileInputStream(privateMessageFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return "(Error loading private messages!)";
			}
		}
		return privateMessageProps.getProperty(messageID + ".message", "(Error getting private message!)");
	}
	//}}}

	//{{{ deletePrivateMessage
	/**
	 *  Deletes a private message
	 *
	 *@param  messageID  The message ID to delete
	 *@return            Reply code to the client
	 */
	public int deletePrivateMessage(String messageID) {
		Properties privateMessageProps  = new Properties();
		File privateMessageFile         = new File(project.getPrivateMessagesURI().getPath() + File.separator + user.getName());
		if (privateMessageFile.exists()) {
			try {
				privateMessageProps.load(new FileInputStream(privateMessageFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return RequestType.PRIVATE_MESSAGE_DELETE_FAILED;
			}
		}

		String nextIndex                = privateMessageProps.getProperty(messageID + ".next");
		String prevIndex                = privateMessageProps.getProperty(messageID + ".previous");

		if (privateMessageProps.getProperty("current").equals(messageID)) {
			privateMessageProps.setProperty("current", prevIndex);
		}

		if (nextIndex != null) {
			privateMessageProps.setProperty(prevIndex + ".next", nextIndex);
			privateMessageProps.setProperty(nextIndex + ".previous", prevIndex);
		}
		privateMessageProps.remove(messageID + ".sender");
		privateMessageProps.remove(messageID + ".date");
		privateMessageProps.remove(messageID + ".previous");
		privateMessageProps.remove(messageID + ".next");
		privateMessageProps.remove(messageID + ".subject");
		privateMessageProps.remove(messageID + ".message");
		privateMessageProps.remove(messageID + ".replyto");

		try {
			privateMessageProps.store(new FileOutputStream(privateMessageFile), "Private Messages for " + user.getName());
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing private message queue for " + user.getName());
			return RequestType.PRIVATE_MESSAGE_DELETE_FAILED;
		}

		getPrivateMessageList();

		return RequestType.PRIVATE_MESSAGE_OK;
	}
	//}}}

	//{{{ addPublicMessage
	/**
	 *  Adds a public message to the list
	 *
	 *@param  sender   The senter
	 *@param  replyTo  The previous message being replied to, if any
	 *@param  subject  The subject
	 *@param  message  The message
	 *@return          Reply code to the client
	 */
	public int addPublicMessage(String sender, String replyTo, String subject, String message) {
		File publicMessageFile         = new File(project.getPublicMessagesURI().getPath());
		Properties publicMessageProps  = new Properties();
		if (publicMessageFile.exists()) {
			try {
				publicMessageProps.load(new FileInputStream(publicMessageFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading public message queue");
				return RequestType.PUBLIC_MESSAGE_ADD_FAILED;
			}
		}

		int currentID                  = Integer.parseInt(publicMessageProps.getProperty("current", "0"));
		currentID++;
		publicMessageProps.setProperty("current", "" + currentID);
		publicMessageProps.setProperty(currentID + ".sender", sender);
		publicMessageProps.setProperty(currentID + ".date", "" + new SimpleDateFormat(RunProColServer.getServer().getProperty("options.procol.server.logtimeformat", "MM/dd/yy HH:mm:ss")).format(new Date()));
		publicMessageProps.setProperty(currentID + ".previous", "" + (currentID - 1));
		publicMessageProps.setProperty((currentID - 1) + ".next", "" + currentID);
		publicMessageProps.setProperty(currentID + ".subject", subject);
		publicMessageProps.setProperty(currentID + ".message", message);
		publicMessageProps.setProperty(currentID + ".replyto", replyTo);

		try {
			publicMessageProps.store(new FileOutputStream(publicMessageFile), "Public messages");
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing public messages");
			return RequestType.PUBLIC_MESSAGE_ADD_FAILED;
		}

		Enumeration connections        = project.getConnections().elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection != null) {
				currentConnection.getPacketFactory().addToQueue(RequestType.NEW_PUBLIC_MESSAGE, PacketFactory.MAX_PRIORITY);
			}
		}

		return RequestType.PUBLIC_MESSAGE_OK;
	}
	//}}}

	//{{{ getPublicMessageList
	/**  Send public message list to the client */
	public void getPublicMessageList() {
		StringBuffer tempBuffer        = new StringBuffer();
		File publicMessageFile         = new File(project.getPublicMessagesURI().getPath());
		Properties publicMessageProps  = new Properties();
		if (publicMessageFile.exists()) {
			try {
				publicMessageProps.load(new FileInputStream(publicMessageFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return;
			}
		}

		String current                 = publicMessageProps.getProperty("current", "0");
		tempBuffer.append("current=" + current);
		while (!current.equals("0")) {
			tempBuffer.append("\n" + current + ".date=" + publicMessageProps.getProperty(current + ".date"));
			tempBuffer.append("\n" + current + ".subject=" + publicMessageProps.getProperty(current + ".subject"));
			tempBuffer.append("\n" + current + ".sender=" + publicMessageProps.getProperty(current + ".sender"));
			tempBuffer.append("\n" + current + ".previous=" + publicMessageProps.getProperty(current + ".previous"));
			tempBuffer.append("\n" + current + ".replyto=" + publicMessageProps.getProperty(current + ".replyto"));
			current = publicMessageProps.getProperty(current + ".previous", "0");
		}

		packetFactory.addToQueue(RequestType.PUBLIC_MESSAGE_LIST, tempBuffer.toString(), PacketFactory.MAX_PRIORITY);
	}
	//}}}

	//{{{ getPublicMessage
	/**
	 *  Return contents of a message to the client
	 *
	 *@param  messageID  The message to get
	 *@return            The message
	 */
	public String getPublicMessage(String messageID) {
		Properties publicMessageProps  = new Properties();
		File publicMessageFile         = new File(project.getPublicMessagesURI().getPath());
		if (publicMessageFile.exists()) {
			try {
				publicMessageProps.load(new FileInputStream(publicMessageFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return "(Error loading private messages!)";
			}
		}
		return publicMessageProps.getProperty(messageID + ".message", "(Error getting private message!)");
	}
	//}}}

	//{{{ deletePublicMessage
	/**
	 *  Deletes a public message
	 *
	 *@param  messageID  The message ID to delete
	 *@return            Reply code to the client
	 */
	public int deletePublicMessage(String messageID) {
		Properties publicMessageProps  = new Properties();
		File publicMessageFile         = new File(project.getPublicMessagesURI().getPath());
		if (publicMessageFile.exists()) {
			try {
				publicMessageProps.load(new FileInputStream(publicMessageFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading public message list");
				return RequestType.PUBLIC_MESSAGE_DELETE_FAILED;
			}
		}

		String nextIndex               = publicMessageProps.getProperty(messageID + ".next");
		String prevIndex               = publicMessageProps.getProperty(messageID + ".previous");

		if (publicMessageProps.getProperty("current").equals(messageID)) {
			publicMessageProps.setProperty("current", prevIndex);
		}

		if (nextIndex != null) {
			publicMessageProps.setProperty(prevIndex + ".next", nextIndex);
			publicMessageProps.setProperty(nextIndex + ".previous", prevIndex);
		}

		publicMessageProps.remove(messageID + ".sender");
		publicMessageProps.remove(messageID + ".date");
		publicMessageProps.remove(messageID + ".previous");
		publicMessageProps.remove(messageID + ".next");
		publicMessageProps.remove(messageID + ".subject");
		publicMessageProps.remove(messageID + ".message");
		publicMessageProps.remove(messageID + ".replyto");

		try {
			publicMessageProps.store(new FileOutputStream(publicMessageFile), "Public Messages");
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing public list");
			return RequestType.PUBLIC_MESSAGE_DELETE_FAILED;
		}

		Enumeration connections        = project.getConnections().elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection != null) {
				currentConnection.getPublicMessageList();
			}
		}

		return RequestType.PUBLIC_MESSAGE_OK;
	}
	//}}}

	//{{{ addTodoItem
	/**
	 *  Add a todo item to the list
	 *
	 *@param  sender       The sender
	 *@param  subject      The subject
	 *@param  priority     The priority
	 *@param  assignedTo   The assignee
	 *@param  dueDate      The due date
	 *@param  complete     percent complete
	 *@param  description  Description of item
	 *@return              Reply code to client
	 */
	public int addTodoItem(String sender, String subject, String priority, String assignedTo, String dueDate, String complete, String description) {
		File todoItemFile         = new File(project.getTodoListURI().getPath());
		Properties todoItemProps  = new Properties();
		if (todoItemFile.exists()) {
			try {
				todoItemProps.load(new FileInputStream(todoItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading todo list");
				return RequestType.TODO_ITEM_ADD_FAILED;
			}
		}

		int currentID             = Integer.parseInt(todoItemProps.getProperty("current", "0"));
		currentID++;
		todoItemProps.setProperty("current", "" + currentID);
		todoItemProps.setProperty(currentID + ".sender", sender);
		todoItemProps.setProperty(currentID + ".duedate", "" + dueDate);
		todoItemProps.setProperty(currentID + ".previous", "" + (currentID - 1));
		todoItemProps.setProperty((currentID - 1) + ".next", "" + currentID);
		todoItemProps.setProperty(currentID + ".subject", subject);
		todoItemProps.setProperty(currentID + ".priority", priority);
		todoItemProps.setProperty(currentID + ".assigned", assignedTo);
		todoItemProps.setProperty(currentID + ".complete", complete);
		todoItemProps.setProperty(currentID + ".description", description);

		try {
			todoItemProps.store(new FileOutputStream(todoItemFile), "Todo list");
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing todo list");
			return RequestType.TODO_ITEM_ADD_FAILED;
		}

		Enumeration connections   = project.getConnections().elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection != null) {
				currentConnection.getPacketFactory().addToQueue(RequestType.NEW_TODO_ITEM, PacketFactory.MAX_PRIORITY);
			}
		}

		return RequestType.TODO_ITEM_OK;
	}
	//}}}

	//{{{ updateTodoItem
	/**
	 *  Update a todo item in the list
	 *
	 *@param  sender       The sender
	 *@param  subject      The subject
	 *@param  priority     The priority
	 *@param  assignedTo   The assignee
	 *@param  dueDate      The due date
	 *@param  complete     percent complete
	 *@param  description  Description of item
	 *@param  currentID    Description of the Parameter
	 *@return              Reply code to client
	 */
	public int updateTodoItem(String sender, String currentID, String subject, String priority, String assignedTo, String dueDate, String complete, String description) {
		File todoItemFile         = new File(project.getTodoListURI().getPath());
		Properties todoItemProps  = new Properties();
		if (todoItemFile.exists()) {
			try {
				todoItemProps.load(new FileInputStream(todoItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading todo list");
				return RequestType.TODO_ITEM_UPDATE_FAILED;
			}
		}

		todoItemProps.setProperty(currentID + ".sender", sender);
		todoItemProps.setProperty(currentID + ".duedate", dueDate);
		todoItemProps.setProperty(currentID + ".subject", subject);
		todoItemProps.setProperty(currentID + ".priority", priority);
		todoItemProps.setProperty(currentID + ".assigned", assignedTo);
		todoItemProps.setProperty(currentID + ".complete", complete);
		todoItemProps.setProperty(currentID + ".description", description);

		try {
			todoItemProps.store(new FileOutputStream(todoItemFile), "Todo list");
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing todo list");
			return RequestType.TODO_ITEM_UPDATE_FAILED;
		}

		Enumeration connections   = project.getConnections().elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection != null) {
				currentConnection.getPacketFactory().addToQueue(RequestType.NEW_TODO_ITEM, PacketFactory.MAX_PRIORITY);
			}
		}

		return RequestType.TODO_ITEM_OK;
	}
	//}}}

	//{{{ getTodoItemList
	/**  Send todo list to the client */
	public void getTodoItemList() {
		StringBuffer tempBuffer   = new StringBuffer();
		File todoItemFile         = new File(project.getTodoListURI().getPath());
		Properties todoItemProps  = new Properties();
		if (todoItemFile.exists()) {
			try {
				todoItemProps.load(new FileInputStream(todoItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading todo item list");
				return;
			}
		}
		String current            = todoItemProps.getProperty("current", "0");
		tempBuffer.append("current=" + current);
		while (!current.equals("0")) {
			tempBuffer.append("\n" + current + ".duedate=" + todoItemProps.getProperty(current + ".duedate"));
			tempBuffer.append("\n" + current + ".subject=" + todoItemProps.getProperty(current + ".subject"));
			tempBuffer.append("\n" + current + ".sender=" + todoItemProps.getProperty(current + ".sender"));
			tempBuffer.append("\n" + current + ".assigned=" + todoItemProps.getProperty(current + ".assigned"));
			tempBuffer.append("\n" + current + ".previous=" + todoItemProps.getProperty(current + ".previous"));
			tempBuffer.append("\n" + current + ".priority=" + todoItemProps.getProperty(current + ".priority"));
			tempBuffer.append("\n" + current + ".complete=" + todoItemProps.getProperty(current + ".complete"));
			current = todoItemProps.getProperty(current + ".previous", "0");
		}
		packetFactory.addToQueue(RequestType.TODO_ITEM_LIST, tempBuffer.toString(), PacketFactory.MAX_PRIORITY);
	}
	//}}}

	//{{{ getTodoItem
	/**
	 *  Return contents of a todo item to the client
	 *
	 *@param  messageID  The item to get
	 *@return            The description
	 */
	public String getTodoItem(String messageID) {
		Properties todoItemProps  = new Properties();
		File todoItemFile         = new File(project.getTodoListURI().getPath());
		if (todoItemFile.exists()) {
			try {
				todoItemProps.load(new FileInputStream(todoItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return "(Error loading private messages!)";
			}
		}
		return todoItemProps.getProperty(messageID + ".description", "(Error getting todo item description!)");
	}
	//}}}

	//{{{ deleteTodoItem
	/**
	 *  Deletes a todo item
	 *
	 *@param  messageID  The item to delete
	 *@return            Reply code to the client
	 */
	public int deleteTodoItem(String messageID) {
		Properties todoItemProps  = new Properties();
		File todoItemFile         = new File(project.getTodoListURI().getPath());
		if (todoItemFile.exists()) {
			try {
				todoItemProps.load(new FileInputStream(todoItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return RequestType.TODO_ITEM_DELETE_FAILED;
			}
		}

		String nextIndex          = todoItemProps.getProperty(messageID + ".next");
		String prevIndex          = todoItemProps.getProperty(messageID + ".previous");

		if (todoItemProps.getProperty("current").equals(messageID)) {
			todoItemProps.setProperty("current", prevIndex);
		}

		if (nextIndex != null) {
			todoItemProps.setProperty(prevIndex + ".next", nextIndex);
			todoItemProps.setProperty(nextIndex + ".previous", prevIndex);
		}
		todoItemProps.remove(messageID + ".sender");
		todoItemProps.remove(messageID + ".duedate");
		todoItemProps.remove(messageID + ".previous");
		todoItemProps.remove(messageID + ".next");
		todoItemProps.remove(messageID + ".subject");
		todoItemProps.remove(messageID + ".priority");
		todoItemProps.remove(messageID + ".assigned");
		todoItemProps.remove(messageID + ".complete");
		todoItemProps.remove(messageID + ".description");
		try {
			todoItemProps.store(new FileOutputStream(todoItemFile), "Private Messages for " + user.getName());
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing private message queue for " + user.getName());
			return RequestType.TODO_ITEM_DELETE_FAILED;
		}

		getTodoItemList();

		return RequestType.TODO_ITEM_OK;
	}
	//}}}

	//{{{ addBugItem
	/**
	 *  Add a bug item to the list
	 *
	 *@param  sender       The sender
	 *@param  subject      The subject
	 *@param  priority     The priority
	 *@param  assignedTo   The assignee
	 *@param  dueDate      The due date
	 *@param  complete     percent complete
	 *@param  description  Description of item
	 *@return              Reply code to client
	 */
	public int addBugItem(String sender, String subject, String priority, String assignedTo, String dueDate, String complete, String description) {
		File bugItemFile         = new File(project.getBugListURI().getPath());
		Properties bugItemProps  = new Properties();
		if (bugItemFile.exists()) {
			try {
				bugItemProps.load(new FileInputStream(bugItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading public message queue");
				return RequestType.BUG_ITEM_ADD_FAILED;
			}
		}

		int currentID            = Integer.parseInt(bugItemProps.getProperty("current", "0"));
		currentID++;
		bugItemProps.setProperty("current", "" + currentID);
		bugItemProps.setProperty(currentID + ".sender", sender);
		bugItemProps.setProperty(currentID + ".duedate", "" + dueDate);
		bugItemProps.setProperty(currentID + ".previous", "" + (currentID - 1));
		bugItemProps.setProperty((currentID - 1) + ".next", "" + currentID);
		bugItemProps.setProperty(currentID + ".subject", subject);
		bugItemProps.setProperty(currentID + ".priority", priority);
		bugItemProps.setProperty(currentID + ".assigned", assignedTo);
		bugItemProps.setProperty(currentID + ".complete", complete);
		bugItemProps.setProperty(currentID + ".description", description);

		try {
			bugItemProps.store(new FileOutputStream(bugItemFile), "Public messages");
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing public messages");
			return RequestType.BUG_ITEM_ADD_FAILED;
		}

		Enumeration connections  = project.getConnections().elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection != null) {
				currentConnection.getPacketFactory().addToQueue(RequestType.NEW_BUG_ITEM, PacketFactory.MAX_PRIORITY);
			}
		}

		return RequestType.BUG_ITEM_OK;
	}
	//}}}

	//{{{ updateBugItem
	/**
	 *  Update a bug item in the list
	 *
	 *@param  sender       The sender
	 *@param  subject      The subject
	 *@param  priority     The priority
	 *@param  assignedTo   The assignee
	 *@param  dueDate      The due date
	 *@param  complete     percent complete
	 *@param  description  Description of item
	 *@param  currentID    Description of the Parameter
	 *@return              Reply code to client
	 */
	public int updateBugItem(String sender, String currentID, String subject, String priority, String assignedTo, String dueDate, String complete, String description) {
		File bugItemFile         = new File(project.getBugListURI().getPath());
		Properties bugItemProps  = new Properties();
		if (bugItemFile.exists()) {
			try {
				bugItemProps.load(new FileInputStream(bugItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading bug list");
				return RequestType.BUG_ITEM_UPDATE_FAILED;
			}
		}

		bugItemProps.setProperty(currentID + ".sender", sender);
		bugItemProps.setProperty(currentID + ".duedate", dueDate);
		bugItemProps.setProperty(currentID + ".subject", subject);
		bugItemProps.setProperty(currentID + ".priority", priority);
		bugItemProps.setProperty(currentID + ".assigned", assignedTo);
		bugItemProps.setProperty(currentID + ".complete", complete);
		bugItemProps.setProperty(currentID + ".description", description);

		try {
			bugItemProps.store(new FileOutputStream(bugItemFile), "Bug list");
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing bug list");
			return RequestType.BUG_ITEM_UPDATE_FAILED;
		}

		Enumeration connections  = project.getConnections().elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection != null) {
				currentConnection.getPacketFactory().addToQueue(RequestType.NEW_BUG_ITEM, PacketFactory.MAX_PRIORITY);
			}
		}

		return RequestType.BUG_ITEM_OK;
	}
	//}}}

	//{{{ getBugItemList
	/**  Send bug list to the client */
	public void getBugItemList() {
		StringBuffer tempBuffer  = new StringBuffer();
		File bugItemFile         = new File(project.getBugListURI().getPath());
		Properties bugItemProps  = new Properties();
		if (bugItemFile.exists()) {
			try {
				bugItemProps.load(new FileInputStream(bugItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading bug item list");
				return;
			}
		}
		String current           = bugItemProps.getProperty("current", "0");
		tempBuffer.append("current=" + current);
		while (!current.equals("0")) {
			tempBuffer.append("\n" + current + ".duedate=" + bugItemProps.getProperty(current + ".duedate"));
			tempBuffer.append("\n" + current + ".subject=" + bugItemProps.getProperty(current + ".subject"));
			tempBuffer.append("\n" + current + ".sender=" + bugItemProps.getProperty(current + ".sender"));
			tempBuffer.append("\n" + current + ".assigned=" + bugItemProps.getProperty(current + ".assigned"));
			tempBuffer.append("\n" + current + ".previous=" + bugItemProps.getProperty(current + ".previous"));
			tempBuffer.append("\n" + current + ".priority=" + bugItemProps.getProperty(current + ".priority"));
			tempBuffer.append("\n" + current + ".complete=" + bugItemProps.getProperty(current + ".complete"));
			current = bugItemProps.getProperty(current + ".previous", "0");
		}
		packetFactory.addToQueue(RequestType.BUG_ITEM_LIST, tempBuffer.toString(), PacketFactory.MAX_PRIORITY);
	}
	//}}}

	//{{{ getBugItem
	/**
	 *  Return contents of a todo item to the client
	 *
	 *@param  messageID  The item to get
	 *@return            The description
	 */
	public String getBugItem(String messageID) {
		Properties bugItemProps  = new Properties();
		File bugItemFile         = new File(project.getBugListURI().getPath());
		if (bugItemFile.exists()) {
			try {
				bugItemProps.load(new FileInputStream(bugItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading private message list for " + user.getName());
				return "(Error loading private messages!)";
			}
		}
		return bugItemProps.getProperty(messageID + ".description", "(Error getting todo item description!)");
	}
	//}}}

	//{{{ deleteBugItem
	/**
	 *  Deletes a bug item
	 *
	 *@param  messageID  The item to delete
	 *@return            Reply code to the client
	 */
	public int deleteBugItem(String messageID) {
		Properties bugItemProps  = new Properties();
		File bugItemFile         = new File(project.getBugListURI().getPath());
		if (bugItemFile.exists()) {
			try {
				bugItemProps.load(new FileInputStream(bugItemFile));
			} catch (IOException ioe) {
				ProColServer.printErr("IOException loading public message list");
				return RequestType.BUG_ITEM_DELETE_FAILED;
			}
		}

		String nextIndex         = bugItemProps.getProperty(messageID + ".next");
		String prevIndex         = bugItemProps.getProperty(messageID + ".previous");

		if (bugItemProps.getProperty("current").equals(messageID)) {
			bugItemProps.setProperty("current", prevIndex);
		}

		if (nextIndex != null) {
			bugItemProps.setProperty(prevIndex + ".next", nextIndex);
			bugItemProps.setProperty(nextIndex + ".previous", prevIndex);
		}
		bugItemProps.remove(messageID + ".sender");
		bugItemProps.remove(messageID + ".duedate");
		bugItemProps.remove(messageID + ".previous");
		bugItemProps.remove(messageID + ".next");
		bugItemProps.remove(messageID + ".subject");
		bugItemProps.remove(messageID + ".priority");
		bugItemProps.remove(messageID + ".assigned");
		bugItemProps.remove(messageID + ".complete");
		bugItemProps.remove(messageID + ".description");
		try {
			bugItemProps.store(new FileOutputStream(bugItemFile), "Public Messages");
		} catch (IOException ioe) {
			ProColServer.printErr("IOException storing public list");
			return RequestType.BUG_ITEM_DELETE_FAILED;
		}

		Enumeration connections  = project.getConnections().elements();
		while (connections.hasMoreElements()) {
			ServerConnection currentConnection  = (ServerConnection)connections.nextElement();
			if (currentConnection != null) {
				currentConnection.getBugItemList();
			}
		}

		return RequestType.BUG_ITEM_OK;
	}
	//}}}

	//}}}
}
