using System;
using System.Collections;
using Tao.Platform;
using Tao.OpenGl;


namespace QCGM.Geometry
{
	#region Public Interfaces
	
	/// <summary>
	/// Interface for a 2D integer-based point.
	/// </summary>
	public interface IPoint2Di
	{
		int X
		{
			get;
			set;
		}

		int Y
		{
			get;
			set;
		}
	}

	
	/// <summary>
	/// Interface for a 3D integer-based point.
	/// </summary>
	public interface IPoint3Di
	{
		int X
		{
			get;
			set;
		}

		int Y
		{
			get;
			set;
		}

		int Z
		{
			get;
			set;
		}
	}

	/// <summary>
	/// Interface for a floating point-based 2D point.
	/// </summary>
	public interface IPoint2Df
	{
		float X
		{
			get;
			set;
		}

		float Y
		{
			get;
			set;
		}
	}

	/// <summary>
	/// Interface for a floating point-based 3D point.
	/// </summary>
	public interface IPoint3Df
	{
		float X
		{
			get;
			set;
		}

		float Y
		{
			get;
			set;
		}

		float Z
		{
			get;
			set;
		}
	}

	public interface ISelectable
	{
		bool Selected
		{
			get;
			set;
		}
	}

	#endregion

	#region 2D and 3D points

	/// <summary>
	/// Integer-based 2D point.
	/// </summary>
	public class Point2Di : IPoint2Di
	{
		protected int myX;
		protected int myY;

		/// <summary>
		/// Gets and sets the Y value of this point
		/// </summary>
		public int Y
		{
			get
			{
				return myY;
			}
			set
			{
				myY = value;
			}
		}

		/// <summary>
		/// Gets and sets the X value of this point
		/// </summary>
		public int X
		{
			get
			{
				return myX;
			}
			set
			{
				myX = value;
			}
		}

		/// <summary>
		/// Creates a new 2D integer-based point
		/// </summary>
		/// <param name="x">The x value for this point</param>
		/// <param name="y">The y value for this point</param>
		public Point2Di(int x, int y)
		{
			this.X = x;
			this.Y = y;
		}

		/// <summary>
		/// Creates a new 2D integer-based point, initializing to (0,0)
		/// </summary>
		public Point2Di()
		{
			X = 0;
			Y = 0;
		}
	}


	/// <summary>
	/// Floating point-based 2D point.
	/// </summary>
	public class Point2Df : IPoint2Df
	{
		protected float myX;
		protected float myY;

		/// <summary>
		/// Gets and sets the Y value of this point
		/// </summary>
		public float Y
		{
			get
			{
				return myY;
			}

			set
			{
				myY = value;
			}
		}

		public float X
		{
			get
			{
				return myX;
			}

			set
			{
				myX = value;
			}
		}

		/// <summary>
		/// Creates a new floating point-based 2D point
		/// </summary>
		/// <param name="x">Initial x value for this point.</param>
		/// <param name="y">Initial y value for this point.</param>
		public Point2Df(float x, float y)
		{
			this.X = x;
			this.Y = y;
		}

		/// <summary>
		/// Creates a new floating point-based 2D point, initialized to (0,0)
		/// </summary>
		public Point2Df()
		{
			X = 0.0f;
			Y = 0.0f;
		}
	}

	
	/// <summary>
	/// Integer-based 3D point.
	/// </summary>
	public class Point3Di : IPoint3Di
	{
		protected int myX;
		protected int myY;
		protected int myZ;

		/// <summary>
		/// Gets and sets the x value of this point.
		/// </summary>
		public int X
		{
			get
			{
				return myX;
			}
			
			set
			{
				myX = value;
			}
		}

		/// <summary>
		/// Gets and sets the Y value of this point.
		/// </summary>
		public int Y
		{
			get
			{
				return myY;
			}
			set
			{
				myY = value;
			}
		}

		/// <summary>
		/// Gets and sets the Z value of this point.
		/// </summary>
		public int Z
		{
			get
			{
				return myZ;
			}

			set
			{
				myZ = value;
			}
		}

		/// <summary>
		/// Creates a new integer-based 3D point
		/// </summary>
		/// <param name="x">Initial x value for this point.</param>
		/// <param name="y">Initial y value for this point.</param>
		/// <param name="z">Initial z value for this point.</param>
		public Point3Di(int x, int y, int z)
		{
			X = x;
			Y = y;
			Z = z;
		}

		/// <summary>
		/// Creates a new integer-based 3D point, initialized to (0,0,0).
		/// </summary>
		public Point3Di()
		{
			X = 0;
			Y = 0;
			Z = 0;
		}
	}


	/// <summary>
	/// Floating point-based 3D point.
	/// </summary>
	public class Point3Df : IPoint3Df
	{
		protected float myX;
		protected float myY;
		protected float myZ;

		/// <summary>
		/// Gets and sets the x value of this point.
		/// </summary>
		public float X
		{
			get
			{
				return myX;
			}
			set
			{
				myX = value;
			}
		}

		/// <summary>
		/// Gets and sets the y value of this point.
		/// </summary>
		public float Y
		{
			get
			{
				return myY;
			}
			set
			{
				myY = value;
			}
		}

		/// <summary>
		/// Gets and sets the z value of this point.
		/// </summary>
		public float Z
		{
			get
			{
				return myZ;
			}
			set
			{
				myZ = value;
			}
		}

		/// <summary>
		/// Creates a new floating point-based 3D point.
		/// </summary>
		/// <param name="x">Initial x value for this point.</param>
		/// <param name="y">Initial y value for this point.</param>
		/// <param name="z">Initial z value for this point.</param>
		public Point3Df(float x, float y, float z)
		{
			X = x;
			Y = y;
			Z = z;
		}

		/// <summary>
		/// Creates a new floating point-based 3D point, initialized to (0,0,0)
		/// </summary>
		public Point3Df()
		{
			X = 0.0f;
			Y = 0.0f;
			Z = 0.0f;
		}

		public void Clone(Point3Df B)
		{
			this.X = B.X;
			this.Y = B.Y;
			this.Z = B.Z;
		}
	}


	#endregion

	#region Colors

	public class Color4f
	{
		protected float MyRed, MyGreen, MyBlue, MyAlpha;

		public Color4f(float r, float g, float b, float a)
		{
			MyRed = r;
			MyGreen = g;
			MyBlue = b;
			MyAlpha = a;
		}

		public Color4f()
		{
			MyRed = 0;
			MyGreen = 0;
			MyBlue = 0;
			MyAlpha = 0;
		}

		public float Red
		{
			get
			{
				return MyRed;
			}
			set
			{
				MyRed = value;
			}
		}

		public float Green
		{
			get
			{
				return MyGreen;
			}
			set
			{
				MyGreen = value;
			}
		}

		public float Blue
		{
			get
			{
				return MyBlue;
			}
			set
			{
				MyBlue = value;
			}
		}

		public float Alpha
		{
			get
			{
				return MyAlpha;
			}
			set
			{
				MyAlpha = value;
			}
		}

		/// <summary>
		/// Returns a 4-tuple array: {R, G, B, A}
		/// </summary>
		public float [] ToRGBA_Array
		{
			get
			{
				float [] a = new float[4];
				a[0] = MyRed;
				a[1] = MyGreen;
				a[2] = MyBlue;
				a[3] = MyAlpha;
				return a;
			}
		}

		/// <summary>
		/// Returns a 3-tuple array: {R, G, B}
		/// </summary>
		public float [] ToRGB_Array
		{
			get
			{
				float [] a = new float[3];
				a[0] = MyRed;
				a[1] = MyGreen;
				a[2] = MyBlue;
				return a;
			}
		}
	}


	public class Colors
	{
		public static Color4f Player1
		{
			get
			{
				return new Color4f( 0.42f, 0.78f, 0.98f, 1.0f);
			}
		}

		public static Color4f Player2
		{
			get
			{
				return new Color4f( 1.0f, 0.85f, 0.54f, 1.0f);
			}
		}

		public static Color4f Tiles
		{
			get
			{
				return new Color4f( 1.00f, 1.00f, 1.0f, 0.6f);
			}
		}

		public static Color4f LimeJello
		{
			get
			{
				return new Color4f(0.0f, 1.0f, 0.0f, 0.6f);
			}
		}
	
		public static Color4f Green
		{
			get
			{
				return new Color4f( 0.0f, 1.0f, 0.0f, 1.0f);
			}
		}

		public static Color4f Board
		{
			get
			{
				return new Color4f( 0.88f, 0.88f, 0.88f, 1.0f);
			}
		}
		
		public static Color4f Wallpaper1
		{
			get
			{
				return new Color4f( 0.5f, 0.54f, 0.85f, 0.8f);
			}
		}
		
		public static Color4f Wallpaper2
		{
			get
			{
				return new Color4f( 1.0f, 0.91f, 0.1f, 0.8f);
			}
		}

		
		public static Color4f Red
		{
			get
			{
				return new Color4f(1.0f, 0.0f, 0.0f, 1.0f);
			}
		}

		public static Color4f Yellow
		{
			get
			{
				return new Color4f(0.0f, 1.0f, 1.0f, 1.0f);
			}
		}

		public static Color4f Gray
		{
			get
			{
				return new Color4f(0.5f, 0.5f, 0.5f, 1.0f);
			}
		}

		public static Color4f ForestGreen
		{
			get
			{
				return new Color4f(0.0f, 0.7f, 0.2f, 1.0f);
			}
		}

		public static Color4f Brown
		{
			get
			{
				return new Color4f(0.6f, 0.5f, 0.4f, 1.0f);
			}
		}

		public static void SetGlColor(Color4f Color)
		{
			Gl.glColor4f(Color.Red, Color.Green, Color.Blue, Color.Alpha);
		}
	}

	#endregion

	public class Namestack
	{
		public static ArrayList al = new ArrayList();
	}

	/// <summary>
	/// An object that holds the geometry for a 3D object.
	/// </summary>
	public class Object3Df
	{
		public Point3Df [][] tris;
		public Point3Df [] norms;
	}


	#region Static Gemoetry-Rendering Classes

	/// <summary>
	/// Provides several static methods to render simple geometry using OpenGL. Most functions require a call to
	/// GlBegin() with the appropriate drawing mode.
	/// </summary>
	public class Shapes
	{
		/*private void GetFaceNormal(ref float [] norm, float [] pointa, float [] pointb, float [] pointc)
		{
			float [][] vect = new float[2];
				vect[0] = new float[3];
				vect[1] = new float[3];
			float [][] point = new float[3];
				point[0] = new float [3];
				point[1] = new float[3];
				point[2] = new float[3];

			for(int i=0; i < 3; i++)
			{
				point[0,i] = pointa[i];//copies points into point[][]
				point[1,i] = pointb[i];
				point[2,i] = pointc[i];
			}

			for(int i=0; i < 2; i++)//calculates vectors from point[0] to point[1]
			{
				for(int b=0; b < 3; b++)//and point[0] to point[2]
				{
					vect[i,b] = point[2-i,b] - point[0,b];
				}
			}

			crossProduct(ref norm, vect[0], vect[1]);	//calculates vector at 90 to to 2 vectors
			normalize(norm);							//makes the vector length 1
		}

		private void normalize(ref float [] vect)	//scales a vector a length of 1
		{
			float length;
			int a;

			length=(float)Math.Sqrt(					//A^2 + B^2 + C^2 = length^2
				Math.Pow((double)vect[0],2)+	
				Math.Pow((double)vect[1],2)+
				Math.Pow((double)vect[2],2)
				);

			for (a=0;a<3;++a)				//divides vector by its length to normalise
			{
				vect[a]/=length;
			}
		}*/

		/// <summary>
		/// Draws a two-dimensional square in 3-Space.
		/// </summary>
		/// <param name="Position">The position of the square in 3D world coordinates</param>
		/// <param name="Dimension">The dimension (width, height, depth) of the square in 3D world coordinates</param>
		/// <remarks>You must call GlBegin() with the appropriate drawing mode, or this function will not do very much.</remarks>
		public static void DrawSquare3Df(Point3Df Position, Point2Df Dimension)
		{
			Gl.glVertex3f(Position.X, Position.Y, Position.Z);
			Gl.glVertex3f(Position.X + Dimension.X, Position.Y, Position.Z);
			Gl.glVertex3f(Position.X + Dimension.X, Position.Y + Dimension.Y, Position.Z);
			Gl.glVertex3f(Position.X, Position.Y + Dimension.Y, Position.Z);
		}

		public static void DrawBox3Df(Point3Df Position, Point3Df Dimension)
		{
			Gl.glBegin(Gl.GL_QUADS);

			// top chk
			Gl.glNormal3f(Position.X, Position.Y+Dimension.Y, Position.Z+Dimension.Z);
			Gl.glVertex3f(Position.X,				Position.Y+Dimension.Y,	Position.Z+Dimension.Z);
			Gl.glVertex3f(Position.X,				Position.Y,				Position.Z+Dimension.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y,				Position.Z+Dimension.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y+Dimension.Y,	Position.Z+Dimension.Z);

			// back fixed
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y+Dimension.Y,	Position.Z+Dimension.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y+Dimension.Y,	Position.Z);
			Gl.glVertex3f(Position.X,				Position.Y+Dimension.Y,	Position.Z);
			Gl.glVertex3f(Position.X,				Position.Y+Dimension.Y,	Position.Z+Dimension.Z);

			// bottom chk
			Gl.glVertex3f(Position.X,				Position.Y,				Position.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y,				Position.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y+Dimension.Y, Position.Z);
			Gl.glVertex3f(Position.X,				Position.Y+Dimension.Y, Position.Z);

			// left side fixd
			Gl.glVertex3f(Position.X,				Position.Y+Dimension.Y,	Position.Z+Dimension.Z);
			Gl.glVertex3f(Position.X,				Position.Y+Dimension.Y,	Position.Z);
			Gl.glVertex3f(Position.X,				Position.Y,				Position.Z);
			Gl.glVertex3f(Position.X,				Position.Y,				Position.Z+Dimension.Z);

			// right side fixd
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y+Dimension.Y,	Position.Z+Dimension.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y+Dimension.Y,	Position.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y,				Position.Z);
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y,				Position.Z+Dimension.Z);

			// front chk
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y,				Dimension.Z);
			Gl.glVertex3f(Position.X,				Position.Y,				Dimension.Z);
			Gl.glVertex3f(Position.X,				Position.Y,				Position.Z+Dimension.Z);			
			Gl.glVertex3f(Position.X+Dimension.X,	Position.Y,				Position.Z+Dimension.Z);
			
			Gl.glEnd();
		}

		public static void DrawTetrahedron3f(Point3Df Position, float width, float height)
		{
			Gl.glBegin(Gl.GL_TRIANGLES);
			float x = Position.X;
			float y = Position.Y;
			float z = Position.Z;
		
			/////// * /////////// * /////////// * NEW * /////// * /////////// * /////////// *

			// Below we add something new to our pyramid code, NORMALS.  In order for OpenGL
			// to know how much it needs to light a certain polygon, it depends on the normal
			// of that polygon.  A normal is the direction of the polygon.  You will notice
			// that we give the BACK polgon a normal of (0, 1, -1).  This means that the triangle
			// is facing slightly up and down the negative Z axis (into the screen).  Remember,
			// These are directions, not actuall coordinates.  There is more discussed about
			// Normals at the bottom of this file in the QUICK NOTES section.  There will be
			// a seperate tutorial just devoted to this subject, since it is very important for lighting.

			// The Gl.glNormal3f() functions let's us specify the normal for the next vertices passed in.
			// To compute the normal of a poly is a bit complicated at first so I will not go into it.
			// To avoid having to discuss to much in this tutorial, we will just do an approximation.

			// These vertices create the Back Side
			Gl.glNormal3f(0, 1, -1);	 // This polygon is facing backwards and up
			Gl.glColor3ub(255, 0, 0);   Gl.glVertex3f(x, y + height, z);					// Top point
			Gl.glColor3ub(0, 255, 255); Gl.glVertex3f(x - width, y - height, z - width);	// Bottom left point
			Gl.glColor3ub(255, 0, 255); Gl.glVertex3f(x + width, y - height, z - width);  // Bottom right point

			// These vertices create the Front Side
			Gl.glNormal3f(0, 1, 1);	 // This polygon is facing towards us and slightly up.
			Gl.glColor3ub(255, 0, 0);   Gl.glVertex3f(x, y + height, z);					// Top point
			Gl.glColor3ub(0, 255, 255); Gl.glVertex3f(x + width, y - height, z + width);  // Bottom right point
			Gl.glColor3ub(255, 0, 255); Gl.glVertex3f(x - width, y - height, z + width);	// Bottom left point

			// These vertices create the Front Left Side
			Gl.glNormal3f(-1, 1, 0);	 // This polygon is facing slightly up and to the left
			Gl.glColor3ub(255, 0, 0);   Gl.glVertex3f(x, y + height, z);					// Top point
			Gl.glColor3ub(255, 0, 255); Gl.glVertex3f(x - width, y - height, z + width);	// Front bottom point
			Gl.glColor3ub(0, 255, 255); Gl.glVertex3f(x - width, y - height, z - width);	// Bottom back point

			// These vertices create the Front right Side
			Gl.glNormal3f(1, 1, 0);	// This polygon is facing slightly up and to the right
			Gl.glColor3ub(255, 0, 0);   Gl.glVertex3f(x, y + height, z);					// Top point
			Gl.glColor3ub(255, 0, 255); Gl.glVertex3f(x + width, y - height, z - width);	// Bottom back point
			Gl.glColor3ub(0, 255, 255); Gl.glVertex3f(x + width, y - height, z + width);	// Front bottom point
			
			Gl.glEnd();

			// Now render the bottom of our pyramid

			Gl.glBegin(Gl.GL_QUADS);

			// These vertices create the bottom of the pyramid
			Gl.glNormal3f(0, -1, 0);	// This polygon is facing straight down
			Gl.glColor3ub(0, 0, 255); Gl.glVertex3f(x - width, y - height, z + width);	// Front left point
			Gl.glColor3ub(0, 0, 255); Gl.glVertex3f(x + width, y - height, z + width);    // Front right point
			Gl.glColor3ub(0, 0, 255); Gl.glVertex3f(x + width, y - height, z - width);    // Back right point
			Gl.glColor3ub(0, 0, 255); Gl.glVertex3f(x - width, y - height, z - width);    // Back left point
			Gl.glEnd();

			/////// * /////////// * /////////// * NEW * /////// * /////////// * /////////// *

		}

		public static void DrawCube(Point3Df Position, Point3Df Dimension)
		{
			Point3Df P = Position;
			Point3Df D = Dimension;

			// Here we create 6 QUADS (RectanGl.gles) to form a cube
			// With the passed in radius, we determine the width and height of the cube
			// You will notice, I add some color at each vertice to make it more interesting

			Gl.glBegin(Gl.GL_QUADS);		
			
			// These vertices create the Back Side
			//Gl.glColor3ub(0, 0, 255);
			Gl.glVertex3f(P.X, P.Y, P.Z);
			//Gl.glColor3ub(255, 0, 255);
			Gl.glVertex3f(P.X, P.Y + D.Y, P.Z);
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y + D.Y, P.Z); 
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y, P.Z);

			// These vertices create the Front Side
			//Gl.glColor3ub(0, 0, 255);
			Gl.glVertex3f(P.X, P.Y, P.Z + D.Z);
			//Gl.glColor3ub(255, 0, 255);
			Gl.glVertex3f(P.X, P.Y + D.Y, P.Z + D.Z);
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y + D.Y, P.Z + D.Z); 
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y, P.Z + D.Z);

			// These vertices create the Bottom Face
			//Gl.glColor3ub(0, 0, 255);
			Gl.glVertex3f(P.X, P.Y, P.Z);
			//Gl.glColor3ub(255, 0, 255);
			Gl.glVertex3f(P.X, P.Y, P.Z + D.Z);
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y, P.Z + D.Z); 
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y, P.Z);
				
			// These vertices create the Top Face
			//Gl.glColor3ub(0, 0, 255);
			Gl.glVertex3f(P.X, P.Y + D.Y, P.Z);
			//Gl.glColor3ub(255, 0, 255);
			Gl.glVertex3f(P.X, P.Y + D.Y, P.Z + D.Z);
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y + D.Y, P.Z + D.Z); 
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y + D.Y, P.Z);

			// These vertices create the Left Face
			//Gl.glColor3ub(0, 0, 255);
			Gl.glVertex3f(P.X, P.Y, P.Z);
			//Gl.glColor3ub(255, 0, 255);
			Gl.glVertex3f(P.X, P.Y, P.Z + D.Z);
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X, P.Y + D.Y, P.Z + D.Z); 
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X, P.Y + D.Y, P.Z);

			// These vertices create the Right Face
			//Gl.glColor3ub(0, 0, 255);
			Gl.glVertex3f(P.X + D.X, P.Y, P.Z);
			//Gl.glColor3ub(255, 0, 255);
			Gl.glVertex3f(P.X + D.X, P.Y, P.Z + D.Z);
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y + D.Y, P.Z + D.Z); 
			//Gl.glColor3ub(0, 255, 255);
			Gl.glVertex3f(P.X + D.X, P.Y + D.Y, P.Z);

			Gl.glEnd();
		}

		/*
		private class Cube
		{
			public Ver [] ver = new Ver[8];
			public Quad quad = new Quad[6];
		}
		
		private class Ver
		{
			public float [] pos = new float[3];
			public float [] col = new float[3];
		}

		private class Quad
		{
			public uint [] ver = new float[6];
			public float [] norm = new float[3];
		}


		#endregion

		public static void DrawCube()
		{
			float [,] vertexPosDat = new float[8,3]
				{
					{0.0f, 1.0f, 1.0f}, //left,top,front
					{1.0f,  1.0f, 1.0f}, //right,top,front
					{1.0f,  1.0f,0.0f}, //right,top,back
					{0.0f, 1.0f,0.0f}, //left, top,back
					{0.0f,0.0f, 1.0f}, //left,bottom,front
					{1.0f, 0.0f, 1.0f}, //right,bottom,front
					{1.0f, 0.0f,0.0f}, //right,bottom,back
					{0.0f,0.0f,0.0f}  //left,bottom,back
				};

			float [,] vertexColDat = new float [8,3]
					{
						{0.5f,	0.0f,	0.0f  }, //dark red
						{1.0f,	1.0f,	0.3f}, //yellow
						{1.0f,	0.0f,	0.0f  }, //red
						{0.5f,	1.0f,	0.2f}, //dull yellow??
						{1.0f,	1.0f,	0.0f  }, //yellow
						{0.9f,	0.5f,	0.0f  }, //orange
						{1.0f,	0.9f,	0.1f }, //yellow
						{1.0f,	0.0f,	0.0f  }, //red
			};
	
			uint [,] quadVerDat = new uint [6,4]
					{
						{0,1,2,3}, //top
						{0,3,7,4}, //left
						{3,2,6,7}, //back
						{2,1,5,6}, //right
						{0,4,5,1}, //front
						{4,7,6,5}, //bottom
			};

			Cube cube;

			for(int a = 0; a < 8; a++)
			{
				for(int b=0; b < 3; b++)
				{
					cube.ver[a].pos[b] = vertexPosDat[a][b];
					cube.ver[a].col[b] = vertexColDat[a][b];
				}
			}

			for(int i=0; i < 6; i++)
			{
				for(int j=0; j < 4; j++)
				{
					cube.quad[i].ver[j] = quadVerDat[a][b];
				}
			}

			for(int i=0; i < 6; i++)
			{
				getFaceNormal(cube.quad[a].norm,	cube.ver[ cube.quad[a].ver[2] ].pos,
					cube.ver[ cube.quad[a].ver[1] ].pos,
					cube.ver[ cube.quad[a].ver[0] ].pos);
			}
		}*/
	}

	#endregion
	
}