using System;
using QCGM.Boards;
using QCGM.Geometry;
using Tao.Platform;
using Tao.OpenGl;

namespace QCGM.Boards
{
	/// <summary>
	/// This class is the graphical representation of a player.
	/// </summary>
	public class Pawn : IDrawable, ISelectable
	{
		protected Color4f MyColor = new Color4f();			// The color of the object.
		
		protected Point3Df MyOrigin = new Point3Df();		// The original location of the object.
		protected Point3Df MyPosition = new Point3Df();		// The current position of the object.
		protected Point3Df MyDestination = new Point3Df();	// The destination of the object.
		protected float MovementProgress = 0.0f;			// The coefficient of progress toward the destination (1.0 = there)
		protected bool Moving = false;						// Is the pawn moving?
		
		protected Point3Df MyDimension = new Point3Df();	// The dimension (size) of the object.
		protected float Scale = 1.0f;						// The scale of the object.

		public string Name="Unset";

		public Pawn(string name)
		{
			MyDimension.X = 0.8f;
			MyDimension.Y = 0.8f;
			MyDimension.Z = 0.5f;

			this.Name = name;
		}

		public void MoveTo(Point3Df NewPoint)
		{
			MyOrigin = MyPosition;
			MyDestination = NewPoint;
			Moving = true;
			MovementProgress = 0.0f;
		}

		public bool MoveTowardDestination(float factor)
		{
			if(Moving)
			{
				this.MovementProgress += factor;
				UpdatePosition();
				if (MovementProgress >= 1.0f)
				{
					MovementProgress = 1.0f;
					Moving = false;
				}
				
				return true;
			}
			else
			{
				return false;
			}
		}

		protected void UpdatePosition()
		{
			float t = MovementProgress;
			MyPosition.X = (1.0f - t) * MyOrigin.X + t * MyDestination.X;
			MyPosition.Y = (1.0f - t) * MyOrigin.Y + t * MyDestination.Y;
			MyPosition.Z = (1.0f - t) * MyOrigin.Z + t * MyDestination.Z;
		}
		
		#region IDrawable Members
		public void Draw()
		{
			Colors.SetGlColor(MyColor);
			Gl.glPushName((int)Geometry.Namestack.al.IndexOf(this));
			Shapes.DrawCube(MyPosition, MyDimension);
			Gl.glPopName();
		}

		/// <summary>
		/// Gets and sets the color value for this Tile.
		/// </summary>
		public Color4f Color
		{
			get
			{
				return MyColor;
			}

			set
			{
				MyColor = value;
			}
		}

		public Point3Df Position
		{
			get
			{
				return MyPosition;
			}
			set
			{
				MyPosition = value;
			}
		}

		public Point3Df Dimension
		{
			get
			{
				return MyDimension;
			}
			set
			{
				MyDimension = value;
			}
		}

		#endregion

		#region ISelectable Members

		private bool isSelected = false;
		private bool isHovering = false;

		public bool Selected
		{
			get
			{
				// TODO:  Add Pawn.IsSelected getter implementation
				return isSelected;
			}
			set
			{
				// TODO:  Add Pawn.IsSelected setter implementation
				isSelected = value;
			}
		}

		public bool Hovering
		{
			get
			{
				return isHovering;
			}
			set
			{
				isHovering = value;
			}
		}

		#endregion
	}
}
