using System;
using QCGM.Boards;
using QCGM.Geometry;
using Tao.Platform;
using Tao.OpenGl;

namespace QCGM.Boards
{
	/// <summary>
	/// Summary description for Wall.
	/// </summary>
	public class Wall : IDrawable
	{
		protected Color4f MyColor = new Color4f();			// The color of the object.
		
		protected Point3Df MyOrigin = new Point3Df();		// The original location of the object.
		protected Point3Df MyPosition = new Point3Df();		// The current position of the object.
		protected Point3Df MyDestination = new Point3Df();	// The destination of the object.
		protected float MovementProgress = 0.0f;			// The coefficient of progress toward the destination (1.0 = there)
		protected bool Moving = false;						// Is the pawn moving?
		
		protected Point3Df MyDimension = new Point3Df();	// The dimension (size) of the object.
		protected float Scale = 1.0f;						// The scale of the object.

		public bool Moved = false;

		protected int index;
		public int Index
		{
			get
			{
				return this.index;
			}
		}

		public enum Orientations { Vertical, Horizontal };
		protected Orientations MyOrientation = Orientations.Vertical;

		public Orientations Orientation
		{
			get
			{
				return MyOrientation;
			}
			set
			{
				if(MyOrientation != value)
				{
					MyOrientation = value;
									
					switch(value)
					{
						case Orientations.Horizontal:
							Dimension.X = 2.23f;
							Dimension.Y = 0.23f;
							Dimension.Z = 1.25f;
							break;
						case Orientations.Vertical:
							MyDimension.X = 0.23f;
							MyDimension.Y = 2.23f;
							MyDimension.Z = 1.25f;
							break;
					}
				}
			}
		}

		/// <summary>
		/// Creates a new wall at the specified 3D location, and with the index specified.
		/// </summary>
		/// <param name="Position">A point in 3D space.</param>
		/// <param name="index">The index (1-10) of the wall. Each player has 10 walls. This index refers to which
		/// Wall this object represents.</param>
		public Wall(Point3Df Position, int index, Color4f col)
		{
			MyPosition = Position;

			// Walls all have the same dimension.
			MyDimension.X = 0.23f;
			MyDimension.Y = 2.23f;
			MyDimension.Z = 1.25f;

			this.Orientation = Wall.Orientations.Horizontal;

			MyColor = col;

			this.index = index;
		}

		public void MoveTo(Orientations Orient, Point3Df NewPoint)
		{
			MyOrigin = MyPosition;
			Orientation = Orient;
			MyDestination = NewPoint;
			Moving = true;
			MovementProgress = 0.0f;
		}

		public bool MoveTowardDestination(float factor)
		{
			if(Moving)
			{
				MovementProgress += factor;
				float t = MovementProgress;
				MyPosition.X = (1.0f - t) * MyOrigin.X + t * MyDestination.X;
				MyPosition.Y = (1.0f - t) * MyOrigin.Y + t * MyDestination.Y;
				MyPosition.Z = (1.0f - t) * MyOrigin.Z + t * MyDestination.Z;

				if (MovementProgress >= 1.0f)
				{
					MovementProgress = 1.0f;
					Moving = false;
				}
				return true;
			}
			else
			{
				return false;
			}
		}

		public void Draw()
		{
			Colors.SetGlColor(MyColor);
			Gl.glPushName((int)Geometry.Namestack.al.IndexOf(this));
			Shapes.DrawCube(MyPosition, MyDimension);
			Gl.glPopName();
		}

		/// <summary>
		/// Gets and sets the color value for this Tile.
		/// </summary>
		public Color4f Color
		{
			get
			{
				return MyColor;
			}

			set
			{
				MyColor = value;
			}
		}

		public Point3Df Position
		{
			get
			{
				return MyPosition;
			}
			set
			{
				MyPosition = value;
			}
		}

		public Point3Df Dimension
		{
			get
			{
				return MyDimension;
			}
			set
			{
				MyDimension = value;
			}
		}
	}
}
