package fortytwo.client;

/**
 * <p>Title: Forty Two Unlimited</p>
 * <p>Description: Texas Forty Two Game</p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: 42Unlimited</p>
 * @author Nathaniel Normandin
 * @version 0.00.01
 */

import java.util.ArrayList;
import fortytwo.network.*;

public class ClientMessageManager implements MessageManager {

  public static final String SERVER_ADDRESS = "209.193.52.222";
  public static final int SERVER_PORT = 5432;

  /////////////////////////////////////////
  // SINGLETON PATTERN FOR GLOBAL ACCESS
  private static ClientMessageManager clientMessageManager = null;

  public static ClientMessageManager getInstance() {
    if ( clientMessageManager == null )
      clientMessageManager = new ClientMessageManager();
    return clientMessageManager;
  }
  // SINGLETON PATTERN END
  /////////////////////////////////////////

  // GAME STATES
  private static final int STATE_DISCONNECTED = 0;
  private static final int STATE_CONNECTED = 1;
  private static final int STATE_LOGGEDIN = 2;
  private static final int STATE_PLAYING = 3;
  // GAME STATES END

  // the current state of the game
  private int currentState;
  private boolean playingState;

  // ProtocolHandler lets us send and received application layer PDUs ( Message )
  private ProtocolHandler protocolHandler;

  // Thread for handling the processing of messages
  private Thread messageProcessingThread;

  // List of listeners connected to the different panels
  private ArrayList chatListeners;
  private ArrayList buttonListeners;
  private ArrayList tablePanelListeners;
  private ArrayList tableFrameListeners;

  private String username;

  // constructor
  public ClientMessageManager() {
    chatListeners = new ArrayList();
    buttonListeners = new ArrayList();
    tablePanelListeners = new ArrayList();
    tableFrameListeners = new ArrayList();

  }
  //////////////////////////////////////////////
  // ADDING THE LISTENERS
  // Add a ChatListener to the ArrayList of ChatListeners
  public void addChatListener ( ChatListener listener ) {
    synchronized ( chatListeners ) {
      if (!chatListeners.contains(listener)) {
        chatListeners.add(listener);
      }
    }
  }

  // Add a ButtonListener to the ArrayList of ButtonListeners
  public void addButtonListener ( ButtonListener listener ) {
    synchronized ( buttonListeners ) {
      if (!buttonListeners.contains(listener)) {
        buttonListeners.add(listener);
      }
    }
  }

  // Add a TableFrameListener to the ArrayList of TableFrameListeners
  public void addTablePanelListener(TablePanelListener listener) {
    synchronized ( tablePanelListeners ) {
      if (!tablePanelListeners.contains(listener)) {
        tablePanelListeners.add(listener);
      }
    }
  }

  public void addTableFrameListener( TableFrameListener listener ) {
    synchronized( tableFrameListeners ) {
      if ( !tableFrameListeners.contains( listener ) ){
        tableFrameListeners.add( listener );
      }
    }
  }
  // ADDING THE LISTENERS END
  /////////////////////////////////////////////////

  ////////////////////////////////////////////////
  // REMOVING THE LISTENERS
  public void removeTableFrameListener( TableFrameListener oListener ) {
    synchronized( tableFrameListeners ) {
      if ( tableFrameListeners.contains( oListener ) )
        tableFrameListeners.remove( oListener );
    }
  }

  // return true if the game is connected
  public boolean isConnected() {
    return ( currentState == STATE_CONNECTED || currentState == STATE_LOGGEDIN );
  }

  public boolean isLoggedIn() {
    return ( currentState == STATE_LOGGEDIN );
  }
  public void connect( ) {
    connect( SERVER_ADDRESS, SERVER_PORT );
  }

  public void connect( String oHostname, int iPort ){
    if (currentState == STATE_DISCONNECTED) {
      // attempt to connect to the server
      protocolHandler = new ProtocolHandler(oHostname, iPort);
      if (protocolHandler.isConnected()) {
        // if it worked then we are connected, set the state to reflect so
        currentState = STATE_CONNECTED;
        // since we are connected, start the thread to process incomming messages
        messageProcessingThread = new Thread(new MessageProcessor());
        messageProcessingThread.start();
        return;
      }
      else {
        // failed to connect to the server
        // TODO: throw a ConnectionFailedException
      }
    }
    else {
      // Already connected
      // TODO: throw an AlreadyConnectedException
    }
  }


  public void disconnect() {
    if (currentState != STATE_DISCONNECTED) {
      if (protocolHandler.isConnected()) {
        // Friendly disconnect means we tell the server we are disconnecting
        // before we drop the connection
        protocolHandler.sendMessage(new Disconnect());
        // reflect the change in state, lets the MessageProcessor thread
        // know to exit
        currentState = STATE_DISCONNECTED;
      }
      else {
        // Already lost the connection, set state to reflect this
        currentState = STATE_DISCONNECTED;
      }
      // tell the listeners we are disconnected either way
      for ( int i = 0; i < chatListeners.size(); i ++ ) {
        ChatListener oListener = (ChatListener) chatListeners.get(i);
        oListener.disconnectReceived();
      }

      for ( int i = 0; i < tablePanelListeners.size(); i ++ ) {
        TablePanelListener oListener = (TablePanelListener) tablePanelListeners.get(i);
        oListener.disconnectReceived();
      }
      for ( int i = 0; i < buttonListeners.size(); i ++ ) {
        ButtonListener oListener = (ButtonListener) buttonListeners.get(i);
        oListener.disconnectReceived();
      }
    }
    else {
      // Already disconnected
      // Dont throw an exception, if someone wants to disconnect just remain
      // disconnected.
    }
  }

  public void setUsername( String oUsername ) {
    username = oUsername;
    for ( int i = 0; i < chatListeners.size(); i ++ ) {
      ChatListener oListener = (ChatListener) chatListeners.get( i );
      oListener.setUsername( username );
    }
  }

  public String getUsername() {
    return username;
  }

  public void playerLogin(String oName, String oPassword) {
    if (currentState == STATE_CONNECTED) {
      protocolHandler.sendMessage(new PlayerLogin(oName, oPassword));
    }
    else {
      if (currentState == STATE_DISCONNECTED) {
        // Not connected, please connect before attempting to log in
        // TODO: throw a NotConnectedException
      }
      else if (currentState == STATE_LOGGEDIN) {
        // Already logged in
        // TODO: throw an AlreadyLoggedInException
      }
    }
  }

  public void lobbyChat( LobbyChat lobbyChat ) {
    if (currentState == STATE_LOGGEDIN) {
      protocolHandler.sendMessage( lobbyChat );
    }
  }

  public void tableChat( String oChat ) {
    TableChat oTableChat = new TableChat( username, oChat );
    if ( currentState == STATE_LOGGEDIN ) {
      protocolHandler.sendMessage( oTableChat );
    }
  }

  public void createTable( Rules rules ) {
    if ( currentState == STATE_LOGGEDIN ) {
      protocolHandler.sendMessage( new CreateTable( rules ) );
    }
  }

  public void joinTable( int iTableNum ) {
    if ( currentState == STATE_LOGGEDIN ) {
      protocolHandler.sendMessage( new JoinTable( iTableNum ) );
    }
  }

  class MessageProcessor implements Runnable {
    public void run() {
      while( protocolHandler.isConnected() ) {
        if ( protocolHandler.hasMessage() ) {
          Message oMessage = protocolHandler.nextMessage();
          switch( currentState ) {
            case STATE_CONNECTED:
            {
              // in the connected state we can receive 4 types of messages:
              // WelcomePlayer, Error, and Disconnect.
              if ( oMessage.getHeader().equals( WelcomePlayer.type ) ) {
                WelcomePlayer oCommand = new WelcomePlayer( oMessage );
                currentState = STATE_LOGGEDIN;
                for( int i = 0; i < buttonListeners.size(); i ++ ) {
                  ButtonListener oListener = (ButtonListener) buttonListeners.get( i );
                  oListener.welcomePlayerReceived( oCommand );
                }
                for( int i = 0; i < chatListeners.size(); i ++ ) {
                  ChatListener oListener = (ChatListener) chatListeners.get( i );
                  oListener.welcomePlayerReceived( oCommand );
                }
                for( int i = 0; i < tablePanelListeners.size(); i ++ ) {
                  TablePanelListener oListener = (TablePanelListener) tablePanelListeners.get( i );
                  oListener.welcomePlayerReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( fortytwo.network.ErrorMessage.type ) ) {
                fortytwo.network.ErrorMessage oCommand = new fortytwo.network.ErrorMessage( oMessage );
              }
              else if ( oMessage.getHeader().equals( Disconnect.type ) ) {
                Disconnect oCommand = new Disconnect( oMessage );
                for ( int i = 0; i < buttonListeners.size(); i ++ ) {
                  ButtonListener oListener = ( ButtonListener ) buttonListeners.get( i );
                  oListener.disconnectReceived();
                }
                for ( int i = 0; i < tablePanelListeners.size(); i ++ ) {
                  TablePanelListener oListener = ( TablePanelListener ) tablePanelListeners.get( i );
                  oListener.disconnectReceived();
                }
                for ( int i = 0; i < chatListeners.size(); i ++ ) {
                  ChatListener oListener = ( ChatListener ) chatListeners.get( i );
                  oListener.disconnectReceived();
                }
                currentState = STATE_DISCONNECTED;
              }
              else {
                // Unsupported message type received for the connected state
                // might tell the user, might not...
              }
            }// case end

            case STATE_LOGGEDIN:
            {
              // in the logged in state we can receive # types of messages:
              if ( oMessage.getHeader().equals( LobbyChatList.type ) ) {
                LobbyChatList oCommand = new LobbyChatList( oMessage );
                for ( int i = 0; i < chatListeners.size(); i ++ ) {
                  ChatListener oListener = ( ChatListener ) chatListeners.get( i );
                  oListener.lobbyChatListReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( LobbyChat.type ) ) {
                LobbyChat oCommand = new LobbyChat( oMessage );
                for ( int i = 0; i < chatListeners.size(); i ++ ) {
                  ChatListener oListener = ( ChatListener ) chatListeners.get( i );
                  oListener.lobbyChatReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( LobbyTableList.type ) ) {
                LobbyTableList oCommand = new LobbyTableList( oMessage );
                for( int i=0; i < tablePanelListeners.size(); i ++) {
                  TablePanelListener oListener = (TablePanelListener) tablePanelListeners.get( i );
                  oListener.lobbyTableListReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( LoadTable.type ) ) {
                LoadTable oCommand = new LoadTable( oMessage );
                new TableFrame( oCommand );
              }
              else if ( oMessage.getHeader().equals( TableChat.type ) ) {
                TableChat oCommand = new TableChat( oMessage );
                for ( int i = 0; i < tableFrameListeners.size() ; i ++ ) {
                  TableFrameListener oListener = (TableFrameListener) tableFrameListeners.get(i);
                  oListener.tableChatReceived( oCommand.getUsername(), oCommand.getChat() );
                }
              }
              else if ( oMessage.getHeader().equals( fortytwo.network.ErrorMessage.type ) ) {
                ErrorMessage oCommand = new ErrorMessage( oMessage );
              }
              else if ( oMessage.getHeader().equals( Disconnect.type ) ) {
                Disconnect oCommand = new Disconnect( oMessage );
                for ( int i = 0; i < tableFrameListeners.size(); i ++ ) {
                  TableFrameListener oListener = ( TableFrameListener ) tableFrameListeners.get( i );
                  oListener .disconnectReceived();
                }
                for ( int i = 0; i < buttonListeners.size(); i ++ ) {
                  ButtonListener oListener = ( ButtonListener ) buttonListeners.get( i );
                  oListener.disconnectReceived();
                }
                for ( int i = 0; i < tablePanelListeners.size(); i ++ ) {
                  TablePanelListener oListener = ( TablePanelListener ) tablePanelListeners.get( i );
                  oListener.disconnectReceived();
                }
                for ( int i = 0; i < chatListeners.size(); i ++ ) {
                  ChatListener oListener = ( ChatListener ) chatListeners.get( i );
                  oListener.disconnectReceived();
                }
                currentState = STATE_DISCONNECTED;
              }
            }
            case STATE_DISCONNECTED:
            {

            }
          }// switch end
        }// if end
        else {
          // Sleep some so as to not hog the CPU
          try {
            Thread.sleep(25);
          }
          catch (InterruptedException oException) {}
        }

      }// while end
    }// run() end
  }

}
