#include "DIInput.h"
#include <sstream>

ChaosEngine::DIInput::DIInput( System* pSystem ) : m_pSystem( pSystem )
{
	SetInputEventHandler( dynamic_cast< IInputEventHandler* >( pSystem ) );

	m_pDI = NULL;
	m_pDIKeyboardDevice = NULL;
	m_pDIMouseDevice = NULL;

	m_hInstance = NULL;
	m_hWindowMain = NULL;

	m_bReady = false;
	m_bAquired = false;
	m_bKeyboardPolled = false;
	m_bMousePolled = false;
};

ChaosEngine::DIInput::~DIInput( void )
{
	m_hInstance = NULL;
	m_hWindowMain = NULL;
	Shutdown();
};

void ChaosEngine::DIInput::SetInputEventHandler( IInputEventHandler* pHandler )
{
	m_pHandler = pHandler;
};

void ChaosEngine::DIInput::Startup( void )
{
	if ( ( m_hInstance != NULL ) && ( m_hWindowMain != NULL ) )
	{
		HRESULT hr; 
		// Create the DirectInput object. 
		hr = DirectInput8Create( m_hInstance , DIRECTINPUT_VERSION , IID_IDirectInput8 , (void**)&m_pDI , NULL ); 
		if ( !FAILED( hr ) )
		{
			// Retrieve a pointer to an IDirectInputDevice8 interface 
			hr = m_pDI->CreateDevice( GUID_SysKeyboard , &m_pDIKeyboardDevice , NULL ); 
			if ( !FAILED( hr ) ) 
			{ 
				// Now that you have an IDirectInputDevice8 interface, get 
				// it ready to use. 

				// Set the data format using the predefined keyboard data 
				// format provided by the DirectInput object for keyboards. 

				hr = m_pDIKeyboardDevice->SetDataFormat( &c_dfDIKeyboard ); 
				if ( !FAILED( hr ) ) 
				{ 
					// Set the cooperative level 
					hr = m_pDIKeyboardDevice->SetCooperativeLevel( m_hWindowMain , DISCL_FOREGROUND | DISCL_NONEXCLUSIVE ); 
					if ( !FAILED( hr ) ) 
					{ 
						hr = m_pDI->CreateDevice( GUID_SysMouse , &m_pDIMouseDevice , NULL );
						if ( !FAILED( hr ) )
						{
							hr = m_pDIMouseDevice->SetDataFormat( &c_dfDIMouse2 );
							if ( !FAILED( hr ) )
							{
								hr = m_pDIMouseDevice->SetCooperativeLevel( m_hWindowMain , DISCL_EXCLUSIVE | DISCL_FOREGROUND );
								if ( !FAILED( hr ) )
								{
									m_bReady = true;
								}
								else
								{
									ChaosEngine::Log::WriteMessage( "DIInput - failed to set mouse cooperative level" , ChaosEngine::Log::Error );
								}
							}
							else
							{
								ChaosEngine::Log::WriteMessage( "DIInput - failed to set mouse data format" , ChaosEngine::Log::Error );
							}
						}
						else
						{
							ChaosEngine::Log::WriteMessage( "DIInput - failed to create mouse device" , ChaosEngine::Log::Error );
						}
					}
					else
					{
						ChaosEngine::Log::WriteMessage( "DIInput - failed to set keyboard cooperative level" , ChaosEngine::Log::Error );
					}
				}
				else
				{
					ChaosEngine::Log::WriteMessage( "DIInput - failed to set keyboard data format" , ChaosEngine::Log::Error );
				}
			} 
			else
			{
				ChaosEngine::Log::WriteMessage( "DIInput - failed to create keyboard device" , ChaosEngine::Log::Error );
			}
		}
		else
		{
			ChaosEngine::Log::WriteMessage( "DIInput - failed to create DirectInput object" , ChaosEngine::Log::Error );
		}
	}
};

bool ChaosEngine::DIInput::IsReady( void )
{
	return m_bReady;
};

void ChaosEngine::DIInput::Shutdown( void )
{
	UnAquire();
	if ( m_pDI ) 
	{ 
		if ( m_pDIMouseDevice ) 
		{ 
			m_pDIMouseDevice->Release();
			m_pDIMouseDevice = NULL; 
		} 
		if ( m_pDIKeyboardDevice ) 
		{ 
			m_pDIKeyboardDevice->Release();
			m_pDIKeyboardDevice = NULL; 
		} 
		m_pDI->Release();
		m_pDI = NULL; 
	}
	m_bReady = false;
};

void ChaosEngine::DIInput::Aquire( void )
{
	HRESULT hr; 
	if ( m_bReady )
	{
		hr = m_pDIKeyboardDevice->Acquire();
		if ( !FAILED( hr ) )
		{
			hr = m_pDIMouseDevice->Acquire();
			if ( !FAILED( hr ) )
			{
				m_bKeyboardPolled = false;
				m_bMousePolled = false;
				m_bAquired = true;
			}
			else
			{
				ChaosEngine::Log::WriteMessage( "DIInput - failed to aquire mouse device" , ChaosEngine::Log::Error );
			}
		}
		else
		{
			ChaosEngine::Log::WriteMessage( "DIInput - failed to aquire keyboard device" , ChaosEngine::Log::Error );
		}
	}
};

bool ChaosEngine::DIInput::IsAquired( void )
{
	return m_bAquired;
};

void ChaosEngine::DIInput::UnAquire( void )
{
	if ( m_bReady && m_bAquired )
	{
		m_pDIMouseDevice->Unacquire(); 
		m_pDIKeyboardDevice->Unacquire(); 
		m_bAquired = false;
	}
};

void ChaosEngine::DIInput::Poll( void )
{
	if ( m_bReady && m_bAquired )
	{
		HRESULT hr; 
		hr = m_pDIKeyboardDevice->GetDeviceState( sizeof( m_aKeyboardData ) , ( LPVOID )&m_aKeyboardData );
		if ( !FAILED( hr ) ) 
		{
			hr = m_pDIMouseDevice->GetDeviceState( sizeof( m_oMouseState ) , ( LPVOID )&m_oMouseState );
			if ( !FAILED( hr ) ) 
			{
				if ( m_bKeyboardPolled )
				{
					for ( int i = 0 ; i < 256 ; ++i )
					{
						if ( m_aKeyboardData[ i ] > m_aLastKeyboardData[ i ] )
						{
							m_pHandler->KeyUp( GetKeyName( i ) );
						}
						else if ( m_aKeyboardData[ i ] < m_aLastKeyboardData[ i ] )
						{
							m_pHandler->KeyDown( GetKeyName( i ) );
						}
					}
				}
				else
				{
					m_bKeyboardPolled = true;
				}
				if ( m_bMousePolled )
				{
					if ( ( m_oMouseState.lX != 0 ) || ( m_oMouseState.lY != 0 ) || ( m_oMouseState.lZ != 0 ) )
					{
						float* pAxisDeltas = new float[ 3 ];
						pAxisDeltas[ 0 ] = static_cast< float >( m_oMouseState.lX );
						pAxisDeltas[ 1 ] = static_cast< float >( m_oMouseState.lY );
						pAxisDeltas[ 2 ] = static_cast< float >( m_oMouseState.lZ );
						m_pHandler->MouseMoved( pAxisDeltas );
						delete[] pAxisDeltas;
					}
					for ( int i = 0 ; i < sizeof( m_oMouseState.rgbButtons ) ; ++i )
					{
						if ( m_oMouseState.rgbButtons[ i ] > m_oLastMouseState.rgbButtons[ i ] )
						{
							m_pHandler->ButtonDown( GetButtonName( i ) );
						}
						else if ( m_oMouseState.rgbButtons[ i ] < m_oLastMouseState.rgbButtons[ i ] )
						{
							m_pHandler->ButtonUp( GetButtonName( i ) );
						}
					}
				}
				else
				{
					m_bMousePolled = true;
				}
				memcpy( static_cast< void* >( m_aLastKeyboardData ) , static_cast< void* >( m_aKeyboardData ) , sizeof( m_aLastKeyboardData ) );
				memcpy( static_cast< void* >( &m_oLastMouseState ) , static_cast< void* >( &m_oMouseState ) , sizeof( m_oLastMouseState ) );
			}
			else if ( hr == DIERR_INPUTLOST )
			{
				Aquire();
			}
			else
			{
				ChaosEngine::Log::WriteMessage( "DIInput - failed to get mouse device state" , ChaosEngine::Log::Error );
			}
		}
		else if ( hr == DIERR_INPUTLOST )
		{
			Aquire();
		}
		else
		{
			ChaosEngine::Log::WriteMessage( "DIInput - failed to get keyboard device state" , ChaosEngine::Log::Error );
		}
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "DIInput - Input polled but not ready and/or aquired" , ChaosEngine::Log::Error );
	}
};

void ChaosEngine::DIInput::SetInstanceHandle( HINSTANCE hInstance )
{
	m_hInstance = hInstance;
};

void ChaosEngine::DIInput::SetWindowHandle( HWND hWindow )
{
	m_hWindowMain = hWindow;
};

std::string ChaosEngine::DIInput::GetKeyName( int i )
{
	switch ( i )
	{
	case 0x01:
		return std::string( "ESCAPE" );
	case 0x02:
		return std::string( "1" );
	case 0x03:
		return std::string( "2" );
	case 0x04:
		return std::string( "3" );
	case 0x05:
		return std::string( "4" );
	case 0x06:
		return std::string( "5" );
	case 0x07:
		return std::string( "6" );
	case 0x08:
		return std::string( "7" );
	case 0x09:
		return std::string( "8" );
	case 0x0A:
		return std::string( "9" );
	case 0x0B:
		return std::string( "0" );
	case 0x0C:
		return std::string( "MINUS" );
	case 0x0D:
		return std::string( "EQUALS" );
	case 0x0E:
		return std::string( "BACK" );
	case 0x0F:
		return std::string( "TAB" );
	case 0x10:
		return std::string( "Q" );
	case 0x11:
		return std::string( "W" );
	case 0x12:
		return std::string( "E" );
	case 0x13:
		return std::string( "R" );
	case 0x14:
		return std::string( "T" );
	case 0x15:
		return std::string( "Y" );
	case 0x16:
		return std::string( "U" );
	case 0x17:
		return std::string( "I" );
	case 0x18:
		return std::string( "O" );
	case 0x19:
		return std::string( "P" );
	case 0x1A:
		return std::string( "LBRACKET" );
	case 0x1B:
		return std::string( "RBRACKET" );
	case 0x1C:
		return std::string( "RETURN" );
	case 0x1D:
		return std::string( "LCONTROL" );
	case 0x1E:
		return std::string( "A" );
	case 0x1F:
		return std::string( "S" );
	case 0x20:
		return std::string( "D" );
	case 0x21:
		return std::string( "F" );
	case 0x22:
		return std::string( "G" );
	case 0x23:
		return std::string( "H" );
	case 0x24:
		return std::string( "J" );
	case 0x25:
		return std::string( "K" );
	case 0x26:
		return std::string( "L" );
	case 0x27:
		return std::string( "SEMICOLON" );
	case 0x28:
		return std::string( "APOSTROPHE" );
	case 0x29:
		return std::string( "GRAVE" );
	case 0x2A:
		return std::string( "LSHIFT" );
	case 0x2B:
		return std::string( "BACKSLASH" );
	case 0x2C:
		return std::string( "Z" );
	case 0x2D:
		return std::string( "X" );
	case 0x2E:
		return std::string( "C" );
	case 0x2F:
		return std::string( "V" );
	case 0x30:
		return std::string( "B" );
	case 0x31:
		return std::string( "N" );
	case 0x32:
		return std::string( "M" );
	case 0x33:
		return std::string( "COMMA" );
	case 0x34:
		return std::string( "PERIOD" );
	case 0x35:
		return std::string( "SLASH" );
	case 0x36:
		return std::string( "RSHIFT" );
	case 0x37:
		return std::string( "MULTIPLY" );
	case 0x38:
		return std::string( "LMENU" );
	case 0x39:
		return std::string( "SPACE" );
	case 0x3A:
		return std::string( "CAPITAL" );
	case 0x3B:
		return std::string( "F1" );
	case 0x3C:
		return std::string( "F2" );
	case 0x3D:
		return std::string( "F3" );
	case 0x3E:
		return std::string( "F4" );
	case 0x3F:
		return std::string( "F5" );
	case 0x40:
		return std::string( "F6" );
	case 0x41:
		return std::string( "F7" );
	case 0x42:
		return std::string( "F8" );
	case 0x43:
		return std::string( "F9" );
	case 0x44:
		return std::string( "F10" );
	case 0x45:
		return std::string( "NUMLOCK" );
	case 0x46:
		return std::string( "SCROLL" );
	case 0x47:
		return std::string( "NUMPAD7" );
	case 0x48:
		return std::string( "NUMPAD8" );
	case 0x49:
		return std::string( "NUMPAD9" );
	case 0x4A:
		return std::string( "SUBTRACT" );
	case 0x4B:
		return std::string( "NUMPAD4" );
	case 0x4C:
		return std::string( "NUMPAD5" );
	case 0x4D:
		return std::string( "NUMPAD6" );
	case 0x4E:
		return std::string( "ADD" );
	case 0x4F:
		return std::string( "NUMPAD1" );
	case 0x50:
		return std::string( "NUMPAD2" );
	case 0x51:
		return std::string( "NUMPAD3" );
	case 0x52:
		return std::string( "NUMPAD0" );
	case 0x53:
		return std::string( "DECIMAL" );
	case 0x56:
		return std::string( "OEM_102" );
	case 0x57:
		return std::string( "F11" );
	case 0x58:
		return std::string( "F12" );
	case 0x64:
		return std::string( "F13" );
	case 0x65:
		return std::string( "F14" );
	case 0x66:
		return std::string( "F15" );
	case 0x70:
		return std::string( "KANA" );
	case 0x73:
		return std::string( "ABNT_C1" );
	case 0x79:
		return std::string( "CONVERT" );
	case 0x7B:
		return std::string( "NOCONVERT" );
	case 0x7D:
		return std::string( "YEN" );
	case 0x7E:
		return std::string( "ABNT_C2" );
	case 0x8D:
		return std::string( "NUMPADEQUALS" );
	case 0x90:
		return std::string( "PREVTRACK" );
	case 0x91:
		return std::string( "AT" );
	case 0x92:
		return std::string( "COLON" );
	case 0x93:
		return std::string( "UNDERLINE" );
	case 0x94:
		return std::string( "KANJI" );
	case 0x95:
		return std::string( "STOP" );
	case 0x96:
		return std::string( "AX" );
	case 0x97:
		return std::string( "UNLABELED" );
	case 0x99:
		return std::string( "NEXTTRACK" );
	case 0x9C:
		return std::string( "NUMPADENTER" );
	case 0x9D:
		return std::string( "RCONTROL" );
	case 0xA0:
		return std::string( "MUTE" );
	case 0xA1:
		return std::string( "CALCULATOR" );
	case 0xA2:
		return std::string( "PLAYPAUSE" );
	case 0xA4:
		return std::string( "MEDIASTOP" );
	case 0xAE:
		return std::string( "VOLUMEDOWN" );
	case 0xB0:
		return std::string( "VOLUMEUP" );
	case 0xB2:
		return std::string( "WEBHOME" );
	case 0xB3:
		return std::string( "NUMPADCOMMA" );
	case 0xB5:
		return std::string( "DIVIDE" );
	case 0xB7:
		return std::string( "SYSRQ" );
	case 0xB8:
		return std::string( "RMENU" );
	case 0xC5:
		return std::string( "PAUSE" );
	case 0xC7:
		return std::string( "HOME" );
	case 0xC8:
		return std::string( "UP" );
	case 0xC9:
		return std::string( "PRIOR" );
	case 0xCB:
		return std::string( "LEFT" );
	case 0xCD:
		return std::string( "RIGHT" );
	case 0xCF:
		return std::string( "END" );
	case 0xD0:
		return std::string( "DOWN" );
	case 0xD1:
		return std::string( "NEXT" );
	case 0xD2:
		return std::string( "INSERT" );
	case 0xD3:
		return std::string( "DELETE" );
	case 0xDB:
		return std::string( "LWIN" );
	case 0xDC:
		return std::string( "RWIN" );
	case 0xDD:
		return std::string( "APPS" );
	case 0xDE:
		return std::string( "POWER" );
	case 0xDF:
		return std::string( "SLEEP" );
	case 0xE3:
		return std::string( "WAKE" );
	case 0xE5:
		return std::string( "WEBSEARCH" );
	case 0xE6:
		return std::string( "WEBFAVORITES" );
	case 0xE7:
		return std::string( "WEBREFRESH" );
	case 0xE8:
		return std::string( "WEBSTOP" );
	case 0xE9:
		return std::string( "WEBFORWARD" );
	case 0xEA:
		return std::string( "WEBBACK" );
	case 0xEB:
		return std::string( "MYCOMPUTER" );
	case 0xEC:
		return std::string( "MAIL" );
	case 0xED:
		return std::string( "MEDIASELECT" );
	default:
		{
			std::stringstream oNameStream;
			oNameStream << "Key " << i + 1;
			return oNameStream.str();
		}
	}
};

std::string ChaosEngine::DIInput::GetButtonName( int i )
{
	std::stringstream oNameStream;
	oNameStream << "Mouse Button " << i + 1;
	return oNameStream.str();
};
