#include "FMODAudio.h"

ChaosEngine::FMODAudio::FMODAudio( System* pSystem ) : m_pSystem( pSystem )
{
	m_pFMODSystem = NULL;
	FMOD_RESULT iFMOD_Result = FMOD::System_Create( &m_pFMODSystem );
	if ( iFMOD_Result != FMOD_OK ) 
	{
		ChaosEngine::Log::WriteMessage( "FMODAudio - failed to create system" , ChaosEngine::Log::Error );
		m_pFMODSystem = NULL;
	}
	else
	{
		iFMOD_Result = m_pFMODSystem->init( 100 , FMOD_INIT_NORMAL , 0 );
		if ( iFMOD_Result != FMOD_OK )
		{
			ChaosEngine::Log::WriteMessage( "FMODAudio - failed to init system" , ChaosEngine::Log::Error );
			m_pFMODSystem->release();
			m_pFMODSystem = NULL;
		}
	}
};

ChaosEngine::FMODAudio::~FMODAudio( void )
{
	if ( m_pFMODSystem != NULL )
	{
		m_pFMODSystem->release();
		m_pFMODSystem = NULL;
	}
	m_oSounds.clear();
	m_oChannels.clear();
};

void ChaosEngine::FMODAudio::Update( void )
{
	if ( m_pFMODSystem != NULL )
	{
		m_pFMODSystem->update();
	}
};

void ChaosEngine::FMODAudio::LoadSound( const std::string& oSoundName , bool bLooping , const std::string& oFilename )
{
	if ( m_pFMODSystem != NULL )
	{
		FMOD::Sound* pSound;
		FMOD_RESULT iFMOD_Result;
		if ( bLooping )
		{
			iFMOD_Result = m_pFMODSystem->createSound( oFilename.c_str() , FMOD_3D | FMOD_LOOP_NORMAL | FMOD_HARDWARE , NULL , &pSound );
		}
		else
		{
			iFMOD_Result = m_pFMODSystem->createSound( oFilename.c_str() , FMOD_3D | FMOD_HARDWARE , NULL , &pSound );
		}
		if ( iFMOD_Result != FMOD_OK )
		{
			ChaosEngine::Log::WriteMessage( "FMODAudio - failed to create sound" , ChaosEngine::Log::Error );
		}
		else
		{
			pSound->set3DMinMaxDistance( 2.0f , 10000.0f );
			if ( m_oSounds.find( oSoundName ) == m_oSounds.end() )
			{
				m_oSounds[ oSoundName ] = pSound;
			}
			else
			{
				ChaosEngine::Log::WriteMessage( "FMODAudio - sound already exists" , ChaosEngine::Log::Error );
			}
		}
	}
};

void ChaosEngine::FMODAudio::CreateSoundSource( const std::string& oSoundSourceName , const std::string& oSoundName , float* pPosition )
{
	if ( m_pFMODSystem != NULL )
	{
		FMOD::Channel* pChannel;
		SoundHash::iterator oSoundIterator = m_oSounds.find( oSoundName );
		if ( oSoundIterator != m_oSounds.end() )
		{
			ChannelHash::iterator oChannelIterator = m_oChannels.find( oSoundSourceName );
			if ( oChannelIterator == m_oChannels.end() )
			{
				FMOD_RESULT iFMOD_Result = m_pFMODSystem->playSound( FMOD_CHANNEL_FREE , oSoundIterator->second , true , &pChannel );
				if ( iFMOD_Result != FMOD_OK )
				{
					ChaosEngine::Log::WriteMessage( "FMODAudio - failed to play sound" , ChaosEngine::Log::Error );
				}
				else
				{
					FMOD_VECTOR oPosition;
					oPosition.x = pPosition[ 0 ];
					oPosition.y = pPosition[ 1 ];
					oPosition.z = pPosition[ 2 ];
					iFMOD_Result = pChannel->set3DAttributes( &oPosition , NULL );
					if ( iFMOD_Result != FMOD_OK )
					{
						ChaosEngine::Log::WriteMessage( "FMODAudio - failed to set 3D sound attributes" , ChaosEngine::Log::Error );
					}
					else
					{
						iFMOD_Result = pChannel->setPaused( false );
						if ( iFMOD_Result != FMOD_OK )
						{
							ChaosEngine::Log::WriteMessage( "FMODAudio - failed to unpause sound" , ChaosEngine::Log::Error );
						}
						else
						{
							FMOD_MODE oMode;
							oSoundIterator->second->getMode( &oMode );
							if ( oMode & FMOD_LOOP_NORMAL )
							{
								m_oChannels[ oSoundSourceName ] = pChannel;
							}
						}
					}
				}
			}
			else
			{
				ChaosEngine::Log::WriteMessage( "FMODAudio - sound source already exists" , ChaosEngine::Log::Error );
			}
		}
		else
		{
			ChaosEngine::Log::WriteMessage( "FMODAudio - sound doesn't exist" , ChaosEngine::Log::Error );
		}
	}
};

void ChaosEngine::FMODAudio::DestroySoundSource( const std::string& oSoundSourceName )
{
	ChannelHash::iterator oChannelIterator = m_oChannels.find( oSoundSourceName );
	if ( oChannelIterator != m_oChannels.end() )
	{
		FMOD_RESULT iFMOD_Result = oChannelIterator->second->stop();
		if ( iFMOD_Result != FMOD_OK )
		{
			ChaosEngine::Log::WriteMessage( "FMODAudio - failed to stop channel" , ChaosEngine::Log::Error );
		}
		m_oChannels.erase( oChannelIterator );
	}
};

void ChaosEngine::FMODAudio::SetListenerPosition( float* pPosition , float* pForward , float* pUp )
{
	if ( m_pFMODSystem != NULL )
	{
		FMOD_VECTOR oPosition;
		oPosition.x = pPosition[ 0 ];
		oPosition.y = pPosition[ 1 ];
		oPosition.z = pPosition[ 2 ];
		FMOD_VECTOR oForward;
		oForward.x = pForward[ 0 ];
		oForward.y = pForward[ 1 ];
		oForward.z = pForward[ 2 ];
		FMOD_VECTOR oUp;
		oUp.x = pUp[ 0 ];
		oUp.y = pUp[ 1 ];
		oUp.z = pUp[ 2 ];
		FMOD_RESULT iFMOD_Result = m_pFMODSystem->set3DListenerAttributes( 0 , &oPosition , NULL , &oForward , &oUp );
		if ( iFMOD_Result != FMOD_OK )
		{
			ChaosEngine::Log::WriteMessage( "FMODAudio - failed to set 3D listener attributes" , ChaosEngine::Log::Error );
		}
	}
};

void ChaosEngine::FMODAudio::SetSoundSourcePosition( const std::string& oSoundSourceName , float* pPosition )
{
	ChannelHash::iterator oChannelIterator = m_oChannels.find( oSoundSourceName );
	if ( oChannelIterator != m_oChannels.end() )
	{
		FMOD_VECTOR oPosition;
		oPosition.x = pPosition[ 0 ];
		oPosition.y = pPosition[ 1 ];
		oPosition.z = pPosition[ 2 ];
		FMOD_RESULT iFMOD_Result = oChannelIterator->second->set3DAttributes( &oPosition , NULL );
		if ( iFMOD_Result != FMOD_OK )
		{
			ChaosEngine::Log::WriteMessage( "FMODAudio - failed to set 3D sound attributes" , ChaosEngine::Log::Error );
		}
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "FMODAudio - sound source doesn't exist" , ChaosEngine::Log::Error );
	}
};

void ChaosEngine::FMODAudio::SetMasterVolume( float fVolume )
{
	if ( m_pFMODSystem != NULL )
	{
		FMOD::ChannelGroup* pGroup;
		FMOD_RESULT iFMOD_Result = m_pFMODSystem->getMasterChannelGroup( &pGroup );
		if ( iFMOD_Result != FMOD_OK )
		{
			ChaosEngine::Log::WriteMessage( "FMODAudio - failed to get master channel group" , ChaosEngine::Log::Error );
		}
		else
		{
			iFMOD_Result = pGroup->setVolume( fVolume );
			if ( iFMOD_Result != FMOD_OK )
			{
				ChaosEngine::Log::WriteMessage( "FMODAudio - failed to set group volume" , ChaosEngine::Log::Error );
			}
		}
	};
};
