#include "System.h"

// constructor - initializes a system, after creation the system has no states and no subsystems
ChaosEngine::System::System( void )
{
	m_pGraphics = NULL;
	m_pAudio    = NULL;
	m_pInput    = NULL;
	m_pPhysics  = NULL;
	m_pStateManager = new StateManager();
	m_pStateManager->Init();
};

// destructor - cleans up the states and subsystems of this system
ChaosEngine::System::~System( void )
{
	if ( m_pStateManager != NULL ) { delete m_pStateManager; m_pStateManager = NULL; }
	if ( m_pGraphics != NULL )     { delete m_pGraphics;     m_pGraphics = NULL; }
	if ( m_pAudio != NULL )        { delete m_pAudio;        m_pAudio = NULL; }
	if ( m_pInput != NULL )        { delete m_pInput;        m_pInput = NULL; }
	if ( m_pPhysics != NULL )      { delete m_pPhysics;      m_pPhysics = NULL; }
};

// graphics subsystem mutator
void ChaosEngine::System::SetGraphics( IGraphics* pGraphics )
{
	m_pGraphics = pGraphics;
};

// graphics subsystem accessor
ChaosEngine::IGraphics* ChaosEngine::System::GetGraphics( void )
{
	return m_pGraphics;
};

// audio subsystem mutator
void ChaosEngine::System::SetAudio( IAudio* pAudio )
{
	m_pAudio = pAudio;
};

// audio subsystem accessor
ChaosEngine::IAudio* ChaosEngine::System::GetAudio( void )
{
	return m_pAudio;
};

// input subsystem mutator
void ChaosEngine::System::SetInput( IInput* pInput )
{
	m_pInput = pInput;
};

// input subsystem accessor
ChaosEngine::IInput* ChaosEngine::System::GetInput( void )
{
	return m_pInput;
};

// physics subsystem mutator
void ChaosEngine::System::SetPhysics( IPhysics* pPhysics )
{
	m_pPhysics = pPhysics;
};

// physics subsystem accessor
ChaosEngine::IPhysics* ChaosEngine::System::GetPhysics( void )
{
	return m_pPhysics;
};

// add a state to this system's state machine
void ChaosEngine::System::AddState( IBaseState* pState )
{
	std::stringstream oMessage;
	oMessage << "System - State Registered " << pState->GetName();
	ChaosEngine::Log::WriteMessage( oMessage.str() , ChaosEngine::Log::Debug );

	m_pStateManager->RegisterState( pState->GetName() , pState );
};

// change this system's state machine state
void ChaosEngine::System::ChangeState( const std::string& oNewStateName )
{
	m_pStateManager->ChangeState( oNewStateName );
};

void ChaosEngine::System::AddEntity( const std::string& oEntityName , Entity* pEntity )
{
	EntityHash::iterator oIterator = m_oEntities.find( oEntityName );
	if ( oIterator == m_oEntities.end() )
	{
		m_oEntities[ oEntityName ] = pEntity;
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Entity already exists" , ChaosEngine::Log::Warning );
	}
};

ChaosEngine::Entity* ChaosEngine::System::GetEntity( const std::string& oEntityName )
{
	EntityHash::iterator oIterator = m_oEntities.find( oEntityName );
	if ( oIterator != m_oEntities.end() )
	{
		return oIterator->second;
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Entity doesn't exist" , ChaosEngine::Log::Warning );
	}
	return NULL;
};

void ChaosEngine::System::RemoveEntity( const std::string& oEntityName )
{
	EntityHash::iterator oIterator = m_oEntities.find( oEntityName );
	if ( oIterator != m_oEntities.end() )
	{
		delete oIterator->second;
		m_oEntities.erase( oIterator );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Entity doesn't exist" , ChaosEngine::Log::Warning );
	}
};

// System Event handlers #############################################################################################

// Update the system based on a change in time
void ChaosEngine::System::Update( float fDeltaTime )
{
	m_pStateManager->Update( std::min< float >( fDeltaTime , 0.1f ) );
};

// IPhysicsEventHandler ##############################################################################################

// pass the handling of step collision events to the current state
bool ChaosEngine::System::ShouldCreateContact( const std::string& oBody1Name , const std::string& oBody2Name )
{
	if ( m_pStateManager->GetCurrentStateClass() != NULL )
	{
		return dynamic_cast< IPhysicsEventHandler* >( m_pStateManager->GetCurrentStateClass() )->ShouldCreateContact( oBody1Name , oBody2Name );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Physics event with NULL physics subsystem" , ChaosEngine::Log::Warning );
	}
	return false;
};

// pass the handling of manual collision events to the current state
void ChaosEngine::System::Collision( const std::string& oGeometry1Name , const std::string& oGeometry2Name )
{
	if ( m_pStateManager->GetCurrentStateClass() != NULL )
	{
		dynamic_cast< IPhysicsEventHandler* >( m_pStateManager->GetCurrentStateClass() )->Collision( oGeometry1Name , oGeometry2Name );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Physics event with NULL physics subsystem" , ChaosEngine::Log::Warning );
	}
};

// IInputEventHandler ################################################################################################
// pass the handling of keyboard key down events to the current state
void ChaosEngine::System::KeyDown( const std::string& oKeyName )
{
	if ( m_pStateManager->GetCurrentStateClass() != NULL )
	{
		dynamic_cast< IInputEventHandler* >( m_pStateManager->GetCurrentStateClass() )->KeyDown( oKeyName );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Input event with NULL input subsystem" , ChaosEngine::Log::Warning );
	}
};

// pass the handling of keyboard key up events to the current state
void ChaosEngine::System::KeyUp( const std::string& oKeyName )
{
	if ( m_pStateManager->GetCurrentStateClass() != NULL )
	{
		dynamic_cast< IInputEventHandler* >( m_pStateManager->GetCurrentStateClass() )->KeyUp( oKeyName );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Input event with NULL input subsystem" , ChaosEngine::Log::Warning );
	}
};

// pass the handling of mouse button down events to the current state
void ChaosEngine::System::ButtonDown( const std::string& oButtonName )
{
	if ( m_pStateManager->GetCurrentStateClass() != NULL )
	{
		dynamic_cast< IInputEventHandler* >( m_pStateManager->GetCurrentStateClass() )->ButtonDown( oButtonName );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Input event with NULL input subsystem" , ChaosEngine::Log::Warning );
	}
};

// pass the handling of mouse button up events to the current state
void ChaosEngine::System::ButtonUp( const std::string& oButtonName )
{
	if ( m_pStateManager->GetCurrentStateClass() != NULL )
	{
		dynamic_cast< IInputEventHandler* >( m_pStateManager->GetCurrentStateClass() )->ButtonUp( oButtonName );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Input event with NULL input subsystem" , ChaosEngine::Log::Warning );
	}
};

// pass the handling of mouse move events to the current state
void ChaosEngine::System::MouseMoved( float* pAxisDeltas )
{
	if ( m_pStateManager->GetCurrentStateClass() != NULL )
	{
		dynamic_cast< IInputEventHandler* >( m_pStateManager->GetCurrentStateClass() )->MouseMoved( pAxisDeltas );
	}
	else
	{
		ChaosEngine::Log::WriteMessage( "System - Input event with NULL input subsystem" , ChaosEngine::Log::Warning );
	}
};

