/*
FILE: DungeonQuest.java
AUTHOR: Trevor Evans
CREATED: 2-24-06
MODIFIED: 4-24-06
*/

import java.awt.*;
import java.awt.event.*;
import java.text.*;
import java.io.*;
import java.util.Timer;
import java.util.TimerTask;
import java.applet.*;
import java.net.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.imageio.*;

/*
CLASS: DungeonQuest
PURPOSE: Contains the main method. This is the starting point for the entire application.
*/
public class DungeonQuest
{
	public static void main(String[] args)
	{
		DQFrame frame = new DQFrame();
		frame.show();
	}
}

/*
CLASS: DQFrame
PURPOSE: This is the largest module in the application. It contains all of the interface
related modules.
*/
class DQFrame extends JFrame
{
	private Container contentPane;
	private MapPanel mapPanel; //the panel where the game graphics are displayed
	private Hero hero; //the hero controlled directly by the player
	private ItemLookUpTable itemTable; //a look-up table containing all items
	private SpellLookUpTable spellTable; //a look-up table containing all spells
	private AudioClip musicClip; //stores the current music clip
	private AudioClip soundEffectClip; //stores the current sound effect clip
	private int dungeonLevel; //stores the current dungeon level (0 = not in dungeon)
	private int encounterCounter; //hidden counter used to count down to a random encounter
	private boolean gameOver; //set to true when hero slain in battle

	public static final int DEFAULT_WIDTH = 326; //window width
	public static final int DEFAULT_HEIGHT = 320; //window height

	public DQFrame()
	{	//set frame parameters
		setTitle("Dungeon Quest");
		setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
		setResizable(false);
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		//create objects
		hero = new Hero();
		itemTable = new ItemLookUpTable();
		spellTable = new SpellLookUpTable();
		try
		{
			musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
													"audio_files/blank.wav"));
			soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
														  "audio_files/blank.wav"));
		}
		catch(IOException exception)
		{
			musicClip = null;
			soundEffectClip = null;
		}
		contentPane = getContentPane();
		mapPanel = new MapPanel();
		contentPane.add(mapPanel);
	}

	/*
	METHOD: getMapPanel
	PURPOSE: Returns the map panel.
	*/
	public MapPanel getMapPanel()
	{
		return mapPanel;
	}

	/*
	METHOD: getMusicClip
	PURPOSE: Returns the music clip.
	*/
	public AudioClip getMusicClip()
	{
		return musicClip;
	}

	/*
	METHOD: setMusicClip
	PURPOSE: Sets the music clip to the specified audio clip.
	*/
	public void setMusicClip(AudioClip clipIn)
	{
		musicClip = clipIn;
	}

	/*
	METHOD: getSoundEffectClip
	PURPOSE: Returns the sound effect clip.
	*/
	public AudioClip getSoundEffectClip()
	{
		return soundEffectClip;
	}

	public class MapPanel extends JPanel
	{
		private int upperLeftX; //x-coordinate of top-left corner of the displayed portion of map
		private int upperLeftY; //y-coordinate of top-left corner of the displayed portion of map
		private int lowerRightX; //x-coordinate of lower-right corner of the displayed portion of map
		private int lowerRightY; //y-coordinate of lower-right corner of the displayed portion of map
		private int mapWidth; //width of current map in pixels
		private int mapHeight; //height of current map in pixels
		private int heroFacing; //direction the hero is currently facing
		private int currentMap; //used to determine the current map the hero is on
		private Map map; //stores the current map
		private Image mapImage; //stores the current map's image
		private Image heroImage; //stores the hero's image, based on his current orientation

		public static final int TITLE = 0;
		public static final int WORLD = 1;
		public static final int TOWN = 2;
		public static final int DUNGEON = 3;
		public static final int FRONT = 0;
		public static final int BACK = 1;
		public static final int RIGHT = 2;
		public static final int LEFT = 3;

		/*
		METHOD: MapPanel
		PURPOSE: This is the constructor. Initializes the map panel to display the title screen.
		*/
		public MapPanel()
		{	//set up map panel to display title screen
			heroFacing = FRONT;
			currentMap = TITLE;
			dungeonLevel = 0;
			encounterCounter = -1;
			gameOver = false;
			addKeyListener(new KeyHandler());
			setFocusable(true);
			//play title theme
			try
			{
				mapImage = ImageIO.read(new File("image_files/title_screen.gif"));
				musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
														"audio_files/title.mid"));
			}
			catch (IOException exception)
			{
				musicClip = null;
			}
			musicClip.play();
		}

		/*
		METHOD: paintComponent
		PURPOSE: This is the draw function for the map panel.
		*/
		public void paintComponent(Graphics g)
		{
			super.paintComponent(g);

			if (mapImage == null) return;

			if(currentMap == TITLE)
			{
				g.drawImage(mapImage, 0, 0, null);
			}
			else
			{
				map.drawMap(g, upperLeftX, upperLeftY, lowerRightX, lowerRightY);
				try
				{
					if(heroFacing == FRONT)
						heroImage = ImageIO.read(new File("image_files/hero1.gif"));
					if(heroFacing == BACK)
						heroImage = ImageIO.read(new File("image_files/hero3.gif"));
					if(heroFacing == RIGHT)
						heroImage = ImageIO.read(new File("image_files/hero6.gif"));
					if(heroFacing == LEFT)
						heroImage = ImageIO.read(new File("image_files/hero7.gif"));
				}
				catch (IOException exception)
				{
					System.out.println("Caught exception: " + exception);
				}
				g.drawImage(heroImage, 160, 128, null);
			}
		}

		/*
		METHOD: setUpWorldMap
		PURPOSE: Sets up the map panel to display the world map.
		*/
		public void setUpWorldMap(int heroColumnIn, int heroRowIn)
		{	//set up world map to display the world map
			currentMap = WORLD;
			map = new WorldMap(heroColumnIn, heroRowIn);
			upperLeftX = (map.getHeroColumn() - 5) * Tile.TILE_SIZE;
			upperLeftY = (map.getHeroRow() - 4) * Tile.TILE_SIZE;
			lowerRightX = (map.getHeroColumn() + 5) * Tile.TILE_SIZE;
			lowerRightY = (map.getHeroRow() + 5) * Tile.TILE_SIZE;
			mapImage = map.getImage();
			mapWidth = map.getWidth() * Tile.TILE_SIZE;
			mapHeight = map.getHeight() * Tile.TILE_SIZE;
			musicClip.stop();
			//play world theme
			try
			{
				musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
														"audio_files/world.mid"));
			}
			catch (IOException exception)
			{
				musicClip = null;
			}
			musicClip.loop();
		}

		/*
		METHOD: setUpTownMap
		PURPOSE: Sets up the map panel to display the town map.
		*/
		public void setUpTownMap(int heroColumnIn, int heroRowIn)
		{	//sets up the map panel to display the town map
			currentMap = TOWN;
			map = new TownMap(heroColumnIn, heroRowIn);
			upperLeftX = (map.getHeroColumn() - 5) * Tile.TILE_SIZE;
			upperLeftY = (map.getHeroRow() - 4) * Tile.TILE_SIZE;
			lowerRightX = (map.getHeroColumn() + 5) * Tile.TILE_SIZE;
			lowerRightY = (map.getHeroRow() + 5) * Tile.TILE_SIZE;
			mapImage = map.getImage();
			mapWidth = map.getWidth() * Tile.TILE_SIZE;
			mapHeight = map.getHeight() * Tile.TILE_SIZE;
			//play town theme
			musicClip.stop();
			try
			{
				musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
														"audio_files/town.mid"));
			}
			catch (IOException exception)
			{
				musicClip = null;
			}
			musicClip.loop();
		}

		/*
		METHOD: setUpDungeonMap
		PURPOSE: Sets up the map panel to display the current dungeon floor.
		*/

		public void setUpDungeonMap(int levelChange)
		{	//play walking up/down stairs sound effect
			try
			{
				soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
														  	  "audio_files/stairs.wav"));
			}
			catch (IOException exception)
			{
				soundEffectClip = null;
			}
			soundEffectClip.play();
			//set current map as dungeon and change dungeon level
			currentMap = DUNGEON;
			dungeonLevel += levelChange;
			if(dungeonLevel > 0)
			{	//set up map panel to display the current dungeon floor
				map = new DungeonMap(dungeonLevel, levelChange);
				upperLeftX = (map.getHeroColumn() - 5) * Tile.TILE_SIZE;
				upperLeftY = (map.getHeroRow() - 4) * Tile.TILE_SIZE;
				lowerRightX = (map.getHeroColumn() + 5) * Tile.TILE_SIZE;
				lowerRightY = (map.getHeroRow() + 5) * Tile.TILE_SIZE;
				mapImage = map.getImage();
				mapWidth = map.getWidth() * Tile.TILE_SIZE;
				mapHeight = map.getHeight() * Tile.TILE_SIZE;
			}
			else //if hero no longer in dungeon
			{
				setUpWorldMap(15, 5);
			}
		}

		/*
		METHOD: getCurrentMap
		PURPOSE: Returns the integer that specifies the current map.
		*/
		public int getCurrentMap()
		{
			return currentMap;
		}

		/*
		METHOD: getDungeonLevel
		PURPOSE: Returns the current dungeon level.
		*/
		public int getDungeonLevel()
		{
			return dungeonLevel;
		}

		/*
		METHOD: setDungeonLevel(int)
		PURPOSE: Sets the current dungeon level to the specified level.
		*/
		public void setDungeonLevel(int levelIn)
		{
			dungeonLevel = levelIn;
		}

		/*
		METHOD: setHeroFacing(int)
		PURPOSE: Sets the direction the hero is facing to the specified direction.
		*/
		public void setHeroFacing(int directionIn)
		{
			heroFacing = directionIn;
		}

		/*
		CLASS: KeyHandler
		PURPOSE: This is the key listener for the map panel.
		*/
		private class KeyHandler extends KeyAdapter
		{
			public void keyReleased(KeyEvent event)
			{
				int key = event.getKeyCode();

				if(gameOver) //if hero has been slain...
				{	//...restart game from title screen
					hero = new Hero();
					itemTable = new ItemLookUpTable();
					spellTable = new SpellLookUpTable();
					heroFacing = FRONT;
					currentMap = TITLE;
					dungeonLevel = 0;
					encounterCounter = -1;
					gameOver = false;
					musicClip.stop();
					try
					{
						mapImage = ImageIO.read(new File("image_files/title_screen.gif"));
						musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																"audio_files/title.mid"));
					}
					catch (IOException exception)
					{
						musicClip = null;
					}
					musicClip.play();
				}
				else if(key == KeyEvent.VK_LEFT) //if left arrow key pressed...
				{	//if tile in front of hero is passable...
					if(map.getTile(map.getHeroColumn() - 1, map.getHeroRow()).isPassable())
					{	//...move hero left by one tile
						upperLeftX -= Tile.TILE_SIZE;
						lowerRightX -= Tile.TILE_SIZE;
						map.moveHeroLeft();
						if(currentMap == DUNGEON) encounterCounter--;
					}
					else //if tile in front of hero is not passable...
					{	//...play "bump" sound effect
						soundEffectClip.stop();
						try
						{
							soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																	  	  "audio_files/bump.wav"));
						}
						catch (IOException exception)
						{
							soundEffectClip = null;
						}
						soundEffectClip.play();
					}
					heroFacing = LEFT;
				}
				else if(key == KeyEvent.VK_RIGHT) //if right arrow key pressed...
				{	//if tile in front of hero is passable...
					if(map.getTile(map.getHeroColumn() + 1, map.getHeroRow()).isPassable())
					{	//...move hero right by one tile
						upperLeftX += Tile.TILE_SIZE;
						lowerRightX += Tile.TILE_SIZE;
						map.moveHeroRight();
						if(currentMap == DUNGEON) encounterCounter--;
					}
					else //if tile in front of hero is not passable...
					{	//...play "bump" sound effect
						soundEffectClip.stop();
						try
						{
							soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																	  	  "audio_files/bump.wav"));
						}
						catch (IOException exception)
						{
							soundEffectClip = null;
						}
						soundEffectClip.play();
					}
					heroFacing = RIGHT;
				}
				else if(key == KeyEvent.VK_UP) //if up arrow key pressed...
				{	//...if tile in front of hero is passable...
					if(map.getTile(map.getHeroColumn(), map.getHeroRow() - 1).isPassable())
					{	//...move hero up by one tile
						upperLeftY -= Tile.TILE_SIZE;
						lowerRightY -= Tile.TILE_SIZE;
						map.moveHeroUp();
						if(currentMap == DUNGEON) encounterCounter--;
					}
					else //if tile in front of hero is not passable...
					{	//...play "bump" sound effect
						soundEffectClip.stop();
						try
						{
							soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																	  	  "audio_files/bump.wav"));
						}
						catch (IOException exception)
						{
							soundEffectClip = null;
						}
						soundEffectClip.play();
					}
					heroFacing = BACK;
				}
				else if(key == KeyEvent.VK_DOWN) //if down arrow key pressed...
				{	//if tile in front of hero is passable...
					if(map.getTile(map.getHeroColumn(), map.getHeroRow() + 1).isPassable())
					{	//...move hero down by one tile
						upperLeftY += Tile.TILE_SIZE;
						lowerRightY += Tile.TILE_SIZE;
						map.moveHeroDown();
						if(currentMap == DUNGEON) encounterCounter--;
					}
					else //if tile in front of hero is not passable...
					{	//...play "bump" sound effect
						soundEffectClip.stop();
						try
						{
							soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																	  	  "audio_files/bump.wav"));
						}
						catch (IOException exception)
						{
							soundEffectClip = null;
						}
						soundEffectClip.play();
					}
					heroFacing = FRONT;
				}
				else if(key == KeyEvent.VK_ENTER) //if enter key pressed...
				{	//if at tile screen...
					if(currentMap == TITLE)
					{	//...open title menu
						new TitleMenu(DQFrame.this).show();
					}//if hero standing on maze exit (descending stairs)...
					else if(currentMap == DUNGEON && ((MazeTile) map.getTile(map.getHeroColumn(), map.getHeroRow())).isExit())
					{	//...reset encounter count down and move hero to next dungeon floor
						encounterCounter = (int) (10 + Math.random() * 15);
						setUpDungeonMap(1);
					}//if hero standing on maze entrance (ascending stairs)...
					else if(currentMap == DUNGEON && ((MazeTile) map.getTile(map.getHeroColumn(), map.getHeroRow())).isEntrance())
					{	//..reset encounter count down and move hero to previous dungeon floor
						encounterCounter = (int) (10 + Math.random() * 15);
						setUpDungeonMap(-1);
					}
					else if(currentMap == DUNGEON && heroFacing == FRONT)
					{	//if hero is standing in front of and facing treasure chest...
						if(((MazeTile) map.getTile(map.getHeroColumn(), map.getHeroRow() + 1)).isTreasureChest())
						{	//..play opening chest sound effect...
							try
							{
								soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																		  	  "audio_files/treasure_chest.wav"));
							}
							catch (IOException exception)
							{
								soundEffectClip = null;
							}
							soundEffectClip.play();
							//...and display prompt to take or leave chest contents
							new TreasureChestPrompt(DQFrame.this, ((DungeonMap) map).getChest(map.getHeroColumn(), map.getHeroRow() + 1)).show();
						}
					}
					else if(currentMap == DUNGEON && heroFacing == BACK)
					{	//if hero is standing in front of and facing treasure chest...
						if(((MazeTile) map.getTile(map.getHeroColumn(), map.getHeroRow() - 1)).isTreasureChest())
						{	//..play opening chest sound effect...
							try
							{
								soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																		  	  "audio_files/treasure_chest.wav"));
							}
							catch (IOException exception)
							{
								soundEffectClip = null;
							}
							soundEffectClip.play();
							//...and display prompt to take or leave chest contents
							new TreasureChestPrompt(DQFrame.this, ((DungeonMap) map).getChest(map.getHeroColumn(), map.getHeroRow() - 1)).show();

						}
					}
					else if(currentMap == DUNGEON && heroFacing == RIGHT)
					{	//if hero is standing in front of and facing treasure chest...
						if(((MazeTile) map.getTile(map.getHeroColumn() + 1, map.getHeroRow())).isTreasureChest())
						{	//..play opening chest sound effect...
							try
							{
								soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																		  	  "audio_files/treasure_chest.wav"));
							}
							catch (IOException exception)
							{
								soundEffectClip = null;
							}
							soundEffectClip.play();
							//...and display prompt to take or leave chest contents
							new TreasureChestPrompt(DQFrame.this, ((DungeonMap) map).getChest(map.getHeroColumn() + 1, map.getHeroRow())).show();
						}
					}
					else if(currentMap == DUNGEON && heroFacing == LEFT)
					{	//if hero is standing in front of and facing treasure chest...
						if(((MazeTile) map.getTile(map.getHeroColumn() - 1, map.getHeroRow())).isTreasureChest())
						{	//..play opening chest sound effect...
							try
							{
								soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																		  	  "audio_files/treasure_chest.wav"));
							}
							catch (IOException exception)
							{
								soundEffectClip = null;
							}
							soundEffectClip.play();
							//...and display prompt to take or leave chest contents
							new TreasureChestPrompt(DQFrame.this, ((DungeonMap) map).getChest(map.getHeroColumn() - 1, map.getHeroRow())).show();
						}
					}
					//if hero standing in front of and facing church alter...
					else if(currentMap == TOWN && map.getHeroColumn() == 27 && map.getHeroRow() == 8 && heroFacing == RIGHT)
					{	//...open up prompt to save game progress
						new ChurchMenu(DQFrame.this).show();
					}
					//if hero standing in front of and facing inn front counter...
					else if(currentMap == TOWN && map.getHeroColumn() == 13 && map.getHeroRow() == 25 && heroFacing == RIGHT)
					{	//...open up prompt to stay at inn
						new InnMenu(DQFrame.this).show();
					}
					//if hero standing in front of and facing weapon and armor shop counter...
					else if(currentMap == TOWN && map.getHeroColumn() == 9 && map.getHeroRow() == 10 && heroFacing == BACK)
					{	//...open up shop menu
						new WeaponAndArmorShopMenu(DQFrame.this).show();
					}
					//if hero standing in front of and facing item shop counter
					else if(currentMap == TOWN && map.getHeroColumn() == 28 && map.getHeroRow() == 29 && heroFacing == RIGHT)
					{	//...open up shop menu
						new ItemShopMenu(DQFrame.this).show();
					}
				}
				else if(key == KeyEvent.VK_SPACE) //if space bar pressed...
				{	//if not on title screen, open main menu
					if(currentMap != TITLE) new MainMenu(DQFrame.this).show();
				}
				//if hero now past town gates...
				if(currentMap == TOWN && map.getHeroRow() == 3)
				{	//...play exiting sound effect and move hero to world map
					try
					{
						soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																  	  "audio_files/stairs.wav"));
					}
					catch (IOException exception)
					{
						soundEffectClip = null;
					}
					soundEffectClip.play();
					setUpWorldMap(5, 14);
				}
				//if hero now standing on town entrance...
				if(currentMap == WORLD && map.getHeroColumn() == 5 && map.getHeroRow() == 15)
				{	//play entering sound effect
					try
					{
						soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																  	  "audio_files/stairs.wav"));
					}
					catch (IOException exception)
					{
						soundEffectClip = null;
					}
					soundEffectClip.play();
					setUpTownMap(19, 4);
				}
				//if hero now standing on dungeon entrance...
				if(currentMap == WORLD && map.getHeroColumn() == 15 && map.getHeroRow() == 4)
				{	//...play dungeon theme...
					musicClip.stop();
					try
					{
						musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																"audio_files/dungeon.mid"));
					}
					catch (IOException exception)
					{
						musicClip = null;
					}
					musicClip.loop();
					//...reset encounter count down and set up first dungeon floor
					encounterCounter = (int) (10 + Math.random() * 15);
					setUpDungeonMap(1);
				}
				//if encounter count down now zero...
				if(encounterCounter == 0)
				{	//...open up battle window and reset encounter count down
					new BattleWindow(DQFrame.this).show();
					encounterCounter = (int) (10 + Math.random() * 15);
				}
				//redraw map panel
				repaint();
			}
		}

		/*
		CLASS: TreasureChestPrompt
		PURPOSE: If hero has room in inventory, displays contents of treasure chest and
		prompts player to take it. Otherwise, notifies player that inventory is full and
		asks whether or not they want to make room.
		*/
		private class TreasureChestPrompt extends JDialog
		{
			private JTextArea promptArea; //displays the appropriate message
			private JButton takeButton; //clicking this button adds chest contents to hero's inventory
			private JButton yesButton; //clicking this button closes dialog window
			private JButton noButton; //clicking this button destroys chest and closes dialog window
			private TreasureChest chest; //treasure chest object associated with this dialog

			public TreasureChestPrompt(JFrame owner, TreasureChest chestIn)
			{	//set dialog window parameters
				super(owner, "", true);
				setSize(326, 157);
				setLocation(owner.getX(), owner.getY());
				setResizable(false);
				setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
				setLayout(new FlowLayout());
				chest = chestIn;
				//set promptArea parameters and add it to dialog window
				promptArea = new JTextArea(hero.getName() + " opens the chest. It contains:     " + chest.getContents().getName() + "\n");
				promptArea.setLineWrap(true);
				promptArea.setWrapStyleWord(true);
				promptArea.setEditable(false);
				promptArea.setFocusable(false);
				promptArea.setSize(320, 100);
				add(promptArea);
				//create buttons and add action listeners
				takeButton = new JButton("Take it");
				yesButton = new JButton("Yes");
				noButton = new JButton("No");
				takeButton.addActionListener(new TakeAction());
				yesButton.addActionListener(new YesAction());
				noButton.addActionListener(new NoAction());
				//if hero has room in inventory for chest contents...
				if(hero.firstEmptySlotInInventory() != -1 ||
				   hero.hasInInventory(chest.getContents().getName()) != -1)
				{
					promptArea.append("\n\n\n");
					add(takeButton);
				}
				else //if hero's inventory is full...
				{	//...notify player, display warning, and add buttons
					promptArea.append(hero.getName() + "'s inventory is full. Would you like to make room for this item?\n" +
									  "Warning: Choosing \"No\" will destroy the chest along with its contents.");
					add(yesButton);
					add(noButton);
				}
			}

			/*
			CLASS: YesAction
			PURPOSE: Closes dialog window.
			*/
			private class YesAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					dispose();
				}
			}

			/*
			CLASS: TakeAction
			PURPOSE: Takes contents from chest and adds it to hero's inventory
			*/
			private class TakeAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					String itemName = chest.getContents().getName();
					//if hero has chest contents in inventory...
					if(hero.hasInInventory(itemName) != -1)
					{	//...increase item's quantity in inventory
						int itemQuantity = hero.getItem(hero.hasInInventory(itemName)).getQuantity();
						hero.getItem(hero.hasInInventory(itemName)).setQuantity(itemQuantity + 1);
					}
					else //if hero does not have chest contents in inventory...
					{	//...add chest contents to first empty slot in inventory
						hero.setItem(chest.getContents(), hero.firstEmptySlotInInventory());
					}
					//remove chest from map and close dialog window
					((MazeTile) map.getTile(chest.getColumn(), chest.getRow())).setTreasureChest(false);
					((MazeTile) map.getTile(chest.getColumn(), chest.getRow())).setPassable(true);
					dispose();
				}
			}

			/*
			CLASS: NoAction
			PURPOSE: Removes chest from map and closes dialog window.
			*/
			private class NoAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					((MazeTile) map.getTile(chest.getColumn(), chest.getRow())).setTreasureChest(false);
					((MazeTile) map.getTile(chest.getColumn(), chest.getRow())).setPassable(true);
					dispose();
				}
			}
		}

		/*
		CLASS: TitleMenu
		PURPOSE: Gives player two options: start a new game, or load a saved game.
		*/
		private class TitleMenu extends JDialog
		{
			public TitleMenu(JFrame owner)
			{
				super(owner, "", true);
				//set dialog window parameters
				setSize(150, 100);
				setLocation(owner.getX() + 88, owner.getY() + 160);
				setResizable(false);
				setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
				setLayout(new FlowLayout());
				//create buttons
				JButton newGameButton = new JButton("Start New Game");
				JButton loadGameButton = new JButton("Load Saved Game");
				//set button parameters
				newGameButton.setMargin(new Insets(2, 15, 2, 15));
				loadGameButton.setMargin(new Insets(2, 10, 2, 10));
				//add action listeners
				newGameButton.addActionListener(new NewGameAction());
				loadGameButton.addActionListener(new LoadGameAction());
				//add buttons to window
				add(newGameButton);
				add(loadGameButton);
			}

			/*
			CLASS: NewGameAction
			PURPOSE: Opens name entry window.
			*/
			private class NewGameAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					new NameEntryPrompt(TitleMenu.this).show();
				}
			}

			/*
			CLASS: LoadGameAction
			PURPOSE: Loads a Dungeon Quest Save (.dq) file and sets up town map with
			hero standing in front of and facing church alter.
			*/
			private class LoadGameAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{	//Open file chooser and load selected file
					String fileName;
					JFileChooser chooser = new JFileChooser();
					chooser.setFileFilter(new DQFileFilter());
					chooser.setAcceptAllFileFilterUsed(false);
					chooser.setCurrentDirectory(new File("."));
					chooser.showOpenDialog(DQFrame.this);
					fileName = chooser.getSelectedFile().getPath();
					try
					{
						ObjectInputStream objIn = new ObjectInputStream(new FileInputStream(fileName));
						hero = ((Hero) objIn.readObject());
					}
					catch(IOException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					catch(ClassNotFoundException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					//set up town map with hero standing in front of and facing churh alter
					heroFacing = RIGHT;
					setUpTownMap(27, 8);
					//close window
					dispose();
				}
			}
		}

		/*
		CLASS: NameEntryPrompt
		PURPOSE: Prompts player to enter their name.
		*/
		private class NameEntryPrompt extends JDialog
		{
			private JDialog owner; //the parent window over which this window will be displayed
			private JFormattedTextField nameField; //player enters name here

			public NameEntryPrompt(JDialog ownerIn)
			{
				super(ownerIn, "", true);
				owner = ownerIn;
				//set dialog window parameters
				setSize(150, 100);
				setLocation(owner.getX(), owner.getY());
				setResizable(false);
				setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
				setLayout(new FlowLayout());
				//create components
				JLabel promptLabel = new JLabel("Enter your name:");
				try
				{	//format textfield to allow only 10 characters
					nameField = new JFormattedTextField(new MaskFormatter("**********"));
					nameField.setColumns(8);
				}
				catch(Exception exception)
				{
					System.out.println("Caught exception: " + exception);
				}
				JButton doneButton = new JButton("DONE");
				//set button parameters
				doneButton.setMargin(new Insets(0, 0, 0, 0));
				doneButton.addActionListener(new DoneAction());
				//add components to window
				add(promptLabel);
				add(nameField);
				add(doneButton);
			}

			/*
			CLASS: DoneAction
			PURPOSE: Creates a new hero named for the player and starts a new game.
			*/
			private class DoneAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					hero.setName(nameField.getText().trim());
					setUpWorldMap(5, 14);
					owner.dispose();
					dispose();
				}
			}
		}
	}

	/*
	CLASS: MainMenu
	PURPOSE: This is the main menu window. Contains 3 tabbed panels: Status, Inventory, and Spells.
	*/
	private class MainMenu extends JDialog
	{
		private JTextField hpField; //displays Hero's current HP on status panel
		private JTextField mpField; //displays Hero's current MP on status panel
		private JFormattedTextField[] statField; //displays Hero's other 9 stats on status panel
		private JFormattedTextField spField; //displays Hero's current SP on status panel
		private JTextField spellField[]; //displays Hero's spell list on spells panel
		private JTextField mpRemainingField; //displays Hero's remaining MP on spells panel
		private JButton[] incStatButton; //buttons that allocate SP on status panel
		private JButton[] decStatButton; //buttons that deallocate SP on status panel
		private JButton applyButton; //button that finalizes changes to stats on status panel
		private int[] startStat; //stores initial stat values before inc/dec buttons are clicked on status panel
		private int startSP; //stores initial SP value before inc/dec buttons are clicked on status panel
		private int remainingSP; //stores SP remaining after inc/dec button is clicked on status panel

		public MainMenu(JFrame owner)
		{
			super(owner, "Main Menu", true);
			//set main menu window parameters
			setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
			setLocation(owner.getX(), owner.getY());
			setResizable(false);
			setFocusable(true);
			setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			//create tabbed panels within main menu
			JTabbedPane tabbedMenuPane = new JTabbedPane();
			StatusPanel statusPanel = new StatusPanel();
			tabbedMenuPane.add("Status", statusPanel);
			InventoryPanel inventoryPanel = new InventoryPanel();
			tabbedMenuPane.add("Inventory", inventoryPanel);
			SpellsPanel spellsPanel = new SpellsPanel();
			tabbedMenuPane.add("Spells", spellsPanel);
			add(tabbedMenuPane);
			addKeyListener(new KeyHandler());
		}

		/*
		CLASS: KeyHandler
		PURPOSE: Handles keyboard input while the main menu is open.
		*/
		private class KeyHandler extends KeyAdapter
		{
			public void keyPressed(KeyEvent event)
			{
				int key = event.getKeyCode();
				//if escape key is pressed...
				if(key == KeyEvent.VK_ESCAPE)
				{	//...close main menu
					dispose();
				}
			}
		}

		/*
		CLASS: StatusPanel
		PURPOSE: This is the status panel within the main menu. This is where the player
		can view the hero's stats and allocate SP to them.
		*/
		private class StatusPanel extends JPanel
		{
			public StatusPanel()
			{
				setLayout(null); //no layout manager used for this panel

				//initialize startStat, startSP, remainingSP
				startStat = new int[9];
				for(int i = 0; i < 9; i++)
				{
					startStat[i] = hero.getStat(i);
				}
				startSP = hero.getSP();
				remainingSP = startSP;
				//create labels for header and footer and stat text fields
				JLabel nameLabel = new JLabel(hero.getName());
				JLabel lvlLabel = new JLabel("LVL " + hero.getLVL());
				JLabel hpLabel = new JLabel("HP");
				JLabel mpLabel = new JLabel("MP");
				JLabel atkLabel = new JLabel("ATK");
				JLabel accLabel = new JLabel("ACC");
				JLabel defLabel = new JLabel("DEF");
				JLabel evaLabel = new JLabel("EVA");
				JLabel wisLabel = new JLabel("WIS");
				JLabel mdefLabel = new JLabel("MDEF");
				JLabel spdLabel = new JLabel("SPD");
				JLabel spLabel = new JLabel("SP");
				JLabel expLabel = new JLabel("EXP");
				JLabel expValueLabel = new JLabel("" + hero.getEXP());
				//...set parameters
				nameLabel.setSize(100, 10);
				nameLabel.setLocation(20, 15);
				lvlLabel.setSize(100, 10);
				lvlLabel.setLocation(100, 15);
				hpLabel.setSize(100, 10);
				hpLabel.setLocation(20, 35);
				mpLabel.setSize(100, 10);
				mpLabel.setLocation(20, 55);
				atkLabel.setSize(100, 10);
				atkLabel.setLocation(20, 75);
				accLabel.setSize(100, 10);
				accLabel.setLocation(20, 95);
				defLabel.setSize(100, 10);
				defLabel.setLocation(20, 115);
				evaLabel.setSize(100, 10);
				evaLabel.setLocation(20, 135);
				wisLabel.setSize(100, 10);
				wisLabel.setLocation(20, 155);
				mdefLabel.setSize(100, 10);
				mdefLabel.setLocation(20, 175);
				spdLabel.setSize(100, 10);
				spdLabel.setLocation(20, 195);
				spLabel.setSize(100, 10);
				spLabel.setLocation(20, 215);
				expLabel.setSize(100, 10);
				expLabel.setLocation(20, 235);
				expValueLabel.setSize(100, 10);
				expValueLabel.setLocation(100, 235);
				//...and place on panel
				add(nameLabel);
				add(lvlLabel);
				add(hpLabel);
				add(mpLabel);
				add(atkLabel);
				add(accLabel);
				add(defLabel);
				add(evaLabel);
				add(wisLabel);
				add(mdefLabel);
				add(spdLabel);
				add(spLabel);
				add(expLabel);;
				add(expValueLabel);
				//create dec buttons, set parameters, add action, and place on panel
				decStatButton = new JButton[9];
				for(int i = 0; i < 9; i++)
				{
					decStatButton[i] = new JButton("-");
					decStatButton[i].setMargin(new Insets(0, 0, 0, 0));
					decStatButton[i].setSize(16, 16);
					decStatButton[i].setLocation(84, 32 + 20*i);
					decStatButton[i].addActionListener(new DecStatAction(i));
					add(decStatButton[i]);
				}
				//create stat text fields, set parameters, and place on panel
				hpField = new JTextField(hero.getHP() + "/");
				mpField = new JTextField(hero.getMP() + "/");
				spField = new JFormattedTextField(NumberFormat.getIntegerInstance());
				hpField.setEditable(false);
				hpField.setFocusable(false);
				hpField.setSize(30, 16);
				hpField.setLocation(100, 32);
				mpField.setEditable(false);
				mpField.setFocusable(false);
				mpField.setSize(30, 16);
				mpField.setLocation(100, 52);
				spField.setValue(new Integer(remainingSP));
				spField.setEditable(false);
				spField.setFocusable(false);
				spField.setSize(25, 16);
				spField.setLocation(100, 212);
				add(hpField);
				add(mpField);
				add(spField);
				statField = new JFormattedTextField[9];
				for(int i = 0; i < 9; i++)
				{
					statField[i] = new JFormattedTextField(NumberFormat.getIntegerInstance());
					statField[i].setValue(new Integer(hero.getStat(i)));
					statField[i].setEditable(false);
					statField[i].setFocusable(false);
					statField[i].setSize(25, 16);
					if(i == 0 || i == 1) statField[i].setLocation(130, 32 + 20*i);
					else statField[i].setLocation(100, 32 + 20*i);
					add(statField[i]);
				}
				//create inc buttons, set parameters, add action, and place on panel
				incStatButton = new JButton[9];
				for(int i = 0; i < 9; i++)
				{
					incStatButton[i] = new JButton("+");
					incStatButton[i].setMargin(new Insets(0, 0, 0, 0));
					incStatButton[i].setSize(16, 16);
					if(i == 0 || i == 1) incStatButton[i].setLocation(155, 32 + 20*i);
					else incStatButton[i].setLocation(125, 32 + 20*i);
					incStatButton[i].addActionListener(new IncStatAction(i));
					add(incStatButton[i]);
				}
				//create apply button, set parameters, add action, and place on panel
				applyButton = new JButton("APPLY");
				applyButton.setMargin(new Insets(0, 0, 0, 0));
				applyButton.setSize(56, 22);
				applyButton.setLocation(236, 218);
				applyButton.addActionListener(new ApplyAction());
				add(applyButton);

				if(remainingSP == 0) //if no SP available...
				{	//...disable all increment buttons
					for(int i = 0; i < 9; i++)
					{
						incStatButton[i].setEnabled(false);
					}
				}

				if(remainingSP == startSP) //if no SP allocated...
				{	//...disable all decrement buttons and the apply button
					for(int i = 0; i < 9; i++)
					{
						decStatButton[i].setEnabled(false);
					}
					applyButton.setEnabled(false);
				}
			}

			/*
			CLASS: DecStatAction
			PURPOSE: Deallocates SP when a decrement button is clicked.
			*/
			private class DecStatAction implements ActionListener
			{
				private int stat; //stores which stat is being acted upon

				public DecStatAction(int statIn)
				{
					super();
					stat = statIn;
				}

				public void actionPerformed(ActionEvent event)
				{
					if(remainingSP < startSP) //if some SP has been allocated...
					{	//...get current value in textfield of stat being acted upon
						Number value = (Number) statField[stat].getValue();
						int v = value.intValue();

						if(v > 0) //if that value > 0 (negative values not allowed)...
						{	//...dec by 10 if max HP, by 5 if max MP...
							if(stat == GameCharacter.MAX_HP) v -= 10;
							else if(stat == GameCharacter.MAX_MP) v -= 5;
							else v--; //...otherwise, dec by 1
							statField[stat].setValue(new Integer(v));
							remainingSP++;
							spField.setValue(new Integer(remainingSP));
							//enable all increment buttons
							for(int i = 0; i < 9; i++)
							{
								incStatButton[i].setEnabled(true);
							}

							if(remainingSP == startSP) //if SP is now at initial value...
							{	//...disable all decrement buttons and the apply button
								for(int i = 0; i < 9; i++)
								{
									decStatButton[i].setEnabled(false);
								}
								applyButton.setEnabled(false);
							}

							if(v == startStat[stat]) //if stat is now at initial value...
							{   //...disable the decrement button for this stat
								decStatButton[stat].setEnabled(false);
							}
						}
					}
				}
			}

			/*
			CLASS: IncStatAction
			PURPOSE: Allocates SP when an increment button is clicked.
			*/
			private class IncStatAction implements ActionListener
			{
				private int stat; //stores which stat is being acted upon

				public IncStatAction(int statIn)
				{
					super();
					stat = statIn;
				}

				public void actionPerformed(ActionEvent event)
				{
					if(remainingSP > 0) //if some SP available...
					{	//...get current value in textfield of stat being acted upon
						Number value = (Number) statField[stat].getValue();
						int v = value.intValue();
						//inc by 10 if max HP, by 5 if max MP...
						if(stat == GameCharacter.MAX_HP) v += 10;
						else if(stat == GameCharacter.MAX_MP) v += 5;
						else v++; //...otherwise inc by 1
						statField[stat].setValue(new Integer(v));
						remainingSP--;
						spField.setValue(new Integer(remainingSP));
						decStatButton[stat].setEnabled(true);
						if(remainingSP == 0) //if SP is now at 0...
						{	//...disable all increment buttons
							for(int i = 0; i < 9; i++)
							{
								incStatButton[i].setEnabled(false);
							}
						}
						applyButton.setEnabled(true);
					}
				}
			}

			/*
			CLASS: ApplyAction
			PURPOSE: Finalizes changes to stats when apply button is clicked.
			*/
			private class ApplyAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					Number value; //stores value of a textfield

					//set the value of each stat to the value in that stat's textfield and
					//disable all decrement buttons
					for(int i = 0; i < 9; i++)
					{
						value = (Number) statField[i].getValue();
						hero.setStat(new Integer(value.intValue()), i);
						startStat[i] = hero.getStat(i);
						decStatButton[i].setEnabled(false);
					}
					hero.setSP(remainingSP); //remove appropriate amount of SP
					startSP = hero.getSP(); //reinitialize startSP
					applyButton.setEnabled(false);
				}
			}
		}

		/*
		CLASS: InventoryPanel
		PURPOSE: This is the inventory panel within the main menu. This is where the player
		can use/remove items and equip/remove weapons and armor.
		*/
		private class InventoryPanel extends JPanel
		{
			private JTextArea descriptionArea; //displays a short description of the selected item
			private JTextField itemField[]; //displays names of items in inventory
			private JTextField itemQuantityField[]; //displays the quantity of each item
			private JTextField equipmentField[]; //displays the item equipped in each slot
			private JButton useButton; //uses/equips the selected item when clicked
			private JButton removeButton; //removes the selected item when clicked

			public InventoryPanel()
			{
				setLayout (null); //no layout manager used for this panel
				//create item textfields, set parameters, add action, and place on panel
				itemField = new JTextField[10];
				itemQuantityField = new JTextField[10];
				for(int i = 0; i < 10; i++)
				{
					itemField[i] = new JTextField(hero.getItem(i).getName());
					itemField[i].setEditable(false);
					itemField[i].setLocation(20, 20 + i*16);
					itemField[i].setSize(100, 16);
					itemField[i].addMouseListener(new ItemFieldClicked(itemField[i]));
					add(itemField[i]);
					//create quantity textfields, set parameters, and place on panel
					if(!hero.getItem(i).getName().equals("")) //if item slot not empty...
					{	//...display quantity
						itemQuantityField[i] = new JTextField("x" + hero.getItem(i).getQuantity());
					}
					else //if item slot is empty...
					{	//...display nothing
						itemQuantityField[i] = new JTextField();
					}
					itemQuantityField[i].setEditable(false);
					itemQuantityField[i].setFocusable(false);
					itemQuantityField[i].setLocation(120, 20 + i*16);
					itemQuantityField[i].setSize(25, 16);
					add(itemQuantityField[i]);
				}
				//create labels for description, equipment, and gold textfields...
				JLabel descriptionLabel = new JLabel("DESCRIPTION");
				JLabel weaponLabel = new JLabel("WEAPON");
				JLabel shieldLabel = new JLabel("SHIELD");
				JLabel headLabel = new JLabel("HEAD");
				JLabel bodyLabel = new JLabel("BODY");
				JLabel accessoryLabel = new JLabel("ACCESSORY");
				JLabel goldLabel = new JLabel("GOLD");
				//...set parameters...
				descriptionLabel.setSize(100, 10);
				descriptionLabel.setLocation(20, 185);
				weaponLabel.setSize(100, 10);
				weaponLabel.setLocation(165, 20);
				shieldLabel.setSize(100, 10);
				shieldLabel.setLocation(165, 52);
				headLabel.setSize(100, 10);
				headLabel.setLocation(165, 84);
				bodyLabel.setSize(100, 10);
				bodyLabel.setLocation(165, 116);
				accessoryLabel.setSize(100, 10);
				accessoryLabel.setLocation(165, 148);
				goldLabel.setSize(100, 10);
				goldLabel.setLocation(165, 180);
				//...and place on panel
				add(descriptionLabel);
				add(weaponLabel);
				add(shieldLabel);
				add(headLabel);
				add(bodyLabel);
				add(accessoryLabel);
				add(goldLabel);
				//create description textarea, set parameters, and place on panel
				descriptionArea = new JTextArea();
				descriptionArea.setLineWrap(true);
				descriptionArea.setWrapStyleWord(true);
				descriptionArea.setEditable(false);
				descriptionArea.setFocusable(false);
				descriptionArea.setSize(125, 40);
				descriptionArea.setLocation(20, 200);
				descriptionArea.setFont(new Font("Arial", Font.PLAIN, 10));
				add(descriptionArea);
				//create equipment textfields, set parameters, add action, and place on panel
				equipmentField = new JTextField[5];
				for(int i = 0; i < 5; i++)
				{
					equipmentField[i] = new JTextField(hero.getEquipment(i).getName());
					equipmentField[i].setEditable(false);
					equipmentField[i].setLocation(165, 33 + 32*i);
					equipmentField[i].setSize(125, 16);
					equipmentField[i].addMouseListener(new EquipmentFieldClicked(equipmentField[i]));
					add(equipmentField[i]);
				}
				//create gold textfield, set parameters, and place on panel
				JTextField goldField = new JTextField("" + hero.getGold());
				goldField.setEditable(false);
				goldField.setFocusable(false);
				goldField.setLocation(165, 193);
				goldField.setSize(125, 16);
				add(goldField);
				//create remove and use buttons, set parameters, add actions, and place on panel
				removeButton = new JButton("DROP");
				useButton = new JButton("USE");
				removeButton.setMargin(new Insets(0, 0, 0, 0));
				removeButton.setSize(56, 22);
				removeButton.setLocation(165, 218);
				removeButton.setEnabled(false);
				useButton.setMargin(new Insets(0, 0, 0, 0));
				useButton.setSize(56, 22);
				useButton.setLocation(236, 218);
				useButton.setEnabled(false);
				removeButton.addActionListener(new RemoveAction());
				useButton.addActionListener(new UseAction());
				add(removeButton);
				add(useButton);
			}

			/*
			CLASS: ItemFieldClicked
			PURPOSE: Highlights name of item in inventory when its textfield is clicked.
			*/
			private class ItemFieldClicked extends MouseAdapter
			{
				private JTextField textField; //stores which textfield has been clicked

				public ItemFieldClicked(JTextField fieldClicked)
				{
					textField = fieldClicked;
				}

				public void mouseReleased(MouseEvent event)
				{	//remove highlighting from all textfields
					for(int i = 0; i < 10; i++)
					{
						itemField[i].setCaretPosition(0);
					}
					for(int i = 0; i < 5; i++)
					{
						equipmentField[i].setCaretPosition(0);
					}

					//highlight clicked textfield and display item's description
					textField.selectAll();
					descriptionArea.setText(itemTable.getItem(textField.getText()).getDescription());

					if(textField.getText().equals("")) //if item slot is empty...
					{
						useButton.setText("USE");
						removeButton.setText("DROP");
						useButton.setEnabled(false);
						removeButton.setEnabled(false);
					}
					else if(hero.hasEquipped(textField.getSelectedText()))
					{
						useButton.setText("EQUIP");
						removeButton.setText("DROP");
						useButton.setEnabled(false);
						removeButton.setEnabled(true);
					}
					//...otherwise, if highlighted item is a magic scroll...
					else if(itemTable.getItem(textField.getText()).getType() == Item.MAGIC_SCROLL)
					{	//...get name of spell learned from using scroll
						String thisScrollsSpell = ((MagicScroll) itemTable.getItem(textField.getText())).getSpell().getName();
						if(hero.hasInSpellList(thisScrollsSpell))//if Hero already knows this spell...
						{	//...don't let player use scroll
							useButton.setText("USE");
							useButton.setEnabled(false);
						}
						else
						{	//...update buttons
							useButton.setText("USE");
							useButton.setEnabled(true);
						}
						//update buttons
						removeButton.setText("DROP");
						removeButton.setEnabled(true);
					}
					//...otherwise, if highlighted item is a healing item or warp scroll...
					else if(itemTable.getItem(textField.getText()).getType() == Item.HEALING ||
							itemTable.getItem(textField.getText()).getType() == Item.WARP_SCROLL)
					{	//...update buttons
						useButton.setText("USE");
						removeButton.setText("DROP");
						useButton.setEnabled(true);
						removeButton.setEnabled(true);
					}
					else //if highlighted item is a piece of equipment...
					{	//...update buttons
						useButton.setText("EQUIP");
						removeButton.setText("DROP");
						useButton.setEnabled(true);
						removeButton.setEnabled(true);
					}
				}
			}

			/*
			CLASS: EquipmentFieldClicked
			PURPOSE: Highlights name of item equipped when a slot's textfield is clicked.
			*/
			private class EquipmentFieldClicked extends MouseAdapter
			{
				private JTextField textField; //stores which textfield has been clicked

				public EquipmentFieldClicked(JTextField fieldClicked)
				{
					textField = fieldClicked;
				}

				public void mouseReleased(MouseEvent event)
				{	//remove highlighting from all textfields
					for(int i = 0; i < 10; i++)
					{
						itemField[i].setCaretPosition(0);
					}
					for(int i = 0; i < 5; i++)
					{
						equipmentField[i].setCaretPosition(0);
					}
					//highlight clicked textfield, display item's description, and disable use button
					textField.selectAll();
					descriptionArea.setText(itemTable.getItem(textField.getText()).getDescription());
					useButton.setEnabled(false);
					if(textField.getText().equals("")) //if equipment slot is empty...
					{
						useButton.setText("EQUIP");
						removeButton.setText("REMOVE");
						removeButton.setEnabled(false);
					}
					else
					{
						useButton.setText("EQUIP");
						removeButton.setText("REMOVE");
						removeButton.setEnabled(true);
					}
				}
			}

			/*
			CLASS: UseAction
			PURPOSE: Uses/equips highlighted item when use button is clicked
			*/
			private class UseAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					String selectedItem; //stores name of highlighted item

					for(int i = 0; i < 10; i++)
					{	//get highlighted item name
						selectedItem = itemField[i].getSelectedText();

						try
						{	//if highlighted text found...
							if(!selectedItem.equals(""))
							{	//if highlighted item is healing item...
								if(itemTable.getItem(selectedItem).getType() == Item.HEALING)
								{	//...use item
									descriptionArea.setText(((HealingItem) hero.getItem(i)).use(hero));
									//if item's quantity still greater than 1...
									if(hero.getItem(i).getQuantity() > 1)
									{	//decrement quantity and update quantity field
										hero.getItem(i).setQuantity(hero.getItem(i).getQuantity() - 1);
										itemQuantityField[i].setText("x" + hero.getItem(i).getQuantity());
									}
									else //if item's quantity now 0...
									{	//remove item from inventory completely...
										hero.setItem((Equipment) itemTable.getItem(""), i);
										//...update appropriate textfields...
										itemField[i].setText("");
										itemQuantityField[i].setText("");
										//...and disable use and remove buttons
										removeButton.setEnabled(false);
										useButton.setEnabled(false);
									}
								}
								//else if highlighted item is a magic scroll...
								else if(itemTable.getItem(selectedItem).getType() == Item.MAGIC_SCROLL)
								{	//...use magic scroll
									descriptionArea.setText(((MagicScroll) hero.getItem(i)).use(hero));
									//if item's quantity still greater than 1...
									if(hero.getItem(i).getQuantity() > 1)
									{	//decrement quantity and update quantity field
										hero.getItem(i).setQuantity(hero.getItem(i).getQuantity() - 1);
										itemQuantityField[i].setText("x" + hero.getItem(i).getQuantity());
									}
									else //if item's quantity now 0...
									{	//remove item from inventory completely...
										hero.setItem((Equipment) itemTable.getItem(""), i);
										//...update appropriate textfields...
										itemField[i].setText("");
										itemQuantityField[i].setText("");
										//...and disable remove buttons
										removeButton.setEnabled(false);
									}
									useButton.setEnabled(false); //disable use button
									//...update spell list on spell panel
									for(int j = 0; j < 10; j++)
									{
										spellField[j].setText(hero.getSpell(j).getName());
									}
								}
								//else if highlighted item is a warp scroll...
								else if(itemTable.getItem(selectedItem).getType() == Item.WARP_SCROLL)
								{
									((WarpScroll) hero.getItem(i)).use(hero, DQFrame.this);
									//if item's quantity still greater than 1...
									if(hero.getItem(i).getQuantity() > 1)
									{	//decrement quantity
										hero.getItem(i).setQuantity(hero.getItem(i).getQuantity() - 1);
									}
									else //if item's quantity now 0...
									{	//remove item from inventory completely...
										hero.setItem((Equipment) itemTable.getItem(""), i);
									}
									mapPanel.repaint();
									dispose();
								}
								else //else if highlighted item is a piece of equipment
								{	//...equip item...
									((Equipment) hero.getItem(i)).equip(hero);
									//...update equipment textfield...
									equipmentField[hero.getItem(i).getType()].setText(selectedItem);
									//...move highlighting to equipment slot...
									equipmentField[hero.getItem(i).getType()].selectAll();
									itemField[i].setCaretPosition(0);
									//...and enable remove button and disable use button
									removeButton.setText("REMOVE");
									removeButton.setEnabled(true);
									useButton.setEnabled(false);
								}
								//update stat textfields on other panels
								hpField.setText(hero.getHP() + "/");
								mpField.setText(hero.getMP() + "/");
								for(int j = 0; j < 9; j++)
								{
									statField[j].setValue(new Integer(hero.getStat(j)));
								}
								mpRemainingField.setText("" + hero.getMP());
								for(int j = 0; j < 9; j++)
								{
									startStat[j] = hero.getStat(j);
								}
								startSP = hero.getSP();
								remainingSP = startSP;
								spField.setValue(new Integer(remainingSP));
								if(remainingSP == 0) //if no SP available...
								{	//...disable all increment buttons
									for(int j = 0; j < 9; ++j)
									{
										incStatButton[j].setEnabled(false);
									}
								}
								else
								{
									for(int j = 0; j < 9; j++)
									{
										incStatButton[j].setEnabled(true);
									}
								}

								if(remainingSP == startSP) //if no SP allocated...
								{	//...disable all decrement buttons and the apply button
									for(int j = 0; j < 9; j++)
									{
										decStatButton[j].setEnabled(false);
									}
									applyButton.setEnabled(false);
								}
								//break from loop (only 1 item highlighted at a time)
								break;
							}
						}
						catch(NullPointerException exception)
						{
							System.out.println("Caught exception: " + exception);
						}
					}
				}
			}

			/*
			CLASS: RemovesAction
			PURPOSE: Removes/unequips highlighted item when remove button is clicked
			*/
			private class RemoveAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					String selectedItem; //stores name of highlighted item

					for(int i = 0; i < 5; i++)
					{	//get highlighted item name
						selectedItem = equipmentField[i].getSelectedText();

						try
						{	//if highlighted text found in equipment textfield...
							if(!selectedItem.equals(""))
							{	//unequip item...
								hero.removeEquipment(i);
								//...update appropriate textfields...
								equipmentField[i].setText("");
								hpField.setText(hero.getHP() + "/");
								mpField.setText(hero.getMP() + "/");
								for(int j = 0; j < 9; j++)
								{
									statField[j].setValue(new Integer(hero.getStat(j)));
								}
								mpRemainingField.setText("" + hero.getMP());
								for(int j = 0; j < 9; j++)
								{
									startStat[j] = hero.getStat(j);
								}
								startSP = hero.getSP();
								remainingSP = startSP;
								spField.setValue(new Integer(remainingSP));
								if(remainingSP == 0) //if no SP available...
								{	//...disable all increment buttons
									for(int j = 0; j < 9; ++j)
									{
										incStatButton[j].setEnabled(false);
									}
								}
								else
								{
									for(int j = 0; j < 9; j++)
									{
										incStatButton[j].setEnabled(true);
									}
								}

								if(remainingSP == startSP) //if no SP allocated...
								{	//...disable all decrement buttons and the apply button
									for(int j = 0; j < 9; j++)
									{
										decStatButton[j].setEnabled(false);
									}
									applyButton.setEnabled(false);
								}
								descriptionArea.setText("");
								//...and disable use and remove buttons
								useButton.setEnabled(false);
								removeButton.setEnabled(false);
								//break from loop (only 1 item highlighted at a time)
								break;
							}
						}
						catch(NullPointerException exception)
						{
							System.out.println("Caught exception: " + exception);
						}
					}

					for(int i = 0; i < 10; i++)
					{	//get highlighted item name
						selectedItem = itemField[i].getSelectedText();

						try
						{	//if highlighted item found in inventory textfield...
							if(!selectedItem.equals(""))
							{	//if quantity greater than 1...
								if(hero.getItem(i).getQuantity() > 1)
								{	//decrement quantity and update quantity field
									hero.getItem(i).setQuantity(hero.getItem(i).getQuantity() - 1);
									itemQuantityField[i].setText("x" + hero.getItem(i).getQuantity());
								}
								else //else if quantity is 1...
								{	//...if item is equipped, unequip item...
									if(selectedItem.equals(hero.getEquipment(Item.WEAPON).getName()))
									{
										hero.removeEquipment(Item.WEAPON);
										equipmentField[Item.WEAPON].setText("");
									}
									else if(selectedItem.equals(hero.getEquipment(Item.SHIELD).getName()))
									{
										hero.removeEquipment(Item.SHIELD);
										equipmentField[Item.SHIELD].setText("");
									}
									else if(selectedItem.equals(hero.getEquipment(Item.HEAD).getName()))
									{
										hero.removeEquipment(Item.HEAD);
										equipmentField[Item.HEAD].setText("");
									}
									else if(selectedItem.equals(hero.getEquipment(Item.BODY).getName()))
									{
										hero.removeEquipment(Item.BODY);
										equipmentField[Item.BODY].setText("");
									}
									else if(selectedItem.equals(hero.getEquipment(Item.ACCESSORY).getName()))
									{
										hero.removeEquipment(Item.ACCESSORY);
										equipmentField[Item.ACCESSORY].setText("");
									}
									//...remove item from inventory completely...
									hero.setItem((Equipment) itemTable.getItem(""), i);
									//...update appropriate textfields...
									itemField[i].setText("");
									itemQuantityField[i].setText("");
									descriptionArea.setText("");
									//...and disable use and remove buttons
									useButton.setEnabled(false);
									removeButton.setEnabled(false);
								}
								//update textfields on status panel
								hpField.setText(hero.getHP() + "/");
								mpField.setText(hero.getMP() + "/");
								for(int j = 0; j < 9; j++)
								{
									statField[j].setValue(new Integer(hero.getStat(j)));
								}
								mpRemainingField.setText("" + hero.getMP());
								for(int j = 0; j < 9; j++)
								{
									startStat[j] = hero.getStat(j);
								}
								startSP = hero.getSP();
								remainingSP = startSP;
								spField.setValue(new Integer(remainingSP));
								if(remainingSP == 0) //if no SP available...
								{	//...disable all increment buttons
									for(int j = 0; j < 9; ++j)
									{
										incStatButton[j].setEnabled(false);
									}
								}
								else
								{
									for(int j = 0; j < 9; j++)
									{
										incStatButton[j].setEnabled(true);
									}
								}

								if(remainingSP == startSP) //if no SP allocated...
								{	//...disable all decrement buttons and the apply button
									for(int j = 0; j < 9; j++)
									{
										decStatButton[j].setEnabled(false);
									}
									applyButton.setEnabled(false);
								}
								//break from loop (only 1 item highlighted at a time)
								break;
							}
						}
						catch(NullPointerException exception)
						{
							System.out.println("Caught exception: " + exception);
						}
					}
				}
			}
		}

		/*
		CLASS: SpellsPanel
		PURPOSE: This is the spells panel within the main menu. This is where the player
		can cast/remove spells.
		*/
		private class SpellsPanel extends JPanel
		{
			private JTextArea descriptionArea; //displays a short description of the selected spell
			private JButton removeButton; //removes the selected spell when clicked
			private JButton castButton; //casts the selected spell when clicked

			public SpellsPanel()
			{
				setLayout(null); //no layout manager used for this panel

				//create spell textfields, set parameters, add action, and place on panel
				spellField = new JTextField[10];
				for(int i = 0; i < 10; i++)
				{
					spellField[i] = new JTextField(hero.getSpell(i).getName());
					spellField[i].setEditable(false);
					spellField[i].setLocation(20, 20 + 16*i);
					spellField[i].setSize(125, 16);
					spellField[i].addMouseListener(new SpellFieldClicked(spellField[i]));
					add(spellField[i]);
				}
				//create labels for description and mp remaining textfields...
				JLabel descriptionLabel = new JLabel("DESCRIPTION");
				JLabel mpLabel = new JLabel("MP REMAINING");
				//...set parameters...
				descriptionLabel.setSize(100, 10);
				descriptionLabel.setLocation(20, 185);
				mpLabel.setSize(100, 10);
				mpLabel.setLocation(165, 200);
				//...and place on panel
				add(descriptionLabel);
				add(mpLabel);
				//create description textarea, set parameters, and place on panel
				descriptionArea = new JTextArea();
				descriptionArea.setLineWrap(true);
				descriptionArea.setWrapStyleWord(true);
				descriptionArea.setEditable(false);
				descriptionArea.setFocusable(false);
				descriptionArea.setSize(125, 40);
				descriptionArea.setLocation(20, 200);
				descriptionArea.setFont(new Font("Arial", Font.PLAIN, 10));
				add(descriptionArea);
				//create mp remaining textfield, set parameters, and place on panel
				mpRemainingField = new JTextField("" + hero.getMP());
				mpRemainingField.setEditable(false);
				mpRemainingField.setFocusable(false);
				mpRemainingField.setLocation(255, 197);
				mpRemainingField.setSize(35, 16);
				add(mpRemainingField);
				//create remove and cast buttons, set parameters, add actions, and place on panel
				removeButton = new JButton("REMOVE");
				castButton = new JButton("CAST");
				removeButton.setMargin(new Insets(0, 0, 0, 0));
				removeButton.setSize(56, 22);
				removeButton.setLocation(165, 218);
				removeButton.setEnabled(false);
				castButton.setMargin(new Insets(0, 0, 0, 0));
				castButton.setSize(56, 22);
				castButton.setLocation(236, 218);
				castButton.setEnabled(false);
				removeButton.addActionListener(new RemoveAction());
				castButton.addActionListener(new CastAction());
				add(removeButton);
				add(castButton);
			}

			/*
			CLASS: SpellFieldClicked
			PURPOSE: Highlights name of spell in list when its textfield is clicked.
			*/
			private class SpellFieldClicked extends MouseAdapter
			{
				private JTextField textField; //stores which textfield has been clicked

				public SpellFieldClicked(JTextField fieldClicked)
				{
					textField = fieldClicked;
				}

				public void mouseReleased(MouseEvent event)
				{	//remove highlighting from all textfields
					for(int i = 0; i < 10; i++)
					{
						spellField[i].setCaretPosition(0);
					}
					//highlight clicked textfield and display item's description
					textField.selectAll();
					descriptionArea.setText(spellTable.getSpell(textField.getText()).getDescription());

					if(textField.getText().equals("")) //if spell slot is empty...
					{
						castButton.setEnabled(false);
						removeButton.setEnabled(false);
					}
					//else if Hero has enough MP to cast highlighted spell, and spell is a
					//healing spell...
					else if(hero.getMP() >= spellTable.getSpell(textField.getText()).getMPcost() &&
							spellTable.getSpell(textField.getText()).getType() == Spell.HEALING)
					{
						castButton.setEnabled(true);
						removeButton.setEnabled(true);
					}
					else
					{
						castButton.setEnabled(false);
						removeButton.setEnabled(true);
					}
				}
			}

			/*
			CLASS: CastAction
			PURPOSE: Casts highlighted spell when cast button is clicked
			*/
			private class CastAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					String selectedSpell = ""; //stores name of highlighted item

					for(int i = 0; i < 10; i++)
					{	//get highlighted spell name
						selectedSpell = spellField[i].getSelectedText();

						try
						{	//if highlighted spell found...
							if(!selectedSpell.equals(""))
							{	//...cast spell...
								descriptionArea.setText(((HealingSpell) hero.getSpell(i)).cast(hero, hero));
								try
								{
									soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																			      "audio_files/spell.wav"));
								}
								catch (IOException exception)
								{
									soundEffectClip = null;
								}
								soundEffectClip.play();
								//...update HP textfield on status panel
								hpField.setText(hero.getHP() + "/");
								//break from loop (only 1 spell highlighted at a time)
								break;
							}
						}
						catch(NullPointerException exception)
						{
							System.out.println("Caught exception: " + exception);
						}
					}
					//update MP textfields on status and spells panels
					mpField.setText(hero.getMP() + "/");
					mpRemainingField.setText("" + hero.getMP());
					//if Hero's MP now less than MP cost of highlighted spell...
					if(hero.getMP() < spellTable.getSpell(selectedSpell).getMPcost())
					{	//...disable casting
						castButton.setEnabled(false);
					}
				}
			}

			/*
			CLASS: RemovesAction
			PURPOSE: Removes highlighted spell when remove button is clicked
			*/
			private class RemoveAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					String selectedSpell = ""; //stores name of highlighted spell

					for(int i = 0; i < 10; i++)
					{	//get highlighted spell name
						selectedSpell = spellField[i].getSelectedText();

						try
						{	//if highlighted spell found...
							if(!selectedSpell.equals(""))
							{	//...remove spell from list...
								hero.removeSpell(i);
								//...update spell list and clear description textarea...
								spellField[i].setText(hero.getSpell(i).getName());
								descriptionArea.setText("");
								//...and disable remove button
								removeButton.setEnabled(false);
								//break from loop (only 1 spell highlighted at a time)
								break;
							}
						}
						catch(NullPointerException exception)
						{
							System.out.println("Caught exception: " + exception);
						}
					}
				}
			}
		}
	}

	/*
	CLASS: ChurchMenu
	PURPOSE: Prompts player to save their game progress.
	*/
	private class ChurchMenu extends JDialog
	{
		private JTextArea promptArea; //displays dialog text
		private JButton yesButton; //clicking this will save game progress
		private JButton noButton; //clicking this will bypass saving game progress
		private JButton goodbyeButton; //clicking this will close the dialog window

		public ChurchMenu(JFrame owner)
		{
			super(owner, "", true);
			//set dialog window parameters
			setSize(326, 157);
			setLocation(owner.getX(), owner.getY());
			setResizable(false);
			setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
			setLayout(new FlowLayout());
			//create components
			promptArea = new JTextArea("\"Welcome. Have you come for confession?\"\n\n\n\n");
			yesButton = new JButton("\"Yes.\"");
			noButton = new JButton("\"Uhhh...not today.\"");
			goodbyeButton = new JButton("\"Goodbye.\"");
			//set prompt area parameters
			promptArea.setLineWrap(true);
			promptArea.setWrapStyleWord(true);
			promptArea.setEditable(false);
			promptArea.setFocusable(false);
			promptArea.setSize(320, 100);
			//add action listeners to buttons
			yesButton.addActionListener(new YesAction());
			noButton.addActionListener(new NoAction());
			goodbyeButton.addActionListener(new GoodbyeAction());
			//add components to dialog window
			add(promptArea);
			add(yesButton);
			add(noButton);
		}

		/*
		CLASS: YesAction
		PURPOSE: Saves hero object as a Dungeon Quest Save (.dq) file.
		*/
		private class YesAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{	//save hero object as a Dungeon Quest Save (.dq) file
				try
				{
					ObjectOutputStream objOut = new ObjectOutputStream(new FileOutputStream(hero.getName() + ".dq"));
					objOut.writeObject(hero);
				}
				catch(IOException exception)
				{
					System.out.println("Caught exception: " + exception);
				}
				//change text and buttons in dialog window
				promptArea.setText("\"Bless you. May the gods watch over you during all your adventures.\"\n\n\n");
				remove(yesButton);
				remove(noButton);
				add(goodbyeButton);
			}
		}

		/*
		CLASS: NoAction
		PURPOSE: Bypasses saving game progress.
		*/
		private class NoAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{	//change text and button in dialog window
				promptArea.setText("\"Hmmm. You should try to come to confession more often. May the gods watch over you during all your adventures.\"\n\n\n");
				remove(yesButton);
				remove(noButton);
				add(goodbyeButton);
			}
		}

		/*
		CLASS: GoodbyeAction
		PURPOSE: Closes dialog window.
		*/
		private class GoodbyeAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{
				dispose();
			}
		}
	}

	/*
	CLASS: InnMenu
	PURPOSE: Prompts player to stay at the inn. Staying fully recovers HP and MP.
	*/
	private class InnMenu extends JDialog
	{
		private int fee; //how much gold it will cost to stay at the inn
		private JTextArea promptArea; //the dialog text
		private JButton yesButton; //clicking this will recover hero's HP and MP if gold is sufficient
		private JButton noButton; //clicking this will bypass staying at the inn
		private JButton goodbyeButton; //clicking this will close the dialog window

		public InnMenu(JFrame owner)
		{
			super(owner, "", true);
			//set dialog window parameters
			setSize(326, 157);
			setLocation(owner.getX(), owner.getY());
			setResizable(false);
			setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
			setLayout(new FlowLayout());

			fee = hero.getLVL()*10;
			//create components
			promptArea = new JTextArea("\"Welcome weary traveler. Would you like to rest for the night? The fee is " + fee + " gold.\"\n\n\n");
			yesButton = new JButton("\"Yes please.\"");
			noButton = new JButton("\"No thanks.\"");
			goodbyeButton = new JButton("\"Goodbye.\"");
			//set prompt area parameters
			promptArea.setLineWrap(true);
			promptArea.setWrapStyleWord(true);
			promptArea.setEditable(false);
			promptArea.setFocusable(false);
			promptArea.setSize(320, 100);
			//add action listeners to buttons
			yesButton.addActionListener(new YesAction());
			noButton.addActionListener(new NoAction());
			goodbyeButton.addActionListener(new GoodbyeAction());
			//add components to dialog window
			add(promptArea);
			add(yesButton);
			add(noButton);
		}

		/*
		CLASS: YesAction
		PURPOSE: If Hero has enough gold, fully recovers Hero's HP and MP and subtracts fee
		from Hero's gold.
		*/
		private class YesAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{
				if(hero.getGold() >= fee)
				{	//fully recover Hero's HP and MP and subtract fee from Hero's gold
					hero.setHP(hero.getStat(GameCharacter.MAX_HP));
					hero.setMP(hero.getStat(GameCharacter.MAX_MP));
					hero.setGold(hero.getGold() - fee);
					promptArea.setText("\"I hope you enjoyed your stay. Goodbye.\"\n\n\n\n");
				}
				else //if not enough gold
				{
					promptArea.setText("\"I'm sorry, but you don't have enough gold. Please come again when you do. Goodbye.\"\n\n\n");
				}
				//change buttons in dialog window
				remove(yesButton);
				remove(noButton);
				add(goodbyeButton);
			}
		}

		/*
		CLASS: NoAction
		PURPOSE: Bypasses staying at the inn.
		*/
		private class NoAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{	//change text and buttons in dialog window
				promptArea.setText("\"Alright, have a good day then.\"\n\n\n\n");
				remove(yesButton);
				remove(noButton);
				add(goodbyeButton);
			}
		}

		/*
		CLASS: GoodbyeAction
		PURPOSE: Closes dialog window.
		*/
		private class GoodbyeAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{
				dispose();
			}
		}
	}

	/*
	CLASS: WeaponAndArmorShopMenu
	PURPOSE: This is the weapon and armor shop menu. Contains 2 tabbed panels: Buy and Sell.
	*/
	private class WeaponAndArmorShopMenu extends JDialog
	{
		private JTextField itemField[]; //displays name of items in Hero's inventory in Sell panel
		private JTextField itemQuantityField[]; //displays quantity of each item in Hero's inventory
		private JTextField buyGoldField; //displays Hero's gold in buy panel
		private JTextField sellGoldField; //displays Hero's gold in sell panel

		public WeaponAndArmorShopMenu(JFrame owner)
		{
			super(owner, "Weapon Shop", true);
			//set window parameters
			setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
			setLocation(owner.getX(), owner.getY());
			setResizable(false);
			setFocusable(true);
			setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			//create tabbed pane
			JTabbedPane tabbedMenuPane = new JTabbedPane();
			//create panels
			BuyPanel buyPanel = new BuyPanel();
			SellPanel sellPanel = new SellPanel();
			//add panels to tabbed pane
			tabbedMenuPane.add("Buy", buyPanel);
			tabbedMenuPane.add("Sell", sellPanel);
			//add tabbed pane to window
			add(tabbedMenuPane);
		}

		/*
		CLASS: BuyPanel
		PURPOSE: Displays the shop's inventory and lets the player purchase items.
		*/
		private class BuyPanel extends JPanel
		{
			private JTextArea inventoryArea; //displays the shop's inventory
			private JTextArea descriptionArea; //displays a short description of the selected item
			private JButton buyButton; //if Hero has enough gold, adds selected item to his inventory

			public BuyPanel()
			{
				setLayout (null);
				//add items to shop's inventory
				inventoryArea = new JTextArea("");
				inventoryArea.append("copper sword\n");
				inventoryArea.append("leather shield\n");
				inventoryArea.append("leather hat\n");
				inventoryArea.append("leather armor\n");
				inventoryArea.append("copper bracers\n");
				inventoryArea.append("copper dagger\n");
				inventoryArea.append("leather cap\n");
				inventoryArea.append("leather tunic\n");
				inventoryArea.append("leather boots\n");
				inventoryArea.append("walking stick\n");
				inventoryArea.append("fur hood\n");
				inventoryArea.append("fur robe\n");
				inventoryArea.append("magic ring\n");
				inventoryArea.append("bronze sword\n");
				inventoryArea.append("bronze shield\n");
				inventoryArea.append("bronze helmet\n");
				inventoryArea.append("bronze armor\n");
				inventoryArea.append("bronze bracers\n");
				inventoryArea.append("bronze knife\n");
				inventoryArea.append("bandit bandana\n");
				inventoryArea.append("bandit tunic\n");
				inventoryArea.append("bandit shoes\n");
				inventoryArea.append("novice staff\n");
				inventoryArea.append("novice hood\n");
				inventoryArea.append("novice robe\n");
				inventoryArea.append("novice ring\n");
				inventoryArea.append("iron sword\n");
				inventoryArea.append("iron shield\n");
				inventoryArea.append("iron helmet\n");
				inventoryArea.append("iron armor\n");
				inventoryArea.append("iron gauntlets\n");
				inventoryArea.append("iron dagger\n");
				inventoryArea.append("robber mask\n");
				inventoryArea.append("robber tunic\n");
				inventoryArea.append("robber boots\n");
				inventoryArea.append("apprentice staff\n");
				inventoryArea.append("apprentice hood\n");
				inventoryArea.append("apprentice robe\n");
				inventoryArea.append("apprentice ring\n");
				inventoryArea.append("steel sword\n");
				inventoryArea.append("steel shield\n");
				inventoryArea.append("steel helmet\n");
				inventoryArea.append("steel armor\n");
				inventoryArea.append("steel gauntlets\n");
				inventoryArea.append("steel dagger\n");
				inventoryArea.append("burglar mask\n");
				inventoryArea.append("burglar tunic\n");
				inventoryArea.append("burglar shoes\n");
				inventoryArea.append("journeyman staff\n");
				inventoryArea.append("journeyman hat\n");
				inventoryArea.append("journeyman robe\n");
				inventoryArea.append("journeyman ring\n");
				inventoryArea.append("silver sword\n");
				inventoryArea.append("silver shield\n");
				inventoryArea.append("silver helmet\n");
				inventoryArea.append("silver armor\n");
				inventoryArea.append("silver bracers\n");
				inventoryArea.append("thief dagger\n");
				inventoryArea.append("thief cap\n");
				inventoryArea.append("thief tunic\n");
				inventoryArea.append("thief boots\n");
				inventoryArea.append("master staff\n");
				inventoryArea.append("master hat\n");
				inventoryArea.append("master robe\n");
				inventoryArea.append("master ring\n");
				inventoryArea.append("platinum sword\n");
				inventoryArea.append("platinum shield\n");
				inventoryArea.append("platinum helmet\n");
				inventoryArea.append("platinum armor\n");
				inventoryArea.append("platinum bracers\n");
				inventoryArea.append("hunting bow\n");
				inventoryArea.append("hunting cap\n");
				inventoryArea.append("hunting vest\n");
				inventoryArea.append("hunting boots\n");
				inventoryArea.append("hermit staff\n");
				inventoryArea.append("hermit hood\n");
				inventoryArea.append("hermit robe\n");
				inventoryArea.append("hermit ring\n");
				inventoryArea.append("mythril sword\n");
				inventoryArea.append("mythril shield\n");
				inventoryArea.append("mythril helmet\n");
				inventoryArea.append("mythril armor\n");
				inventoryArea.append("mythril gauntlets\n");
				inventoryArea.append("katana\n");
				inventoryArea.append("ninja mask\n");
				inventoryArea.append("ninja gi\n");
				inventoryArea.append("ninja soles\n");
				inventoryArea.append("wizard staff\n");
				inventoryArea.append("wizard hat\n");
				inventoryArea.append("wizard robe\n");
				inventoryArea.append("wizard ring");
				//set inventory area parameters and add mouse listener
				inventoryArea.setCaretPosition(0);
				inventoryArea.setEditable(false);
				inventoryArea.addMouseListener(new InventoryAreaClicked());
				//create scroll pane, set parameters, and add it to panel
				JScrollPane inventoryScrollPane = new JScrollPane(inventoryArea, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				inventoryScrollPane.setLocation(20, 20);
				inventoryScrollPane.setSize(125, 160);
				add(inventoryScrollPane);
				//create other components
				descriptionArea = new JTextArea("\"Do you see anything you would like to buy?\"");
				JLabel goldLabel = new JLabel("GOLD");
				buyGoldField = new JTextField("" + hero.getGold());
				buyButton = new JButton("Buy");
				//set component parameters
				descriptionArea.setLineWrap(true);
				descriptionArea.setWrapStyleWord(true);
				descriptionArea.setEditable(false);
				descriptionArea.setFocusable(false);
				descriptionArea.setLocation(20, 200);
				descriptionArea.setSize(200, 40);
				descriptionArea.setFont(new Font("Arial", Font.PLAIN, 10));
				goldLabel.setLocation(165, 167);
				goldLabel.setSize(100, 10);
				buyGoldField.setEditable(false);
				buyGoldField.setFocusable(false);
				buyGoldField.setLocation(200, 164);
				buyGoldField.setSize(90, 16);
				buyButton.setEnabled(false);
				buyButton.setLocation(236, 218);
				buyButton.setSize(56, 22);
				buyButton.setMargin(new Insets(0, 0, 0, 0));
				buyButton.addActionListener(new BuyAction());
				//add components to panel
				add(descriptionArea);
				add(goldLabel);
				add(buyGoldField);
				add(buyButton);
			}

			/*
			CLASS: InventoryAreaClicked
			PURPOSE: Highlights the clicked on string.
			*/
			private class InventoryAreaClicked extends MouseAdapter
			{
				public void mouseReleased(MouseEvent event)
				{	//highlight the clicked on string
					try
					{
						inventoryArea.select(inventoryArea.getLineStartOffset(inventoryArea.getLineOfOffset(inventoryArea.getCaretPosition())),
										 inventoryArea.getLineEndOffset(inventoryArea.getLineOfOffset(inventoryArea.getCaretPosition())));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}

					try
					{
						if(!inventoryArea.getSelectedText().equals(""))
						{	//display item price and description
							String selectedItem = inventoryArea.getSelectedText().trim();
							descriptionArea.setText("Price: " + itemTable.getItem(selectedItem).getValue() + " gold\n" +
													itemTable.getItem(selectedItem).getDescription());
							//if Hero has enough gold...
							if(hero.getGold() >= itemTable.getItem(selectedItem).getValue())
							{
								buyButton.setEnabled(true);
							}
							else //if Hero does not have enough gold...
							{
								buyButton.setEnabled(false);
							}
						}
					}
					catch(NullPointerException exception)
					{
						descriptionArea.setText("");
						buyButton.setEnabled(false);
					}
				}
			}

			/*
			CLASS: BuyAction
			PURPOSE: If Hero has enough gold, adds selected item to Hero's inventory.
			*/
			private class BuyAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					try
					{
						String selectedItem = inventoryArea.getSelectedText().trim();
						//if Hero has enough gold...
						if(itemTable.getItem(selectedItem).getValue() <= hero.getGold())
						{	//if Hero has item in inventory...
							if(hero.hasInInventory(selectedItem) != -1)
							{	//increase item's quantity in inventory and subtract price from gold
								int slot = hero.hasInInventory(selectedItem);
								hero.getItem(slot).setQuantity(hero.getItem(slot).getQuantity() + 1);
								hero.setGold(hero.getGold() - itemTable.getItem(selectedItem).getValue());
							}
							//if Hero has room in inventory...
							else if(hero.firstEmptySlotInInventory() != -1)
							{	//add selected item to first empty slot and subtract price from gold
								hero.setItem(itemTable.getItem(selectedItem), hero.firstEmptySlotInInventory());
								hero.setGold(hero.getGold() - itemTable.getItem(selectedItem).getValue());
							}
							else //if not enough room in inventory...
							{
								descriptionArea.setText("\"I'm sorry, but it looks like your inventory is full. You'll have to make room before you can purchase this item.\"");
							}
							//update appropriate textfields
							buyGoldField.setText("" + hero.getGold());
							sellGoldField.setText("" + hero.getGold());
							for(int i = 0; i < 10; i++)
							{
								if(!hero.getItem(i).getName().equals(""))
								{
									itemField[i].setText(hero.getItem(i).getName());
									itemQuantityField[i].setText("x" + hero.getItem(i).getQuantity());
								}
							}
							//if Hero no longer has enough gold...
							if(hero.getGold() < itemTable.getItem(selectedItem).getValue())
							{
								buyButton.setEnabled(false);
							}
						}
					}
					catch(NullPointerException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
				}
			}
		}

		/*
		CLASS: SellPanel
		PURPOSE: Displays the hero's inventory and lets the player sell items.
		*/
		private class SellPanel extends JPanel
		{
			private JTextArea descriptionArea; //displays a short description of the selected item
			private JButton sellButton; //removes selected item from inventory and adds price to gold

			public SellPanel()
			{
				setLayout (null);

				itemField = new JTextField[10]; //displays Hero's inventory
				itemQuantityField = new JTextField[10]; //displays quantity of each item in inventory

				for(int i = 0; i < 10; i++)
				{	//create textfields
					itemField[i] = new JTextField(hero.getItem(i).getName());
					if(!hero.getItem(i).getName().equals(""))
					{
						itemQuantityField[i] = new JTextField("x" + hero.getItem(i).getQuantity());
					}
					else
					{
						itemQuantityField[i] = new JTextField();
					}
					//set textfield parameters
					itemField[i].setEditable(false);
					itemField[i].setLocation(20, 20 + i*16);
					itemField[i].setSize(100, 16);
					itemQuantityField[i].setEditable(false);
					itemQuantityField[i].setFocusable(false);
					itemQuantityField[i].setLocation(120, 20 + i*16);
					itemQuantityField[i].setSize(25, 16);
					//add mouse listener
					itemField[i].addMouseListener(new ItemFieldClicked(itemField[i]));
					//add textfields to panel
					add(itemField[i]);
					add(itemQuantityField[i]);
				}
				//create other components
				descriptionArea = new JTextArea("\"Do you have anything you would like to sell?\"");
				JLabel goldLabel = new JLabel("GOLD");
				sellGoldField = new JTextField("" + hero.getGold());
				sellButton = new JButton("Sell");
				//set parameters
				descriptionArea.setLineWrap(true);
				descriptionArea.setWrapStyleWord(true);
				descriptionArea.setEditable(false);
				descriptionArea.setFocusable(false);
				descriptionArea.setLocation(20, 200);
				descriptionArea.setSize(200, 40);
				descriptionArea.setFont(new Font("Arial", Font.PLAIN, 10));
				goldLabel.setLocation(165, 167);
				goldLabel.setSize(100, 10);
				sellGoldField.setEditable(false);
				sellGoldField.setFocusable(false);
				sellGoldField.setLocation(200, 164);
				sellGoldField.setSize(90, 16);
				sellButton.setEnabled(false);
				sellButton.setLocation(236, 218);
				sellButton.setSize(56, 22);
				sellButton.setMargin(new Insets(0, 0, 0, 0));
				//add action listener
				sellButton.addActionListener(new SellAction());
				//add components
				add(descriptionArea);
				add(goldLabel);
				add(sellGoldField);
				add(sellButton);
			}

			/*
			CLASS: ItemFieldClicked
			PURPOSE: Highlights the clicked on textfield.
			*/
			private class ItemFieldClicked extends MouseAdapter
			{
				private JTextField textField; //the clicked on textfield

				public ItemFieldClicked(JTextField fieldClicked)
				{
					textField = fieldClicked;
				}

				public void mouseReleased(MouseEvent event)
				{	//highlight the clicked on textfield
					for(int i = 0; i < 10; i++)
					{
						itemField[i].setCaretPosition(0);
					}
					textField.selectAll();
					//display item price and description
					descriptionArea.setText("Price: " + itemTable.getItem(textField.getText()).getValue() / 2 + " gold\n" +
											itemTable.getItem(textField.getText()).getDescription());
					//if textfield blank...
					if(textField.getText().equals(""))
					{
						sellButton.setEnabled(false);
						descriptionArea.setText("");
					}
					else
					{
						sellButton.setEnabled(true);
					}
				}
			}

			/*
			CLASS: SellAction
			PURPOSE: Removes the selected item from the hero's inventory and adds half of
			the item's value to the hero's gold.
			*/
			private class SellAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					String selectedItem;
					//search each item in inventory
					for(int i = 0; i < 10; i++)
					{
						selectedItem = itemField[i].getSelectedText();
						try
						{	//if selected item found...
							if(!selectedItem.equals(""))
							{
								if(hero.getItem(i).getQuantity() > 1)
								{	//decrease item quantity by one
									hero.getItem(i).setQuantity(hero.getItem(i).getQuantity() - 1);
									itemQuantityField[i].setText("x" + hero.getItem(i).getQuantity());
								}
								else //if last copy of item in inventory...
								{	//...remove item if equipped...
									if(selectedItem.equals(hero.getEquipment(Item.WEAPON).getName()))
									{
										hero.removeEquipment(Item.WEAPON);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.SHIELD).getName()))
									{
										hero.removeEquipment(Item.SHIELD);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.HEAD).getName()))
									{
										hero.removeEquipment(Item.HEAD);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.BODY).getName()))
									{
										hero.removeEquipment(Item.BODY);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.ACCESSORY).getName()))
									{
										hero.removeEquipment(Item.ACCESSORY);
									}
									//...and remove item from inventory completely
									hero.setItem((Equipment) itemTable.getItem(""), i);
									itemField[i].setText("");
									itemQuantityField[i].setText("");
									descriptionArea.setText("");
									sellButton.setEnabled(false);
								}
								//add price to gold
								hero.setGold(hero.getGold() + itemTable.getItem(selectedItem).getValue() / 2);
								buyGoldField.setText("" + hero.getGold());
								sellGoldField.setText("" + hero.getGold());
								break;
							}
						}
						catch(NullPointerException exception)
						{
							System.out.println("Caught exception: " + exception);
						}
					}
				}
			}
		}
	}

	/*
	CLASS: ItemShopMenu
	PURPOSE:  This is the item shop menu. Contains 2 tabbed panels: Buy and Sell.
	*/
	private class ItemShopMenu extends JDialog
	{
		private JTextField itemField[]; //displays name of items in Hero's inventory in Sell panel
		private JTextField itemQuantityField[]; //displays quantity of each item in Hero's inventory
		private JTextField buyGoldField; //displays Hero's gold in buy panel
		private JTextField sellGoldField; //displays Hero's gold in sell panel

		public ItemShopMenu(JFrame owner)
		{
			super(owner, "Item Shop", true);
			//set window parameters
			setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
			setLocation(owner.getX(), owner.getY());
			setResizable(false);
			setFocusable(true);
			setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			//create tabbed pane
			JTabbedPane tabbedMenuPane = new JTabbedPane();
			//create panels
			BuyPanel buyPanel = new BuyPanel();
			SellPanel sellPanel = new SellPanel();
			//add panels to tabbed pane
			tabbedMenuPane.add("Buy", buyPanel);
			tabbedMenuPane.add("Sell", sellPanel);
			//add tabbed pane to window
			add(tabbedMenuPane);
		}

		/*
		CLASS: BuyPanel
		PURPOSE: Displays the shop's inventory and lets the player purchase items.
		*/
		private class BuyPanel extends JPanel
		{
			private JTextArea inventoryArea; //displays the shop's inventory
			private JTextArea descriptionArea; //displays a short description of the selected item
			private JButton buyButton; //if Hero has enough gold, adds selected item to his inventory

			public BuyPanel()
			{
				setLayout (null);
				//add items to shop's inventory
				inventoryArea = new JTextArea("");
				inventoryArea.append("herb\n");
				inventoryArea.append("potion\n");
				inventoryArea.append("strong herb\n");
				inventoryArea.append("strong potion\n");
				inventoryArea.append("special herb\n");
				inventoryArea.append("special potion\n");
				inventoryArea.append("warp scroll\n");
				inventoryArea.append("minor heal\n");
				inventoryArea.append("minor blast\n");
				inventoryArea.append("minor strength\n");
				inventoryArea.append("minor weakness\n");
				inventoryArea.append("minor shield\n");
				inventoryArea.append("minor exposure\n");
				inventoryArea.append("minor sight\n");
				inventoryArea.append("minor blind\n");
				inventoryArea.append("minor agility\n");
				inventoryArea.append("minor paralysis\n");
				inventoryArea.append("minor barrier\n");
				inventoryArea.append("minor opening\n");
				inventoryArea.append("minor accelerate\n");
				inventoryArea.append("minor brakes\n");
				inventoryArea.append("minor stupidity\n");
				inventoryArea.append("mid heal\n");
				inventoryArea.append("mid blast\n");
				inventoryArea.append("mid strength\n");
				inventoryArea.append("mid weakness\n");
				inventoryArea.append("mid shield\n");
				inventoryArea.append("mid exposure\n");
				inventoryArea.append("mid sight\n");
				inventoryArea.append("mid blind\n");
				inventoryArea.append("mid agility\n");
				inventoryArea.append("mid paralysis\n");
				inventoryArea.append("mid barrier\n");
				inventoryArea.append("mid opening\n");
				inventoryArea.append("mid accelerate\n");
				inventoryArea.append("mid brakes\n");
				inventoryArea.append("mid stupidity\n");
				inventoryArea.append("major heal\n");
				inventoryArea.append("major blast\n");
				inventoryArea.append("major strength\n");
				inventoryArea.append("major weakness\n");
				inventoryArea.append("major shield\n");
				inventoryArea.append("major exposure\n");
				inventoryArea.append("major sight\n");
				inventoryArea.append("major blind\n");
				inventoryArea.append("major agility\n");
				inventoryArea.append("major paralysis\n");
				inventoryArea.append("major barrier\n");
				inventoryArea.append("major opening\n");
				inventoryArea.append("major accelerate\n");
				inventoryArea.append("major brakes\n");
				inventoryArea.append("major stupidity");
				//set inventory area parameters and add mouse listener
				inventoryArea.setCaretPosition(0);
				inventoryArea.setEditable(false);
				inventoryArea.addMouseListener(new InventoryAreaClicked());
				//create scroll pane, set parameters, and add it to panel
				JScrollPane inventoryScrollPane = new JScrollPane(inventoryArea, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				inventoryScrollPane.setLocation(20, 20);
				inventoryScrollPane.setSize(125, 160);
				add(inventoryScrollPane);
				//create other components
				descriptionArea = new JTextArea("\"Do you see anything you would like to buy?\"");
				JLabel goldLabel = new JLabel("GOLD");
				buyGoldField = new JTextField("" + hero.getGold());
				buyButton = new JButton("Buy");
				//set component parameters
				descriptionArea.setLineWrap(true);
				descriptionArea.setWrapStyleWord(true);
				descriptionArea.setEditable(false);
				descriptionArea.setFocusable(false);
				descriptionArea.setLocation(20, 200);
				descriptionArea.setSize(200, 40);
				descriptionArea.setFont(new Font("Arial", Font.PLAIN, 10));
				goldLabel.setLocation(165, 167);
				goldLabel.setSize(100, 10);
				buyGoldField.setEditable(false);
				buyGoldField.setFocusable(false);
				buyGoldField.setLocation(200, 164);
				buyGoldField.setSize(90, 16);
				buyButton.setEnabled(false);
				buyButton.setLocation(236, 218);
				buyButton.setSize(56, 22);
				buyButton.setMargin(new Insets(0, 0, 0, 0));
				buyButton.addActionListener(new BuyAction());
				//add components to panel
				add(descriptionArea);
				add(goldLabel);
				add(buyGoldField);
				add(buyButton);
			}

			/*
			CLASS: InventoryAreaClicked
			PURPOSE: Highlights the clicked on string.
			*/
			private class InventoryAreaClicked extends MouseAdapter
			{
				public void mouseReleased(MouseEvent event)
				{	//highlight the clicked on string
					try
					{
						inventoryArea.select(inventoryArea.getLineStartOffset(inventoryArea.getLineOfOffset(inventoryArea.getCaretPosition())),
										 inventoryArea.getLineEndOffset(inventoryArea.getLineOfOffset(inventoryArea.getCaretPosition())));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}

					try
					{
						if(!inventoryArea.getSelectedText().equals(""))
						{	//display item price and description
							String selectedItem = inventoryArea.getSelectedText().trim();
							descriptionArea.setText("Price: " + itemTable.getItem(selectedItem).getValue() + " gold\n" +
													itemTable.getItem(selectedItem).getDescription());
							//if Hero has enough gold...
							if(hero.getGold() >= itemTable.getItem(selectedItem).getValue())
							{
								buyButton.setEnabled(true);
							}
							else //if Hero does not have enough gold...
							{
								buyButton.setEnabled(false);
							}
						}
					}
					catch(NullPointerException exception)
					{
						descriptionArea.setText("");
						buyButton.setEnabled(false);
					}
				}
			}

			/*
			CLASS: BuyAction
			PURPOSE: If Hero has enough gold, adds selected item to Hero's inventory.
			*/
			private class BuyAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					try
					{
						String selectedItem = inventoryArea.getSelectedText().trim();
						//if Hero has enough gold...
						if(itemTable.getItem(selectedItem).getValue() <= hero.getGold())
						{	//if Hero has item in inventory...
							if(hero.hasInInventory(selectedItem) != -1)
							{	//increase item's quantity in inventory and subtract price from gold
								int slot = hero.hasInInventory(selectedItem);
								hero.getItem(slot).setQuantity(hero.getItem(slot).getQuantity() + 1);
								hero.setGold(hero.getGold() - itemTable.getItem(selectedItem).getValue());
							}
							//if Hero has room in inventory...
							else if(hero.firstEmptySlotInInventory() != -1)
							{	//add selected item to first empty slot and subtract price from gold
								hero.setItem(itemTable.getItem(selectedItem), hero.firstEmptySlotInInventory());
								hero.setGold(hero.getGold() - itemTable.getItem(selectedItem).getValue());
							}
							else //if not enough room in inventory...
							{
								descriptionArea.setText("\"I'm sorry, but it looks like your inventory is full. You'll have to make room before you can purchase this item.\"");
							}
							//update appropriate textfields
							buyGoldField.setText("" + hero.getGold());
							sellGoldField.setText("" + hero.getGold());
							for(int i = 0; i < 10; i++)
							{
								if(!hero.getItem(i).getName().equals(""))
								{
									itemField[i].setText(hero.getItem(i).getName());
									itemQuantityField[i].setText("x" + hero.getItem(i).getQuantity());
								}
							}
							//if Hero no longer has enough gold...
							if(hero.getGold() < itemTable.getItem(selectedItem).getValue())
							{
								buyButton.setEnabled(false);
							}
						}
					}
					catch(NullPointerException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
				}
			}
		}

		/*
		CLASS: SellPanel
		PURPOSE: Displays the hero's inventory and lets the player sell items.
		*/
		private class SellPanel extends JPanel
		{
			private JTextArea descriptionArea; //displays a short description of the selected item
			private JButton sellButton; //removes selected item from inventory and adds price to gold

			public SellPanel()
			{
				setLayout (null);

				itemField = new JTextField[10]; //displays Hero's inventory
				itemQuantityField = new JTextField[10]; //displays quantity of each item in inventory

				for(int i = 0; i < 10; i++)
				{	//create textfields
					itemField[i] = new JTextField(hero.getItem(i).getName());
					if(!hero.getItem(i).getName().equals(""))
					{
						itemQuantityField[i] = new JTextField("x" + hero.getItem(i).getQuantity());
					}
					else
					{
						itemQuantityField[i] = new JTextField();
					}
					//set textfield parameters
					itemField[i].setEditable(false);
					itemField[i].setLocation(20, 20 + i*16);
					itemField[i].setSize(100, 16);
					itemQuantityField[i].setEditable(false);
					itemQuantityField[i].setFocusable(false);
					itemQuantityField[i].setLocation(120, 20 + i*16);
					itemQuantityField[i].setSize(25, 16);
					//add mouse listener
					itemField[i].addMouseListener(new ItemFieldClicked(itemField[i]));
					//add textfields to panel
					add(itemField[i]);
					add(itemQuantityField[i]);
				}
				//create other components
				descriptionArea = new JTextArea("\"Do you have anything you would like to sell?\"");
				JLabel goldLabel = new JLabel("GOLD");
				sellGoldField = new JTextField("" + hero.getGold());
				sellButton = new JButton("Sell");
				//set parameters
				descriptionArea.setLineWrap(true);
				descriptionArea.setWrapStyleWord(true);
				descriptionArea.setEditable(false);
				descriptionArea.setFocusable(false);
				descriptionArea.setLocation(20, 200);
				descriptionArea.setSize(200, 40);
				descriptionArea.setFont(new Font("Arial", Font.PLAIN, 10));
				goldLabel.setLocation(165, 167);
				goldLabel.setSize(100, 10);
				sellGoldField.setEditable(false);
				sellGoldField.setFocusable(false);
				sellGoldField.setLocation(200, 164);
				sellGoldField.setSize(90, 16);
				sellButton.setEnabled(false);
				sellButton.setLocation(236, 218);
				sellButton.setSize(56, 22);
				sellButton.setMargin(new Insets(0, 0, 0, 0));
				//add action listener
				sellButton.addActionListener(new SellAction());
				//add components
				add(descriptionArea);
				add(goldLabel);
				add(sellGoldField);
				add(sellButton);
			}

			/*
			CLASS: ItemFieldClicked
			PURPOSE: Highlights the clicked on textfield.
			*/
			private class ItemFieldClicked extends MouseAdapter
			{
				private JTextField textField; //the clicked on textfield

				public ItemFieldClicked(JTextField fieldClicked)
				{
					textField = fieldClicked;
				}

				public void mouseReleased(MouseEvent event)
				{	//highlight the clicked on textfield
					for(int i = 0; i < 10; i++)
					{
						itemField[i].setCaretPosition(0);
					}
					textField.selectAll();
					//display item price and description
					descriptionArea.setText("Price: " + itemTable.getItem(textField.getText()).getValue() / 2 + " gold\n" +
											itemTable.getItem(textField.getText()).getDescription());
					//if textfield blank...
					if(textField.getText().equals(""))
					{
						sellButton.setEnabled(false);
						descriptionArea.setText("");
					}
					else
					{
						sellButton.setEnabled(true);
					}
				}
			}

			/*
			CLASS: SellAction
			PURPOSE: Removes the selected item from the hero's inventory and adds half of
			the item's value to the hero's gold.
			*/
			private class SellAction implements ActionListener
			{
				public void actionPerformed(ActionEvent event)
				{
					String selectedItem;
					//search each item in inventory
					for(int i = 0; i < 10; i++)
					{
						selectedItem = itemField[i].getSelectedText();
						try
						{	//if selected item found...
							if(!selectedItem.equals(""))
							{
								if(hero.getItem(i).getQuantity() > 1)
								{	//decrease item quantity by one
									hero.getItem(i).setQuantity(hero.getItem(i).getQuantity() - 1);
									itemQuantityField[i].setText("x" + hero.getItem(i).getQuantity());
								}
								else //if last copy of item in inventory...
								{	//...remove item if equipped...
									if(selectedItem.equals(hero.getEquipment(Item.WEAPON).getName()))
									{
										hero.removeEquipment(Item.WEAPON);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.SHIELD).getName()))
									{
										hero.removeEquipment(Item.SHIELD);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.HEAD).getName()))
									{
										hero.removeEquipment(Item.HEAD);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.BODY).getName()))
									{
										hero.removeEquipment(Item.BODY);
									}
									else if(selectedItem.equals(hero.getEquipment(Item.ACCESSORY).getName()))
									{
										hero.removeEquipment(Item.ACCESSORY);
									}
									//...and remove item from inventory completely
									hero.setItem((Equipment) itemTable.getItem(""), i);
									itemField[i].setText("");
									itemQuantityField[i].setText("");
									descriptionArea.setText("");
									sellButton.setEnabled(false);
								}
								//add price to gold
								hero.setGold(hero.getGold() + itemTable.getItem(selectedItem).getValue() / 2);
								buyGoldField.setText("" + hero.getGold());
								sellGoldField.setText("" + hero.getGold());
								break;
							}
						}
						catch(NullPointerException exception)
						{
							System.out.println("Caught exception: " + exception);
						}
					}
				}
			}
		}
	}

	/*
	CLASS: BattleWindow
	PURPOSE: Contains four sections: the monster's image, the command buttons, the hero's
	HP and MP status, and the battle log.
	*/
	private class BattleWindow extends JDialog
	{
		private JPanel battlePanel; //contains all the window components
		private JButton attackButton; //commands the hero to attack with his weapon
		private JButton spellsButton; //displays the hero's spell list
		private JButton itemsButton; //displays the hero's inventory
		private JButton fleeButton; //commands the hero to flee
		private JTextField hpField; //displays the hero's current HP
		private JTextField mpField; //displays the hero's current MP
		private JFormattedTextField maxHpField; //displays the hero's maximum HP
		private JFormattedTextField maxMpField; //displays the hero's maximum MP
		private JTextArea battleLog; //displays the results of each battle action
		private MonsterLookUpTable monsterTable; //a look-up table containing all monsters
		private Monster monster; //the randomly chosen monster for this battle
		private Timer timer; //used to schedule battle action tasks
		private int random; //acts as dice in battle
		private boolean battleOver; //used to determine if battle is over
		private int[] startStat; //used to record the hero's stats before battle began

		public BattleWindow(JFrame owner)
		{
			super(owner, "", true);
			//initialize variables
			timer = new Timer();
			battleOver = false;
			//get a random monster from look-up table, based on current dungeon level
			monsterTable = new MonsterLookUpTable();
			monster = monsterTable.getRandomMonster(dungeonLevel);
			//record the hero's current stats
			startStat = new int[9];
			for(int i = 0; i < 9; i++)
			{
				startStat[i] = hero.getStat(i);
			}
			//set window parameters
			setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
			setLocation(owner.getX(), owner.getY());
			setResizable(false);
			setFocusable(true);
			setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
			//create battle panel and add it to window
			battlePanel = new BattlePanel();
			add(battlePanel);
		}

		/*
		CLASS: BattlePanel
		PURPOSE: Creates the battle window components.
		*/
		private class BattlePanel extends JPanel
		{
			public BattlePanel()
			{
				setLayout (null);
				//create window components
				attackButton = new JButton("ATTACK");
				spellsButton = new JButton("SPELLS");
				itemsButton = new JButton("ITEMS");
				fleeButton = new JButton("FLEE");
				JLabel nameLabel = new JLabel(hero.getName());
				JLabel hpLabel = new JLabel("HP");
				JLabel mpLabel = new JLabel("MP");
				hpField = new JTextField(hero.getHP() + "/");
				mpField = new JTextField(hero.getMP() + "/");
				maxHpField = new JFormattedTextField(new Integer(hero.getStat(GameCharacter.MAX_HP)));
				maxMpField = new JFormattedTextField(new Integer(hero.getStat(GameCharacter.MAX_MP)));
				if(monster.getName().charAt(0) == 'a' ||
				   monster.getName().charAt(0) == 'e' ||
				   monster.getName().charAt(0) == 'i' ||
				   monster.getName().charAt(0) == 'o' ||
				   monster.getName().charAt(0) == 'u')
				{
					battleLog = new JTextArea("An " + monster.getName() + " approaches. ");
				}
				else
				{
					battleLog = new JTextArea("A " + monster.getName() + " approaches. ");
				}

				if(hero.getLVL() - monster.getLVL() >= 5)
				{
					battleLog.append("This fight looks way too easy.\n");
				}
				else if(hero.getLVL() - monster.getLVL() >= 2 && hero.getLVL() - monster.getLVL() <= 4)
				{
					battleLog.append("This looks like an easy fight.\n");
				}
				else if(hero.getLVL() - monster.getLVL() >= -1 && hero.getLVL() - monster.getLVL() <= 1)
				{
					battleLog.append("This looks like a fair fight.\n");
				}
				else if(hero.getLVL() - monster.getLVL() >= -4 && hero.getLVL() - monster.getLVL() <= -2)
				{
					battleLog.append("This looks like a tough fight.\n");
				}
				else
				{
					battleLog.append("This fight looks hopeless. Fleeing is highly recommended.\n");
				}
				JScrollPane battleLogScrollPane = new JScrollPane(battleLog, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				//set component parameters
				attackButton.setMargin(new Insets(0, 0, 0, 0));
				attackButton.setSize(60, 20);
				attackButton.setLocation(20, 160);
				spellsButton.setMargin(new Insets(0, 0, 0, 0));
				spellsButton.setSize(60, 20);
				spellsButton.setLocation(20, 190);
				itemsButton.setMargin(new Insets(0, 0, 0, 0));
				itemsButton.setSize(60, 20);
				itemsButton.setLocation(20, 220);
				fleeButton.setMargin(new Insets(0, 0, 0, 0));
				fleeButton.setSize(60, 20);
				fleeButton.setLocation(20, 250);
				nameLabel.setSize(100, 10);
				nameLabel.setLocation(100, 160);
				hpLabel.setSize(20, 10);
				hpLabel.setLocation(100, 175);
				mpLabel.setSize(20, 10);
				mpLabel.setLocation(200, 175);
				hpField.setEditable(false);
				hpField.setFocusable(false);
				hpField.setSize(30, 16);
				hpField.setLocation(120, 172);
				mpField.setEditable(false);
				mpField.setFocusable(false);
				mpField.setSize(30, 16);
				mpField.setLocation(220, 172);
				maxHpField.setEditable(false);
				maxHpField.setFocusable(false);
				maxHpField.setSize(25, 16);
				maxHpField.setLocation(150, 172);
				maxMpField.setEditable(false);
				maxMpField.setFocusable(false);
				maxMpField.setSize(25, 16);
				maxMpField.setLocation(250, 172);
				battleLog.setEditable(false);
				battleLog.setFocusable(false);
				battleLog.setLineWrap(true);
				battleLog.setWrapStyleWord(true);
				battleLogScrollPane.setSize(200, 80);
				battleLogScrollPane.setLocation(100, 190);
				if(hero.spellListIsEmpty())
				{
					spellsButton.setEnabled(false);
				}
				if(hero.hasNoHealingItems())
				{
					itemsButton.setEnabled(false);
				}
				//add action listeners to buttons
				attackButton.addActionListener(new AttackAction());
				spellsButton.addActionListener(new SpellsAction());
				itemsButton.addActionListener(new ItemsAction());
				fleeButton.addActionListener(new FleeAction());
				//add components to panel
				add(attackButton);
				add(spellsButton);
				add(itemsButton);
				add(fleeButton);
				add(nameLabel);
				add(hpLabel);
				add(hpField);
				add(maxHpField);
				add(mpLabel);
				add(mpField);
				add(maxMpField);
				add(battleLogScrollPane);
				//play battle theme
				musicClip.stop();
				try
				{
					musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
															"audio_files/battle.mid"));
				}
				catch (IOException exception)
				{
					musicClip = null;
				}
				musicClip.loop();
			}

			/*
			METHOD: paintComponent
			PURPOSE: This is the draw function for the battle panel.
			*/
			public void paintComponent(Graphics g)
			{
				super.paintComponent(g);

				Image monsterImage = null;
				try
				{
					monsterImage = ImageIO.read(new File("image_files/" + monster.getImageFileName()));
				}
				catch(IOException exception)
				{
					System.out.println("Caught exception: " + exception);
				}
				g.drawImage(monsterImage, 80, 0, null);
			}
		}

		/*
		CLASS: AttackAction
		PURPOSE: Schedules the appropriate battle action tasks for when the player issues
		the attack command.
		*/
		private class AttackAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{	//disable commands during action sequence
				attackButton.setEnabled(false);
				spellsButton.setEnabled(false);
				itemsButton.setEnabled(false);
				fleeButton.setEnabled(false);
				//if the hero is faster than the monster...
				if(hero.getStat(GameCharacter.SPD) >= monster.getStat(GameCharacter.SPD))
				{	//...the hero's action is executed first, the monster's second
					timer.schedule(new HeroAttackFirst(), 0);
					timer.schedule(new CheckIfMonsterSlain(), 1000);
					timer.schedule(new MonsterActionSecond(), 2000);
					timer.schedule(new EndRound(), 4000);
				}
				else //if the monster is faster than the hero...
				{	//...the monster's action is executed first, the hero's second
					timer.schedule(new MonsterActionFirst(), 0);
					timer.schedule(new CheckIfHeroSlain(), 1000);
					timer.schedule(new HeroAttackSecond(), 2000);
					timer.schedule(new EndRound(), 4000);
				}
			}
		}

		/*
		CLASS: SpellsAction
		PURPOSE: Displays the hero's spell list in the form of a pop-up menu.
		*/
		private class SpellsAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{	//create pop-up menu and menu items
				JPopupMenu popup = new JPopupMenu();
				JMenuItem[] menuItem = new JMenuItem[10];
				//for each spell list slot...
				for(int i = 0; i < 10; i++)
				{
					menuItem[i] = new JMenuItem(hero.getSpell(i).getName());
					//if the slot is not empty...
					if(!hero.getSpell(i).getName().equals(""))
					{	//...create tool tip for displaying spell description...
						menuItem[i].setToolTipText(hero.getSpell(i).getDescription());
						//...add action listener to menu item...
						menuItem[i].addActionListener(new SpellsMenuItemAction(hero.getSpell(i)));
						//if the hero doesn't have enough MP to cast the spell...
						if(hero.getMP() < hero.getSpell(i).getMPcost())
						{
							menuItem[i].setEnabled(false);
						}
						//...and add menu item to pop-up menu
						popup.add(menuItem[i]);
					}
				}
				//display pop-up menu
				popup.show(spellsButton, spellsButton.getWidth() - 1, 0);
			}

			/*
			CLASS: SpellsMenuItemAction
			PURPOSE: Schedules the appropriate battle action tasks for when the player
			chooses a spell to cast.
			*/
			private class SpellsMenuItemAction implements ActionListener
			{
				Spell spell; //the spell the player has chosen to cast

				public SpellsMenuItemAction(Spell spellIn)
				{
					spell = spellIn;
				}

				public void actionPerformed(ActionEvent event)
				{	//disable commands during action sequence
					attackButton.setEnabled(false);
					spellsButton.setEnabled(false);
					itemsButton.setEnabled(false);
					fleeButton.setEnabled(false);
					//if the hero is faster than the monster...
					if(hero.getStat(GameCharacter.SPD) >= monster.getStat(GameCharacter.SPD))
					{	//...the hero's action is executed first, the monster's second
						timer.schedule(new HeroCastSpellFirst(spell), 0);
						timer.schedule(new CheckIfMonsterSlain(), 1000);
						timer.schedule(new MonsterActionSecond(), 2000);
						timer.schedule(new EndRound(), 4000);
					}
					else //if the monster is faster than the hero...
					{	//...the monster's action is executed first, the hero's second
						timer.schedule(new MonsterActionFirst(), 0);
						timer.schedule(new CheckIfHeroSlain(), 1000);
						timer.schedule(new HeroCastSpellSecond(spell), 2000);
						timer.schedule(new EndRound(), 4000);
					}
				}
			}
		}

		/*
		CLASS: ItemsAction
		PURPOSE: Displays the hero's inventory in the form of a pop-up menu.
		*/
		private class ItemsAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{	//create pop-up menu and menu items
				JPopupMenu popup = new JPopupMenu();
				JMenuItem[] menuItem = new JMenuItem[10];
				//for each item in inventory...
				for(int i = 0; i < 10; i++)
				{
					menuItem[i] = new JMenuItem(hero.getItem(i).getName() + " x" + hero.getItem(i).getQuantity());
					//if the slot is not empty...
					if(!hero.getItem(i).getName().equals("") && hero.getItem(i).getType() == Item.HEALING)
					{	//...create tool tip for displaying spell description...
						menuItem[i].setToolTipText(hero.getItem(i).getDescription());
						//...add action listener to menu item...
						menuItem[i].addActionListener(new ItemsMenuItemAction(i));
						//...and add menu item to pop-up menu
						popup.add(menuItem[i]);
					}
				}
				//display pop-up menu
				popup.show(itemsButton, itemsButton.getWidth() - 1, 0);
			}

			/*
			CLASS: ItemsMenuItemAction
			PURPOSE: Schedules the appropriate battle action tasks for when the player
			chooses an item to use.
			*/
			private class ItemsMenuItemAction implements ActionListener
			{
				private int inventorySlot; //inventory slot of the item the player has chosen to use

				public ItemsMenuItemAction(int slotIn)
				{
					inventorySlot = slotIn;
				}

				public void actionPerformed(ActionEvent event)
				{	//disable commands during action sequence
					attackButton.setEnabled(false);
					spellsButton.setEnabled(false);
					itemsButton.setEnabled(false);
					fleeButton.setEnabled(false);
					//if the hero is faster than the monster...
					if(hero.getStat(GameCharacter.SPD) >= monster.getStat(GameCharacter.SPD))
					{	//...the hero's action is executed first, the monster's second
						timer.schedule(new HeroUseItemFirst(inventorySlot), 0);
						timer.schedule(new MonsterActionSecond(), 2000);
						timer.schedule(new EndRound(), 4000);
					}
					else //if the monster is faster than the hero...
					{	//...the monster's action is executed first, the hero's second
						timer.schedule(new MonsterActionFirst(), 0);
						timer.schedule(new CheckIfHeroSlain(), 1000);
						timer.schedule(new HeroUseItemSecond(inventorySlot), 2000);
						timer.schedule(new EndRound(), 4000);
					}
				}
			}
		}

		/*
		CLASS: FleeAction
		PURPOSE: Hero attempts to flee from battle. If successful, battle comes to an end.
		If unsuccessful, the appropriate battle action tasks for when the player chooses
		the flee command are scheduled.
		*/
		private class FleeAction implements ActionListener
		{
			public void actionPerformed(ActionEvent event)
			{	//disable commands during action sequence
				attackButton.setEnabled(false);
				spellsButton.setEnabled(false);
				itemsButton.setEnabled(false);
				fleeButton.setEnabled(false);

				//Hero's odds of successfully fleeing are determined by the difference
				//between his level and the monster's. The odds are 50-50 if levels are
				//equal, 51-49 if hero's level is one higher, and so on.
				random = (int) (Math.random() * 100);
				if(random >= (50 - (hero.getLVL() - monster.getLVL())))
				{	//Hero successfully flees from battle
					battleLog.append(hero.getName() + " flees from the " + monster.getName() + ".\n");
                                        //play dungeon theme
					musicClip.stop();
                  			try
					{
						musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																"audio_files/dungeon.mid"));
					}
					catch (IOException exception)
					{
						musicClip = null;
					}
					musicClip.loop();
					try
					{
						soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																  	  "audio_files/stairs.wav"));
					}
					catch (IOException exception)
					{
						soundEffectClip = null;
					}
					soundEffectClip.play();
					timer.schedule(new CloseTask(), 2000);
				}
				else
				{	//Hero unsuccessful in fleeing, monster's action executed
					battleLog.append(hero.getName() + " tries to escape, but the " + monster.getName() + " blocks his path.\n");
					timer.schedule(new MonsterActionFirst(), 2000);
					timer.schedule(new CheckIfHeroSlain(), 3000);
					timer.schedule(new EndRound(), 4000);
				}
			}
		}

		/*
		CLASS: HeroAttackFirst
		PURPOSE: This task is scheduled when the player issues the attack command and the
		hero is faster than the monster.
		*/
		private class HeroAttackFirst extends TimerTask
		{
			public void run()
			{
				int damage; //the amount of damage dealt by the hero's attack

				//The odds of the hero's attack landing successfully are determined by the
				//difference between the hero's ACC and the monster's EVA. The odds are
				//50-50 if they are equal, 51-49 if the hero's ACC is one higher than the
				//monster's EVA, and so on.
				random = (int) (Math.random() * 100);
				if(random >= (50 - (hero.getStat(GameCharacter.ACC) - monster.getStat(GameCharacter.EVA))))
				{	//the hero's attack is successful
					damage = hero.getStat(GameCharacter.ATK) - monster.getStat(GameCharacter.DEF);
					if(damage <= 0) damage = 1; //melee attacks always do at least 1 HP of damage
					monster.setHP(monster.getHP() - damage);
					battleLog.append(hero.getName() + " hits the " + monster.getName() + " for " + damage + " HP.\n");
					//play hit sound effect
					try
					{
						soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																  	  "audio_files/hit.wav"));
					}
					catch (IOException exception)
					{
						soundEffectClip = null;
					}
					soundEffectClip.play();
				}
				else
				{	//the hero's attack is unsuccessful; the monster dodges the attack
					battleLog.append(hero.getName() + " takes a swing at the " + monster.getName() + ", but it evades his attack.\n");
					//play dodge sound effect
					try
					{
						soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																  	  "audio_files/dodge.wav"));
					}
					catch (IOException exception)
					{
						soundEffectClip = null;
					}
					soundEffectClip.play();
				}
				//update caret position in battle log
				try
				{
					battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
				}
				catch(BadLocationException exception)
				{
					System.out.println("Caught exception: " + exception);
				}
			}
		}

		/*
		CLASS: HeroAttackSecond
		PURPOSE: This task is scheduled when the player issues the attack command and the
		hero is slower than the monster.
		*/
		private class HeroAttackSecond extends TimerTask
		{
			public void run()
			{
				if(!battleOver) //this task does nothing if the battle is already over
				{
					int damage; //the amount of damage dealt by the hero's attack

					//The odds of the hero's attack landing successfully are determined by the
					//difference between the hero's ACC and the monster's EVA. The odds are
					//50-50 is they are equal, 51-49 if the hero's ACC one higher than the
					//monster's EVA, and so on.
					random = (int) (Math.random() * 100);
					if(random >= (50 - (hero.getStat(GameCharacter.ACC) - monster.getStat(GameCharacter.EVA))))
					{	//the hero's attack is successful
						damage = hero.getStat(GameCharacter.ATK) - monster.getStat(GameCharacter.DEF);
						if(damage <= 0) damage = 1; //melee attacks always do at least 1 HP of damage
						monster.setHP(monster.getHP() - damage);
						battleLog.append(hero.getName() + " hits the " + monster.getName() + " for " + damage + " HP.\n");
						try
						{
							soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																	  	  "audio_files/hit.wav"));
						}
						catch (IOException exception)
						{
							soundEffectClip = null;
						}
						soundEffectClip.play();
					}
					else
					{	//the hero's attack is unsuccessful; the monster dodges the attack
						battleLog.append(hero.getName() + " takes a swing at the " + monster.getName() + ", but it evades his attack.\n");
						try
						{
							soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																	  	  "audio_files/dodge.wav"));
						}
						catch (IOException exception)
						{
							soundEffectClip = null;
						}
						soundEffectClip.play();
					}
					//update caret position in battle log
					try
					{
						battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					timer.schedule(new CheckIfMonsterSlain(), 1000);
				}
			}
		}

		/*
		CLASS: HeroCastSpellFirst
		PURPOSE: This task is scheduled when the player chooses a spell to cast and the
		hero is faster than the monster.
		*/
		private class HeroCastSpellFirst extends TimerTask
		{
			Spell spell; //the spell the player has chosen to cast

			public HeroCastSpellFirst(Spell spellIn)
			{
				spell = spellIn;
			}

			public void run()
			{	//cast spell
				if(spell.getType() == Spell.HEALING)
				{
					battleLog.append(((HealingSpell) spell).cast(hero, hero) + "\n");
				}
				else if(spell.getType() == Spell.OFFENSIVE)
				{
					battleLog.append(((OffensiveSpell) spell).cast(hero, monster) + "\n");
				}
				else if(spell.getType() == Spell.BUFFING)
				{
					battleLog.append(((BuffingSpell) spell).cast(hero, hero) + "\n");
				}
				else if(spell.getType() == Spell.DEBUFFING)
				{
					battleLog.append(((DebuffingSpell) spell).cast(hero, monster) + "\n");
				}
				//update caret position in battle log
				try
				{
					battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
				}
				catch(BadLocationException exception)
				{
					System.out.println("Caught exception: " + exception);
				}
				//update hero HP and MP textfields
				hpField.setText(hero.getHP() + "/");
				mpField.setText(hero.getMP() + "/");
			}
		}

		/*
		CLASS: HeroCastSpellSecond
		PURPOSE: This task is scheduled when the player chooses a spell to cast and the
		hero is slower than the monster.
		*/
		private class HeroCastSpellSecond extends TimerTask
		{
			Spell spell; //the spell the player has chosen to cast

			public HeroCastSpellSecond(Spell spellIn)
			{
				spell = spellIn;
			}

			public void run()
			{
				if(!battleOver) //this task does nothing if the battle is already over
				{	//cast spell
					if(spell.getType() == Spell.HEALING)
					{
						battleLog.append(((HealingSpell) spell).cast(hero, hero) + "\n");
					}
					else if(spell.getType() == Spell.OFFENSIVE)
					{
						battleLog.append(((OffensiveSpell) spell).cast(hero, monster) + "\n");
					}
					else if(spell.getType() == Spell.BUFFING)
					{
						battleLog.append(((BuffingSpell) spell).cast(hero, hero) + "\n");
					}
					else if(spell.getType() == Spell.DEBUFFING)
					{
						battleLog.append(((DebuffingSpell) spell).cast(hero, monster) + "\n");
					}
					//update caret postion in battle log
					try
					{
						battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					//update hero HP and MP textfields
					hpField.setText(hero.getHP() + "/");
					mpField.setText(hero.getMP() + "/");

					timer.schedule(new CheckIfMonsterSlain(), 1000);
				}
			}
		}

		/*
		CLASS: HeroUseItemFirst
		PURPOSE: This task is scheduled when the player chooses an item to use and the
		hero is faster than the monster.
		*/
		private class HeroUseItemFirst extends TimerTask
		{
			private int inventorySlot; //the inventory slot of the item the player has chosen to use
			private Item item; //the item the player has chosen to use

			public HeroUseItemFirst(int slotIn)
			{
				inventorySlot = slotIn;
				item = hero.getItem(inventorySlot);
			}

			public void run()
			{	//use item
				battleLog.append(((HealingItem) item).use(hero) + "\n");
				if(item.getQuantity() > 1)
				{	//decrement quantity
					item.setQuantity(item.getQuantity() - 1);
				}
				else //if item's quantity now 0...
				{	//remove item from inventory completely...
					hero.setItem((Equipment) itemTable.getItem(""), inventorySlot);
				}
				//update caret position in battle log
				try
				{
					battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
				}
				catch(BadLocationException exception)
				{
					System.out.println("Caught exception: " + exception);
				}
				//update hero HP and MP textfields
				hpField.setText(hero.getHP() + "/");
				mpField.setText(hero.getMP() + "/");
			}
		}

		/*
		CLASS: HeroUseItemSecond
		PURPOSE: This task is scheduled when the player chooses an item to use and the
		hero is slower than the monster.
		*/
		private class HeroUseItemSecond extends TimerTask
		{
			private int inventorySlot; //the inventory slot of the item the player has chosen to use
			private Item item; //the item the player has chosen to use

			public HeroUseItemSecond(int slotIn)
			{
				inventorySlot = slotIn;
				item = hero.getItem(inventorySlot);
			}

			public void run()
			{
				if(!battleOver) //this task does nothing if the battle is already over
				{	//use item
					battleLog.append(((HealingItem) item).use(hero) + "\n");
					if(item.getQuantity() > 1)
					{	//decrement quantity
						item.setQuantity(item.getQuantity() - 1);
					}
					else //if item's quantity now 0...
					{	//remove item from inventory completely...
						hero.setItem((Equipment) itemTable.getItem(""), inventorySlot);
					}
					//update caret position in battle log
					try
					{
						battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					//update hero HP and MP textfields
					hpField.setText(hero.getHP() + "/");
					mpField.setText(hero.getMP() + "/");
				}
			}
		}

		/*
		CLASS: MonsterActionFirst
		PURPOSE: This task is scheduled when the monster is faster than the hero.
		*/
		private class MonsterActionFirst extends TimerTask
		{
			public void run()
			{	//randomly choose an action to perform
				Spell monsterAction = monster.chooseRandomAction();
				//perform action
				if(monsterAction.getType() == Spell.HEALING)
				{
					battleLog.append(((HealingSpell) monsterAction).cast(monster, monster) + "\n");
				}
				else if(monsterAction.getType() == Spell.OFFENSIVE)
				{
					battleLog.append(((OffensiveSpell) monsterAction).cast(monster, hero) + "\n");
				}
				else if(monsterAction.getType() == Spell.BUFFING)
				{
					battleLog.append(((BuffingSpell) monsterAction).cast(monster, monster) + "\n");
				}
				else if(monsterAction.getType() == Spell.DEBUFFING)
				{
					battleLog.append(((DebuffingSpell) monsterAction).cast(monster, hero) + "\n");
				}
				else if(monsterAction.getType() == Spell.MONSTER_MELEE)
				{
					battleLog.append(((MonsterMelee) monsterAction).attack(monster, hero) + "\n");
				}
				//update caret position in battle log
				try
				{
					battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
				}
				catch(BadLocationException exception)
				{
					System.out.println("Caught exception: " + exception);
				}
				//update hero HP textfield
				hpField.setText(hero.getHP() + "/");
			}
		}

		/*
		CLASS: MonsterActionSecond
		PURPOSE: This task is scheduled when the monster is slower than the hero.
		*/
		private class MonsterActionSecond extends TimerTask
		{
			public void run()
			{
				if(!battleOver) //this task does nothing if the battle is already over
				{	//randomly choose an action to perform
					Spell monsterAction = monster.chooseRandomAction();
					//perform action
					if(monsterAction.getType() == Spell.HEALING)
					{
						battleLog.append(((HealingSpell) monsterAction).cast(monster, monster) + "\n");
					}
					else if(monsterAction.getType() == Spell.OFFENSIVE)
					{
						battleLog.append(((OffensiveSpell) monsterAction).cast(monster, hero) + "\n");
					}
					else if(monsterAction.getType() == Spell.BUFFING)
					{
						battleLog.append(((BuffingSpell) monsterAction).cast(monster, monster) + "\n");
					}
					else if(monsterAction.getType() == Spell.DEBUFFING)
					{
						battleLog.append(((DebuffingSpell) monsterAction).cast(monster, hero) + "\n");
					}
					else if(monsterAction.getType() == Spell.MONSTER_MELEE)
					{
						battleLog.append(((MonsterMelee) monsterAction).attack(monster, hero) + "\n");
					}
					//update caret position in battle log
					try
					{
						battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					//update hero HP textfield
					hpField.setText(hero.getHP() + "/");

					timer.schedule(new CheckIfHeroSlain(), 1000);
				}
			}
		}

		/*
		CLASS: CheckIfMonsterSlain
		PURPOSE: Determines if the hero has slain the monster. If so, the hero is awarded
		EXP and gold and the battle ends.
		*/
		private class CheckIfMonsterSlain extends TimerTask
		{
			public void run()
			{
				if(monster.getHP() == 0)
				{	//remove monster image
					monster.setImageFileName("blank.gif");
					battlePanel.repaint();
					//award the hero with EXP and gold
					battleLog.append("The " + monster.getName() + " is slain.\n");
					battleLog.append("The " + monster.getName() + hero.addGold(monster.getGold()));
					battleLog.append(hero.addEXP(monster.getEXP()));
					//update caret position in battle log
					try
					{
						battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					//end battle
					battleOver = true;
                                        //play dungeon theme
					musicClip.stop();
					try
					{
						musicClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																"audio_files/dungeon.mid"));
					}
					catch (IOException exception)
					{
						musicClip = null;
					}
					musicClip.loop();
                                        //play victory sound effect
					try
					{
						soundEffectClip = Applet.newAudioClip(new URL("file:" + System.getProperty("user.dir") + "/" +
																  	  "audio_files/win_battle.wav"));
					}
					catch (IOException exception)
					{
						soundEffectClip = null;
					}
					soundEffectClip.play();
					timer.schedule(new CloseTask(), 5000);
				}
			}
		}

		/*
		CLASS: CheckIfHeroSlain
		PURPOSE: Determines if the monster has slain the hero. If so, the battle ends and
		game over is set.
		*/
		private class CheckIfHeroSlain extends TimerTask
		{
			public void run()
			{
				if(hero.getHP() == 0)
				{
					battleLog.append(hero.getName() + " is slain.\n");
					//update the caret position in the battle log
					try
					{
						battleLog.setCaretPosition(battleLog.getLineEndOffset(battleLog.getLineCount() - 1));
					}
					catch(BadLocationException exception)
					{
						System.out.println("Caught exception: " + exception);
					}
					//end battle and end game
					battleOver = true;
					gameOver = true;
					musicClip.stop();
					timer.schedule(new CloseTask(), 5000);
				}
			}
		}

		/*
		CLASS: EndRound
		PURPOSE: Enables the command buttons so that the player may issue his next command.
		*/
		private class EndRound extends TimerTask
		{
			public void run()
			{
				if(!battleOver) //this task does nothing if the battle is already over
				{
					attackButton.setEnabled(true);
					if(!hero.spellListIsEmpty())
					{
						spellsButton.setEnabled(true);
					}
					if(!hero.hasNoHealingItems())
					{
						itemsButton.setEnabled(true);
					}
					fleeButton.setEnabled(true);
				}
			}
		}

		/*
		CLASS: CloseTask
		PURPOSE: Closes battle window
		*/
		private class CloseTask extends TimerTask
		{
			public void run()
			{	//reset any buffed or debuffed stats back to normal
				for(int i = 0; i < 9; i++)
				{
					hero.setStat(startStat[i], i);
				}
				//close timer and battle window
				timer.cancel();
				dispose();
			}
		}
	}
}