// Much of this code is derived from poker.eval (look for it on sourceforge.net).
// This library is covered by the GPL Gnu license. See http://www.gnu.org/licenses/old-licenses/gpl-2.0.html 
// for more information on this license.

// This code is a very fast, native C# Texas Holdem mask evaluator (containing no interop or unsafe code). 
// This code can enumarate 35 million 5 card hands per second and 29 million 7 card hands per second on my desktop machine.
// That's not nearly as fast as the heavily macro-ed poker.eval C library. However, this implementation is
// in roughly the same ballpark for speed and is quite usable in C#.

// The speed ups are mostly table driven. That means that there are several very large tables included in this file. 
// The code is divided up into several files they are:
//      HandEvaluator.cs - base mask evaluator
//      HandIterator.cs - methods that support IEnumerable and methods that validate the mask evaluator
//      HandAnalysis.cs - methods to aid in analysis of Texas Holdem Hands.
//      PocketHands.cs - a class to manipulate pocket hands.
//      PocketQueryParser - a parser used to interprete pocket mask query language statements.

// Written (ported) by Keith Rule - Sept 2005, updated May 2007

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;

namespace Private.Logic.Poker.Probability
{
    /// <summary>
    /// Represents a set of pocket hands and operations that
    /// can be applied to them.
    /// </summary>
    public partial class PocketHands : IEnumerable<ulong>
    {
        #region Fields
        /// <summary>
        /// Contains a list of the masks contains in this hole card collection.
        /// </summary>
        private List<ulong> list = new List<ulong>();
        #endregion

        #region PreCalculated Tables

        #region Pocket Grouping Mask Table

        /// <summary>
        /// Sklansky groupings 
        /// </summary>
        internal static readonly ulong[][] _PocketGroupings = {
            new ulong [] {
	            0x8004000000000, 0x8000002000000, 0x8000000001000, 0x4002000000, 0x4000001000, 
	            0x2001000, 0x4002000000000, 0x4000001000000, 0x4000000000800, 0x2001000000, 
	            0x2000000800, 0x1000800, 0x2001000000000, 0x2000000800000, 0x2000000000400, 
	            0x1000800000, 0x1000000400, 0x800400, 0x1000800000000, 0x1000000400000, 
	            0x1000000000200, 0x800400000, 0x800000200, 0x400200, 0xc000000000000, 
	            0x6000000000, 0x3000000, 0x1800
            },
            new ulong [] {
	            0x800400000000, 0x800000200000, 0x800000000100, 0x400200000, 0x400000100, 
	            0x200100, 0xa000000000000, 0x5000000000, 0x2800000, 0x1400, 
	            0x9000000000000, 0x4800000000, 0x2400000, 0x1200, 0x6000000000000, 
	            0x3000000000, 0x1800000, 0xc00, 0x8002000000000, 0x8000001000000, 
	            0x8000000000800, 0x4004000000000, 0x4000002000000, 0x4000000001000, 0x4001000000, 
	            0x4000000800, 0x2002000000, 0x2000001000, 0x2000800, 0x1001000
	        },
            new ulong [] {
	            0x400200000000, 0x400000100000, 0x400000000080, 0x200100000, 0x200000080, 
	            0x100080, 0x1800000000000, 0xc00000000, 0x600000, 0x300, 
	            0x3000000000000, 0x1800000000, 0xc00000, 0x600, 0x5000000000000, 
	            0x2800000000, 0x1400000, 0xa00, 0x8800000000000, 0x4400000000, 
	            0x2200000, 0x1100, 0x8001000000000, 0x8000000800000, 0x8000000000400, 
	            0x2004000000000, 0x2000002000000, 0x2000000001000, 0x4000800000, 0x4000000400, 
	            0x1002000000, 0x1000001000, 0x2000400, 0x801000
            },
            new ulong [] {
	            0xc00000000000, 0x600000000, 0x300000, 0x180, 0x4001000000000, 
	            0x4000000800000, 0x4000000000400, 0x2002000000000, 0x2000001000000, 0x2000000000800, 
	            0x2000800000, 0x2000000400, 0x1001000000, 0x1000000800, 0x1000400, 
	            0x800800, 0x200100000000, 0x200000080000, 0x200000000040, 0x100080000, 
	            0x100000040, 0x80040, 0x2800000000000, 0x1400000000, 0xa00000, 
	            0x500, 0x600000000000, 0x300000000, 0x180000, 0xc0, 
	            0x1400000000000, 0xa00000000, 0x500000, 0x280, 0x8000800000000, 
	            0x8000000400000, 0x8000000000200, 0x1004000000000, 0x1000002000000, 0x1000000001000, 
	            0x4000400000, 0x4000000200, 0x802000000, 0x800001000, 0x2000200, 
	            0x401000, 0x4800000000000, 0x2400000000, 0x1200000, 0x900 
	        },
            new ulong [] {
	            0x100080000000, 0x100000040000, 0x100000000020, 0x80040000, 0x80000020, 
	            0x40020, 0x300000000000, 0x180000000, 0xc0000, 0x60, 
	            0x2400000000000, 0x1200000000, 0x900000, 0x480, 0xa00000000000, 
	            0x500000000, 0x280000, 0x140, 0x4000800000000, 0x4000000400000, 
	            0x4000000000200, 0x1002000000000, 0x1000001000000, 0x1000000000800, 0x2000400000, 
	            0x2000000200, 0x801000000, 0x800000800, 0x1000200, 0x400800, 
	            0x2000800000000, 0x2000000400000, 0x2000000000200, 0x1001000000000, 0x1000000800000, 
	            0x1000000000400, 0x1000400000, 0x1000000200, 0x800800000, 0x800000400, 
	            0x800200, 0x400400, 0x1000400000000, 0x1000000200000, 0x1000000000100, 
	            0x800800000000, 0x800000400000, 0x800000000200, 0x800200000, 0x800000100, 
	            0x400400000, 0x400000200, 0x400100, 0x200200, 0x180000000000, 
	            0xc0000000, 0x60000, 0x30, 0x500000000000, 0x280000000, 
	            0x140000, 0xa0, 0x8400000000000, 0x4200000000, 0x2100000, 
	            0x1080, 0x8200000000000, 0x4100000000, 0x2080000, 0x1040, 
	            0x8100000000000, 0x4080000000, 0x2040000, 0x1020, 0x8080000000000, 
	            0x4040000000, 0x2020000, 0x1010, 0x8040000000000, 0x4020000000, 
	            0x2010000, 0x1008, 0x8020000000000, 0x4010000000, 0x2008000, 
	            0x1004, 0x8010000000000, 0x4008000000, 0x2004000, 0x1002, 
	            0x8008000000000, 0x4004000000, 0x2002000, 0x1001, 0xc0000000000, 
	            0x60000000, 0x30000, 0x18
            },
            new ulong [] {
	            0x80040000000, 0x80000020000, 0x80000000010, 0x40020000, 0x40000010, 
	            0x20010, 0x8000400000000, 0x8000000200000, 0x8000000000100, 0x804000000000, 
	            0x800002000000, 0x800000001000, 0x4000200000, 0x4000000100, 0x402000000, 
	            0x400001000, 0x2000100, 0x201000, 0x40020000000, 0x40000010000, 
	            0x40000000008, 0x20010000, 0x20000008, 0x10008, 0x280000000000, 
	            0x140000000, 0xa0000, 0x50, 0x4000400000000, 0x4000000200000, 
	            0x4000000000100, 0x802000000000, 0x800001000000, 0x800000000800, 0x2000200000, 
	            0x2000000100, 0x401000000, 0x400000800, 0x1000100, 0x200800, 
	            0x2000400000000, 0x2000000200000, 0x2000000000100, 0x801000000000, 0x800000800000, 
	            0x800000000400, 0x1000200000, 0x1000000100, 0x400800000, 0x400000400, 
	            0x800100, 0x200400, 0x60000000000, 0x30000000, 0x18000, 
	            0xc, 0x4400000000000, 0x2200000000, 0x1100000, 0x880, 
	            0x1200000000000, 0x900000000, 0x480000, 0x240
            },
            new ulong [] {
	            0x20010000000, 0x20000008000, 0x20000000004, 0x10008000, 0x10000004, 
	            0x8004, 0x1000200000000, 0x1000000100000, 0x1000000000080, 0x400800000000, 
	            0x400000400000, 0x400000000200, 0x800100000, 0x800000080, 0x200400000, 
	            0x200000200, 0x400080, 0x100200, 0x30000000000, 0x18000000, 
	            0xc000, 0x6, 0x140000000000, 0xa0000000, 0x50000, 
	            0x28, 0x800200000000, 0x800000100000, 0x800000000080, 0x400400000000, 
	            0x400000200000, 0x400000000100, 0x400100000, 0x400000080, 0x200200000, 
	            0x200000100, 0x200080, 0x100100, 0x10008000000, 0x10000004000, 
	            0x10000000002, 0x8004000, 0x8000002, 0x4002, 0x400100000000, 
	            0x400000080000, 0x400000000040, 0x200200000000, 0x200000100000, 0x200000000080, 
	            0x200080000, 0x200000040, 0x100100000, 0x100000080, 0x100040, 
	            0x80080, 0xa0000000000, 0x50000000, 0x28000, 0x14, 
	            0x8004000000, 0x8000002000, 0x8000000001, 0x4002000, 0x4000001, 
	            0x2001, 0x4200000000000, 0x2100000000, 0x1080000, 0x840, 
	            0x4100000000000, 0x2080000000, 0x1040000, 0x820, 0x4080000000000, 
	            0x2040000000, 0x1020000, 0x810, 0x4040000000000, 0x2020000000, 
	            0x1010000, 0x808, 0x4020000000000, 0x2010000000, 0x1008000, 
	            0x804, 0x4010000000000, 0x2008000000, 0x1004000, 0x802, 
	            0x4008000000000, 0x2004000000, 0x1002000, 0x801, 0x2200000000000, 
	            0x1100000000, 0x880000, 0x440
            },
            new ulong [] {
	            0x200080000000, 0x200000040000, 0x200000000020, 0x100100000000, 0x100000080000, 
	            0x100000000040, 0x100040000, 0x100000020, 0x80080000, 0x80000040, 
	            0x80020, 0x40040, 0x50000000000, 0x28000000, 0x14000, 
	            0xa, 0x8000200000000, 0x8000000100000, 0x8000000000080, 0x404000000000, 
	            0x400002000000, 0x400000001000, 0x4000100000, 0x4000000080, 0x202000000, 
	            0x200001000, 0x2000080, 0x101000, 0x2000200000000, 0x2000000100000, 
	            0x2000000000080, 0x401000000000, 0x400000800000, 0x400000000400, 0x1000100000, 
	            0x1000000080, 0x200800000, 0x200000400, 0x800080, 0x100400, 
	            0x100040000000, 0x100000020000, 0x100000000010, 0x80080000000, 0x80000040000, 
	            0x80000000020, 0x80020000, 0x80000010, 0x40040000, 0x40000020, 
	            0x40010, 0x20020, 0x28000000000, 0x14000000, 0xa000, 
	            0x5, 0x18000000000, 0xc000000, 0x6000, 0x3, 
	            0x480000000000, 0x240000000, 0x120000, 0x90, 0x240000000000, 
	            0x120000000, 0x90000, 0x48, 0x1000100000000, 0x1000000080000, 
	            0x1000000000040, 0x200800000000, 0x200000400000, 0x200000000200, 0x800080000, 
	            0x800000040, 0x100400000, 0x100000200, 0x400040, 0x80200, 
	            0x1100000000000, 0x880000000, 0x440000, 0x220, 0x80020000000, 
	            0x80000010000, 0x80000000008, 0x40040000000, 0x40000020000, 0x40000000010, 
	            0x40010000, 0x40000008, 0x20020000, 0x20000010, 0x20008, 
	            0x10010, 0x40010000000, 0x40000008000, 0x40000000004, 0x20020000000, 
	            0x20000010000, 0x20000000008, 0x20008000, 0x20000004, 0x10010000, 
	            0x10000008, 0x10004, 0x8008, 0x4000200000000, 0x4000000100000, 
	            0x4000000000080, 0x402000000000, 0x400001000000, 0x400000000800, 0x2000100000, 
	            0x2000000080, 0x201000000, 0x200000800, 0x1000080, 0x100800, 
	            0x800100000000, 0x800000080000, 0x800000000040, 0x200400000000, 0x200000200000, 
	            0x200000000100, 0x400080000, 0x400000040, 0x100200000, 0x100000100, 
	            0x200040, 0x80100
            },
            new ulong [] {
	            0x8000100000000, 0x8000000080000, 0x8000000000040, 0x204000000000, 0x200002000000, 
	            0x200000001000, 0x4000080000, 0x4000000040, 0x102000000, 0x100001000, 
	            0x2000040, 0x81000, 0x8000080000000, 0x8000000040000, 0x8000000000020, 
	            0x104000000000, 0x100002000000, 0x100000001000, 0x4000040000, 0x4000000020, 
	            0x82000000, 0x80001000, 0x2000020, 0x41000, 0x8000040000000, 
	            0x8000000020000, 0x8000000000010, 0x84000000000, 0x80002000000, 0x80000001000, 
	            0x4000020000, 0x4000000010, 0x42000000, 0x40001000, 0x2000010, 
	            0x21000, 0x8000020000000, 0x8000000010000, 0x8000000000008, 0x44000000000, 
	            0x40002000000, 0x40000001000, 0x4000010000, 0x4000000008, 0x22000000, 
	            0x20001000, 0x2000008, 0x11000, 0x8000010000000, 0x8000000008000, 
	            0x8000000000004, 0x24000000000, 0x20002000000, 0x20000001000, 0x4000008000, 
	            0x4000000004, 0x12000000, 0x10001000, 0x2000004, 0x9000, 
	            0x8000008000000, 0x8000000004000, 0x8000000000002, 0x14000000000, 0x10002000000, 
	            0x10000001000, 0x4000004000, 0x4000000002, 0xa000000, 0x8001000, 
	            0x2000002, 0x5000, 0x8000004000000, 0x8000000002000, 0x8000000000001, 
	            0xc000000000, 0x8002000000, 0x8000001000, 0x4000002000, 0x4000000001, 
	            0x6000000, 0x4001000, 0x2000001, 0x3000, 0x4000100000000, 
	            0x4000000080000, 0x4000000000040, 0x202000000000, 0x200001000000, 0x200000000800, 
	            0x2000080000, 0x2000000040, 0x101000000, 0x100000800, 0x1000040, 
	            0x80800, 0x4000080000000, 0x4000000040000, 0x4000000000020, 0x102000000000, 
	            0x100001000000, 0x100000000800, 0x2000040000, 0x2000000020, 0x81000000, 
	            0x80000800, 0x1000020, 0x40800, 0x4000040000000, 0x4000000020000, 
	            0x4000000000010, 0x82000000000, 0x80001000000, 0x80000000800, 0x2000020000, 
	            0x2000000010, 0x41000000, 0x40000800, 0x1000010, 0x20800, 
	            0x4000020000000, 0x4000000010000, 0x4000000000008, 0x42000000000, 0x40001000000, 
	            0x40000000800, 0x2000010000, 0x2000000008, 0x21000000, 0x20000800, 
	            0x1000008, 0x10800, 0x4000010000000, 0x4000000008000, 0x4000000000004, 
	            0x22000000000, 0x20001000000, 0x20000000800, 0x2000008000, 0x2000000004, 
	            0x11000000, 0x10000800, 0x1000004, 0x8800, 0x4000008000000, 
	            0x4000000004000, 0x4000000000002, 0x12000000000, 0x10001000000, 0x10000000800, 
	            0x2000004000, 0x2000000002, 0x9000000, 0x8000800, 0x1000002, 
	            0x4800, 0x4000004000000, 0x4000000002000, 0x4000000000001, 0xa000000000, 
	            0x8001000000, 0x8000000800, 0x2000002000, 0x2000000001, 0x5000000, 
	            0x4000800, 0x1000001, 0x2800, 0x2000100000000, 0x2000000080000, 
	            0x2000000000040, 0x201000000000, 0x200000800000, 0x200000000400, 0x1000080000, 
	            0x1000000040, 0x100800000, 0x100000400, 0x800040, 0x80400, 
	            0x2100000000000, 0x1080000000, 0x840000, 0x420, 0x2000080000000, 
	            0x2000000040000, 0x2000000000020, 0x101000000000, 0x100000800000, 0x100000000400, 
	            0x1000040000, 0x1000000020, 0x80800000, 0x80000400, 0x800020, 
	            0x40400, 0x2080000000000, 0x1040000000, 0x820000, 0x410, 
	            0x2000040000000, 0x2000000020000, 0x2000000000010, 0x81000000000, 0x80000800000, 
	            0x80000000400, 0x1000020000, 0x1000000010, 0x40800000, 0x40000400, 
	            0x800010, 0x20400, 0x2040000000000, 0x1020000000, 0x810000, 
	            0x408, 0x2000020000000, 0x2000000010000, 0x2000000000008, 0x41000000000, 
	            0x40000800000, 0x40000000400, 0x1000010000, 0x1000000008, 0x20800000, 
	            0x20000400, 0x800008, 0x10400, 0x2020000000000, 0x1010000000, 
	            0x808000, 0x404, 0x2000010000000, 0x2000000008000, 0x2000000000004, 
	            0x21000000000, 0x20000800000, 0x20000000400, 0x1000008000, 0x1000000004, 
	            0x10800000, 0x10000400, 0x800004, 0x8400, 0x2010000000000, 
	            0x1008000000, 0x804000, 0x402, 0x2000008000000, 0x2000000004000, 
	            0x2000000000002, 0x11000000000, 0x10000800000, 0x10000000400, 0x1000004000, 
	            0x1000000002, 0x8800000, 0x8000400, 0x800002, 0x4400, 
	            0x2008000000000, 0x1004000000, 0x802000, 0x401, 0x2000004000000, 
	            0x2000000002000, 0x2000000000001, 0x9000000000, 0x8000800000, 0x8000000400, 
	            0x1000002000, 0x1000000001, 0x4800000, 0x4000400, 0x800001, 
	            0x2400, 0x1000080000000, 0x1000000040000, 0x1000000000020, 0x100800000000, 
	            0x100000400000, 0x100000000200, 0x800040000, 0x800000020, 0x80400000, 
	            0x80000200, 0x400020, 0x40200, 0x1080000000000, 0x840000000, 
	            0x420000, 0x210, 0x1000040000000, 0x1000000020000, 0x1000000000010, 
	            0x80800000000, 0x80000400000, 0x80000000200, 0x800020000, 0x800000010, 
	            0x40400000, 0x40000200, 0x400010, 0x20200, 0x1040000000000, 
	            0x820000000, 0x410000, 0x208, 0x1000020000000, 0x1000000010000, 
	            0x1000000000008, 0x40800000000, 0x40000400000, 0x40000000200, 0x800010000, 
	            0x800000008, 0x20400000, 0x20000200, 0x400008, 0x10200, 
	            0x1020000000000, 0x810000000, 0x408000, 0x204, 0x1000010000000, 
	            0x1000000008000, 0x1000000000004, 0x20800000000, 0x20000400000, 0x20000000200, 
	            0x800008000, 0x800000004, 0x10400000, 0x10000200, 0x400004, 
	            0x8200, 0x1010000000000, 0x808000000, 0x404000, 0x202, 
	            0x1000008000000, 0x1000000004000, 0x1000000000002, 0x10800000000, 0x10000400000, 
	            0x10000000200, 0x800004000, 0x800000002, 0x8400000, 0x8000200, 
	            0x400002, 0x4200, 0x1008000000000, 0x804000000, 0x402000, 
	            0x201, 0x1000004000000, 0x1000000002000, 0x1000000000001, 0x8800000000, 
	            0x8000400000, 0x8000000200, 0x800002000, 0x800000001, 0x4400000, 
	            0x4000200, 0x400001, 0x2200, 0x900000000000, 0x480000000, 
	            0x240000, 0x120, 0x800080000000, 0x800000040000, 0x800000000020, 
	            0x100400000000, 0x100000200000, 0x100000000100, 0x400040000, 0x400000020, 
	            0x80200000, 0x80000100, 0x200020, 0x40100, 0x880000000000, 
	            0x440000000, 0x220000, 0x110, 0x800040000000, 0x800000020000, 
	            0x800000000010, 0x80400000000, 0x80000200000, 0x80000000100, 0x400020000, 
	            0x400000010, 0x40200000, 0x40000100, 0x200010, 0x20100, 
	            0x840000000000, 0x420000000, 0x210000, 0x108, 0x800020000000, 
	            0x800000010000, 0x800000000008, 0x40400000000, 0x40000200000, 0x40000000100, 
	            0x400010000, 0x400000008, 0x20200000, 0x20000100, 0x200008, 
	            0x10100, 0x820000000000, 0x410000000, 0x208000, 0x104, 
	            0x800010000000, 0x800000008000, 0x800000000004, 0x20400000000, 0x20000200000, 
	            0x20000000100, 0x400008000, 0x400000004, 0x10200000, 0x10000100, 
	            0x200004, 0x8100, 0x810000000000, 0x408000000, 0x204000, 
	            0x102, 0x800008000000, 0x800000004000, 0x800000000002, 0x10400000000, 
	            0x10000200000, 0x10000000100, 0x400004000, 0x400000002, 0x8200000, 
	            0x8000100, 0x200002, 0x4100, 0x808000000000, 0x404000000, 
	            0x202000, 0x101, 0x800004000000, 0x800000002000, 0x800000000001, 
	            0x8400000000, 0x8000200000, 0x8000000100, 0x400002000, 0x400000001, 
	            0x4200000, 0x4000100, 0x200001, 0x2100, 0x400080000000, 
	            0x400000040000, 0x400000000020, 0x100200000000, 0x100000100000, 0x100000000080, 
	            0x200040000, 0x200000020, 0x80100000, 0x80000080, 0x100020, 
	            0x40080, 0x400040000000, 0x400000020000, 0x400000000010, 0x80200000000, 
	            0x80000100000, 0x80000000080, 0x200020000, 0x200000010, 0x40100000, 
	            0x40000080, 0x100010, 0x20080, 0x440000000000, 0x220000000, 
	            0x110000, 0x88, 0x400020000000, 0x400000010000, 0x400000000008, 
	            0x40200000000, 0x40000100000, 0x40000000080, 0x200010000, 0x200000008, 
	            0x20100000, 0x20000080, 0x100008, 0x10080, 0x420000000000, 
	            0x210000000, 0x108000, 0x84, 0x400010000000, 0x400000008000, 
	            0x400000000004, 0x20200000000, 0x20000100000, 0x20000000080, 0x200008000, 
	            0x200000004, 0x10100000, 0x10000080, 0x100004, 0x8080, 
	            0x410000000000, 0x208000000, 0x104000, 0x82, 0x400008000000, 
	            0x400000004000, 0x400000000002, 0x10200000000, 0x10000100000, 0x10000000080, 
	            0x200004000, 0x200000002, 0x8100000, 0x8000080, 0x100002, 
	            0x4080, 0x408000000000, 0x204000000, 0x102000, 0x81, 
	            0x400004000000, 0x400000002000, 0x400000000001, 0x8200000000, 0x8000100000, 
	            0x8000000080, 0x200002000, 0x200000001, 0x4100000, 0x4000080, 
	            0x100001, 0x2080, 0x200040000000, 0x200000020000, 0x200000000010, 
	            0x80100000000, 0x80000080000, 0x80000000040, 0x100020000, 0x100000010, 
	            0x40080000, 0x40000040, 0x80010, 0x20040, 0x200020000000, 
	            0x200000010000, 0x200000000008, 0x40100000000, 0x40000080000, 0x40000000040, 
	            0x100010000, 0x100000008, 0x20080000, 0x20000040, 0x80008, 
	            0x10040, 0x220000000000, 0x110000000, 0x88000, 0x44, 
	            0x200010000000, 0x200000008000, 0x200000000004, 0x20100000000, 0x20000080000, 
	            0x20000000040, 0x100008000, 0x100000004, 0x10080000, 0x10000040, 
	            0x80004, 0x8040, 0x210000000000, 0x108000000, 0x84000, 
	            0x42, 0x200008000000, 0x200000004000, 0x200000000002, 0x10100000000, 
	            0x10000080000, 0x10000000040, 0x100004000, 0x100000002, 0x8080000, 
	            0x8000040, 0x80002, 0x4040, 0x208000000000, 0x104000000, 
	            0x82000, 0x41, 0x200004000000, 0x200000002000, 0x200000000001, 
	            0x8100000000, 0x8000080000, 0x8000000040, 0x100002000, 0x100000001, 
	            0x4080000, 0x4000040, 0x80001, 0x2040, 0x100020000000, 
	            0x100000010000, 0x100000000008, 0x40080000000, 0x40000040000, 0x40000000020, 
	            0x80010000, 0x80000008, 0x20040000, 0x20000020, 0x40008, 
	            0x10020, 0x120000000000, 0x90000000, 0x48000, 0x24, 
	            0x100010000000, 0x100000008000, 0x100000000004, 0x20080000000, 0x20000040000, 
	            0x20000000020, 0x80008000, 0x80000004, 0x10040000, 0x10000020, 
	            0x40004, 0x8020, 0x110000000000, 0x88000000, 0x44000, 
	            0x22, 0x100008000000, 0x100000004000, 0x100000000002, 0x10080000000, 
	            0x10000040000, 0x10000000020, 0x80004000, 0x80000002, 0x8040000, 
	            0x8000020, 0x40002, 0x4020, 0x108000000000, 0x84000000, 
	            0x42000, 0x21, 0x100004000000, 0x100000002000, 0x100000000001, 
	            0x8080000000, 0x8000040000, 0x8000000020, 0x80002000, 0x80000001, 
	            0x4040000, 0x4000020, 0x40001, 0x2020, 0x80010000000, 
	            0x80000008000, 0x80000000004, 0x20040000000, 0x20000020000, 0x20000000010, 
	            0x40008000, 0x40000004, 0x10020000, 0x10000010, 0x20004, 
	            0x8010, 0x90000000000, 0x48000000, 0x24000, 0x12, 
	            0x80008000000, 0x80000004000, 0x80000000002, 0x10040000000, 0x10000020000, 
	            0x10000000010, 0x40004000, 0x40000002, 0x8020000, 0x8000010, 
	            0x20002, 0x4010, 0x88000000000, 0x44000000, 0x22000, 
	            0x11, 0x80004000000, 0x80000002000, 0x80000000001, 0x8040000000, 
	            0x8000020000, 0x8000000010, 0x40002000, 0x40000001, 0x4020000, 
	            0x4000010, 0x20001, 0x2010, 0x40008000000, 0x40000004000, 
	            0x40000000002, 0x10020000000, 0x10000010000, 0x10000000008, 0x20004000, 
	            0x20000002, 0x8010000, 0x8000008, 0x10002, 0x4008, 
	            0x48000000000, 0x24000000, 0x12000, 0x9, 0x40004000000, 
	            0x40000002000, 0x40000000001, 0x8020000000, 0x8000010000, 0x8000000008, 
	            0x20002000, 0x20000001, 0x4010000, 0x4000008, 0x10001, 
	            0x2008, 0x20008000000, 0x20000004000, 0x20000000002, 0x10010000000, 
	            0x10000008000, 0x10000000004, 0x10004000, 0x10000002, 0x8008000, 
	            0x8000004, 0x8002, 0x4004, 0x20004000000, 0x20000002000, 
	            0x20000000001, 0x8010000000, 0x8000008000, 0x8000000004, 0x10002000, 
	            0x10000001, 0x4008000, 0x4000004, 0x8001, 0x2004, 
	            0x10004000000, 0x10000002000, 0x10000000001, 0x8008000000, 0x8000004000, 
	            0x8000000002, 0x8002000, 0x8000001, 0x4004000, 0x4000002, 
	            0x4001, 0x2002
            }
        };

        #endregion

        #region Connected Table

        /// <summary>
        /// The card masked of pocket card sets that are connected such as 23o
        /// </summary>
        static internal readonly ulong[] _connectedTable = 
        {
            0xc000000000000, 0x8008000000000, 0x8002000000000, 0x8000004000000, 0x8000001000000, 
            0x8000000002000, 0x8000000000800, 0x8000000000001, 0x6000000000000, 0x4004000000000, 
            0x4001000000000, 0x4000002000000, 0x4000000800000, 0x4000000001000, 0x4000000000400, 
            0x3000000000000, 0x2002000000000, 0x2000800000000, 0x2000001000000, 0x2000000400000, 
            0x2000000000800, 0x2000000000200, 0x1800000000000, 0x1001000000000, 0x1000400000000, 
            0x1000000800000, 0x1000000200000, 0x1000000000400, 0x1000000000100, 0xc00000000000, 
            0x800800000000, 0x800200000000, 0x800000400000, 0x800000100000, 0x800000000200, 
            0x800000000080, 0x600000000000, 0x400400000000, 0x400100000000, 0x400000200000, 
            0x400000080000, 0x400000000100, 0x400000000040, 0x300000000000, 0x200200000000, 
            0x200080000000, 0x200000100000, 0x200000040000, 0x200000000080, 0x200000000020, 
            0x180000000000, 0x100100000000, 0x100040000000, 0x100000080000, 0x100000020000, 
            0x100000000040, 0x100000000010, 0xc0000000000, 0x80080000000, 0x80020000000, 
            0x80000040000, 0x80000010000, 0x80000000020, 0x80000000008, 0x60000000000, 
            0x40040000000, 0x40010000000, 0x40000020000, 0x40000008000, 0x40000000010, 
            0x40000000004, 0x30000000000, 0x20020000000, 0x20008000000, 0x20000010000, 
            0x20000004000, 0x20000000008, 0x20000000002, 0x18000000000, 0x10010000000, 
            0x10004000000, 0x10000008000, 0x10000002000, 0x10000000004, 0x10000000001, 
            0xc000000000, 0x8008000000, 0x8002000000, 0x8000004000, 0x8000001000, 
            0x8000000002, 0x6000000000, 0x4004000000, 0x4001000000, 0x4000002000, 
            0x4000000800, 0x4000000001, 0x3000000000, 0x2002000000, 0x2000800000, 
            0x2000001000, 0x2000000400, 0x1800000000, 0x1001000000, 0x1000400000, 
            0x1000000800, 0x1000000200, 0xc00000000, 0x800800000, 0x800200000, 
            0x800000400, 0x800000100, 0x600000000, 0x400400000, 0x400100000, 
            0x400000200, 0x400000080, 0x300000000, 0x200200000, 0x200080000, 
            0x200000100, 0x200000040, 0x180000000, 0x100100000, 0x100040000, 
            0x100000080, 0x100000020, 0xc0000000, 0x80080000, 0x80020000, 
            0x80000040, 0x80000010, 0x60000000, 0x40040000, 0x40010000, 
            0x40000020, 0x40000008, 0x30000000, 0x20020000, 0x20008000, 
            0x20000010, 0x20000004, 0x18000000, 0x10010000, 0x10004000, 
            0x10000008, 0x10000002, 0xc000000, 0x8008000, 0x8002000, 
            0x8000004, 0x8000001, 0x6000000, 0x4004000, 0x4001000, 
            0x4000002, 0x3000000, 0x2002000, 0x2000800, 0x2000001, 
            0x1800000, 0x1001000, 0x1000400, 0xc00000, 0x800800, 
            0x800200, 0x600000, 0x400400, 0x400100, 0x300000, 
            0x200200, 0x200080, 0x180000, 0x100100, 0x100040, 
            0xc0000, 0x80080, 0x80020, 0x60000, 0x40040, 
            0x40010, 0x30000, 0x20020, 0x20008, 0x18000, 
            0x10010, 0x10004, 0xc000, 0x8008, 0x8002, 
            0x6000, 0x4004, 0x4001, 0x3000, 0x2002, 
            0x1800, 0x1001, 0xc00, 0x600, 0x300, 
            0x180, 0xc0, 0x60, 0x30, 0x18, 
            0xc, 0x6, 0x3
        };
        #endregion

        #region Suited Table
        /// <summary>
        /// The list of all pocket hands that are suited.
        /// </summary>
        static internal readonly ulong[] _suitedTable = 
        {
            	0xc000000000000,	0x6000000000,	    0x3000000,	        0x1800,	            0xa000000000000,
	            0x5000000000,	    0x2800000,	        0x1400,	            0x9000000000000,	0x4800000000,
	            0x2400000,	        0x1200,	            0x8800000000000,	0x4400000000,	    0x2200000,
	            0x1100,	            0x8400000000000,	0x4200000000,	    0x2100000,	        0x1080,
	            0x8200000000000,	0x4100000000,	    0x2080000,	        0x1040,	            0x8100000000000,
	            0x4080000000,	    0x2040000,	        0x1020,	            0x8080000000000,	0x4040000000,
	            0x2020000,	        0x1010,	            0x8040000000000,	0x4020000000,	    0x2010000,
	            0x1008,	            0x8020000000000,	0x4010000000,	    0x2008000,	        0x1004,
	            0x8010000000000,	0x4008000000,	    0x2004000,	        0x1002,	            0x8008000000000,
	            0x4004000000,	    0x2002000,	        0x1001,	            0x6000000000000,	0x3000000000,
	            0x1800000,	        0xc00,	            0x5000000000000,	0x2800000000,	    0x1400000,
	            0xa00,	            0x4800000000000,	0x2400000000,	    0x1200000,	        0x900,
	            0x4400000000000,	0x2200000000,	    0x1100000,	        0x880,	            0x4200000000000,
	            0x2100000000,	    0x1080000,	        0x840,	            0x4100000000000,	0x2080000000,
	            0x1040000,	        0x820,	            0x4080000000000,	0x2040000000,	    0x1020000,
	            0x810,	            0x4040000000000,	0x2020000000,	    0x1010000,	        0x808,
	            0x4020000000000,	0x2010000000,	    0x1008000,	        0x804,	            0x4010000000000,
	            0x2008000000,	    0x1004000,	        0x802,	            0x4008000000000,	0x2004000000,
	            0x1002000,	        0x801,	            0x3000000000000,	0x1800000000,	    0xc00000,
	            0x600,	0x2800000000000,	0x1400000000,	0xa00000,	0x500,
	            0x2400000000000,	0x1200000000,	0x900000,	0x480,	0x2200000000000,
	            0x1100000000,	0x880000,	0x440,	0x2100000000000,	0x1080000000,
	            0x840000,	0x420,	0x2080000000000,	0x1040000000,	0x820000,
	            0x410,	0x2040000000000,	0x1020000000,	0x810000,	0x408,
	            0x2020000000000,	0x1010000000,	0x808000,	0x404,	0x2010000000000,
	            0x1008000000,	0x804000,	0x402,	0x2008000000000,	0x1004000000,
	            0x802000,	0x401,	0x1800000000000,	0xc00000000,	0x600000,
	            0x300,	0x1400000000000,	0xa00000000,	0x500000,	0x280,
	            0x1200000000000,	0x900000000,	0x480000,	0x240,	0x1100000000000,
	            0x880000000,	0x440000,	0x220,	0x1080000000000,	0x840000000,
	            0x420000,	0x210,	0x1040000000000,	0x820000000,	0x410000,
	            0x208,	0x1020000000000,	0x810000000,	0x408000,	0x204,
	            0x1010000000000,	0x808000000,	0x404000,	0x202,	0x1008000000000,
	            0x804000000,	0x402000,	0x201,	0xc00000000000,	0x600000000,
	            0x300000,	0x180,	0xa00000000000,	0x500000000,	0x280000,
	            0x140,	0x900000000000,	0x480000000,	0x240000,	0x120,
	            0x880000000000,	0x440000000,	0x220000,	0x110,	0x840000000000,
	            0x420000000,	0x210000,	0x108,	0x820000000000,	0x410000000,
	            0x208000,	0x104,	0x810000000000,	0x408000000,	0x204000,
	            0x102,	0x808000000000,	0x404000000,	0x202000,	0x101,
	            0x600000000000,	0x300000000,	0x180000,	0xc0,	0x500000000000,
	            0x280000000,	0x140000,	0xa0,	0x480000000000,	0x240000000,
	            0x120000,	0x90,	0x440000000000,	0x220000000,	0x110000,
	            0x88,	0x420000000000,	0x210000000,	0x108000,	0x84,
	            0x410000000000,	0x208000000,	0x104000,	0x82,	0x408000000000,
	            0x204000000,	0x102000,	0x81,	0x300000000000,	0x180000000,
	            0xc0000,	0x60,	0x280000000000,	0x140000000,	0xa0000,
	            0x50,	0x240000000000,	0x120000000,	0x90000,	0x48,
	            0x220000000000,	0x110000000,	0x88000,	0x44,	0x210000000000,
	            0x108000000,	0x84000,	0x42,	0x208000000000,	0x104000000,
	            0x82000,	0x41,	0x180000000000,	0xc0000000,	0x60000,
	            0x30,	0x140000000000,	0xa0000000,	0x50000,	0x28,
	            0x120000000000,	0x90000000,	0x48000,	0x24,	0x110000000000,
	            0x88000000,	0x44000,	0x22,	0x108000000000,	0x84000000,
	            0x42000,	0x21,	0xc0000000000,	0x60000000,	0x30000,
	            0x18,	0xa0000000000,	0x50000000,	0x28000,	0x14,
	            0x90000000000,	0x48000000,	0x24000,	0x12,	0x88000000000,
	            0x44000000,	0x22000,	0x11,	0x60000000000,	0x30000000,
	            0x18000,	0xc,	0x50000000000,	0x28000000,	0x14000,
	            0xa,	0x48000000000,	0x24000000,	0x12000,	0x9,
	            0x30000000000,	0x18000000,	0xc000,	0x6,	0x28000000000,
	            0x14000000,	0xa000,	0x5,	0x18000000000,	0xc000000,
	            0x6000,	0x3
        };
        #endregion

        #region Pair Table
        /// <summary>
        /// A list of all pocket cards combinations that represent pairs
        /// </summary>
        static internal readonly ulong[] _pairTable = 
        {
            	0x8004000000000,	0x8000002000000,	0x8000000001000,	0x4002000000,	0x4000001000,
	            0x2001000,	0x4002000000000,	0x4000001000000,	0x4000000000800,	0x2001000000,
	            0x2000000800,	0x1000800,	0x2001000000000,	0x2000000800000,	0x2000000000400,
	            0x1000800000,	0x1000000400,	0x800400,	0x1000800000000,	0x1000000400000,
	            0x1000000000200,	0x800400000,	0x800000200,	0x400200,	0x800400000000,
	            0x800000200000,	0x800000000100,	0x400200000,	0x400000100,	0x200100,
	            0x400200000000,	0x400000100000,	0x400000000080,	0x200100000,	0x200000080,
	            0x100080,	0x200100000000,	0x200000080000,	0x200000000040,	0x100080000,
	            0x100000040,	0x80040,	0x100080000000,	0x100000040000,	0x100000000020,
	            0x80040000,	0x80000020,	0x40020,	0x80040000000,	0x80000020000,
	            0x80000000010,	0x40020000,	0x40000010,	0x20010,	0x40020000000,
	            0x40000010000,	0x40000000008,	0x20010000,	0x20000008,	0x10008,
	            0x20010000000,	0x20000008000,	0x20000000004,	0x10008000,	0x10000004,
	            0x8004,	0x10008000000,	0x10000004000,	0x10000000002,	0x8004000,
	            0x8000002,	0x4002,	0x8004000000,	0x8000002000,	0x8000000001,
	            0x4002000,	0x4000001,	0x2001
        };
        #endregion

        #region Gap1 Table
        /// <summary>
        /// A list of all pocket card combintations such as 42o that have a gap of 1 cards.
        /// </summary>
        static internal ulong[] _gap1Table = {
		    0xa000000000000,0x8010000000000,0x8001000000000,0x8000008000000,0x8000000800000,
            0x8000000004000,0x8000000000400,0x8000000000002,0x5000000000000,0x4000800000000,
            0x4000000400000,0x4000000000200,0x2800000000000,0x2004000000000,0x2000400000000,
            0x2000002000000,0x2000000200000,0x2000000001000,0x2000000000100,0x1400000000000,
            0x1002000000000,0x1000200000000,0x1000001000000,0x1000000100000,0x1000000000800,
            0x1000000000080,0xa00000000000,0x801000000000,0x800100000000,0x800000800000,
            0x800000080000,0x800000000400,0x800000000040,0x500000000000,0x400800000000,
            0x400080000000,0x400000400000,0x400000040000,0x400000000200,0x400000000020,
            0x280000000000,0x200400000000,0x200040000000,0x200000200000,0x200000020000,
            0x200000000100,0x200000000010,0x140000000000,0x100200000000,0x100020000000,
            0x100000100000,0x100000010000,0x100000000080,0x100000000008,0xa0000000000,
            0x80100000000,0x80010000000,0x80000080000,0x80000008000,0x80000000040,
            0x80000000004,0x50000000000,0x40080000000,0x40008000000,0x40000040000,
            0x40000004000,0x40000000020,0x40000000002,0x28000000000,0x20040000000,
            0x20004000000,0x20000020000,0x20000002000,0x20000000010,0x20000000001,
            0x14000000000,0x10020000000,0x10002000000,0x10000010000,0x10000001000,
            0x10000000008,0x8010000000,0x8000008000,0x8000000004,0x5000000000,
            0x4008000000,0x4000800000,0x4000004000,0x4000000400,0x4000000002,
            0x2800000000,0x2000400000,0x2000000200,0x1400000000,0x1002000000,
            0x1000200000,0x1000001000,0x1000000100,0xa00000000,0x801000000,
            0x800100000,0x800000800,0x800000080,0x500000000,0x400800000,
            0x400080000,0x400000400,0x400000040,0x280000000,0x200400000,
            0x200040000,0x200000200,0x200000020,0x140000000,0x100200000,
            0x100020000,0x100000100,0x100000010,0xa0000000,0x80100000,
            0x80010000,0x80000080,0x80000008,0x50000000,0x40080000,
            0x40008000,0x40000040,0x40000004,0x28000000,0x20040000,
            0x20004000,0x20000020,0x20000002,0x14000000,0x10020000,
            0x10002000,0x10000010,0x10000001,0xa000000,0x8010000,
            0x8001000,0x8000008,0x4008000,0x4000004,0x2800000,
            0x2004000,0x2000400,0x2000002,0x1400000,0x1000200,
            0xa00000,0x801000,0x800100,0x500000,0x400800,
            0x400080,0x280000,0x200400,0x200040,0x140000,
            0x100200,0x100020,0xa0000,0x80100,0x80010,
            0x50000,0x40080,0x40008,0x28000,0x20040,
            0x20004,0x14000,0x10020,0x10002,0xa000,
            0x8010,0x8001,0x5000,0x4008,0x2004,
            0x1400,0x1002,0xa00,0x500,0x280,
            0x140,0xa0,0x50,0x28,0x14,
            0xa,0x5
        };
        #endregion

        #region Gap2 Table
        /// <summary>
        /// A list of all pocket card combintations such as 52o that have a gap of 2 cards.
        /// </summary>
        static internal ulong[] _gap2Table = {
		    0x9000000000000,0x8020000000000,0x8000800000000,0x8000010000000,0x8000000400000,
            0x8000000008000,0x8000000000200,0x8000000000004,0x4800000000000,0x4000400000000,
            0x4000000200000,0x4000000000100,0x2400000000000,0x2000200000000,0x2000000100000,
            0x2000000000080,0x1200000000000,0x1004000000000,0x1000100000000,0x1000002000000,
            0x1000000080000,0x1000000001000,0x1000000000040,0x900000000000,0x802000000000,
            0x800080000000,0x800001000000,0x800000040000,0x800000000800,0x800000000020,
            0x480000000000,0x401000000000,0x400040000000,0x400000800000,0x400000020000,
            0x400000000400,0x400000000010,0x240000000000,0x200800000000,0x200020000000,
            0x200000400000,0x200000010000,0x200000000200,0x200000000008,0x120000000000,
            0x100400000000,0x100010000000,0x100000200000,0x100000008000,0x100000000100,
            0x100000000004,0x90000000000,0x80200000000,0x80008000000,0x80000100000,
            0x80000004000,0x80000000080,0x80000000002,0x48000000000,0x40100000000,
            0x40004000000,0x40000080000,0x40000002000,0x40000000040,0x40000000001,
            0x24000000000,0x20080000000,0x20002000000,0x20000040000,0x20000001000,
            0x20000000020,0x10040000000,0x10000020000,0x10000000010,0x8020000000,
            0x8000010000,0x8000000008,0x4800000000,0x4010000000,0x4000400000,
            0x4000008000,0x4000000200,0x4000000004,0x2400000000,0x2000200000,
            0x2000000100,0x1200000000,0x1000100000,0x1000000080,0x900000000,
            0x802000000,0x800080000,0x800001000,0x800000040,0x480000000,
            0x401000000,0x400040000,0x400000800,0x400000020,0x240000000,
            0x200800000,0x200020000,0x200000400,0x200000010,0x120000000,
            0x100400000,0x100010000,0x100000200,0x100000008,0x90000000,
            0x80200000,0x80008000,0x80000100,0x80000004,0x48000000,
            0x40100000,0x40004000,0x40000080,0x40000002,0x24000000,
            0x20080000,0x20002000,0x20000040,0x20000001,0x12000000,
            0x10040000,0x10001000,0x10000020,0x8020000,0x8000010,
            0x4010000,0x4000008,0x2400000,0x2008000,0x2000200,
            0x2000004,0x1200000,0x1000100,0x900000,0x800080,
            0x480000,0x401000,0x400040,0x240000,0x200800,
            0x200020,0x120000,0x100400,0x100010,0x90000,
            0x80200,0x80008,0x48000,0x40100,0x40004,
            0x24000,0x20080,0x20002,0x12000,0x10040,
            0x10001,0x9000,0x8020,0x4010,0x2008,
            0x1200,0x1004,0x900,0x480,0x240,
            0x120,0x90,0x48,0x24,0x12,
            0x9
        };
        #endregion

        #region Gap3 Table
        /// <summary>
        /// A list of all pocket card combintations such as 62o that have a gap of 3 cards.
        /// </summary>
        static internal ulong[] _gap3Table = {
		    0x8800000000000,0x8040000000000,0x8000400000000,0x8000020000000,0x8000000200000,
            0x8000000010000,0x8000000000100,0x8000000000008,0x4400000000000,0x4000200000000,
            0x4000000100000,0x4000000000080,0x2200000000000,0x2000100000000,0x2000000080000,
            0x2000000000040,0x1100000000000,0x1000080000000,0x1000000040000,0x1000000000020,
            0x880000000000,0x804000000000,0x800040000000,0x800002000000,0x800000020000,
            0x800000001000,0x800000000010,0x440000000000,0x402000000000,0x400020000000,
            0x400001000000,0x400000010000,0x400000000800,0x400000000008,0x220000000000,
            0x201000000000,0x200010000000,0x200000800000,0x200000008000,0x200000000400,
            0x200000000004,0x110000000000,0x100800000000,0x100008000000,0x100000400000,
            0x100000004000,0x100000000200,0x100000000002,0x88000000000,0x80400000000,
            0x80004000000,0x80000200000,0x80000002000,0x80000000100,0x80000000001,
            0x44000000000,0x40200000000,0x40002000000,0x40000100000,0x40000001000,
            0x40000000080,0x20100000000,0x20000080000,0x20000000040,0x10080000000,
            0x10000040000,0x10000000020,0x8040000000,0x8000020000,0x8000000010,
            0x4400000000,0x4020000000,0x4000200000,0x4000010000,0x4000000100,
            0x4000000008,0x2200000000,0x2000100000,0x2000000080,0x1100000000,
            0x1000080000,0x1000000040,0x880000000,0x800040000,0x800000020,
            0x440000000,0x402000000,0x400020000,0x400001000,0x400000010,
            0x220000000,0x201000000,0x200010000,0x200000800,0x200000008,
            0x110000000,0x100800000,0x100008000,0x100000400,0x100000004,
            0x88000000,0x80400000,0x80004000,0x80000200,0x80000002,
            0x44000000,0x40200000,0x40002000,0x40000100,0x40000001,
            0x22000000,0x20100000,0x20001000,0x20000080,0x10080000,
            0x10000040,0x8040000,0x8000020,0x4020000,0x4000010,
            0x2200000,0x2010000,0x2000100,0x2000008,0x1100000,
            0x1000080,0x880000,0x800040,0x440000,0x400020,
            0x220000,0x201000,0x200010,0x110000,0x100800,
            0x100008,0x88000,0x80400,0x80004,0x44000,
            0x40200,0x40002,0x22000,0x20100,0x20001,
            0x11000,0x10080,0x8040,0x4020,0x2010,
            0x1100,0x1008,0x880,0x440,0x220,
            0x110,0x88,0x44,0x22,0x11
        };
        #endregion

        #region PocketTable Masks

        /// <summary>
        /// The 1326 pocket card combinations.
        /// </summary>
        internal static readonly ulong[] _PocketTableMasks =
        {
		    0xc000000000000, 0xa000000000000, 0x9000000000000, 0x8800000000000, 0x8400000000000, 
            0x8200000000000, 0x8100000000000, 0x8080000000000, 0x8040000000000, 0x8020000000000, 
            0x8010000000000, 0x8008000000000, 0x8004000000000, 0x8002000000000, 0x8001000000000, 
            0x8000800000000, 0x8000400000000, 0x8000200000000, 0x8000100000000, 0x8000080000000, 
            0x8000040000000, 0x8000020000000, 0x8000010000000, 0x8000008000000, 0x8000004000000, 
            0x8000002000000, 0x8000001000000, 0x8000000800000, 0x8000000400000, 0x8000000200000, 
            0x8000000100000, 0x8000000080000, 0x8000000040000, 0x8000000020000, 0x8000000010000, 
            0x8000000008000, 0x8000000004000, 0x8000000002000, 0x8000000001000, 0x8000000000800, 
            0x8000000000400, 0x8000000000200, 0x8000000000100, 0x8000000000080, 0x8000000000040, 
            0x8000000000020, 0x8000000000010, 0x8000000000008, 0x8000000000004, 0x8000000000002, 
            0x8000000000001, 0x6000000000000, 0x5000000000000, 0x4800000000000, 0x4400000000000, 
            0x4200000000000, 0x4100000000000, 0x4080000000000, 0x4040000000000, 0x4020000000000, 
            0x4010000000000, 0x4008000000000, 0x4004000000000, 0x4002000000000, 0x4001000000000, 
            0x4000800000000, 0x4000400000000, 0x4000200000000, 0x4000100000000, 0x4000080000000, 
            0x4000040000000, 0x4000020000000, 0x4000010000000, 0x4000008000000, 0x4000004000000, 
            0x4000002000000, 0x4000001000000, 0x4000000800000, 0x4000000400000, 0x4000000200000, 
            0x4000000100000, 0x4000000080000, 0x4000000040000, 0x4000000020000, 0x4000000010000, 
            0x4000000008000, 0x4000000004000, 0x4000000002000, 0x4000000001000, 0x4000000000800, 
            0x4000000000400, 0x4000000000200, 0x4000000000100, 0x4000000000080, 0x4000000000040, 
            0x4000000000020, 0x4000000000010, 0x4000000000008, 0x4000000000004, 0x4000000000002, 
            0x4000000000001, 0x3000000000000, 0x2800000000000, 0x2400000000000, 0x2200000000000, 
            0x2100000000000, 0x2080000000000, 0x2040000000000, 0x2020000000000, 0x2010000000000, 
            0x2008000000000, 0x2004000000000, 0x2002000000000, 0x2001000000000, 0x2000800000000, 
            0x2000400000000, 0x2000200000000, 0x2000100000000, 0x2000080000000, 0x2000040000000, 
            0x2000020000000, 0x2000010000000, 0x2000008000000, 0x2000004000000, 0x2000002000000, 
            0x2000001000000, 0x2000000800000, 0x2000000400000, 0x2000000200000, 0x2000000100000, 
            0x2000000080000, 0x2000000040000, 0x2000000020000, 0x2000000010000, 0x2000000008000, 
            0x2000000004000, 0x2000000002000, 0x2000000001000, 0x2000000000800, 0x2000000000400, 
            0x2000000000200, 0x2000000000100, 0x2000000000080, 0x2000000000040, 0x2000000000020, 
            0x2000000000010, 0x2000000000008, 0x2000000000004, 0x2000000000002, 0x2000000000001, 
            0x1800000000000, 0x1400000000000, 0x1200000000000, 0x1100000000000, 0x1080000000000, 
            0x1040000000000, 0x1020000000000, 0x1010000000000, 0x1008000000000, 0x1004000000000, 
            0x1002000000000, 0x1001000000000, 0x1000800000000, 0x1000400000000, 0x1000200000000, 
            0x1000100000000, 0x1000080000000, 0x1000040000000, 0x1000020000000, 0x1000010000000, 
            0x1000008000000, 0x1000004000000, 0x1000002000000, 0x1000001000000, 0x1000000800000, 
            0x1000000400000, 0x1000000200000, 0x1000000100000, 0x1000000080000, 0x1000000040000, 
            0x1000000020000, 0x1000000010000, 0x1000000008000, 0x1000000004000, 0x1000000002000, 
            0x1000000001000, 0x1000000000800, 0x1000000000400, 0x1000000000200, 0x1000000000100, 
            0x1000000000080, 0x1000000000040, 0x1000000000020, 0x1000000000010, 0x1000000000008, 
            0x1000000000004, 0x1000000000002, 0x1000000000001, 0xc00000000000, 0xa00000000000, 
            0x900000000000, 0x880000000000, 0x840000000000, 0x820000000000, 0x810000000000, 
            0x808000000000, 0x804000000000, 0x802000000000, 0x801000000000, 0x800800000000, 
            0x800400000000, 0x800200000000, 0x800100000000, 0x800080000000, 0x800040000000, 
            0x800020000000, 0x800010000000, 0x800008000000, 0x800004000000, 0x800002000000, 
            0x800001000000, 0x800000800000, 0x800000400000, 0x800000200000, 0x800000100000, 
            0x800000080000, 0x800000040000, 0x800000020000, 0x800000010000, 0x800000008000, 
            0x800000004000, 0x800000002000, 0x800000001000, 0x800000000800, 0x800000000400, 
            0x800000000200, 0x800000000100, 0x800000000080, 0x800000000040, 0x800000000020, 
            0x800000000010, 0x800000000008, 0x800000000004, 0x800000000002, 0x800000000001, 
            0x600000000000, 0x500000000000, 0x480000000000, 0x440000000000, 0x420000000000, 
            0x410000000000, 0x408000000000, 0x404000000000, 0x402000000000, 0x401000000000, 
            0x400800000000, 0x400400000000, 0x400200000000, 0x400100000000, 0x400080000000, 
            0x400040000000, 0x400020000000, 0x400010000000, 0x400008000000, 0x400004000000, 
            0x400002000000, 0x400001000000, 0x400000800000, 0x400000400000, 0x400000200000, 
            0x400000100000, 0x400000080000, 0x400000040000, 0x400000020000, 0x400000010000, 
            0x400000008000, 0x400000004000, 0x400000002000, 0x400000001000, 0x400000000800, 
            0x400000000400, 0x400000000200, 0x400000000100, 0x400000000080, 0x400000000040, 
            0x400000000020, 0x400000000010, 0x400000000008, 0x400000000004, 0x400000000002, 
            0x400000000001, 0x300000000000, 0x280000000000, 0x240000000000, 0x220000000000, 
            0x210000000000, 0x208000000000, 0x204000000000, 0x202000000000, 0x201000000000, 
            0x200800000000, 0x200400000000, 0x200200000000, 0x200100000000, 0x200080000000, 
            0x200040000000, 0x200020000000, 0x200010000000, 0x200008000000, 0x200004000000, 
            0x200002000000, 0x200001000000, 0x200000800000, 0x200000400000, 0x200000200000, 
            0x200000100000, 0x200000080000, 0x200000040000, 0x200000020000, 0x200000010000, 
            0x200000008000, 0x200000004000, 0x200000002000, 0x200000001000, 0x200000000800, 
            0x200000000400, 0x200000000200, 0x200000000100, 0x200000000080, 0x200000000040, 
            0x200000000020, 0x200000000010, 0x200000000008, 0x200000000004, 0x200000000002, 
            0x200000000001, 0x180000000000, 0x140000000000, 0x120000000000, 0x110000000000, 
            0x108000000000, 0x104000000000, 0x102000000000, 0x101000000000, 0x100800000000, 
            0x100400000000, 0x100200000000, 0x100100000000, 0x100080000000, 0x100040000000, 
            0x100020000000, 0x100010000000, 0x100008000000, 0x100004000000, 0x100002000000, 
            0x100001000000, 0x100000800000, 0x100000400000, 0x100000200000, 0x100000100000, 
            0x100000080000, 0x100000040000, 0x100000020000, 0x100000010000, 0x100000008000, 
            0x100000004000, 0x100000002000, 0x100000001000, 0x100000000800, 0x100000000400, 
            0x100000000200, 0x100000000100, 0x100000000080, 0x100000000040, 0x100000000020, 
            0x100000000010, 0x100000000008, 0x100000000004, 0x100000000002, 0x100000000001, 
            0xc0000000000, 0xa0000000000, 0x90000000000, 0x88000000000, 0x84000000000, 
            0x82000000000, 0x81000000000, 0x80800000000, 0x80400000000, 0x80200000000, 
            0x80100000000, 0x80080000000, 0x80040000000, 0x80020000000, 0x80010000000, 
            0x80008000000, 0x80004000000, 0x80002000000, 0x80001000000, 0x80000800000, 
            0x80000400000, 0x80000200000, 0x80000100000, 0x80000080000, 0x80000040000, 
            0x80000020000, 0x80000010000, 0x80000008000, 0x80000004000, 0x80000002000, 
            0x80000001000, 0x80000000800, 0x80000000400, 0x80000000200, 0x80000000100, 
            0x80000000080, 0x80000000040, 0x80000000020, 0x80000000010, 0x80000000008, 
            0x80000000004, 0x80000000002, 0x80000000001, 0x60000000000, 0x50000000000, 
            0x48000000000, 0x44000000000, 0x42000000000, 0x41000000000, 0x40800000000, 
            0x40400000000, 0x40200000000, 0x40100000000, 0x40080000000, 0x40040000000, 
            0x40020000000, 0x40010000000, 0x40008000000, 0x40004000000, 0x40002000000, 
            0x40001000000, 0x40000800000, 0x40000400000, 0x40000200000, 0x40000100000, 
            0x40000080000, 0x40000040000, 0x40000020000, 0x40000010000, 0x40000008000, 
            0x40000004000, 0x40000002000, 0x40000001000, 0x40000000800, 0x40000000400, 
            0x40000000200, 0x40000000100, 0x40000000080, 0x40000000040, 0x40000000020, 
            0x40000000010, 0x40000000008, 0x40000000004, 0x40000000002, 0x40000000001, 
            0x30000000000, 0x28000000000, 0x24000000000, 0x22000000000, 0x21000000000, 
            0x20800000000, 0x20400000000, 0x20200000000, 0x20100000000, 0x20080000000, 
            0x20040000000, 0x20020000000, 0x20010000000, 0x20008000000, 0x20004000000, 
            0x20002000000, 0x20001000000, 0x20000800000, 0x20000400000, 0x20000200000, 
            0x20000100000, 0x20000080000, 0x20000040000, 0x20000020000, 0x20000010000, 
            0x20000008000, 0x20000004000, 0x20000002000, 0x20000001000, 0x20000000800, 
            0x20000000400, 0x20000000200, 0x20000000100, 0x20000000080, 0x20000000040, 
            0x20000000020, 0x20000000010, 0x20000000008, 0x20000000004, 0x20000000002, 
            0x20000000001, 0x18000000000, 0x14000000000, 0x12000000000, 0x11000000000, 
            0x10800000000, 0x10400000000, 0x10200000000, 0x10100000000, 0x10080000000, 
            0x10040000000, 0x10020000000, 0x10010000000, 0x10008000000, 0x10004000000, 
            0x10002000000, 0x10001000000, 0x10000800000, 0x10000400000, 0x10000200000, 
            0x10000100000, 0x10000080000, 0x10000040000, 0x10000020000, 0x10000010000, 
            0x10000008000, 0x10000004000, 0x10000002000, 0x10000001000, 0x10000000800, 
            0x10000000400, 0x10000000200, 0x10000000100, 0x10000000080, 0x10000000040, 
            0x10000000020, 0x10000000010, 0x10000000008, 0x10000000004, 0x10000000002, 
            0x10000000001, 0xc000000000, 0xa000000000, 0x9000000000, 0x8800000000, 
            0x8400000000, 0x8200000000, 0x8100000000, 0x8080000000, 0x8040000000, 
            0x8020000000, 0x8010000000, 0x8008000000, 0x8004000000, 0x8002000000, 
            0x8001000000, 0x8000800000, 0x8000400000, 0x8000200000, 0x8000100000, 
            0x8000080000, 0x8000040000, 0x8000020000, 0x8000010000, 0x8000008000, 
            0x8000004000, 0x8000002000, 0x8000001000, 0x8000000800, 0x8000000400, 
            0x8000000200, 0x8000000100, 0x8000000080, 0x8000000040, 0x8000000020, 
            0x8000000010, 0x8000000008, 0x8000000004, 0x8000000002, 0x8000000001, 
            0x6000000000, 0x5000000000, 0x4800000000, 0x4400000000, 0x4200000000, 
            0x4100000000, 0x4080000000, 0x4040000000, 0x4020000000, 0x4010000000, 
            0x4008000000, 0x4004000000, 0x4002000000, 0x4001000000, 0x4000800000, 
            0x4000400000, 0x4000200000, 0x4000100000, 0x4000080000, 0x4000040000, 
            0x4000020000, 0x4000010000, 0x4000008000, 0x4000004000, 0x4000002000, 
            0x4000001000, 0x4000000800, 0x4000000400, 0x4000000200, 0x4000000100, 
            0x4000000080, 0x4000000040, 0x4000000020, 0x4000000010, 0x4000000008, 
            0x4000000004, 0x4000000002, 0x4000000001, 0x3000000000, 0x2800000000, 
            0x2400000000, 0x2200000000, 0x2100000000, 0x2080000000, 0x2040000000, 
            0x2020000000, 0x2010000000, 0x2008000000, 0x2004000000, 0x2002000000, 
            0x2001000000, 0x2000800000, 0x2000400000, 0x2000200000, 0x2000100000, 
            0x2000080000, 0x2000040000, 0x2000020000, 0x2000010000, 0x2000008000, 
            0x2000004000, 0x2000002000, 0x2000001000, 0x2000000800, 0x2000000400, 
            0x2000000200, 0x2000000100, 0x2000000080, 0x2000000040, 0x2000000020, 
            0x2000000010, 0x2000000008, 0x2000000004, 0x2000000002, 0x2000000001, 
            0x1800000000, 0x1400000000, 0x1200000000, 0x1100000000, 0x1080000000, 
            0x1040000000, 0x1020000000, 0x1010000000, 0x1008000000, 0x1004000000, 
            0x1002000000, 0x1001000000, 0x1000800000, 0x1000400000, 0x1000200000, 
            0x1000100000, 0x1000080000, 0x1000040000, 0x1000020000, 0x1000010000, 
            0x1000008000, 0x1000004000, 0x1000002000, 0x1000001000, 0x1000000800, 
            0x1000000400, 0x1000000200, 0x1000000100, 0x1000000080, 0x1000000040, 
            0x1000000020, 0x1000000010, 0x1000000008, 0x1000000004, 0x1000000002, 
            0x1000000001, 0xc00000000, 0xa00000000, 0x900000000, 0x880000000, 
            0x840000000, 0x820000000, 0x810000000, 0x808000000, 0x804000000, 
            0x802000000, 0x801000000, 0x800800000, 0x800400000, 0x800200000, 
            0x800100000, 0x800080000, 0x800040000, 0x800020000, 0x800010000, 
            0x800008000, 0x800004000, 0x800002000, 0x800001000, 0x800000800, 
            0x800000400, 0x800000200, 0x800000100, 0x800000080, 0x800000040, 
            0x800000020, 0x800000010, 0x800000008, 0x800000004, 0x800000002, 
            0x800000001, 0x600000000, 0x500000000, 0x480000000, 0x440000000, 
            0x420000000, 0x410000000, 0x408000000, 0x404000000, 0x402000000, 
            0x401000000, 0x400800000, 0x400400000, 0x400200000, 0x400100000, 
            0x400080000, 0x400040000, 0x400020000, 0x400010000, 0x400008000, 
            0x400004000, 0x400002000, 0x400001000, 0x400000800, 0x400000400, 
            0x400000200, 0x400000100, 0x400000080, 0x400000040, 0x400000020, 
            0x400000010, 0x400000008, 0x400000004, 0x400000002, 0x400000001, 
            0x300000000, 0x280000000, 0x240000000, 0x220000000, 0x210000000, 
            0x208000000, 0x204000000, 0x202000000, 0x201000000, 0x200800000, 
            0x200400000, 0x200200000, 0x200100000, 0x200080000, 0x200040000, 
            0x200020000, 0x200010000, 0x200008000, 0x200004000, 0x200002000, 
            0x200001000, 0x200000800, 0x200000400, 0x200000200, 0x200000100, 
            0x200000080, 0x200000040, 0x200000020, 0x200000010, 0x200000008, 
            0x200000004, 0x200000002, 0x200000001, 0x180000000, 0x140000000, 
            0x120000000, 0x110000000, 0x108000000, 0x104000000, 0x102000000, 
            0x101000000, 0x100800000, 0x100400000, 0x100200000, 0x100100000, 
            0x100080000, 0x100040000, 0x100020000, 0x100010000, 0x100008000, 
            0x100004000, 0x100002000, 0x100001000, 0x100000800, 0x100000400, 
            0x100000200, 0x100000100, 0x100000080, 0x100000040, 0x100000020, 
            0x100000010, 0x100000008, 0x100000004, 0x100000002, 0x100000001, 
            0xc0000000, 0xa0000000, 0x90000000, 0x88000000, 0x84000000, 
            0x82000000, 0x81000000, 0x80800000, 0x80400000, 0x80200000, 
            0x80100000, 0x80080000, 0x80040000, 0x80020000, 0x80010000, 
            0x80008000, 0x80004000, 0x80002000, 0x80001000, 0x80000800, 
            0x80000400, 0x80000200, 0x80000100, 0x80000080, 0x80000040, 
            0x80000020, 0x80000010, 0x80000008, 0x80000004, 0x80000002, 
            0x80000001, 0x60000000, 0x50000000, 0x48000000, 0x44000000, 
            0x42000000, 0x41000000, 0x40800000, 0x40400000, 0x40200000, 
            0x40100000, 0x40080000, 0x40040000, 0x40020000, 0x40010000, 
            0x40008000, 0x40004000, 0x40002000, 0x40001000, 0x40000800, 
            0x40000400, 0x40000200, 0x40000100, 0x40000080, 0x40000040, 
            0x40000020, 0x40000010, 0x40000008, 0x40000004, 0x40000002, 
            0x40000001, 0x30000000, 0x28000000, 0x24000000, 0x22000000, 
            0x21000000, 0x20800000, 0x20400000, 0x20200000, 0x20100000, 
            0x20080000, 0x20040000, 0x20020000, 0x20010000, 0x20008000, 
            0x20004000, 0x20002000, 0x20001000, 0x20000800, 0x20000400, 
            0x20000200, 0x20000100, 0x20000080, 0x20000040, 0x20000020, 
            0x20000010, 0x20000008, 0x20000004, 0x20000002, 0x20000001, 
            0x18000000, 0x14000000, 0x12000000, 0x11000000, 0x10800000, 
            0x10400000, 0x10200000, 0x10100000, 0x10080000, 0x10040000, 
            0x10020000, 0x10010000, 0x10008000, 0x10004000, 0x10002000, 
            0x10001000, 0x10000800, 0x10000400, 0x10000200, 0x10000100, 
            0x10000080, 0x10000040, 0x10000020, 0x10000010, 0x10000008, 
            0x10000004, 0x10000002, 0x10000001, 0xc000000, 0xa000000, 
            0x9000000, 0x8800000, 0x8400000, 0x8200000, 0x8100000, 
            0x8080000, 0x8040000, 0x8020000, 0x8010000, 0x8008000, 
            0x8004000, 0x8002000, 0x8001000, 0x8000800, 0x8000400, 
            0x8000200, 0x8000100, 0x8000080, 0x8000040, 0x8000020, 
            0x8000010, 0x8000008, 0x8000004, 0x8000002, 0x8000001, 
            0x6000000, 0x5000000, 0x4800000, 0x4400000, 0x4200000, 
            0x4100000, 0x4080000, 0x4040000, 0x4020000, 0x4010000, 
            0x4008000, 0x4004000, 0x4002000, 0x4001000, 0x4000800, 
            0x4000400, 0x4000200, 0x4000100, 0x4000080, 0x4000040, 
            0x4000020, 0x4000010, 0x4000008, 0x4000004, 0x4000002, 
            0x4000001, 0x3000000, 0x2800000, 0x2400000, 0x2200000, 
            0x2100000, 0x2080000, 0x2040000, 0x2020000, 0x2010000, 
            0x2008000, 0x2004000, 0x2002000, 0x2001000, 0x2000800, 
            0x2000400, 0x2000200, 0x2000100, 0x2000080, 0x2000040, 
            0x2000020, 0x2000010, 0x2000008, 0x2000004, 0x2000002, 
            0x2000001, 0x1800000, 0x1400000, 0x1200000, 0x1100000, 
            0x1080000, 0x1040000, 0x1020000, 0x1010000, 0x1008000, 
            0x1004000, 0x1002000, 0x1001000, 0x1000800, 0x1000400, 
            0x1000200, 0x1000100, 0x1000080, 0x1000040, 0x1000020, 
            0x1000010, 0x1000008, 0x1000004, 0x1000002, 0x1000001, 
            0xc00000, 0xa00000, 0x900000, 0x880000, 0x840000, 
            0x820000, 0x810000, 0x808000, 0x804000, 0x802000, 
            0x801000, 0x800800, 0x800400, 0x800200, 0x800100, 
            0x800080, 0x800040, 0x800020, 0x800010, 0x800008, 
            0x800004, 0x800002, 0x800001, 0x600000, 0x500000, 
            0x480000, 0x440000, 0x420000, 0x410000, 0x408000, 
            0x404000, 0x402000, 0x401000, 0x400800, 0x400400, 
            0x400200, 0x400100, 0x400080, 0x400040, 0x400020, 
            0x400010, 0x400008, 0x400004, 0x400002, 0x400001, 
            0x300000, 0x280000, 0x240000, 0x220000, 0x210000, 
            0x208000, 0x204000, 0x202000, 0x201000, 0x200800, 
            0x200400, 0x200200, 0x200100, 0x200080, 0x200040, 
            0x200020, 0x200010, 0x200008, 0x200004, 0x200002, 
            0x200001, 0x180000, 0x140000, 0x120000, 0x110000, 
            0x108000, 0x104000, 0x102000, 0x101000, 0x100800, 
            0x100400, 0x100200, 0x100100, 0x100080, 0x100040, 
            0x100020, 0x100010, 0x100008, 0x100004, 0x100002, 
            0x100001, 0xc0000, 0xa0000, 0x90000, 0x88000, 
            0x84000, 0x82000, 0x81000, 0x80800, 0x80400, 
            0x80200, 0x80100, 0x80080, 0x80040, 0x80020, 
            0x80010, 0x80008, 0x80004, 0x80002, 0x80001, 
            0x60000, 0x50000, 0x48000, 0x44000, 0x42000, 
            0x41000, 0x40800, 0x40400, 0x40200, 0x40100, 
            0x40080, 0x40040, 0x40020, 0x40010, 0x40008, 
            0x40004, 0x40002, 0x40001, 0x30000, 0x28000, 
            0x24000, 0x22000, 0x21000, 0x20800, 0x20400, 
            0x20200, 0x20100, 0x20080, 0x20040, 0x20020, 
            0x20010, 0x20008, 0x20004, 0x20002, 0x20001, 
            0x18000, 0x14000, 0x12000, 0x11000, 0x10800, 
            0x10400, 0x10200, 0x10100, 0x10080, 0x10040, 
            0x10020, 0x10010, 0x10008, 0x10004, 0x10002, 
            0x10001, 0xc000, 0xa000, 0x9000, 0x8800, 
            0x8400, 0x8200, 0x8100, 0x8080, 0x8040, 
            0x8020, 0x8010, 0x8008, 0x8004, 0x8002, 
            0x8001, 0x6000, 0x5000, 0x4800, 0x4400, 
            0x4200, 0x4100, 0x4080, 0x4040, 0x4020, 
            0x4010, 0x4008, 0x4004, 0x4002, 0x4001, 
            0x3000, 0x2800, 0x2400, 0x2200, 0x2100, 
            0x2080, 0x2040, 0x2020, 0x2010, 0x2008, 
            0x2004, 0x2002, 0x2001, 0x1800, 0x1400, 
            0x1200, 0x1100, 0x1080, 0x1040, 0x1020, 
            0x1010, 0x1008, 0x1004, 0x1002, 0x1001, 
            0xc00, 0xa00, 0x900, 0x880, 0x840, 
            0x820, 0x810, 0x808, 0x804, 0x802, 
            0x801, 0x600, 0x500, 0x480, 0x440, 
            0x420, 0x410, 0x408, 0x404, 0x402, 
            0x401, 0x300, 0x280, 0x240, 0x220, 
            0x210, 0x208, 0x204, 0x202, 0x201, 
            0x180, 0x140, 0x120, 0x110, 0x108, 
            0x104, 0x102, 0x101, 0xc0, 0xa0, 
            0x90, 0x88, 0x84, 0x82, 0x81, 
            0x60, 0x50, 0x48, 0x44, 0x42, 
            0x41, 0x30, 0x28, 0x24, 0x22, 
            0x21, 0x18, 0x14, 0x12, 0x11, 
            0xc, 0xa, 0x9, 0x6, 0x5, 
            0x3
        };
        #endregion
        #endregion

        #region Pocket Groupings

        #region Pocket Grouping Rank Enumeration
        /// <summary>
        /// Pocket Card Groupings (Group1 is best, None is worst). 
        /// </summary>
        public enum GroupTypeEnum : int
        {
            /// <summary>
            /// Strongest
            /// </summary>
            Group1 = 0,
            /// <summary>
            /// Very Strong
            /// </summary>
            Group2 = 1,
            /// <summary>
            /// 
            /// </summary>
            Group3 = 2,

            /// <summary>
            /// 
            /// </summary>
            Group4 = 3,
            /// <summary>
            /// 
            /// </summary>
            Group5 = 4,
            /// <summary>
            /// 
            /// </summary>
            Group6 = 5,
            /// <summary>
            /// 
            /// </summary>
            Group7 = 6,
            /// <summary>
            /// 
            /// </summary>
            Group8 = 7,
            /// <summary>
            /// Not in the group
            /// </summary>
            None = 8
        }
        #endregion

        #region Pocket Grouping Enumerators

        ///<exclude/>
        static private Dictionary<ulong, GroupTypeEnum> groupTypeDict = new Dictionary<ulong, GroupTypeEnum>();

        /// <summary>
        /// Returns pocket grouping info for a given pocket cards. This is similar to but 
        /// different from Sklansky groupings. It may be used just like Sklansky groupings
        /// and for most tables the this grouping and Sklansky groupings are identical.
        /// </summary>
        /// <param name="mask">pocket mask to group</param>
        /// <returns>An enum value representing the pocket grouping rank.</returns>
        public static GroupTypeEnum GroupType(ulong mask)
        {
#if DEBUG
            // Preconditions: must be a pocket card mask
            if (Hand.BitCount(mask) != 2)
                throw new ArgumentOutOfRangeException("mask");
#endif

            // Fill in dictionary
            if (groupTypeDict.Count == 0)
            {
                for (int i = 0; i < _PocketGroupings.Length; i++)
                {
                    foreach (ulong tmask in _PocketGroupings[i])
                    {
                        groupTypeDict.Add(tmask, (GroupTypeEnum)i);
                    }
                }
            }

            return groupTypeDict[mask];
        }
        #endregion

        #endregion

        #region Misc 169 Tables
        /// <summary>
        /// This table contains the gap count for each of the 169 mask types. The index cooresponds PocketHand169Enum.
        /// </summary>
        static internal readonly int[] _PocketCards169Gap = {
		    -1, -1, -1, -1, -1, 
            -1, -1, -1, -1, -1, 
            -1, -1, -1, 0, 0, 
            1, 1, 2, 2, 3, 
            3, -1, -1, -1, -1, 
            -1, -1, -1, -1, 3, 
            3, 2, 2, 1, 1, 
            0, 0, 0, 0, 1, 
            1, 2, 2, 3, 3, 
            -1, -1, -1, -1, -1, 
            -1, -1, -1, -1, -1, 
            -1, -1, -1, -1, 0, 
            0, 1, 1, 2, 2, 
            3, 3, -1, -1, -1, 
            -1, -1, -1, -1, -1, 
            -1, -1, -1, -1, 0, 
            0, 1, 1, 2, 2, 
            3, 3, -1, -1, -1, 
            -1, -1, -1, -1, -1, 
            -1, -1, 0, 0, 1, 
            1, 2, 2, 3, 3, 
            -1, -1, -1, -1, -1, 
            -1, -1, -1, 0, 0, 
            1, 1, 2, 2, 3, 
            3, -1, -1, -1, -1, 
            -1, -1, 0, 0, 1, 
            1, 2, 2, 3, 3, 
            -1, -1, -1, -1, 0, 
            0, 1, 1, 2, 2, 
            3, 3, -1, -1, 0, 
            0, 1, 1, 2, 2, 
            3, 3, 0, 0, 1, 
            1, 2, 2, 0, 0, 
            1, 1, 0, 0
        };

        /// <summary>
        /// This table is true for entries in the PocketCards169 table that are connected.
        /// </summary>
        static internal readonly bool[] _PocketCards169Connected = {
		    false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, true, true, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            true, true, true, true, false, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, false, true, 
            true, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, false, true, 
            true, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, true, true, false, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, false, true, true, 
            false, false, false, false, false, 
            false, false, false, false, false, 
            false, false, true, true, false, 
            false, false, false, false, false, 
            false, false, false, false, true, 
            true, false, false, false, false, 
            false, false, false, false, true, 
            true, false, false, false, false, 
            false, false, true, true, false, 
            false, false, false, true, true, 
            false, false, true, true
        };

        /// <summary>
        /// This table contains the suited boolean value for each of the 169 mask types. The index cooresponds PocketHand169Enum.
        /// </summary>
        static internal readonly bool[] _PocketCards169Suited = {
		    false, false, false, false, false, false, false, false, false, false, false, false, false, 
		    true, false, true, false, true, false, true, false, true, false, 
            true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false, 
            true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,
            true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,  
            true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,
            true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false, true, false, true, false,
		    true, false, true, false, true, false            
        };
        #endregion

        #region Pocket 169 Hand Definitions

        #region Pocket Table
        /// <summary>
        /// The 1326 possible pocket cards ordered by the 169 unique holdem combinations. The
        /// index is equivalent to the number value of Hand.PocketHand169Enum.
        /// </summary>
        internal static readonly ulong[][] _Pocket169Combinations = {
	        new ulong [] {0x8004000000000, 0x8000002000000, 0x8000000001000, 0x4002000000, 0x4000001000, 0x2001000},
	        new ulong [] {0x4002000000000, 0x4000001000000, 0x4000000000800, 0x2001000000, 0x2000000800, 0x1000800},
	        new ulong [] {0x2001000000000, 0x2000000800000, 0x2000000000400, 0x1000800000, 0x1000000400, 0x800400},
	        new ulong [] {0x1000800000000, 0x1000000400000, 0x1000000000200, 0x800400000, 0x800000200, 0x400200},
	        new ulong [] {0x800400000000, 0x800000200000, 0x800000000100, 0x400200000, 0x400000100, 0x200100},
	        new ulong [] {0x400200000000, 0x400000100000, 0x400000000080, 0x200100000, 0x200000080, 0x100080},
	        new ulong [] {0x200100000000, 0x200000080000, 0x200000000040, 0x100080000, 0x100000040, 0x80040},
	        new ulong [] {0x100080000000, 0x100000040000, 0x100000000020, 0x80040000, 0x80000020, 0x40020},
	        new ulong [] {0x80040000000, 0x80000020000, 0x80000000010, 0x40020000, 0x40000010, 0x20010},
	        new ulong [] {0x40020000000, 0x40000010000, 0x40000000008, 0x20010000, 0x20000008, 0x10008},
	        new ulong [] {0x20010000000, 0x20000008000, 0x20000000004, 0x10008000, 0x10000004, 0x8004},
	        new ulong [] {0x10008000000, 0x10000004000, 0x10000000002, 0x8004000, 0x8000002, 0x4002},
	        new ulong [] {0x8004000000, 0x8000002000, 0x8000000001, 0x4002000, 0x4000001, 0x2001},
	        new ulong [] {0xC000000000000, 0x6000000000, 0x3000000, 0x1800},
	        new ulong [] {0x8002000000000, 0x8000001000000, 0x8000000000800, 0x4004000000000, 0x4000002000000, 0x4000000001000, 0x4001000000, 0x4000000800, 0x2002000000, 0x2000001000, 0x2000800, 0x1001000},
	        new ulong [] {0xA000000000000, 0x5000000000, 0x2800000, 0x1400},
	        new ulong [] {0x8001000000000, 0x8000000800000, 0x8000000000400, 0x2004000000000, 0x2000002000000, 0x2000000001000, 0x4000800000, 0x4000000400, 0x1002000000, 0x1000001000, 0x2000400, 0x801000},
	        new ulong [] {0x9000000000000, 0x4800000000, 0x2400000, 0x1200},
	        new ulong [] {0x8000800000000, 0x8000000400000, 0x8000000000200, 0x1004000000000, 0x1000002000000, 0x1000000001000, 0x4000400000, 0x4000000200, 0x802000000, 0x800001000, 0x2000200, 0x401000},
	        new ulong [] {0x8800000000000, 0x4400000000, 0x2200000, 0x1100},
	        new ulong [] {0x8000400000000, 0x8000000200000, 0x8000000000100, 0x804000000000, 0x800002000000, 0x800000001000, 0x4000200000, 0x4000000100, 0x402000000, 0x400001000, 0x2000100, 0x201000},
	        new ulong [] {0x8400000000000, 0x4200000000, 0x2100000, 0x1080},
	        new ulong [] {0x8000200000000, 0x8000000100000, 0x8000000000080, 0x404000000000, 0x400002000000, 0x400000001000, 0x4000100000, 0x4000000080, 0x202000000, 0x200001000, 0x2000080, 0x101000},
	        new ulong [] {0x8200000000000, 0x4100000000, 0x2080000, 0x1040},
	        new ulong [] {0x8000100000000, 0x8000000080000, 0x8000000000040, 0x204000000000, 0x200002000000, 0x200000001000, 0x4000080000, 0x4000000040, 0x102000000, 0x100001000, 0x2000040, 0x81000},
	        new ulong [] {0x8100000000000, 0x4080000000, 0x2040000, 0x1020},
	        new ulong [] {0x8000080000000, 0x8000000040000, 0x8000000000020, 0x104000000000, 0x100002000000, 0x100000001000, 0x4000040000, 0x4000000020, 0x82000000, 0x80001000, 0x2000020, 0x41000},
	        new ulong [] {0x8080000000000, 0x4040000000, 0x2020000, 0x1010},
	        new ulong [] {0x8000040000000, 0x8000000020000, 0x8000000000010, 0x84000000000, 0x80002000000, 0x80000001000, 0x4000020000, 0x4000000010, 0x42000000, 0x40001000, 0x2000010, 0x21000},
	        new ulong [] {0x8040000000000, 0x4020000000, 0x2010000, 0x1008},
	        new ulong [] {0x8000020000000, 0x8000000010000, 0x8000000000008, 0x44000000000, 0x40002000000, 0x40000001000, 0x4000010000, 0x4000000008, 0x22000000, 0x20001000, 0x2000008, 0x11000},
	        new ulong [] {0x8020000000000, 0x4010000000, 0x2008000, 0x1004},
	        new ulong [] {0x8000010000000, 0x8000000008000, 0x8000000000004, 0x24000000000, 0x20002000000, 0x20000001000, 0x4000008000, 0x4000000004, 0x12000000, 0x10001000, 0x2000004, 0x9000},
	        new ulong [] {0x8010000000000, 0x4008000000, 0x2004000, 0x1002},
	        new ulong [] {0x8000008000000, 0x8000000004000, 0x8000000000002, 0x14000000000, 0x10002000000, 0x10000001000, 0x4000004000, 0x4000000002, 0xA000000, 0x8001000, 0x2000002, 0x5000},
	        new ulong [] {0x8008000000000, 0x4004000000, 0x2002000, 0x1001},
	        new ulong [] {0x8000004000000, 0x8000000002000, 0x8000000000001, 0xC000000000, 0x8002000000, 0x8000001000, 0x4000002000, 0x4000000001, 0x6000000, 0x4001000, 0x2000001, 0x3000},
	        new ulong [] {0x6000000000000, 0x3000000000, 0x1800000, 0xC00},
	        new ulong [] {0x4001000000000, 0x4000000800000, 0x4000000000400, 0x2002000000000, 0x2000001000000, 0x2000000000800, 0x2000800000, 0x2000000400, 0x1001000000, 0x1000000800, 0x1000400, 0x800800},
	        new ulong [] {0x5000000000000, 0x2800000000, 0x1400000, 0xA00},
	        new ulong [] {0x4000800000000, 0x4000000400000, 0x4000000000200, 0x1002000000000, 0x1000001000000, 0x1000000000800, 0x2000400000, 0x2000000200, 0x801000000, 0x800000800, 0x1000200, 0x400800},
	        new ulong [] {0x4800000000000, 0x2400000000, 0x1200000, 0x900},
	        new ulong [] {0x4000400000000, 0x4000000200000, 0x4000000000100, 0x802000000000, 0x800001000000, 0x800000000800, 0x2000200000, 0x2000000100, 0x401000000, 0x400000800, 0x1000100, 0x200800},
	        new ulong [] {0x4400000000000, 0x2200000000, 0x1100000, 0x880},
	        new ulong [] {0x4000200000000, 0x4000000100000, 0x4000000000080, 0x402000000000, 0x400001000000, 0x400000000800, 0x2000100000, 0x2000000080, 0x201000000, 0x200000800, 0x1000080, 0x100800},
	        new ulong [] {0x4200000000000, 0x2100000000, 0x1080000, 0x840},
	        new ulong [] {0x4000100000000, 0x4000000080000, 0x4000000000040, 0x202000000000, 0x200001000000, 0x200000000800, 0x2000080000, 0x2000000040, 0x101000000, 0x100000800, 0x1000040, 0x80800},
	        new ulong [] {0x4100000000000, 0x2080000000, 0x1040000, 0x820},
	        new ulong [] {0x4000080000000, 0x4000000040000, 0x4000000000020, 0x102000000000, 0x100001000000, 0x100000000800, 0x2000040000, 0x2000000020, 0x81000000, 0x80000800, 0x1000020, 0x40800},
	        new ulong [] {0x4080000000000, 0x2040000000, 0x1020000, 0x810},
	        new ulong [] {0x4000040000000, 0x4000000020000, 0x4000000000010, 0x82000000000, 0x80001000000, 0x80000000800, 0x2000020000, 0x2000000010, 0x41000000, 0x40000800, 0x1000010, 0x20800},
	        new ulong [] {0x4040000000000, 0x2020000000, 0x1010000, 0x808},
	        new ulong [] {0x4000020000000, 0x4000000010000, 0x4000000000008, 0x42000000000, 0x40001000000, 0x40000000800, 0x2000010000, 0x2000000008, 0x21000000, 0x20000800, 0x1000008, 0x10800},
	        new ulong [] {0x4020000000000, 0x2010000000, 0x1008000, 0x804},
	        new ulong [] {0x4000010000000, 0x4000000008000, 0x4000000000004, 0x22000000000, 0x20001000000, 0x20000000800, 0x2000008000, 0x2000000004, 0x11000000, 0x10000800, 0x1000004, 0x8800},
	        new ulong [] {0x4010000000000, 0x2008000000, 0x1004000, 0x802},
	        new ulong [] {0x4000008000000, 0x4000000004000, 0x4000000000002, 0x12000000000, 0x10001000000, 0x10000000800, 0x2000004000, 0x2000000002, 0x9000000, 0x8000800, 0x1000002, 0x4800},
	        new ulong [] {0x4008000000000, 0x2004000000, 0x1002000, 0x801},
	        new ulong [] {0x4000004000000, 0x4000000002000, 0x4000000000001, 0xA000000000, 0x8001000000, 0x8000000800, 0x2000002000, 0x2000000001, 0x5000000, 0x4000800, 0x1000001, 0x2800},
	        new ulong [] {0x3000000000000, 0x1800000000, 0xC00000, 0x600},
	        new ulong [] {0x2000800000000, 0x2000000400000, 0x2000000000200, 0x1001000000000, 0x1000000800000, 0x1000000000400, 0x1000400000, 0x1000000200, 0x800800000, 0x800000400, 0x800200, 0x400400},
	        new ulong [] {0x2800000000000, 0x1400000000, 0xA00000, 0x500},
	        new ulong [] {0x2000400000000, 0x2000000200000, 0x2000000000100, 0x801000000000, 0x800000800000, 0x800000000400, 0x1000200000, 0x1000000100, 0x400800000, 0x400000400, 0x800100, 0x200400},
	        new ulong [] {0x2400000000000, 0x1200000000, 0x900000, 0x480},
	        new ulong [] {0x2000200000000, 0x2000000100000, 0x2000000000080, 0x401000000000, 0x400000800000, 0x400000000400, 0x1000100000, 0x1000000080, 0x200800000, 0x200000400, 0x800080, 0x100400},
	        new ulong [] {0x2200000000000, 0x1100000000, 0x880000, 0x440},
	        new ulong [] {0x2000100000000, 0x2000000080000, 0x2000000000040, 0x201000000000, 0x200000800000, 0x200000000400, 0x1000080000, 0x1000000040, 0x100800000, 0x100000400, 0x800040, 0x80400},
	        new ulong [] {0x2100000000000, 0x1080000000, 0x840000, 0x420},
	        new ulong [] {0x2000080000000, 0x2000000040000, 0x2000000000020, 0x101000000000, 0x100000800000, 0x100000000400, 0x1000040000, 0x1000000020, 0x80800000, 0x80000400, 0x800020, 0x40400},
	        new ulong [] {0x2080000000000, 0x1040000000, 0x820000, 0x410},
	        new ulong [] {0x2000040000000, 0x2000000020000, 0x2000000000010, 0x81000000000, 0x80000800000, 0x80000000400, 0x1000020000, 0x1000000010, 0x40800000, 0x40000400, 0x800010, 0x20400},
	        new ulong [] {0x2040000000000, 0x1020000000, 0x810000, 0x408},
	        new ulong [] {0x2000020000000, 0x2000000010000, 0x2000000000008, 0x41000000000, 0x40000800000, 0x40000000400, 0x1000010000, 0x1000000008, 0x20800000, 0x20000400, 0x800008, 0x10400},
	        new ulong [] {0x2020000000000, 0x1010000000, 0x808000, 0x404},
	        new ulong [] {0x2000010000000, 0x2000000008000, 0x2000000000004, 0x21000000000, 0x20000800000, 0x20000000400, 0x1000008000, 0x1000000004, 0x10800000, 0x10000400, 0x800004, 0x8400},
	        new ulong [] {0x2010000000000, 0x1008000000, 0x804000, 0x402},
	        new ulong [] {0x2000008000000, 0x2000000004000, 0x2000000000002, 0x11000000000, 0x10000800000, 0x10000000400, 0x1000004000, 0x1000000002, 0x8800000, 0x8000400, 0x800002, 0x4400},
	        new ulong [] {0x2008000000000, 0x1004000000, 0x802000, 0x401},
	        new ulong [] {0x2000004000000, 0x2000000002000, 0x2000000000001, 0x9000000000, 0x8000800000, 0x8000000400, 0x1000002000, 0x1000000001, 0x4800000, 0x4000400, 0x800001, 0x2400},
	        new ulong [] {0x1800000000000, 0xC00000000, 0x600000, 0x300},
	        new ulong [] {0x1000400000000, 0x1000000200000, 0x1000000000100, 0x800800000000, 0x800000400000, 0x800000000200, 0x800200000, 0x800000100, 0x400400000, 0x400000200, 0x400100, 0x200200},
	        new ulong [] {0x1400000000000, 0xA00000000, 0x500000, 0x280},
	        new ulong [] {0x1000200000000, 0x1000000100000, 0x1000000000080, 0x400800000000, 0x400000400000, 0x400000000200, 0x800100000, 0x800000080, 0x200400000, 0x200000200, 0x400080, 0x100200},
	        new ulong [] {0x1200000000000, 0x900000000, 0x480000, 0x240},
	        new ulong [] {0x1000100000000, 0x1000000080000, 0x1000000000040, 0x200800000000, 0x200000400000, 0x200000000200, 0x800080000, 0x800000040, 0x100400000, 0x100000200, 0x400040, 0x80200},
	        new ulong [] {0x1100000000000, 0x880000000, 0x440000, 0x220},
	        new ulong [] {0x1000080000000, 0x1000000040000, 0x1000000000020, 0x100800000000, 0x100000400000, 0x100000000200, 0x800040000, 0x800000020, 0x80400000, 0x80000200, 0x400020, 0x40200},
	        new ulong [] {0x1080000000000, 0x840000000, 0x420000, 0x210},
	        new ulong [] {0x1000040000000, 0x1000000020000, 0x1000000000010, 0x80800000000, 0x80000400000, 0x80000000200, 0x800020000, 0x800000010, 0x40400000, 0x40000200, 0x400010, 0x20200},
	        new ulong [] {0x1040000000000, 0x820000000, 0x410000, 0x208},
	        new ulong [] {0x1000020000000, 0x1000000010000, 0x1000000000008, 0x40800000000, 0x40000400000, 0x40000000200, 0x800010000, 0x800000008, 0x20400000, 0x20000200, 0x400008, 0x10200},
	        new ulong [] {0x1020000000000, 0x810000000, 0x408000, 0x204},
	        new ulong [] {0x1000010000000, 0x1000000008000, 0x1000000000004, 0x20800000000, 0x20000400000, 0x20000000200, 0x800008000, 0x800000004, 0x10400000, 0x10000200, 0x400004, 0x8200},
	        new ulong [] {0x1010000000000, 0x808000000, 0x404000, 0x202},
	        new ulong [] {0x1000008000000, 0x1000000004000, 0x1000000000002, 0x10800000000, 0x10000400000, 0x10000000200, 0x800004000, 0x800000002, 0x8400000, 0x8000200, 0x400002, 0x4200},
	        new ulong [] {0x1008000000000, 0x804000000, 0x402000, 0x201},
	        new ulong [] {0x1000004000000, 0x1000000002000, 0x1000000000001, 0x8800000000, 0x8000400000, 0x8000000200, 0x800002000, 0x800000001, 0x4400000, 0x4000200, 0x400001, 0x2200},
	        new ulong [] {0xC00000000000, 0x600000000, 0x300000, 0x180},
	        new ulong [] {0x800200000000, 0x800000100000, 0x800000000080, 0x400400000000, 0x400000200000, 0x400000000100, 0x400100000, 0x400000080, 0x200200000, 0x200000100, 0x200080, 0x100100},
	        new ulong [] {0xA00000000000, 0x500000000, 0x280000, 0x140},
	        new ulong [] {0x800100000000, 0x800000080000, 0x800000000040, 0x200400000000, 0x200000200000, 0x200000000100, 0x400080000, 0x400000040, 0x100200000, 0x100000100, 0x200040, 0x80100},
	        new ulong [] {0x900000000000, 0x480000000, 0x240000, 0x120},
	        new ulong [] {0x800080000000, 0x800000040000, 0x800000000020, 0x100400000000, 0x100000200000, 0x100000000100, 0x400040000, 0x400000020, 0x80200000, 0x80000100, 0x200020, 0x40100},
	        new ulong [] {0x880000000000, 0x440000000, 0x220000, 0x110},
	        new ulong [] {0x800040000000, 0x800000020000, 0x800000000010, 0x80400000000, 0x80000200000, 0x80000000100, 0x400020000, 0x400000010, 0x40200000, 0x40000100, 0x200010, 0x20100},
	        new ulong [] {0x840000000000, 0x420000000, 0x210000, 0x108},
	        new ulong [] {0x800020000000, 0x800000010000, 0x800000000008, 0x40400000000, 0x40000200000, 0x40000000100, 0x400010000, 0x400000008, 0x20200000, 0x20000100, 0x200008, 0x10100},
	        new ulong [] {0x820000000000, 0x410000000, 0x208000, 0x104},
	        new ulong [] {0x800010000000, 0x800000008000, 0x800000000004, 0x20400000000, 0x20000200000, 0x20000000100, 0x400008000, 0x400000004, 0x10200000, 0x10000100, 0x200004, 0x8100},
	        new ulong [] {0x810000000000, 0x408000000, 0x204000, 0x102},
	        new ulong [] {0x800008000000, 0x800000004000, 0x800000000002, 0x10400000000, 0x10000200000, 0x10000000100, 0x400004000, 0x400000002, 0x8200000, 0x8000100, 0x200002, 0x4100},
	        new ulong [] {0x808000000000, 0x404000000, 0x202000, 0x101},
	        new ulong [] {0x800004000000, 0x800000002000, 0x800000000001, 0x8400000000, 0x8000200000, 0x8000000100, 0x400002000, 0x400000001, 0x4200000, 0x4000100, 0x200001, 0x2100},
	        new ulong [] {0x600000000000, 0x300000000, 0x180000, 0xC0},
	        new ulong [] {0x400100000000, 0x400000080000, 0x400000000040, 0x200200000000, 0x200000100000, 0x200000000080, 0x200080000, 0x200000040, 0x100100000, 0x100000080, 0x100040, 0x80080},
	        new ulong [] {0x500000000000, 0x280000000, 0x140000, 0xA0},
	        new ulong [] {0x400080000000, 0x400000040000, 0x400000000020, 0x100200000000, 0x100000100000, 0x100000000080, 0x200040000, 0x200000020, 0x80100000, 0x80000080, 0x100020, 0x40080},
	        new ulong [] {0x480000000000, 0x240000000, 0x120000, 0x90},
	        new ulong [] {0x400040000000, 0x400000020000, 0x400000000010, 0x80200000000, 0x80000100000, 0x80000000080, 0x200020000, 0x200000010, 0x40100000, 0x40000080, 0x100010, 0x20080},
	        new ulong [] {0x440000000000, 0x220000000, 0x110000, 0x88},
	        new ulong [] {0x400020000000, 0x400000010000, 0x400000000008, 0x40200000000, 0x40000100000, 0x40000000080, 0x200010000, 0x200000008, 0x20100000, 0x20000080, 0x100008, 0x10080},
	        new ulong [] {0x420000000000, 0x210000000, 0x108000, 0x84},
	        new ulong [] {0x400010000000, 0x400000008000, 0x400000000004, 0x20200000000, 0x20000100000, 0x20000000080, 0x200008000, 0x200000004, 0x10100000, 0x10000080, 0x100004, 0x8080},
	        new ulong [] {0x410000000000, 0x208000000, 0x104000, 0x82},
	        new ulong [] {0x400008000000, 0x400000004000, 0x400000000002, 0x10200000000, 0x10000100000, 0x10000000080, 0x200004000, 0x200000002, 0x8100000, 0x8000080, 0x100002, 0x4080},
	        new ulong [] {0x408000000000, 0x204000000, 0x102000, 0x81},
	        new ulong [] {0x400004000000, 0x400000002000, 0x400000000001, 0x8200000000, 0x8000100000, 0x8000000080, 0x200002000, 0x200000001, 0x4100000, 0x4000080, 0x100001, 0x2080},
	        new ulong [] {0x300000000000, 0x180000000, 0xC0000, 0x60},
	        new ulong [] {0x200080000000, 0x200000040000, 0x200000000020, 0x100100000000, 0x100000080000, 0x100000000040, 0x100040000, 0x100000020, 0x80080000, 0x80000040, 0x80020, 0x40040},
	        new ulong [] {0x280000000000, 0x140000000, 0xA0000, 0x50},
	        new ulong [] {0x200040000000, 0x200000020000, 0x200000000010, 0x80100000000, 0x80000080000, 0x80000000040, 0x100020000, 0x100000010, 0x40080000, 0x40000040, 0x80010, 0x20040},
	        new ulong [] {0x240000000000, 0x120000000, 0x90000, 0x48},
	        new ulong [] {0x200020000000, 0x200000010000, 0x200000000008, 0x40100000000, 0x40000080000, 0x40000000040, 0x100010000, 0x100000008, 0x20080000, 0x20000040, 0x80008, 0x10040},
	        new ulong [] {0x220000000000, 0x110000000, 0x88000, 0x44},
	        new ulong [] {0x200010000000, 0x200000008000, 0x200000000004, 0x20100000000, 0x20000080000, 0x20000000040, 0x100008000, 0x100000004, 0x10080000, 0x10000040, 0x80004, 0x8040},
	        new ulong [] {0x210000000000, 0x108000000, 0x84000, 0x42},
	        new ulong [] {0x200008000000, 0x200000004000, 0x200000000002, 0x10100000000, 0x10000080000, 0x10000000040, 0x100004000, 0x100000002, 0x8080000, 0x8000040, 0x80002, 0x4040},
	        new ulong [] {0x208000000000, 0x104000000, 0x82000, 0x41},
	        new ulong [] {0x200004000000, 0x200000002000, 0x200000000001, 0x8100000000, 0x8000080000, 0x8000000040, 0x100002000, 0x100000001, 0x4080000, 0x4000040, 0x80001, 0x2040},
	        new ulong [] {0x180000000000, 0xC0000000, 0x60000, 0x30},
	        new ulong [] {0x100040000000, 0x100000020000, 0x100000000010, 0x80080000000, 0x80000040000, 0x80000000020, 0x80020000, 0x80000010, 0x40040000, 0x40000020, 0x40010, 0x20020},
	        new ulong [] {0x140000000000, 0xA0000000, 0x50000, 0x28},
	        new ulong [] {0x100020000000, 0x100000010000, 0x100000000008, 0x40080000000, 0x40000040000, 0x40000000020, 0x80010000, 0x80000008, 0x20040000, 0x20000020, 0x40008, 0x10020},
	        new ulong [] {0x120000000000, 0x90000000, 0x48000, 0x24},
	        new ulong [] {0x100010000000, 0x100000008000, 0x100000000004, 0x20080000000, 0x20000040000, 0x20000000020, 0x80008000, 0x80000004, 0x10040000, 0x10000020, 0x40004, 0x8020},
	        new ulong [] {0x110000000000, 0x88000000, 0x44000, 0x22},
	        new ulong [] {0x100008000000, 0x100000004000, 0x100000000002, 0x10080000000, 0x10000040000, 0x10000000020, 0x80004000, 0x80000002, 0x8040000, 0x8000020, 0x40002, 0x4020},
	        new ulong [] {0x108000000000, 0x84000000, 0x42000, 0x21},
	        new ulong [] {0x100004000000, 0x100000002000, 0x100000000001, 0x8080000000, 0x8000040000, 0x8000000020, 0x80002000, 0x80000001, 0x4040000, 0x4000020, 0x40001, 0x2020},
	        new ulong [] {0xC0000000000, 0x60000000, 0x30000, 0x18},
	        new ulong [] {0x80020000000, 0x80000010000, 0x80000000008, 0x40040000000, 0x40000020000, 0x40000000010, 0x40010000, 0x40000008, 0x20020000, 0x20000010, 0x20008, 0x10010},
	        new ulong [] {0xA0000000000, 0x50000000, 0x28000, 0x14},
	        new ulong [] {0x80010000000, 0x80000008000, 0x80000000004, 0x20040000000, 0x20000020000, 0x20000000010, 0x40008000, 0x40000004, 0x10020000, 0x10000010, 0x20004, 0x8010},
	        new ulong [] {0x90000000000, 0x48000000, 0x24000, 0x12},
	        new ulong [] {0x80008000000, 0x80000004000, 0x80000000002, 0x10040000000, 0x10000020000, 0x10000000010, 0x40004000, 0x40000002, 0x8020000, 0x8000010, 0x20002, 0x4010},
	        new ulong [] {0x88000000000, 0x44000000, 0x22000, 0x11},
	        new ulong [] {0x80004000000, 0x80000002000, 0x80000000001, 0x8040000000, 0x8000020000, 0x8000000010, 0x40002000, 0x40000001, 0x4020000, 0x4000010, 0x20001, 0x2010},
	        new ulong [] {0x60000000000, 0x30000000, 0x18000, 0xC},
	        new ulong [] {0x40010000000, 0x40000008000, 0x40000000004, 0x20020000000, 0x20000010000, 0x20000000008, 0x20008000, 0x20000004, 0x10010000, 0x10000008, 0x10004, 0x8008},
	        new ulong [] {0x50000000000, 0x28000000, 0x14000, 0xA},
	        new ulong [] {0x40008000000, 0x40000004000, 0x40000000002, 0x10020000000, 0x10000010000, 0x10000000008, 0x20004000, 0x20000002, 0x8010000, 0x8000008, 0x10002, 0x4008},
	        new ulong [] {0x48000000000, 0x24000000, 0x12000, 0x9},
	        new ulong [] {0x40004000000, 0x40000002000, 0x40000000001, 0x8020000000, 0x8000010000, 0x8000000008, 0x20002000, 0x20000001, 0x4010000, 0x4000008, 0x10001, 0x2008},
	        new ulong [] {0x30000000000, 0x18000000, 0xC000, 0x6},
	        new ulong [] {0x20008000000, 0x20000004000, 0x20000000002, 0x10010000000, 0x10000008000, 0x10000000004, 0x10004000, 0x10000002, 0x8008000, 0x8000004, 0x8002, 0x4004},
	        new ulong [] {0x28000000000, 0x14000000, 0xA000, 0x5},
	        new ulong [] {0x20004000000, 0x20000002000, 0x20000000001, 0x8010000000, 0x8000008000, 0x8000000004, 0x10002000, 0x10000001, 0x4008000, 0x4000004, 0x8001, 0x2004},
	        new ulong [] {0x18000000000, 0xC000000, 0x6000, 0x3},
	        new ulong [] {0x10004000000, 0x10000002000, 0x10000000001, 0x8008000000, 0x8000004000, 0x8000000002, 0x8002000, 0x8000001, 0x4004000, 0x4000002, 0x4001, 0x2002}
        };

        /// <summary>
        /// The 1326 pocket card combinations.
        /// </summary>
        internal static ulong[] PocketTableMasks =
        {
		    0xA000000000000,
		    0x9000000000000,
		    0x8800000000000,
		    0x8400000000000,
		    0x8200000000000,
		    0x8100000000000,
		    0x8080000000000,
		    0x8040000000000,
		    0x8020000000000,
		    0x8010000000000,
		    0x8008000000000,
		    0x8004000000000,
		    0x8002000000000,
		    0x8001000000000,
		    0x8000800000000,
		    0x8000400000000,
		    0x8000200000000,
		    0x8000100000000,
		    0x8000080000000,
		    0x8000040000000,
		    0x8000020000000,
		    0x8000010000000,
		    0x8000008000000,
		    0x8000004000000,
		    0x8000002000000,
		    0x8000001000000,
		    0x8000000800000,
		    0x8000000400000,
		    0x8000000200000,
		    0x8000000100000,
		    0x8000000080000,
		    0x8000000040000,
		    0x8000000020000,
		    0x8000000010000,
		    0x8000000008000,
		    0x8000000004000,
		    0x8000000002000,
		    0x8000000001000,
		    0x8000000000800,
		    0x8000000000400,
		    0x8000000000200,
		    0x8000000000100,
		    0x8000000000080,
		    0x8000000000040,
		    0x8000000000020,
		    0x8000000000010,
		    0x8000000000008,
		    0x8000000000004,
		    0x8000000000002,
		    0x8000000000001,
		    0x6000000000000,
		    0x5000000000000,
		    0x4800000000000,
		    0x4400000000000,
		    0x4200000000000,
		    0x4100000000000,
		    0x4080000000000,
		    0x4040000000000,
		    0x4020000000000,
		    0x4010000000000,
		    0x4008000000000,
		    0x4004000000000,
		    0x4002000000000,
		    0x4001000000000,
		    0x4000800000000,
		    0x4000400000000,
		    0x4000200000000,
		    0x4000100000000,
		    0x4000080000000,
		    0x4000040000000,
		    0x4000020000000,
		    0x4000010000000,
		    0x4000008000000,
		    0x4000004000000,
		    0x4000002000000,
		    0x4000001000000,
		    0x4000000800000,
		    0x4000000400000,
		    0x4000000200000,
		    0x4000000100000,
		    0x4000000080000,
		    0x4000000040000,
		    0x4000000020000,
		    0x4000000010000,
		    0x4000000008000,
		    0x4000000004000,
		    0x4000000002000,
		    0x4000000001000,
		    0x4000000000800,
		    0x4000000000400,
		    0x4000000000200,
		    0x4000000000100,
		    0x4000000000080,
		    0x4000000000040,
		    0x4000000000020,
		    0x4000000000010,
		    0x4000000000008,
		    0x4000000000004,
		    0x4000000000002,
		    0x4000000000001,
		    0x3000000000000,
		    0x2800000000000,
		    0x2400000000000,
		    0x2200000000000,
		    0x2100000000000,
		    0x2080000000000,
		    0x2040000000000,
		    0x2020000000000,
		    0x2010000000000,
		    0x2008000000000,
		    0x2004000000000,
		    0x2002000000000,
		    0x2001000000000,
		    0x2000800000000,
		    0x2000400000000,
		    0x2000200000000,
		    0x2000100000000,
		    0x2000080000000,
		    0x2000040000000,
		    0x2000020000000,
		    0x2000010000000,
		    0x2000008000000,
		    0x2000004000000,
		    0x2000002000000,
		    0x2000001000000,
		    0x2000000800000,
		    0x2000000400000,
		    0x2000000200000,
		    0x2000000100000,
		    0x2000000080000,
		    0x2000000040000,
		    0x2000000020000,
		    0x2000000010000,
		    0x2000000008000,
		    0x2000000004000,
		    0x2000000002000,
		    0x2000000001000,
		    0x2000000000800,
		    0x2000000000400,
		    0x2000000000200,
		    0x2000000000100,
		    0x2000000000080,
		    0x2000000000040,
		    0x2000000000020,
		    0x2000000000010,
		    0x2000000000008,
		    0x2000000000004,
		    0x2000000000002,
		    0x2000000000001,
		    0x1800000000000,
		    0x1400000000000,
		    0x1200000000000,
		    0x1100000000000,
		    0x1080000000000,
		    0x1040000000000,
		    0x1020000000000,
		    0x1010000000000,
		    0x1008000000000,
		    0x1004000000000,
		    0x1002000000000,
		    0x1001000000000,
		    0x1000800000000,
		    0x1000400000000,
		    0x1000200000000,
		    0x1000100000000,
		    0x1000080000000,
		    0x1000040000000,
		    0x1000020000000,
		    0x1000010000000,
		    0x1000008000000,
		    0x1000004000000,
		    0x1000002000000,
		    0x1000001000000,
		    0x1000000800000,
		    0x1000000400000,
		    0x1000000200000,
		    0x1000000100000,
		    0x1000000080000,
		    0x1000000040000,
		    0x1000000020000,
		    0x1000000010000,
		    0x1000000008000,
		    0x1000000004000,
		    0x1000000002000,
		    0x1000000001000,
		    0x1000000000800,
		    0x1000000000400,
		    0x1000000000200,
		    0x1000000000100,
		    0x1000000000080,
		    0x1000000000040,
		    0x1000000000020,
		    0x1000000000010,
		    0x1000000000008,
		    0x1000000000004,
		    0x1000000000002,
		    0x1000000000001,
		    0xC00000000000,
		    0xA00000000000,
		    0x900000000000,
		    0x880000000000,
		    0x840000000000,
		    0x820000000000,
		    0x810000000000,
		    0x808000000000,
		    0x804000000000,
		    0x802000000000,
		    0x801000000000,
		    0x800800000000,
		    0x800400000000,
		    0x800200000000,
		    0x800100000000,
		    0x800080000000,
		    0x800040000000,
		    0x800020000000,
		    0x800010000000,
		    0x800008000000,
		    0x800004000000,
		    0x800002000000,
		    0x800001000000,
		    0x800000800000,
		    0x800000400000,
		    0x800000200000,
		    0x800000100000,
		    0x800000080000,
		    0x800000040000,
		    0x800000020000,
		    0x800000010000,
		    0x800000008000,
		    0x800000004000,
		    0x800000002000,
		    0x800000001000,
		    0x800000000800,
		    0x800000000400,
		    0x800000000200,
		    0x800000000100,
		    0x800000000080,
		    0x800000000040,
		    0x800000000020,
		    0x800000000010,
		    0x800000000008,
		    0x800000000004,
		    0x800000000002,
		    0x800000000001,
		    0x600000000000,
		    0x500000000000,
		    0x480000000000,
		    0x440000000000,
		    0x420000000000,
		    0x410000000000,
		    0x408000000000,
		    0x404000000000,
		    0x402000000000,
		    0x401000000000,
		    0x400800000000,
		    0x400400000000,
		    0x400200000000,
		    0x400100000000,
		    0x400080000000,
		    0x400040000000,
		    0x400020000000,
		    0x400010000000,
		    0x400008000000,
		    0x400004000000,
		    0x400002000000,
		    0x400001000000,
		    0x400000800000,
		    0x400000400000,
		    0x400000200000,
		    0x400000100000,
		    0x400000080000,
		    0x400000040000,
		    0x400000020000,
		    0x400000010000,
		    0x400000008000,
		    0x400000004000,
		    0x400000002000,
		    0x400000001000,
		    0x400000000800,
		    0x400000000400,
		    0x400000000200,
		    0x400000000100,
		    0x400000000080,
		    0x400000000040,
		    0x400000000020,
		    0x400000000010,
		    0x400000000008,
		    0x400000000004,
		    0x400000000002,
		    0x400000000001,
		    0x300000000000,
		    0x280000000000,
		    0x240000000000,
		    0x220000000000,
		    0x210000000000,
		    0x208000000000,
		    0x204000000000,
		    0x202000000000,
		    0x201000000000,
		    0x200800000000,
		    0x200400000000,
		    0x200200000000,
		    0x200100000000,
		    0x200080000000,
		    0x200040000000,
		    0x200020000000,
		    0x200010000000,
		    0x200008000000,
		    0x200004000000,
		    0x200002000000,
		    0x200001000000,
		    0x200000800000,
		    0x200000400000,
		    0x200000200000,
		    0x200000100000,
		    0x200000080000,
		    0x200000040000,
		    0x200000020000,
		    0x200000010000,
		    0x200000008000,
		    0x200000004000,
		    0x200000002000,
		    0x200000001000,
		    0x200000000800,
		    0x200000000400,
		    0x200000000200,
		    0x200000000100,
		    0x200000000080,
		    0x200000000040,
		    0x200000000020,
		    0x200000000010,
		    0x200000000008,
		    0x200000000004,
		    0x200000000002,
		    0x200000000001,
		    0x180000000000,
		    0x140000000000,
		    0x120000000000,
		    0x110000000000,
		    0x108000000000,
		    0x104000000000,
		    0x102000000000,
		    0x101000000000,
		    0x100800000000,
		    0x100400000000,
		    0x100200000000,
		    0x100100000000,
		    0x100080000000,
		    0x100040000000,
		    0x100020000000,
		    0x100010000000,
		    0x100008000000,
		    0x100004000000,
		    0x100002000000,
		    0x100001000000,
		    0x100000800000,
		    0x100000400000,
		    0x100000200000,
		    0x100000100000,
		    0x100000080000,
		    0x100000040000,
		    0x100000020000,
		    0x100000010000,
		    0x100000008000,
		    0x100000004000,
		    0x100000002000,
		    0x100000001000,
		    0x100000000800,
		    0x100000000400,
		    0x100000000200,
		    0x100000000100,
		    0x100000000080,
		    0x100000000040,
		    0x100000000020,
		    0x100000000010,
		    0x100000000008,
		    0x100000000004,
		    0x100000000002,
		    0x100000000001,
		    0xC0000000000,
		    0xA0000000000,
		    0x90000000000,
		    0x88000000000,
		    0x84000000000,
		    0x82000000000,
		    0x81000000000,
		    0x80800000000,
		    0x80400000000,
		    0x80200000000,
		    0x80100000000,
		    0x80080000000,
		    0x80040000000,
		    0x80020000000,
		    0x80010000000,
		    0x80008000000,
		    0x80004000000,
		    0x80002000000,
		    0x80001000000,
		    0x80000800000,
		    0x80000400000,
		    0x80000200000,
		    0x80000100000,
		    0x80000080000,
		    0x80000040000,
		    0x80000020000,
		    0x80000010000,
		    0x80000008000,
		    0x80000004000,
		    0x80000002000,
		    0x80000001000,
		    0x80000000800,
		    0x80000000400,
		    0x80000000200,
		    0x80000000100,
		    0x80000000080,
		    0x80000000040,
		    0x80000000020,
		    0x80000000010,
		    0x80000000008,
		    0x80000000004,
		    0x80000000002,
		    0x80000000001,
		    0x60000000000,
		    0x50000000000,
		    0x48000000000,
		    0x44000000000,
		    0x42000000000,
		    0x41000000000,
		    0x40800000000,
		    0x40400000000,
		    0x40200000000,
		    0x40100000000,
		    0x40080000000,
		    0x40040000000,
		    0x40020000000,
		    0x40010000000,
		    0x40008000000,
		    0x40004000000,
		    0x40002000000,
		    0x40001000000,
		    0x40000800000,
		    0x40000400000,
		    0x40000200000,
		    0x40000100000,
		    0x40000080000,
		    0x40000040000,
		    0x40000020000,
		    0x40000010000,
		    0x40000008000,
		    0x40000004000,
		    0x40000002000,
		    0x40000001000,
		    0x40000000800,
		    0x40000000400,
		    0x40000000200,
		    0x40000000100,
		    0x40000000080,
		    0x40000000040,
		    0x40000000020,
		    0x40000000010,
		    0x40000000008,
		    0x40000000004,
		    0x40000000002,
		    0x40000000001,
		    0x30000000000,
		    0x28000000000,
		    0x24000000000,
		    0x22000000000,
		    0x21000000000,
		    0x20800000000,
		    0x20400000000,
		    0x20200000000,
		    0x20100000000,
		    0x20080000000,
		    0x20040000000,
		    0x20020000000,
		    0x20010000000,
		    0x20008000000,
		    0x20004000000,
		    0x20002000000,
		    0x20001000000,
		    0x20000800000,
		    0x20000400000,
		    0x20000200000,
		    0x20000100000,
		    0x20000080000,
		    0x20000040000,
		    0x20000020000,
		    0x20000010000,
		    0x20000008000,
		    0x20000004000,
		    0x20000002000,
		    0x20000001000,
		    0x20000000800,
		    0x20000000400,
		    0x20000000200,
		    0x20000000100,
		    0x20000000080,
		    0x20000000040,
		    0x20000000020,
		    0x20000000010,
		    0x20000000008,
		    0x20000000004,
		    0x20000000002,
		    0x20000000001,
		    0x18000000000,
		    0x14000000000,
		    0x12000000000,
		    0x11000000000,
		    0x10800000000,
		    0x10400000000,
		    0x10200000000,
		    0x10100000000,
		    0x10080000000,
		    0x10040000000,
		    0x10020000000,
		    0x10010000000,
		    0x10008000000,
		    0x10004000000,
		    0x10002000000,
		    0x10001000000,
		    0x10000800000,
		    0x10000400000,
		    0x10000200000,
		    0x10000100000,
		    0x10000080000,
		    0x10000040000,
		    0x10000020000,
		    0x10000010000,
		    0x10000008000,
		    0x10000004000,
		    0x10000002000,
		    0x10000001000,
		    0x10000000800,
		    0x10000000400,
		    0x10000000200,
		    0x10000000100,
		    0x10000000080,
		    0x10000000040,
		    0x10000000020,
		    0x10000000010,
		    0x10000000008,
		    0x10000000004,
		    0x10000000002,
		    0x10000000001,
		    0xC000000000,
		    0xA000000000,
		    0x9000000000,
		    0x8800000000,
		    0x8400000000,
		    0x8200000000,
		    0x8100000000,
		    0x8080000000,
		    0x8040000000,
		    0x8020000000,
		    0x8010000000,
		    0x8008000000,
		    0x8004000000,
		    0x8002000000,
		    0x8001000000,
		    0x8000800000,
		    0x8000400000,
		    0x8000200000,
		    0x8000100000,
		    0x8000080000,
		    0x8000040000,
		    0x8000020000,
		    0x8000010000,
		    0x8000008000,
		    0x8000004000,
		    0x8000002000,
		    0x8000001000,
		    0x8000000800,
		    0x8000000400,
		    0x8000000200,
		    0x8000000100,
		    0x8000000080,
		    0x8000000040,
		    0x8000000020,
		    0x8000000010,
		    0x8000000008,
		    0x8000000004,
		    0x8000000002,
		    0x8000000001,
		    0x6000000000,
		    0x5000000000,
		    0x4800000000,
		    0x4400000000,
		    0x4200000000,
		    0x4100000000,
		    0x4080000000,
		    0x4040000000,
		    0x4020000000,
		    0x4010000000,
		    0x4008000000,
		    0x4004000000,
		    0x4002000000,
		    0x4001000000,
		    0x4000800000,
		    0x4000400000,
		    0x4000200000,
		    0x4000100000,
		    0x4000080000,
		    0x4000040000,
		    0x4000020000,
		    0x4000010000,
		    0x4000008000,
		    0x4000004000,
		    0x4000002000,
		    0x4000001000,
		    0x4000000800,
		    0x4000000400,
		    0x4000000200,
		    0x4000000100,
		    0x4000000080,
		    0x4000000040,
		    0x4000000020,
		    0x4000000010,
		    0x4000000008,
		    0x4000000004,
		    0x4000000002,
		    0x4000000001,
		    0x3000000000,
		    0x2800000000,
		    0x2400000000,
		    0x2200000000,
		    0x2100000000,
		    0x2080000000,
		    0x2040000000,
		    0x2020000000,
		    0x2010000000,
		    0x2008000000,
		    0x2004000000,
		    0x2002000000,
		    0x2001000000,
		    0x2000800000,
		    0x2000400000,
		    0x2000200000,
		    0x2000100000,
		    0x2000080000,
		    0x2000040000,
		    0x2000020000,
		    0x2000010000,
		    0x2000008000,
		    0x2000004000,
		    0x2000002000,
		    0x2000001000,
		    0x2000000800,
		    0x2000000400,
		    0x2000000200,
		    0x2000000100,
		    0x2000000080,
		    0x2000000040,
		    0x2000000020,
		    0x2000000010,
		    0x2000000008,
		    0x2000000004,
		    0x2000000002,
		    0x2000000001,
		    0x1800000000,
		    0x1400000000,
		    0x1200000000,
		    0x1100000000,
		    0x1080000000,
		    0x1040000000,
		    0x1020000000,
		    0x1010000000,
		    0x1008000000,
		    0x1004000000,
		    0x1002000000,
		    0x1001000000,
		    0x1000800000,
		    0x1000400000,
		    0x1000200000,
		    0x1000100000,
		    0x1000080000,
		    0x1000040000,
		    0x1000020000,
		    0x1000010000,
		    0x1000008000,
		    0x1000004000,
		    0x1000002000,
		    0x1000001000,
		    0x1000000800,
		    0x1000000400,
		    0x1000000200,
		    0x1000000100,
		    0x1000000080,
		    0x1000000040,
		    0x1000000020,
		    0x1000000010,
		    0x1000000008,
		    0x1000000004,
		    0x1000000002,
		    0x1000000001,
		    0xC00000000,
		    0xA00000000,
		    0x900000000,
		    0x880000000,
		    0x840000000,
		    0x820000000,
		    0x810000000,
		    0x808000000,
		    0x804000000,
		    0x802000000,
		    0x801000000,
		    0x800800000,
		    0x800400000,
		    0x800200000,
		    0x800100000,
		    0x800080000,
		    0x800040000,
		    0x800020000,
		    0x800010000,
		    0x800008000,
		    0x800004000,
		    0x800002000,
		    0x800001000,
		    0x800000800,
		    0x800000400,
		    0x800000200,
		    0x800000100,
		    0x800000080,
		    0x800000040,
		    0x800000020,
		    0x800000010,
		    0x800000008,
		    0x800000004,
		    0x800000002,
		    0x800000001,
		    0x600000000,
		    0x500000000,
		    0x480000000,
		    0x440000000,
		    0x420000000,
		    0x410000000,
		    0x408000000,
		    0x404000000,
		    0x402000000,
		    0x401000000,
		    0x400800000,
		    0x400400000,
		    0x400200000,
		    0x400100000,
		    0x400080000,
		    0x400040000,
		    0x400020000,
		    0x400010000,
		    0x400008000,
		    0x400004000,
		    0x400002000,
		    0x400001000,
		    0x400000800,
		    0x400000400,
		    0x400000200,
		    0x400000100,
		    0x400000080,
		    0x400000040,
		    0x400000020,
		    0x400000010,
		    0x400000008,
		    0x400000004,
		    0x400000002,
		    0x400000001,
		    0x300000000,
		    0x280000000,
		    0x240000000,
		    0x220000000,
		    0x210000000,
		    0x208000000,
		    0x204000000,
		    0x202000000,
		    0x201000000,
		    0x200800000,
		    0x200400000,
		    0x200200000,
		    0x200100000,
		    0x200080000,
		    0x200040000,
		    0x200020000,
		    0x200010000,
		    0x200008000,
		    0x200004000,
		    0x200002000,
		    0x200001000,
		    0x200000800,
		    0x200000400,
		    0x200000200,
		    0x200000100,
		    0x200000080,
		    0x200000040,
		    0x200000020,
		    0x200000010,
		    0x200000008,
		    0x200000004,
		    0x200000002,
		    0x200000001,
		    0x180000000,
		    0x140000000,
		    0x120000000,
		    0x110000000,
		    0x108000000,
		    0x104000000,
		    0x102000000,
		    0x101000000,
		    0x100800000,
		    0x100400000,
		    0x100200000,
		    0x100100000,
		    0x100080000,
		    0x100040000,
		    0x100020000,
		    0x100010000,
		    0x100008000,
		    0x100004000,
		    0x100002000,
		    0x100001000,
		    0x100000800,
		    0x100000400,
		    0x100000200,
		    0x100000100,
		    0x100000080,
		    0x100000040,
		    0x100000020,
		    0x100000010,
		    0x100000008,
		    0x100000004,
		    0x100000002,
		    0x100000001,
		    0xC0000000,
		    0xA0000000,
		    0x90000000,
		    0x88000000,
		    0x84000000,
		    0x82000000,
		    0x81000000,
		    0x80800000,
		    0x80400000,
		    0x80200000,
		    0x80100000,
		    0x80080000,
		    0x80040000,
		    0x80020000,
		    0x80010000,
		    0x80008000,
		    0x80004000,
		    0x80002000,
		    0x80001000,
		    0x80000800,
		    0x80000400,
		    0x80000200,
		    0x80000100,
		    0x80000080,
		    0x80000040,
		    0x80000020,
		    0x80000010,
		    0x80000008,
		    0x80000004,
		    0x80000002,
		    0x80000001,
		    0x60000000,
		    0x50000000,
		    0x48000000,
		    0x44000000,
		    0x42000000,
		    0x41000000,
		    0x40800000,
		    0x40400000,
		    0x40200000,
		    0x40100000,
		    0x40080000,
		    0x40040000,
		    0x40020000,
		    0x40010000,
		    0x40008000,
		    0x40004000,
		    0x40002000,
		    0x40001000,
		    0x40000800,
		    0x40000400,
		    0x40000200,
		    0x40000100,
		    0x40000080,
		    0x40000040,
		    0x40000020,
		    0x40000010,
		    0x40000008,
		    0x40000004,
		    0x40000002,
		    0x40000001,
		    0x30000000,
		    0x28000000,
		    0x24000000,
		    0x22000000,
		    0x21000000,
		    0x20800000,
		    0x20400000,
		    0x20200000,
		    0x20100000,
		    0x20080000,
		    0x20040000,
		    0x20020000,
		    0x20010000,
		    0x20008000,
		    0x20004000,
		    0x20002000,
		    0x20001000,
		    0x20000800,
		    0x20000400,
		    0x20000200,
		    0x20000100,
		    0x20000080,
		    0x20000040,
		    0x20000020,
		    0x20000010,
		    0x20000008,
		    0x20000004,
		    0x20000002,
		    0x20000001,
		    0x18000000,
		    0x14000000,
		    0x12000000,
		    0x11000000,
		    0x10800000,
		    0x10400000,
		    0x10200000,
		    0x10100000,
		    0x10080000,
		    0x10040000,
		    0x10020000,
		    0x10010000,
		    0x10008000,
		    0x10004000,
		    0x10002000,
		    0x10001000,
		    0x10000800,
		    0x10000400,
		    0x10000200,
		    0x10000100,
		    0x10000080,
		    0x10000040,
		    0x10000020,
		    0x10000010,
		    0x10000008,
		    0x10000004,
		    0x10000002,
		    0x10000001,
		    0xC000000,
		    0xA000000,
		    0x9000000,
		    0x8800000,
		    0x8400000,
		    0x8200000,
		    0x8100000,
		    0x8080000,
		    0x8040000,
		    0x8020000,
		    0x8010000,
		    0x8008000,
		    0x8004000,
		    0x8002000,
		    0x8001000,
		    0x8000800,
		    0x8000400,
		    0x8000200,
		    0x8000100,
		    0x8000080,
		    0x8000040,
		    0x8000020,
		    0x8000010,
		    0x8000008,
		    0x8000004,
		    0x8000002,
		    0x8000001,
		    0x6000000,
		    0x5000000,
		    0x4800000,
		    0x4400000,
		    0x4200000,
		    0x4100000,
		    0x4080000,
		    0x4040000,
		    0x4020000,
		    0x4010000,
		    0x4008000,
		    0x4004000,
		    0x4002000,
		    0x4001000,
		    0x4000800,
		    0x4000400,
		    0x4000200,
		    0x4000100,
		    0x4000080,
		    0x4000040,
		    0x4000020,
		    0x4000010,
		    0x4000008,
		    0x4000004,
		    0x4000002,
		    0x4000001,
		    0x3000000,
		    0x2800000,
		    0x2400000,
		    0x2200000,
		    0x2100000,
		    0x2080000,
		    0x2040000,
		    0x2020000,
		    0x2010000,
		    0x2008000,
		    0x2004000,
		    0x2002000,
		    0x2001000,
		    0x2000800,
		    0x2000400,
		    0x2000200,
		    0x2000100,
		    0x2000080,
		    0x2000040,
		    0x2000020,
		    0x2000010,
		    0x2000008,
		    0x2000004,
		    0x2000002,
		    0x2000001,
		    0x1800000,
		    0x1400000,
		    0x1200000,
		    0x1100000,
		    0x1080000,
		    0x1040000,
		    0x1020000,
		    0x1010000,
		    0x1008000,
		    0x1004000,
		    0x1002000,
		    0x1001000,
		    0x1000800,
		    0x1000400,
		    0x1000200,
		    0x1000100,
		    0x1000080,
		    0x1000040,
		    0x1000020,
		    0x1000010,
		    0x1000008,
		    0x1000004,
		    0x1000002,
		    0x1000001,
		    0xC00000,
		    0xA00000,
		    0x900000,
		    0x880000,
		    0x840000,
		    0x820000,
		    0x810000,
		    0x808000,
		    0x804000,
		    0x802000,
		    0x801000,
		    0x800800,
		    0x800400,
		    0x800200,
		    0x800100,
		    0x800080,
		    0x800040,
		    0x800020,
		    0x800010,
		    0x800008,
		    0x800004,
		    0x800002,
		    0x800001,
		    0x600000,
		    0x500000,
		    0x480000,
		    0x440000,
		    0x420000,
		    0x410000,
		    0x408000,
		    0x404000,
		    0x402000,
		    0x401000,
		    0x400800,
		    0x400400,
		    0x400200,
		    0x400100,
		    0x400080,
		    0x400040,
		    0x400020,
		    0x400010,
		    0x400008,
		    0x400004,
		    0x400002,
		    0x400001,
		    0x300000,
		    0x280000,
		    0x240000,
		    0x220000,
		    0x210000,
		    0x208000,
		    0x204000,
		    0x202000,
		    0x201000,
		    0x200800,
		    0x200400,
		    0x200200,
		    0x200100,
		    0x200080,
		    0x200040,
		    0x200020,
		    0x200010,
		    0x200008,
		    0x200004,
		    0x200002,
		    0x200001,
		    0x180000,
		    0x140000,
		    0x120000,
		    0x110000,
		    0x108000,
		    0x104000,
		    0x102000,
		    0x101000,
		    0x100800,
		    0x100400,
		    0x100200,
		    0x100100,
		    0x100080,
		    0x100040,
		    0x100020,
		    0x100010,
		    0x100008,
		    0x100004,
		    0x100002,
		    0x100001,
		    0xC0000,
		    0xA0000,
		    0x90000,
		    0x88000,
		    0x84000,
		    0x82000,
		    0x81000,
		    0x80800,
		    0x80400,
		    0x80200,
		    0x80100,
		    0x80080,
		    0x80040,
		    0x80020,
		    0x80010,
		    0x80008,
		    0x80004,
		    0x80002,
		    0x80001,
		    0x60000,
		    0x50000,
		    0x48000,
		    0x44000,
		    0x42000,
		    0x41000,
		    0x40800,
		    0x40400,
		    0x40200,
		    0x40100,
		    0x40080,
		    0x40040,
		    0x40020,
		    0x40010,
		    0x40008,
		    0x40004,
		    0x40002,
		    0x40001,
		    0x30000,
		    0x28000,
		    0x24000,
		    0x22000,
		    0x21000,
		    0x20800,
		    0x20400,
		    0x20200,
		    0x20100,
		    0x20080,
		    0x20040,
		    0x20020,
		    0x20010,
		    0x20008,
		    0x20004,
		    0x20002,
		    0x20001,
		    0x18000,
		    0x14000,
		    0x12000,
		    0x11000,
		    0x10800,
		    0x10400,
		    0x10200,
		    0x10100,
		    0x10080,
		    0x10040,
		    0x10020,
		    0x10010,
		    0x10008,
		    0x10004,
		    0x10002,
		    0x10001,
		    0xC000,
		    0xA000,
		    0x9000,
		    0x8800,
		    0x8400,
		    0x8200,
		    0x8100,
		    0x8080,
		    0x8040,
		    0x8020,
		    0x8010,
		    0x8008,
		    0x8004,
		    0x8002,
		    0x8001,
		    0x6000,
		    0x5000,
		    0x4800,
		    0x4400,
		    0x4200,
		    0x4100,
		    0x4080,
		    0x4040,
		    0x4020,
		    0x4010,
		    0x4008,
		    0x4004,
		    0x4002,
		    0x4001,
		    0x3000,
		    0x2800,
		    0x2400,
		    0x2200,
		    0x2100,
		    0x2080,
		    0x2040,
		    0x2020,
		    0x2010,
		    0x2008,
		    0x2004,
		    0x2002,
		    0x2001,
		    0x1800,
		    0x1400,
		    0x1200,
		    0x1100,
		    0x1080,
		    0x1040,
		    0x1020,
		    0x1010,
		    0x1008,
		    0x1004,
		    0x1002,
		    0x1001,
		    0xC00,
		    0xA00,
		    0x900,
		    0x880,
		    0x840,
		    0x820,
		    0x810,
		    0x808,
		    0x804,
		    0x802,
		    0x801,
		    0x600,
		    0x500,
		    0x480,
		    0x440,
		    0x420,
		    0x410,
		    0x408,
		    0x404,
		    0x402,
		    0x401,
		    0x300,
		    0x280,
		    0x240,
		    0x220,
		    0x210,
		    0x208,
		    0x204,
		    0x202,
		    0x201,
		    0x180,
		    0x140,
		    0x120,
		    0x110,
		    0x108,
		    0x104,
		    0x102,
		    0x101,
		    0xC0,
		    0xA0,
		    0x90,
		    0x88,
		    0x84,
		    0x82,
		    0x81,
		    0x60,
		    0x50,
		    0x48,
		    0x44,
		    0x42,
		    0x41,
		    0x30,
		    0x28,
		    0x24,
		    0x22,
		    0x21,
		    0x18,
		    0x14,
		    0x12,
		    0x11,
		    0xC,
		    0xA,
		    0x9,
		    0x6,
		    0x5,
		    0x3
        };
        #endregion

        #region Pocket 169 Hand Definition Strings
        /// <summary>
        /// A string table cooresponding to the PocketHand169Enum enumeration. 
        /// The index of each string should correspond to the integer value
        /// associated with enumerator entry.
        /// </summary>
        static internal readonly string[] PocketCards169Strings = {
		    "AA", "KK", "QQ", "JJ", "TT", "99", "88", "77", "66", "55", "44", "33", "22", 
		    "AKs", "AKo", "AQs", "AQo", "AJs", "AJo", "ATs", "ATo", "A9s", "A9o", 
            "A8s", "A8o", "A7s", "A7o", "A6s", "A6o", "A5s", "A5o", "A4s", "A4o",
		    "A3s", "A3o", "A2s", "A2o", "KQs", "KQo", "KJs", "KJo", "KTs", "KTo",
		    "K9s", "K9o", "K8s", "K8o", "K7s", "K7o", "K6s", "K6o", "K5s", "K5o",
		    "K4s", "K4o", "K3s", "K3o", "K2s", "K2o", "QJs", "QJo", "QTs", "QTo", 
            "Q9s", "Q9o", "Q8s", "Q8o", "Q7s", "Q7o", "Q6s", "Q6o", "Q5s", "Q5o",
		    "Q4s", "Q4o", "Q3s", "Q3o", "Q2s", "Q2o", "JTs", "JTo", "J9s", "J9o",
            "J8s", "J8o", "J7s", "J7o", "J6s", "J6o", "J5s", "J5o", "J4s", "J4o",
		    "J3s", "J3o", "J2s", "J2o", "T9s", "T9o", "T8s", "T8o", "T7s", "T7o",  
            "T6s", "T6o", "T5s", "T5o", "T4s", "T4o", "T3s", "T3o", "T2s", "T2o",
		    "98s", "98o", "97s", "97o", "96s", "96o", "95s", "95o", "94s", "94o",
            "93s", "93o", "92s", "92o", "87s", "87o", "86s", "86o", "85s", "85o",
		    "84s", "84o", "83s", "83o", "82s", "82o", "76s", "76o", "75s", "75o",
		    "74s", "74o", "73s", "73o", "72s", "72o", "65s", "65o", "64s", "64o",
		    "63s", "63o", "62s", "62o", "54s", "54o", "53s", "53o", "52s", "52o",
		    "43s", "43o", "42s", "42o", "32s", "32o"            
        };

        /// <exclude/>
        static private Dictionary<string, int> pocket169StringToEnum = new Dictionary<string, int>();

        /// <summary>
        /// Takes as Pocket Card 169 string definition and returns the cooresponding PocketHand169Enum enum
        /// value.
        /// </summary>
        /// <param name="s">String representing the pocket card 169 type</param>
        /// <returns>The equivalent enum value in PocketHand169Enum</returns>
        static public Hand.PocketHand169Enum PocketCard169StringToEnum(string s)
        {
            if (pocket169StringToEnum.Count == 0)
            {
                for (int i = 0; i < PocketCards169Strings.Length; i++)
                {
                    pocket169StringToEnum.Add(PocketCards169Strings[i].ToLower(), i);
                }
            }

#if DEBUG
            if (!pocket169StringToEnum.ContainsKey(s.ToLower()))
                throw new ArgumentException("Passed string is illegal value");
#endif

            return (Hand.PocketHand169Enum)pocket169StringToEnum[s.ToLower()];
        }

        #endregion

        #region Pocket 169 Win Probablity

        /// <summary>
        /// Probablity of beating one other random player with
        /// This pocket card combination.
        /// </summary>
        static internal readonly double[] _Pocket169ProbTable = {
            /* 0 */	    0.85203713302101,
            /* 1 */	    0.823956797867859,
            /* 2 */	    0.799251640610832,
            /* 3 */	    0.774694729011499,
            /* 4 */	    0.750117799509566,
            /* 5 */	    0.720572519451534,
            /* 6 */     0.691630354690022,
            /* 7 */	    0.662360227947317,
            /* 8 */	    0.632847482165574,
            /* 9 */	    0.603249205128748,
            /* 10 */	0.570228211908204,
            /* 11 */	0.536930763867793,
            /* 12 */	0.503340190784356,
            /* 13 */	0.670446323092352,
            /* 14 */	0.65320071788702,
            /* 15 */	0.662088623973122,
            /* 16 */	0.644318393729818,
            /* 17 */	0.653926790321993,
            /* 18 */	0.635632579118604,
            /* 19 */	0.64602386787698,
            /* 20 */	0.627216546375229,
            /* 21 */	0.627812139166209,
            /* 22 */	0.60772806387994,
            /* 23 */	0.619438106403383,
            /* 24 */	0.598726050886253,
            /* 25 */	0.609839585513234,
            /* 26 */	0.588411954219077,
            /* 27 */	0.599058280419784,
            /* 28 */	0.576824522958063,
            /* 29 */	0.599229256162982,
            /* 30 */	0.576965343603873,
            /* 31 */	0.590336359784292,
            /* 32 */	0.567296776263837,
            /* 33 */	0.582203205953702,
            /* 34 */	0.558446023364915,
            /* 35 */	0.573788978630726,
            /* 36 */	0.549285586757339,
            /* 37 */	0.634004032947802,
            /* 38 */	0.614558000477123,
            /* 39 */	0.625673401309056,
            /* 40 */	0.60568685161952,
            /* 41 */	0.617885581684809,
            /* 42 */	0.597389151382808,
            /* 43 */	0.599884755110241,
            /* 44 */	0.578119246563313,
            /* 45 */	0.583123499336662,
            /* 46 */	0.560201725575718,
            /* 47 */	0.575377375055087,
            /* 48 */	0.55187350172037,
            /* 49 */	0.566407355235986,
            /* 50 */	0.542232789437924,
            /* 51 */	0.557929176365974,
            /* 52 */	0.53313972833548,
            /* 53 */	0.548846365684446,
            /* 54 */	0.523274721053729,
            /* 55 */	0.540549764575468,
            /* 56 */	0.514256896448485,
            /* 57 */	0.532117283293773,
            /* 58 */	0.505087237751603,
            /* 59 */	0.602592051411432,
            /* 60 */	0.581346896774576,
            /* 61 */	0.59467559308084,
            /* 62 */	0.572907825970632,
            /* 63 */	0.576643217178106,
            /* 64 */	0.553604349246777,
            /* 65 */	0.560177329755102,
            /* 66 */	0.535997920739232,
            /* 67 */	0.543022632019758,
            /* 68 */	0.517656659431636,
            /* 69 */	0.536125664363242,
            /* 70 */	0.51024052304464,
            /* 71 */	0.527694108913714,
            /* 72 */	0.501200827918979,
            /* 73 */	0.518553020386805,
            /* 74 */	0.491276841028229,
            /* 75 */	0.510192464870343,
            /* 76 */	0.48219436025188,
            /* 77 */	0.501690352619056,
            /* 78 */	0.472954368821787,
            /* 79 */	0.575278571075783,
            /* 80 */	0.55247703082859,
            /* 81 */	0.556624705540557,
            /* 82 */	0.532511968835974,
            /* 83 */	0.540156441799101,
            /* 84 */	0.514901630093912,
            /* 85 */	0.523247811851453,
            /* 86 */	0.49681933600957,
            /* 87 */	0.50605907142943,
            /* 88 */	0.478442730510756,
            /* 89 */	0.49986849512322,
            /* 90 */	0.471808882258367,
            /* 91 */	0.490704533965073,
            /* 92 */	0.461863845319475,
            /* 93 */	0.48231624019271,
            /* 94 */	0.452755448870323,
            /* 95 */	0.473781524060862,
            /* 96 */	0.443484676142764,
            /* 97 */	0.540275286564602,
            /* 98 */	0.515316723990075,
            /* 99 */	0.52334370723032,
            /* 100 */	0.497212736733187,
            /* 101 */	0.506390375369165,
            /* 102 */	0.479081355189456,
            /* 103 */	0.489406756829943,
            /* 104 */	0.460920032843682,
            /* 105 */	0.472162589715616,
            /* 106 */	0.442509495500608,
            /* 107 */	0.465304936077534,
            /* 108 */	0.435041080107652,
            /* 109 */	0.456925120200857,
            /* 110 */	0.425945508483998,
            /* 111 */	0.448394827277476,
            /* 112 */	0.416683505894719,
            /* 113 */	0.508007553875137,
            /* 114 */	0.480970327651146,
            /* 115 */	0.491177310971483,
            /* 116 */	0.462978064070637,
            /* 117 */	0.474282907707977,
            /* 118 */	0.444913452570219,
            /* 119 */	0.457218745584181,
            /* 120 */	0.426691428148082,
            /* 121 */	0.438619708192194,
            /* 122 */	0.406710535235876,
            /* 123 */	0.432642577915308,
            /* 124 */	0.400195143442963,
            /* 125 */	0.4241517186725,
            /* 126 */	0.390979362857749,
            /* 127 */	0.479363402426538,
            /* 128 */	0.450508122627853,
            /* 129 */	0.462432692668916,
            /* 130 */	0.432409018635066,
            /* 131 */	0.445449927020397,
            /* 132 */	0.414275259819399,
            /* 133 */	0.427016272954392,
            /* 134 */	0.394467914671265,
            /* 135 */	0.408735041040776,
            /* 136 */	0.374838125968858,
            /* 137 */	0.402716344379817,
            /* 138 */	0.368276741007843,
            /* 139 */	0.453717666431919,
            /* 140 */	0.423227468811088,
            /* 141 */	0.436755366346354,
            /* 142 */	0.405119688598115,
            /* 143 */	0.418493118759572,
            /* 144 */	0.385498278390772,
            /* 145 */	0.400359358752051,
            /* 146 */	0.366022569948003,
            /* 147 */	0.381558934747616,
            /* 148 */	0.345836473153442,
            /* 149 */	0.431333862182779,
            /* 150 */	0.399443023039395,
            /* 151 */	0.413333190310856,
            /* 152 */	0.380104882434571,
            /* 153 */	0.395335599381456,
            /* 154 */	0.360776309556705,
            /* 155 */	0.376689641082234,
            /* 156 */	0.34075138336107,
            /* 157 */	0.414534203682314,
            /* 158 */	0.381552870832969,
            /* 159 */	0.396929623978653,
            /* 160 */	0.362647711230373,
            /* 161 */	0.378493279898229,
            /* 162 */	0.342846455025819,
            /* 163 */	0.386419483780393,
            /* 164 */	0.351458933908551,
            /* 165 */	0.36829014721971,
            /* 166 */	0.331997501254307,
            /* 167 */	0.359844305970082,
            /* 168 */	0.323032281269529   
        };

        /// <summary>
        /// Returns the probablity of the specified two card pocket mask winning against
        /// a random opponent. This is just a table lookup so the results are fairly quick.
        /// </summary>
        /// <param name="mask">2 card pocket mask</param>
        static public double WinOdds(ulong mask)
        {
#if DEBUG
            if (Hand.BitCount(mask) != 2) throw new ArgumentOutOfRangeException("mask");
#endif
            return _Pocket169ProbTable[(int) Hand.PocketHand169Type(mask)];
        }
        #endregion

        #region Pocket 169 Mask to Enum Lookup Method

        /// <summary>
        /// Returns the number of elements of the pocket type corresponding
        /// to this mask.
        /// </summary>
        /// <param name="mask"></param>
        /// <returns></returns>
        static public long PocketHand169TypeCount(ulong mask)
        {
            return _Pocket169Combinations[(int)Hand.PocketHand169Type(mask)].Length;
        }

        /// <summary>
        /// Returns true if the 2 card pocket mask passed is connected. This function is a lookup so
        /// it is reasonably fast.
        /// </summary>
        /// <param name="mask"></param>
        /// <returns></returns>
        static public bool IsConnected(ulong mask)
        {
            return _PocketCards169Connected[(int)Hand.PocketHand169Type(mask)];
        }

        /// <summary>
        /// Returns true if the two card pocket mask passed is suited. This method is a lookup so it is 
        /// quite fast.
        /// </summary>
        /// <param name="mask"></param>
        /// <returns></returns>
        static public bool IsSuited(ulong mask)
        {
            return _PocketCards169Suited[(int)Hand.PocketHand169Type(mask)];
        }

        /// <summary>
        /// This method returns the gap count which is the distance between two pocket cards.
        /// For example, As 2h has a gap of 0 while As 3h has a gap of 1. The values returned are
        /// 0, 1, 2, 3 or -1.
        /// </summary>
        /// <param name="mask"></param>
        /// <returns></returns>
        static public int GapCount(ulong mask)
        {
            return _PocketCards169Gap[(int)Hand.PocketHand169Type(mask)];
        }

        #endregion

        #endregion

        #region PocketCard169
        /// <exclude/>
        static internal Dictionary<string, Hand.PocketHand169Enum> _card169ToTypeTable = new Dictionary<string, Hand.PocketHand169Enum>();

        /// <summary>
        /// Given a string, the corresponding PocketHand169Enum is returned.
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        static internal Hand.PocketHand169Enum Pocket169(string s)
        {
            if (_card169ToTypeTable.Count == 0)
            {
                for (int i = 0; i < PocketCards169Strings.Length; i++)
                {
                    _card169ToTypeTable.Add(PocketCards169Strings[i].ToLower(), (Hand.PocketHand169Enum)i);
                }
            }
            return _card169ToTypeTable[FixCard169(s)];
        }
        #endregion

        #region Card169 Fix
        /// <exclude/>
        static internal Dictionary<string, string> _card169fixtable = new Dictionary<string, string>();

        /// <summary>
        /// I goofed up the text description of the 169 card hands. I originally
        /// used my own syntax rather than the one Sklansky uses in his book. I released
        /// after working through is books, that I should have used his syntax (even
        /// though I think it's a bit confusing). This functions converts the Sklansky 
        /// syntax to mine. 
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        static internal bool FindFixCard169(string s)
        {
            BuildFix169Table();
            return _card169fixtable.ContainsKey(s.ToLower());
        }

        /// <summary>
        /// See FindFixCard169() description.
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        static internal string FixCard169(string s)
        {
            BuildFix169Table();
            return _card169fixtable[s.ToLower()];
        }

        /// <summary>
        /// See FindFixCard169 description
        /// </summary>
        static internal void BuildFix169Table()
        {
            // Should only fill the Dictionary once.
            if (_card169fixtable.Count == 0)
            {
                _card169fixtable.Add("aa", "aa");
                _card169fixtable.Add("kk", "kk");
                _card169fixtable.Add("qq", "qq");
                _card169fixtable.Add("jj", "jj");
                _card169fixtable.Add("tt", "tt");
                _card169fixtable.Add("99", "99");
                _card169fixtable.Add("88", "88");
                _card169fixtable.Add("77", "77");
                _card169fixtable.Add("66", "66");
                _card169fixtable.Add("55", "55");
                _card169fixtable.Add("44", "44");
                _card169fixtable.Add("33", "33");
                _card169fixtable.Add("22", "22");
                _card169fixtable.Add("ak", "ako");
                _card169fixtable.Add("ka", "ako");
                _card169fixtable.Add("ako", "ako");
                _card169fixtable.Add("kao", "ako");
                _card169fixtable.Add("aks", "aks");
                _card169fixtable.Add("kas", "aks");
                _card169fixtable.Add("ak*", "ak");
                _card169fixtable.Add("ka*", "ak");
                _card169fixtable.Add("aq", "aqo");
                _card169fixtable.Add("qa", "aqo");
                _card169fixtable.Add("aqo", "aqo");
                _card169fixtable.Add("qao", "aqo");
                _card169fixtable.Add("aqs", "aqs");
                _card169fixtable.Add("qas", "aqs");
                _card169fixtable.Add("aq*", "aq");
                _card169fixtable.Add("qa*", "aq");
                _card169fixtable.Add("aj", "ajo");
                _card169fixtable.Add("ja", "ajo");
                _card169fixtable.Add("ajo", "ajo");
                _card169fixtable.Add("jao", "ajo");
                _card169fixtable.Add("ajs", "ajs");
                _card169fixtable.Add("jas", "ajs");
                _card169fixtable.Add("aj*", "aj");
                _card169fixtable.Add("ja*", "aj");
                _card169fixtable.Add("at", "ato");
                _card169fixtable.Add("ta", "ato");
                _card169fixtable.Add("ato", "ato");
                _card169fixtable.Add("tao", "ato");
                _card169fixtable.Add("ats", "ats");
                _card169fixtable.Add("tas", "ats");
                _card169fixtable.Add("at*", "at");
                _card169fixtable.Add("ta*", "at");
                _card169fixtable.Add("a9", "a9o");
                _card169fixtable.Add("9a", "a9o");
                _card169fixtable.Add("a9o", "a9o");
                _card169fixtable.Add("9ao", "a9o");
                _card169fixtable.Add("a9s", "a9s");
                _card169fixtable.Add("9as", "a9s");
                _card169fixtable.Add("a9*", "a9");
                _card169fixtable.Add("9a*", "a9");
                _card169fixtable.Add("a8", "a8o");
                _card169fixtable.Add("8a", "a8o");
                _card169fixtable.Add("a8o", "a8o");
                _card169fixtable.Add("8ao", "a8o");
                _card169fixtable.Add("a8s", "a8s");
                _card169fixtable.Add("8as", "a8s");
                _card169fixtable.Add("a8*", "a8");
                _card169fixtable.Add("8a*", "a8");
                _card169fixtable.Add("a7", "a7o");
                _card169fixtable.Add("7a", "a7o");
                _card169fixtable.Add("a7o", "a7o");
                _card169fixtable.Add("7ao", "a7o");
                _card169fixtable.Add("a7s", "a7s");
                _card169fixtable.Add("7as", "a7s");
                _card169fixtable.Add("a7*", "a7");
                _card169fixtable.Add("7a*", "a7");
                _card169fixtable.Add("a6", "a6o");
                _card169fixtable.Add("6a", "a6o");
                _card169fixtable.Add("a6o", "a6o");
                _card169fixtable.Add("6ao", "a6o");
                _card169fixtable.Add("a6s", "a6s");
                _card169fixtable.Add("6as", "a6s");
                _card169fixtable.Add("a6*", "a6");
                _card169fixtable.Add("6a*", "a6");
                _card169fixtable.Add("a5", "a5o");
                _card169fixtable.Add("5a", "a5o");
                _card169fixtable.Add("a5o", "a5o");
                _card169fixtable.Add("5ao", "a5o");
                _card169fixtable.Add("a5s", "a5s");
                _card169fixtable.Add("5as", "a5s");
                _card169fixtable.Add("a5*", "a5");
                _card169fixtable.Add("5a*", "a5");
                _card169fixtable.Add("a4", "a4o");
                _card169fixtable.Add("4a", "a4o");
                _card169fixtable.Add("a4o", "a4o");
                _card169fixtable.Add("4ao", "a4o");
                _card169fixtable.Add("a4s", "a4s");
                _card169fixtable.Add("4as", "a4s");
                _card169fixtable.Add("a4*", "a4");
                _card169fixtable.Add("4a*", "a4");
                _card169fixtable.Add("a3", "a3o");
                _card169fixtable.Add("3a", "a3o");
                _card169fixtable.Add("a3o", "a3o");
                _card169fixtable.Add("3ao", "a3o");
                _card169fixtable.Add("a3s", "a3s");
                _card169fixtable.Add("3as", "a3s");
                _card169fixtable.Add("a3*", "a3");
                _card169fixtable.Add("3a*", "a3");
                _card169fixtable.Add("a2", "a2o");
                _card169fixtable.Add("2a", "a2o");
                _card169fixtable.Add("a2o", "a2o");
                _card169fixtable.Add("2ao", "a2o");
                _card169fixtable.Add("a2s", "a2s");
                _card169fixtable.Add("2as", "a2s");
                _card169fixtable.Add("a2*", "a2");
                _card169fixtable.Add("2a*", "a2");
                _card169fixtable.Add("kq", "kqo");
                _card169fixtable.Add("qk", "kqo");
                _card169fixtable.Add("kqo", "kqo");
                _card169fixtable.Add("qko", "kqo");
                _card169fixtable.Add("kqs", "kqs");
                _card169fixtable.Add("qks", "kqs");
                _card169fixtable.Add("kq*", "kq");
                _card169fixtable.Add("qk*", "kq");
                _card169fixtable.Add("kj", "kjo");
                _card169fixtable.Add("jk", "kjo");
                _card169fixtable.Add("kjo", "kjo");
                _card169fixtable.Add("jko", "kjo");
                _card169fixtable.Add("kjs", "kjs");
                _card169fixtable.Add("jks", "kjs");
                _card169fixtable.Add("kj*", "kj");
                _card169fixtable.Add("jk*", "kj");
                _card169fixtable.Add("kt", "kto");
                _card169fixtable.Add("tk", "kto");
                _card169fixtable.Add("kto", "kto");
                _card169fixtable.Add("tko", "kto");
                _card169fixtable.Add("kts", "kts");
                _card169fixtable.Add("tks", "kts");
                _card169fixtable.Add("kt*", "kt");
                _card169fixtable.Add("tk*", "kt");
                _card169fixtable.Add("k9", "k9o");
                _card169fixtable.Add("9k", "k9o");
                _card169fixtable.Add("k9o", "k9o");
                _card169fixtable.Add("9ko", "k9o");
                _card169fixtable.Add("k9s", "k9s");
                _card169fixtable.Add("9ks", "k9s");
                _card169fixtable.Add("k9*", "k9");
                _card169fixtable.Add("9k*", "k9");
                _card169fixtable.Add("k8", "k8o");
                _card169fixtable.Add("8k", "k8o");
                _card169fixtable.Add("k8o", "k8o");
                _card169fixtable.Add("8ko", "k8o");
                _card169fixtable.Add("k8s", "k8s");
                _card169fixtable.Add("8ks", "k8s");
                _card169fixtable.Add("k8*", "k8");
                _card169fixtable.Add("8k*", "k8");
                _card169fixtable.Add("k7", "k7o");
                _card169fixtable.Add("7k", "k7o");
                _card169fixtable.Add("k7o", "k7o");
                _card169fixtable.Add("7ko", "k7o");
                _card169fixtable.Add("k7s", "k7s");
                _card169fixtable.Add("7ks", "k7s");
                _card169fixtable.Add("k7*", "k7");
                _card169fixtable.Add("7k*", "k7");
                _card169fixtable.Add("k6", "k6o");
                _card169fixtable.Add("6k", "k6o");
                _card169fixtable.Add("k6o", "k6o");
                _card169fixtable.Add("6ko", "k6o");
                _card169fixtable.Add("k6s", "k6s");
                _card169fixtable.Add("6ks", "k6s");
                _card169fixtable.Add("k6*", "k6");
                _card169fixtable.Add("6k*", "k6");
                _card169fixtable.Add("k5", "k5o");
                _card169fixtable.Add("5k", "k5o");
                _card169fixtable.Add("k5o", "k5o");
                _card169fixtable.Add("5ko", "k5o");
                _card169fixtable.Add("k5s", "k5s");
                _card169fixtable.Add("5ks", "k5s");
                _card169fixtable.Add("k5*", "k5");
                _card169fixtable.Add("5k*", "k5");
                _card169fixtable.Add("k4", "k4o");
                _card169fixtable.Add("4k", "k4o");
                _card169fixtable.Add("k4o", "k4o");
                _card169fixtable.Add("4ko", "k4o");
                _card169fixtable.Add("k4s", "k4s");
                _card169fixtable.Add("4ks", "k4s");
                _card169fixtable.Add("k4*", "k4");
                _card169fixtable.Add("4k*", "k4");
                _card169fixtable.Add("k3", "k3o");
                _card169fixtable.Add("3k", "k3o");
                _card169fixtable.Add("k3o", "k3o");
                _card169fixtable.Add("3ko", "k3o");
                _card169fixtable.Add("k3s", "k3s");
                _card169fixtable.Add("3ks", "k3s");
                _card169fixtable.Add("k3*", "k3");
                _card169fixtable.Add("3k*", "k3");
                _card169fixtable.Add("k2", "k2o");
                _card169fixtable.Add("2k", "k2o");
                _card169fixtable.Add("k2o", "k2o");
                _card169fixtable.Add("2ko", "k2o");
                _card169fixtable.Add("k2s", "k2s");
                _card169fixtable.Add("2ks", "k2s");
                _card169fixtable.Add("k2*", "k2");
                _card169fixtable.Add("2k*", "k2");
                _card169fixtable.Add("qj", "qjo");
                _card169fixtable.Add("jq", "qjo");
                _card169fixtable.Add("qjo", "qjo");
                _card169fixtable.Add("jqo", "qjo");
                _card169fixtable.Add("qjs", "qjs");
                _card169fixtable.Add("jqs", "qjs");
                _card169fixtable.Add("qj*", "qj");
                _card169fixtable.Add("jq*", "qj");
                _card169fixtable.Add("qt", "qto");
                _card169fixtable.Add("tq", "qto");
                _card169fixtable.Add("qto", "qto");
                _card169fixtable.Add("tqo", "qto");
                _card169fixtable.Add("qts", "qts");
                _card169fixtable.Add("tqs", "qts");
                _card169fixtable.Add("qt*", "qt");
                _card169fixtable.Add("tq*", "qt");
                _card169fixtable.Add("q9", "q9o");
                _card169fixtable.Add("9q", "q9o");
                _card169fixtable.Add("q9o", "q9o");
                _card169fixtable.Add("9qo", "q9o");
                _card169fixtable.Add("q9s", "q9s");
                _card169fixtable.Add("9qs", "q9s");
                _card169fixtable.Add("q9*", "q9");
                _card169fixtable.Add("9q*", "q9");
                _card169fixtable.Add("q8", "q8o");
                _card169fixtable.Add("8q", "q8o");
                _card169fixtable.Add("q8o", "q8o");
                _card169fixtable.Add("8qo", "q8o");
                _card169fixtable.Add("q8s", "q8s");
                _card169fixtable.Add("8qs", "q8s");
                _card169fixtable.Add("q8*", "q8");
                _card169fixtable.Add("8q*", "q8");
                _card169fixtable.Add("q7", "q7o");
                _card169fixtable.Add("7q", "q7o");
                _card169fixtable.Add("q7o", "q7o");
                _card169fixtable.Add("7qo", "q7o");
                _card169fixtable.Add("q7s", "q7s");
                _card169fixtable.Add("7qs", "q7s");
                _card169fixtable.Add("q7*", "q7");
                _card169fixtable.Add("7q*", "q7");
                _card169fixtable.Add("q6", "q6o");
                _card169fixtable.Add("6q", "q6o");
                _card169fixtable.Add("q6o", "q6o");
                _card169fixtable.Add("6qo", "q6o");
                _card169fixtable.Add("q6s", "q6s");
                _card169fixtable.Add("6qs", "q6s");
                _card169fixtable.Add("q6*", "q6");
                _card169fixtable.Add("6q*", "q6");
                _card169fixtable.Add("q5", "q5o");
                _card169fixtable.Add("5q", "q5o");
                _card169fixtable.Add("q5o", "q5o");
                _card169fixtable.Add("5qo", "q5o");
                _card169fixtable.Add("q5s", "q5s");
                _card169fixtable.Add("5qs", "q5s");
                _card169fixtable.Add("q5*", "q5");
                _card169fixtable.Add("5q*", "q5");
                _card169fixtable.Add("q4", "q4o");
                _card169fixtable.Add("4q", "q4o");
                _card169fixtable.Add("q4o", "q4o");
                _card169fixtable.Add("4qo", "q4o");
                _card169fixtable.Add("q4s", "q4s");
                _card169fixtable.Add("4qs", "q4s");
                _card169fixtable.Add("q4*", "q4");
                _card169fixtable.Add("4q*", "q4");
                _card169fixtable.Add("q3", "q3o");
                _card169fixtable.Add("3q", "q3o");
                _card169fixtable.Add("q3o", "q3o");
                _card169fixtable.Add("3qo", "q3o");
                _card169fixtable.Add("q3s", "q3s");
                _card169fixtable.Add("3qs", "q3s");
                _card169fixtable.Add("q3*", "q3");
                _card169fixtable.Add("3q*", "q3");
                _card169fixtable.Add("q2", "q2o");
                _card169fixtable.Add("2q", "q2o");
                _card169fixtable.Add("q2o", "q2o");
                _card169fixtable.Add("2qo", "q2o");
                _card169fixtable.Add("q2s", "q2s");
                _card169fixtable.Add("2qs", "q2s");
                _card169fixtable.Add("q2*", "q2");
                _card169fixtable.Add("2q*", "q2");
                _card169fixtable.Add("jt", "jto");
                _card169fixtable.Add("tj", "jto");
                _card169fixtable.Add("jto", "jto");
                _card169fixtable.Add("tjo", "jto");
                _card169fixtable.Add("jts", "jts");
                _card169fixtable.Add("tjs", "jts");
                _card169fixtable.Add("jt*", "jt");
                _card169fixtable.Add("tj*", "jt");
                _card169fixtable.Add("j9", "j9o");
                _card169fixtable.Add("9j", "j9o");
                _card169fixtable.Add("j9o", "j9o");
                _card169fixtable.Add("9jo", "j9o");
                _card169fixtable.Add("j9s", "j9s");
                _card169fixtable.Add("9js", "j9s");
                _card169fixtable.Add("j9*", "j9");
                _card169fixtable.Add("9j*", "j9");
                _card169fixtable.Add("j8", "j8o");
                _card169fixtable.Add("8j", "j8o");
                _card169fixtable.Add("j8o", "j8o");
                _card169fixtable.Add("8jo", "j8o");
                _card169fixtable.Add("j8s", "j8s");
                _card169fixtable.Add("8js", "j8s");
                _card169fixtable.Add("j8*", "j8");
                _card169fixtable.Add("8j*", "j8");
                _card169fixtable.Add("j7", "j7o");
                _card169fixtable.Add("7j", "j7o");
                _card169fixtable.Add("j7o", "j7o");
                _card169fixtable.Add("7jo", "j7o");
                _card169fixtable.Add("j7s", "j7s");
                _card169fixtable.Add("7js", "j7s");
                _card169fixtable.Add("j7*", "j7");
                _card169fixtable.Add("7j*", "j7");
                _card169fixtable.Add("j6", "j6o");
                _card169fixtable.Add("6j", "j6o");
                _card169fixtable.Add("j6o", "j6o");
                _card169fixtable.Add("6jo", "j6o");
                _card169fixtable.Add("j6s", "j6s");
                _card169fixtable.Add("6js", "j6s");
                _card169fixtable.Add("j6*", "j6");
                _card169fixtable.Add("6j*", "j6");
                _card169fixtable.Add("j5", "j5o");
                _card169fixtable.Add("5j", "j5o");
                _card169fixtable.Add("j5o", "j5o");
                _card169fixtable.Add("5jo", "j5o");
                _card169fixtable.Add("j5s", "j5s");
                _card169fixtable.Add("5js", "j5s");
                _card169fixtable.Add("j5*", "j5");
                _card169fixtable.Add("5j*", "j5");
                _card169fixtable.Add("j4", "j4o");
                _card169fixtable.Add("4j", "j4o");
                _card169fixtable.Add("j4o", "j4o");
                _card169fixtable.Add("4jo", "j4o");
                _card169fixtable.Add("j4s", "j4s");
                _card169fixtable.Add("4js", "j4s");
                _card169fixtable.Add("j4*", "j4");
                _card169fixtable.Add("4j*", "j4");
                _card169fixtable.Add("j3", "j3o");
                _card169fixtable.Add("3j", "j3o");
                _card169fixtable.Add("j3o", "j3o");
                _card169fixtable.Add("3jo", "j3o");
                _card169fixtable.Add("j3s", "j3s");
                _card169fixtable.Add("3js", "j3s");
                _card169fixtable.Add("j3*", "j3");
                _card169fixtable.Add("3j*", "j3");
                _card169fixtable.Add("j2", "j2o");
                _card169fixtable.Add("2j", "j2o");
                _card169fixtable.Add("j2o", "j2o");
                _card169fixtable.Add("2jo", "j2o");
                _card169fixtable.Add("j2s", "j2s");
                _card169fixtable.Add("2js", "j2s");
                _card169fixtable.Add("j2*", "j2");
                _card169fixtable.Add("2j*", "j2");
                _card169fixtable.Add("t9", "t9o");
                _card169fixtable.Add("9t", "t9o");
                _card169fixtable.Add("t9o", "t9o");
                _card169fixtable.Add("9to", "t9o");
                _card169fixtable.Add("t9s", "t9s");
                _card169fixtable.Add("9ts", "t9s");
                _card169fixtable.Add("t9*", "t9");
                _card169fixtable.Add("9t*", "t9");
                _card169fixtable.Add("t8", "t8o");
                _card169fixtable.Add("8t", "t8o");
                _card169fixtable.Add("t8o", "t8o");
                _card169fixtable.Add("8to", "t8o");
                _card169fixtable.Add("t8s", "t8s");
                _card169fixtable.Add("8ts", "t8s");
                _card169fixtable.Add("t8*", "t8");
                _card169fixtable.Add("8t*", "t8");
                _card169fixtable.Add("t7", "t7o");
                _card169fixtable.Add("7t", "t7o");
                _card169fixtable.Add("t7o", "t7o");
                _card169fixtable.Add("7to", "t7o");
                _card169fixtable.Add("t7s", "t7s");
                _card169fixtable.Add("7ts", "t7s");
                _card169fixtable.Add("t7*", "t7");
                _card169fixtable.Add("7t*", "t7");
                _card169fixtable.Add("t6", "t6o");
                _card169fixtable.Add("6t", "t6o");
                _card169fixtable.Add("t6o", "t6o");
                _card169fixtable.Add("6to", "t6o");
                _card169fixtable.Add("t6s", "t6s");
                _card169fixtable.Add("6ts", "t6s");
                _card169fixtable.Add("t6*", "t6");
                _card169fixtable.Add("6t*", "t6");
                _card169fixtable.Add("t5", "t5o");
                _card169fixtable.Add("5t", "t5o");
                _card169fixtable.Add("t5o", "t5o");
                _card169fixtable.Add("5to", "t5o");
                _card169fixtable.Add("t5s", "t5s");
                _card169fixtable.Add("5ts", "t5s");
                _card169fixtable.Add("t5*", "t5");
                _card169fixtable.Add("5t*", "t5");
                _card169fixtable.Add("t4", "t4o");
                _card169fixtable.Add("4t", "t4o");
                _card169fixtable.Add("t4o", "t4o");
                _card169fixtable.Add("4to", "t4o");
                _card169fixtable.Add("t4s", "t4s");
                _card169fixtable.Add("4ts", "t4s");
                _card169fixtable.Add("t4*", "t4");
                _card169fixtable.Add("4t*", "t4");
                _card169fixtable.Add("t3", "t3o");
                _card169fixtable.Add("3t", "t3o");
                _card169fixtable.Add("t3o", "t3o");
                _card169fixtable.Add("3to", "t3o");
                _card169fixtable.Add("t3s", "t3s");
                _card169fixtable.Add("3ts", "t3s");
                _card169fixtable.Add("t3*", "t3");
                _card169fixtable.Add("3t*", "t3");
                _card169fixtable.Add("t2", "t2o");
                _card169fixtable.Add("2t", "t2o");
                _card169fixtable.Add("t2o", "t2o");
                _card169fixtable.Add("2to", "t2o");
                _card169fixtable.Add("t2s", "t2s");
                _card169fixtable.Add("2ts", "t2s");
                _card169fixtable.Add("t2*", "t2");
                _card169fixtable.Add("2t*", "t2");
                _card169fixtable.Add("98", "98o");
                _card169fixtable.Add("89", "98o");
                _card169fixtable.Add("98o", "98o");
                _card169fixtable.Add("89o", "98o");
                _card169fixtable.Add("98s", "98s");
                _card169fixtable.Add("89s", "98s");
                _card169fixtable.Add("98*", "98");
                _card169fixtable.Add("89*", "98");
                _card169fixtable.Add("97", "97o");
                _card169fixtable.Add("79", "97o");
                _card169fixtable.Add("97o", "97o");
                _card169fixtable.Add("79o", "97o");
                _card169fixtable.Add("97s", "97s");
                _card169fixtable.Add("79s", "97s");
                _card169fixtable.Add("97*", "97");
                _card169fixtable.Add("79*", "97");
                _card169fixtable.Add("96", "96o");
                _card169fixtable.Add("69", "96o");
                _card169fixtable.Add("96o", "96o");
                _card169fixtable.Add("69o", "96o");
                _card169fixtable.Add("96s", "96s");
                _card169fixtable.Add("69s", "96s");
                _card169fixtable.Add("96*", "96");
                _card169fixtable.Add("69*", "96");
                _card169fixtable.Add("95", "95o");
                _card169fixtable.Add("59", "95o");
                _card169fixtable.Add("95o", "95o");
                _card169fixtable.Add("59o", "95o");
                _card169fixtable.Add("95s", "95s");
                _card169fixtable.Add("59s", "95s");
                _card169fixtable.Add("95*", "95");
                _card169fixtable.Add("59*", "95");
                _card169fixtable.Add("94", "94o");
                _card169fixtable.Add("49", "94o");
                _card169fixtable.Add("94o", "94o");
                _card169fixtable.Add("49o", "94o");
                _card169fixtable.Add("94s", "94s");
                _card169fixtable.Add("49s", "94s");
                _card169fixtable.Add("94*", "94");
                _card169fixtable.Add("49*", "94");
                _card169fixtable.Add("93", "93o");
                _card169fixtable.Add("39", "93o");
                _card169fixtable.Add("93o", "93o");
                _card169fixtable.Add("39o", "93o");
                _card169fixtable.Add("93s", "93s");
                _card169fixtable.Add("39s", "93s");
                _card169fixtable.Add("93*", "93");
                _card169fixtable.Add("39*", "93");
                _card169fixtable.Add("92", "92o");
                _card169fixtable.Add("29", "92o");
                _card169fixtable.Add("92o", "92o");
                _card169fixtable.Add("29o", "92o");
                _card169fixtable.Add("92s", "92s");
                _card169fixtable.Add("29s", "92s");
                _card169fixtable.Add("92*", "92");
                _card169fixtable.Add("29*", "92");
                _card169fixtable.Add("87", "87o");
                _card169fixtable.Add("78", "87o");
                _card169fixtable.Add("87o", "87o");
                _card169fixtable.Add("78o", "87o");
                _card169fixtable.Add("87s", "87s");
                _card169fixtable.Add("78s", "87s");
                _card169fixtable.Add("87*", "87");
                _card169fixtable.Add("78*", "87");
                _card169fixtable.Add("86", "86o");
                _card169fixtable.Add("68", "86o");
                _card169fixtable.Add("86o", "86o");
                _card169fixtable.Add("68o", "86o");
                _card169fixtable.Add("86s", "86s");
                _card169fixtable.Add("68s", "86s");
                _card169fixtable.Add("86*", "86");
                _card169fixtable.Add("68*", "86");
                _card169fixtable.Add("85", "85o");
                _card169fixtable.Add("58", "85o");
                _card169fixtable.Add("85o", "85o");
                _card169fixtable.Add("58o", "85o");
                _card169fixtable.Add("85s", "85s");
                _card169fixtable.Add("58s", "85s");
                _card169fixtable.Add("85*", "85");
                _card169fixtable.Add("58*", "85");
                _card169fixtable.Add("84", "84o");
                _card169fixtable.Add("48", "84o");
                _card169fixtable.Add("84o", "84o");
                _card169fixtable.Add("48o", "84o");
                _card169fixtable.Add("84s", "84s");
                _card169fixtable.Add("48s", "84s");
                _card169fixtable.Add("84*", "84");
                _card169fixtable.Add("48*", "84");
                _card169fixtable.Add("83", "83o");
                _card169fixtable.Add("38", "83o");
                _card169fixtable.Add("83o", "83o");
                _card169fixtable.Add("38o", "83o");
                _card169fixtable.Add("83s", "83s");
                _card169fixtable.Add("38s", "83s");
                _card169fixtable.Add("83*", "83");
                _card169fixtable.Add("38*", "83");
                _card169fixtable.Add("82", "82o");
                _card169fixtable.Add("28", "82o");
                _card169fixtable.Add("82o", "82o");
                _card169fixtable.Add("28o", "82o");
                _card169fixtable.Add("82s", "82s");
                _card169fixtable.Add("28s", "82s");
                _card169fixtable.Add("82*", "82");
                _card169fixtable.Add("28*", "82");
                _card169fixtable.Add("76", "76o");
                _card169fixtable.Add("67", "76o");
                _card169fixtable.Add("76o", "76o");
                _card169fixtable.Add("67o", "76o");
                _card169fixtable.Add("76s", "76s");
                _card169fixtable.Add("67s", "76s");
                _card169fixtable.Add("76*", "76");
                _card169fixtable.Add("67*", "76");
                _card169fixtable.Add("75", "75o");
                _card169fixtable.Add("57", "75o");
                _card169fixtable.Add("75o", "75o");
                _card169fixtable.Add("57o", "75o");
                _card169fixtable.Add("75s", "75s");
                _card169fixtable.Add("57s", "75s");
                _card169fixtable.Add("75*", "75");
                _card169fixtable.Add("57*", "75");
                _card169fixtable.Add("74", "74o");
                _card169fixtable.Add("47", "74o");
                _card169fixtable.Add("74o", "74o");
                _card169fixtable.Add("47o", "74o");
                _card169fixtable.Add("74s", "74s");
                _card169fixtable.Add("47s", "74s");
                _card169fixtable.Add("74*", "74");
                _card169fixtable.Add("47*", "74");
                _card169fixtable.Add("73", "73o");
                _card169fixtable.Add("37", "73o");
                _card169fixtable.Add("73o", "73o");
                _card169fixtable.Add("37o", "73o");
                _card169fixtable.Add("73s", "73s");
                _card169fixtable.Add("37s", "73s");
                _card169fixtable.Add("73*", "73");
                _card169fixtable.Add("37*", "73");
                _card169fixtable.Add("72", "72o");
                _card169fixtable.Add("27", "72o");
                _card169fixtable.Add("72o", "72o");
                _card169fixtable.Add("27o", "72o");
                _card169fixtable.Add("72s", "72s");
                _card169fixtable.Add("27s", "72s");
                _card169fixtable.Add("72*", "72");
                _card169fixtable.Add("27*", "72");
                _card169fixtable.Add("65", "65o");
                _card169fixtable.Add("56", "65o");
                _card169fixtable.Add("65o", "65o");
                _card169fixtable.Add("56o", "65o");
                _card169fixtable.Add("65s", "65s");
                _card169fixtable.Add("56s", "65s");
                _card169fixtable.Add("65*", "65");
                _card169fixtable.Add("56*", "65");
                _card169fixtable.Add("64", "64o");
                _card169fixtable.Add("46", "64o");
                _card169fixtable.Add("64o", "64o");
                _card169fixtable.Add("46o", "64o");
                _card169fixtable.Add("64s", "64s");
                _card169fixtable.Add("46s", "64s");
                _card169fixtable.Add("64*", "64");
                _card169fixtable.Add("46*", "64");
                _card169fixtable.Add("63", "63o");
                _card169fixtable.Add("36", "63o");
                _card169fixtable.Add("63o", "63o");
                _card169fixtable.Add("36o", "63o");
                _card169fixtable.Add("63s", "63s");
                _card169fixtable.Add("36s", "63s");
                _card169fixtable.Add("63*", "63");
                _card169fixtable.Add("36*", "63");
                _card169fixtable.Add("62", "62o");
                _card169fixtable.Add("26", "62o");
                _card169fixtable.Add("62o", "62o");
                _card169fixtable.Add("26o", "62o");
                _card169fixtable.Add("62s", "62s");
                _card169fixtable.Add("26s", "62s");
                _card169fixtable.Add("62*", "62");
                _card169fixtable.Add("26*", "62");
                _card169fixtable.Add("54", "54o");
                _card169fixtable.Add("45", "54o");
                _card169fixtable.Add("54o", "54o");
                _card169fixtable.Add("45o", "54o");
                _card169fixtable.Add("54s", "54s");
                _card169fixtable.Add("45s", "54s");
                _card169fixtable.Add("54*", "54");
                _card169fixtable.Add("45*", "54");
                _card169fixtable.Add("53", "53o");
                _card169fixtable.Add("35", "53o");
                _card169fixtable.Add("53o", "53o");
                _card169fixtable.Add("35o", "53o");
                _card169fixtable.Add("53s", "53s");
                _card169fixtable.Add("35s", "53s");
                _card169fixtable.Add("53*", "53");
                _card169fixtable.Add("35*", "53");
                _card169fixtable.Add("52", "52o");
                _card169fixtable.Add("25", "52o");
                _card169fixtable.Add("52o", "52o");
                _card169fixtable.Add("25o", "52o");
                _card169fixtable.Add("52s", "52s");
                _card169fixtable.Add("25s", "52s");
                _card169fixtable.Add("52*", "52");
                _card169fixtable.Add("25*", "52");
                _card169fixtable.Add("43", "43o");
                _card169fixtable.Add("34", "43o");
                _card169fixtable.Add("43o", "43o");
                _card169fixtable.Add("34o", "43o");
                _card169fixtable.Add("43s", "43s");
                _card169fixtable.Add("34s", "43s");
                _card169fixtable.Add("43*", "43");
                _card169fixtable.Add("34*", "43");
                _card169fixtable.Add("42", "42o");
                _card169fixtable.Add("24", "42o");
                _card169fixtable.Add("42o", "42o");
                _card169fixtable.Add("24o", "42o");
                _card169fixtable.Add("42s", "42s");
                _card169fixtable.Add("24s", "42s");
                _card169fixtable.Add("42*", "42");
                _card169fixtable.Add("24*", "42");
                _card169fixtable.Add("32", "32o");
                _card169fixtable.Add("23", "32o");
                _card169fixtable.Add("32o", "32o");
                _card169fixtable.Add("23o", "32o");
                _card169fixtable.Add("32s", "32s");
                _card169fixtable.Add("23s", "32s");
                _card169fixtable.Add("32*", "32");
                _card169fixtable.Add("23*", "32");
            }
        }


        #endregion

        #region Constructors
        /// <summary>
        /// Default constructor
        /// </summary>
        public PocketHands()
        {
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="arg"></param>
        public PocketHands(PocketHands arg)
        {
            foreach (ulong mask in arg.list)
            {
#if DEBUG
                System.Diagnostics.Debug.Assert(Hand.BitCount(mask) == 2 && !list.Contains(mask));
#endif
                list.Add(mask);
            }
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="arg"></param>
        public PocketHands(ulong[] arg)
        {
            foreach (ulong mask in arg)
            {
#if DEBUG
                System.Diagnostics.Debug.Assert(Hand.BitCount(mask) == 2 && !list.Contains(mask));
#endif
                list.Add(mask);
            }
        }
        
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="mask"></param>
        public PocketHands(ulong mask)
        {
#if DEBUG
            System.Diagnostics.Debug.Assert(Hand.BitCount(mask) == 2 && !list.Contains(mask));
#endif
            list.Add(mask);
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="arg"></param>
        public PocketHands(List<ulong> arg)
        {
            foreach (ulong mask in arg)
            {
#if DEBUG
                System.Diagnostics.Debug.Assert(Hand.BitCount(mask) == 2 && !list.Contains(mask));
#endif
                list.Add(mask);
            }
        }
        #endregion

        #region Class Factories

        ///<exclude/>
        static private readonly PocketHands _allhands = new PocketHands(_PocketTableMasks);

        /// <summary>
        /// Creates and instance of PocketHands with all 1326 possible pocket cards.
        /// </summary>
        /// <returns></returns>
        static public PocketHands AllHands
        {
            get
            {
                return _allhands;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _connected = new PocketHands(_connectedTable);

        /// <summary>
        /// Creates an instance of PocketHands which are connected.
        /// </summary>
        /// <returns></returns>
        static public PocketHands Connected
        {
            get
            {
                return _connected;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _suited = new PocketHands(_suitedTable);


        /// <summary>
        /// Creates an instance of PocketHands which contain all of the possible suited hands.
        /// </summary>
        /// <returns></returns>
        static public PocketHands Suited
        {
            get
            {
                return _suited;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _offsuit = !Suited;


        /// <summary>
        /// Creates an instance of PocketHands which contain all of the possible offsuit hands.
        /// </summary>
        static public PocketHands Offsuit
        {
            get { return _offsuit; }
        }

        ///<exclude/>
        static private readonly PocketHands _pair = new PocketHands(_pairTable);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket pairs.
        /// </summary>
        /// <returns></returns>
        static public PocketHands TwoOfAKind
        {
            get
            {
                return _pair;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _gap1 = new PocketHands(_gap1Table);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands with gap of one.
        /// </summary>
        /// <returns></returns>
        static public PocketHands Gap1
        {
            get
            {
                return _gap1;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _gap2 = new PocketHands(_gap2Table);

        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands with gap of two.
        /// </summary>
        /// <returns></returns>
        static public PocketHands Gap2
        {
            get
            {
                return _gap2;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _gap3 = new PocketHands(_gap3Table);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands with gap of three.
        /// </summary>
        /// <returns></returns>
        static public PocketHands Gap3
        {
            get
            {
                return _gap3;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _gap = Gap1 | Gap2 | Gap3;


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands with gap a gap of one, two or three.
        /// </summary>
        /// <returns></returns>
        static public PocketHands Gap
        {
            get
            {
                return _gap;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group1 = new PocketHands(_PocketGroupings[0]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group1.
        /// </summary>
        static public PocketHands Group1
        {
            get {
                return _group1;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group2 = new PocketHands(_PocketGroupings[1]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group2.
        /// </summary>
        static public PocketHands Group2
        {
            get
            {
                return _group2;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group3 = new PocketHands(_PocketGroupings[2]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group3.
        /// </summary>
        static public PocketHands Group3
        {
            get
            {
                return _group3;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group4 = new PocketHands(_PocketGroupings[3]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group4.
        /// </summary>
        static public PocketHands Group4
        {
            get
            {
                return _group4;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group5 = new PocketHands(_PocketGroupings[4]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group5.
        /// </summary>
        static public PocketHands Group5
        {
            get
            {
                return _group5;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group6 = new PocketHands(_PocketGroupings[5]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group6.
        /// </summary>
        static public PocketHands Group6
        {
            get
            {
                return _group6;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group7 = new PocketHands(_PocketGroupings[6]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group7.
        /// </summary>
        static public PocketHands Group7
        {
            get
            {
                return _group7;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _group8 = new PocketHands(_PocketGroupings[7]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are in Skalansky group8.
        /// </summary>
        static public PocketHands Group8
        {
            get
            {
                return _group8;
            }
        }

        ///<exclude/>
        static private readonly PocketHands _groupnone = new PocketHands(_PocketGroupings[8]);


        /// <summary>
        /// Creates an instance of PocketHands that contains all possible pocket hands that are not in any of the Skalansky groups.
        /// </summary>
        static public PocketHands GroupNone
        {
            get
            {
               return _groupnone;
            }
        }

        /// <summary>
        /// Creates an instance of PocketHands that contains the specified pocket cards.
        /// Valid strings are "As Kd" for Ace Spaces and a King of Diamonds and so on.
        /// </summary>
        /// <param name="pocket"></param>
        /// <returns></returns>
        static public PocketHands PocketCards(string pocket)
        {
#if DEBUG
            if (!Hand.ValidateHand(pocket)) throw new ArgumentException("pocket");
#endif
            return new PocketHands(Hand.ParseHand(pocket));
        }

        /// <summary>
        /// Creates an instance of PocketHands that contains all of the hands specified by the Card 169 string. 
        /// Valid strings are AKs (Ace/King suited), AA (A pair of aces) and so on.
        /// </summary>
        /// <param name="pocket"></param>
        /// <returns></returns>
        static public PocketHands PocketCards169(string pocket)
        {
            PocketHands retval = new PocketHands();
            if (pocket.Length == 3 && pocket[2] == '*')
            {

                retval |= _Pocket169Combinations[(int)Pocket169(string.Format("{0}{1}s", pocket[0], pocket[1]))];
                retval |= _Pocket169Combinations[(int)Pocket169(string.Format("{0}{1}o", pocket[0], pocket[1]))];
            }
            else
            {
                retval |= _Pocket169Combinations[(int)Pocket169(pocket)];
            }
            
            return retval;
        }

        /// <summary>
        /// This methods reduces the hands passed in hands to a set of that represents one of each of the
        /// 169 mask types (if such a mask exists).
        /// </summary>
        /// <param name="hands"></param>
        /// <returns></returns>
        static public PocketHands Condense169(PocketHands hands)
        {
            ulong[] pocket169 = new ulong[169];

            foreach (ulong mask in hands)
            {
                int index = (int)Hand.PocketHand169Type(mask);
                if (pocket169[index] == 0)
                {
                    pocket169[index] = mask;
                }
            }
            return new PocketHands(pocket169);
        }

        /// <summary>
        /// The method removes all of the cards specified in the bit mask from the PocketHands passed in hands.
        /// </summary>
        /// <param name="dead"></param>
        /// <param name="hands"></param>
        /// <returns></returns>
        static public PocketHands RemoveDead(ulong dead, PocketHands hands)
        {
            return hands - dead;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="arg"></param>
        /// <returns></returns>
        static internal int Card169Max(string arg)
        {
            if (arg.Length < 2 || arg.Length > 3) return -1;

            arg = FixCard169(arg);

            if (arg.Length == 3)
            {
                return (int)PocketCard169StringToEnum(arg);
            }
            else
            {
                if (arg[0] == arg[1])
                {
                    return (int)PocketCard169StringToEnum(arg);
                }
                else
                {
                    return (int)PocketCard169StringToEnum(arg + "o");
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="arg"></param>
        /// <returns></returns>
        static internal int Card169Min(string arg)
        {
            if (arg.Length < 2 || arg.Length > 3) return -1;

            arg = FixCard169(arg);

            if (arg.Length == 3)
            {
                return (int)PocketCard169StringToEnum(arg);
            }
            else
            {
                if (arg[0] == arg[1])
                {
                    return (int)PocketCard169StringToEnum(arg);
                }
                else
                {
                    return (int)PocketCard169StringToEnum(arg + "s");
                }
            }
        }

        /// <summary>
        /// Creates an instance of PocketHands given a 169 wild card string. For example
        /// AX would be and Ace and a rag. K?s would be a King and any other card suited.
        /// </summary>
        /// <param name="arg"></param>
        /// <returns></returns>
        static public PocketHands PocketCards169Wild(string arg)
        {
            string[] cardTbl = { "A", "K", "Q", "J", "T", "9", "8", "7", "6", "5", "4", "3", "2" };
            string[] ragTbl = { "9", "8", "7", "6", "5", "4", "3", "2" };
            PocketHands retval = new PocketHands();

            if (arg.Length != 2 && arg.Length != 3)
                throw new ArgumentOutOfRangeException("arg");

            switch (arg[1])
            {
                case '?':
                    foreach (string s in cardTbl)
                    {
                        if (s == arg[0].ToString())
                        {
                            if ((arg.Length == 3 && arg[2] != 's' && arg[2] != 'S') || arg.Length == 2)
                                retval |= PocketHands.PocketCards169(string.Format("{0}{1}", arg[0], s));
                        }
                        else if (arg.Length == 3)
                        {
                            switch (arg[2])
                            {
                                case 'S':
                                case 's':
                                case 'O':
                                case 'o':
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}{2}", arg[0], s, arg[2]));
                                    break;
                                case '*':
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}o", arg[0], s));
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}s", arg[0], s));
                                    break;
                            }
                        }
                        else
                            retval |= PocketHands.PocketCards169(string.Format("{0}{1}o", arg[0], s));
                    }
                    break;
                case 'x':
                case 'X':
                    foreach (string s in ragTbl)
                    {
                        if (arg.Length == 3)
                        {
                            switch (arg[2])
                            {
                                case 'S':
                                case 's':
                                case 'O':
                                case 'o':
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}{2}", arg[0], s, arg[2]));
                                    break;
                                case '*':
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}o", arg[0], s));
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}s", arg[0], s));
                                    break;
                            }
                        }
                        else
                        {
                            if (arg.Length == 2)
                                if (arg[0] == s[0])
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}", arg[0], s));
                                else
                                    retval |= PocketHands.PocketCards169(string.Format("{0}{1}o", arg[0], s));
                        }
                    }
                    break;
            }

            return retval;
        }

        /// <summary>
        /// Allows a range of 169 cards definitions for example Pocketcard169Range("AA", "22") would create
        /// a representation of all of the possible pairs. This method assumes that the 169 card definitions have
        /// an order. This order is AA-22, AKs, AKo, AQs, AQo and so on.
        /// </summary>
        /// <param name="s"></param>
        /// <param name="e"></param>
        /// <returns></returns>
        static public PocketHands PocketCards169Range(string s, string e)
        {
            Hand.PocketHand169Enum smin, smax, emin, emax;
            if (s.Length == 3 && s[2] == '*')
            {
                smin = Pocket169(string.Format("{0}{1}s", s[0], s[1]));
                smax = Pocket169(string.Format("{0}{1}o", s[0], s[1]));
            }
            else
            {
                smin = smax = Pocket169(s);
            }

            if (e.Length == 3 && e[2] == '*')
            {
                emin = Pocket169(string.Format("{0}{1}s", e[0], e[1]));
                emax = Pocket169(string.Format("{0}{1}o", e[0], e[1]));
            }
            else
            {
                emin = emax = Pocket169(e);
            }

            if (smax > emax)
            {
                return PocketCard169Range(emin, smax);
            }
            else
            {
                return PocketCard169Range(smin, emax);
            }
            
        }

        /// <summary>
        /// Allows a range of 169 cards definitions for example Pocketcard169Range(Hand.PocketHand169Enum.PocketAA, Hand.PocketHand169Enum.Pocket22) would create
        /// a representation of all of the possible pairs. This method assumes that the 169 card definitions have
        /// an order. This order is AA-22, AKs, AKo, AQs, AQo and so on.
        /// </summary>
        /// <param name="s"></param>
        /// <param name="e"></param>
        /// <returns></returns>
        static public PocketHands PocketCard169Range(Hand.PocketHand169Enum s, Hand.PocketHand169Enum e)
        {
            PocketHands retval = new PocketHands();
            int start = (int) s, end = (int) e;
            if (start > end)
            {
                for (int i = end; i <= start; i++)
                {
                    retval |= _Pocket169Combinations[i];
                }
            }
            else
            {
                for (int i = start; i <= end; i++)
                {
                    retval |= _Pocket169Combinations[i];
                }
            }
            return retval;
        }

        /// <summary>
        /// Given a PocketGroupingRank (Sklansky group value) all of the pockethands that are in the
        /// specified group are returned.
        /// </summary>
        /// <param name="rank"></param>
        /// <returns></returns>
        static public PocketHands Group(GroupTypeEnum rank)
        {
            return new PocketHands(_PocketGroupings[(int)rank]);
        }

        /// <summary>
        /// Given a PocketGroupRank range, all of the pockethands that are between (and including) the specfied groups
        /// are returned.
        /// </summary>
        /// <param name="s"></param>
        /// <param name="e"></param>
        /// <returns></returns>
        static public PocketHands GroupRange(GroupTypeEnum s, GroupTypeEnum e)
        {
            PocketHands retval = new PocketHands();
            int start = (int) s, end = (int) e;
            if (start > end)
            {
                for (int i = end; i <= start; i++)
                {
                    retval |= _PocketGroupings[i];
                }
            }
            else
            {
                for (int i = start; i <= end; i++)
                {
                    retval |= _PocketGroupings[i];
                }
            }
            return retval;
        }

        #endregion

        #region Operators

        #region Union (|) Operator 
        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator |(PocketHands arg1, PocketHands arg2)
        {
            return arg1 + arg2;
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator |(PocketHands arg1, ulong[] arg2)
        {
            return arg1 + arg2;
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator |(ulong[] arg1, PocketHands arg2)
        {
            return arg1 + arg2;
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator |(PocketHands arg1, ulong arg2)
        {
            return arg1 | new PocketHands(arg2);
        }
        #endregion

        #region Add (+) Operators

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator +(PocketHands arg1, PocketHands arg2)
        {
            PocketHands retval = new PocketHands(arg1);

            foreach (ulong mask in arg2)
            {
                if (!retval.Contains(mask))
                {
                    retval.list.Add(mask);
                }
            }

            return retval;
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator +(PocketHands arg1, ulong arg2)
        {
            return arg1 + new PocketHands(arg2);
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator +(ulong arg1, PocketHands arg2)
        {
            return new PocketHands(arg1) + arg2;
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator +(PocketHands arg1, ulong[] arg2)
        {
            return arg1 + new PocketHands(arg2);
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator +(ulong[] arg1, PocketHands arg2)
        {
            return new PocketHands(arg1) + arg2;
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator +(PocketHands arg1, List<ulong> arg2)
        {
            return arg1 + new PocketHands(arg2);
        }

        /// <summary>
        /// This is a union operator. It combines all elements of the two collections into
        /// one combined collection.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator +(List<ulong> arg1, PocketHands arg2)
        {
            return new PocketHands(arg1) + arg2;
        }
        #endregion

        #region Interection (&) Operator
        /// <summary>
        /// This is a intersection operator. It produces a collection of the elements that are in
        /// both of the specified arguments.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator &(PocketHands arg1, PocketHands arg2)
        {
            PocketHands retval = new PocketHands();
            for (int i = 0; i < arg1.Count; i++)
            {
                if (arg2.Contains(arg1[i]))
                {
                    retval.list.Add(arg1[i]);
                }
            }
            return retval;
        }

        /// <summary>
        /// This is a intersection operator. It produces a collection of the elements that are in
        /// both of the specified arguments.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator &(PocketHands arg1, ulong[] arg2)
        {
            PocketHands retval = new PocketHands();
            for (int i = 0; i < arg2.Length; i++)
            {
                if (arg1.Contains(arg2[i]))
                {
                    retval.list.Add(arg2[i]);
                }
            }
            return retval;
        }

        /// <summary>
        /// This is a intersection operator. It produces a collection of the elements that are in
        /// both of the specified arguments.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator &(ulong[] arg1, PocketHands arg2)
        {
            PocketHands retval = new PocketHands();
            for (int i = 0; i < arg1.Length; i++)
            {
                if (arg2.Contains(arg1[i]))
                {
                    retval.list.Add(arg1[i]);
                }
            }
            return retval;
        }
        #endregion

        #region Minus (-) Operator

        /// <summary>
        /// This operator returns a collection with all of the cards defined in the dead card mask set removed from the PocketHands collection.
        /// This can be used to remove dead cards from any PocketHands collection.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="dead"></param>
        /// <returns></returns>
        static public PocketHands operator -(PocketHands arg, ulong dead)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg)
            {
                if ((mask & dead) == 0)
                {
                    retval += mask;
                }
            }

            return retval;
        }

        /// <summary>
        /// This operator returns a collection with all of the pocket hands defined in arg2 set removed from the PocketHands collection defined in arg1.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator -(PocketHands arg1, List<ulong> arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (!arg2.Contains(mask))
                {
                    retval |= mask;
                }
            }

            return retval;
        }

        /// <summary>
        /// This operator returns a collection with all of the pocket hands defined in arg2 set removed from the collection defined in arg1.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator -(List<ulong> arg1, PocketHands arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (!arg2.Contains(mask))
                {
                    retval |= mask;
                }
            }

            return retval;
        }

        /// <summary>
        /// This operator returns a collection with all of the pocket hands defined in arg2 set removed from the PocketHands collection defined in arg1.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator -(PocketHands arg1, PocketHands arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (!arg2.Contains(mask))
                {
                    retval |= mask;
                }
            }

            return retval;
        }

        /// <summary>
        /// This operator returns a collection with all of the pocket hands defined in arg2 set removed from the collection defined in arg1.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public PocketHands operator -(PocketHands arg1, ulong [] arg2)
        {
            PocketHands retval = new PocketHands();
            PocketHands a2 = new PocketHands(arg2);

            foreach (ulong mask in arg1)
            {
                if (!a2.Contains(mask))
                {
                    retval |= mask;
                }
            }

            return retval;
        }

        /// <summary>
        /// This operator returns a collection with all of the pocket hands defined in arg2 set removed from the collection defined in arg1.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
       static public PocketHands operator -(ulong [] arg1, PocketHands arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (!arg2.Contains(mask))
                {
                    retval |= mask;
                }
            }

            return retval;
        }

        #endregion

        #region Comparison Operators

        /// <summary>
        /// This operation returns all the possible pocket hands that aren't in arg.
        /// </summary>
        /// <param name="arg"></param>
        /// <returns></returns>
        static public PocketHands operator !(PocketHands arg)
        {
            return PocketHands.AllHands - arg;
        }

        /// <summary>
        /// Compares arg1 to arg2 for equality. Order doesn't matter, but both collections must
        /// contain the same elements.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator ==(PocketHands arg1, PocketHands arg2)
        {
            if (arg1.Count != arg2.Count) return false;

            foreach (ulong mask in arg1)
            {
                if (!arg2.Contains(mask)) return false;
            }
            return true;
        }

        /// <summary>
        /// This operator compares two PocketHands collections for inequality.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator !=(PocketHands arg1, PocketHands arg2)
        {
            return !(arg1 == arg2);
        }

        /// <summary>
        /// This oprator compares two collections of pocket hands for equality. 
        /// Note: order doesn't matter.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator ==(ulong[] arg1, PocketHands arg2)
        {
            return new PocketHands(arg1) == arg2;
        }

        /// <summary>
        /// This operator compares two pocket mask collections for inequality.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator !=(ulong[] arg1, PocketHands arg2)
        {
            return new PocketHands(arg1) != arg2;
        }

        /// <summary>
        /// This oprator compares two collections of pocket hands for equality. 
        /// Note: order doesn't matter.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator ==(List<ulong> arg1, PocketHands arg2)
        {
            return new PocketHands(arg1) == arg2;
        }

        /// <summary>
        /// This operator compares two pocket mask collections for inequality.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator !=(List<ulong> arg1, PocketHands arg2)
        {
            return new PocketHands(arg1) != arg2;
        }

        /// <summary>
        /// This oprator compares two collections of pocket hands for equality. 
        /// Note: order doesn't matter.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator ==(PocketHands arg1, ulong[] arg2)
        {
            return arg1 == new PocketHands(arg2);
        }

        /// <summary>
        /// This operator compares two pocket mask collections for inequality.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator !=(PocketHands arg1, ulong[] arg2)
        {
            return arg1 != new PocketHands(arg2);
        }

        /// <summary>
        /// This oprator compares two collections of pocket hands for equality. 
        /// Note: order doesn't matter.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator ==(PocketHands arg1, List<ulong> arg2)
        {
            return arg1 == new PocketHands(arg2);
        }

        /// <summary>
        /// This operator compares two pocket mask collections for inequality.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="arg2"></param>
        /// <returns></returns>
        static public bool operator !=(PocketHands arg1, List<ulong> arg2)
        {
            return arg1 != new PocketHands(arg2);
        }

        /// <summary>
        /// Compares each element of the PocketHands argument to see if
        /// it's win percentage is less than arg2.
        /// </summary>
        /// <param name="arg1">PocketHands collection to compare</param>
        /// <param name="arg2">Win Value</param>
        /// <returns>A PocketHands collection where each item has win odds less than arg2</returns>
        static public PocketHands operator <(PocketHands arg1, double arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (PocketHands.WinOdds(mask) < arg2)
                    retval += mask;
            }
            return retval;
        }

        /// <summary>
        /// Compares each element of the PocketHands argument to see if
        /// it's win percentage is less than or equal to arg2.
        /// </summary>
        /// <param name="arg1">PocketHands collection to compare</param>
        /// <param name="arg2">Win Value</param>
        /// <returns>A PocketHands collection where each item has win odds less than or equal to arg2</returns>
        static public PocketHands operator <=(PocketHands arg1, double arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (PocketHands.WinOdds(mask) <= arg2)
                    retval += mask;
            }
            return retval;
        }

        /// <summary>
        /// Compares each element of the PocketHands argument to see if
        /// it's win percentage is greater than arg2.
        /// </summary>
        /// <param name="arg1">PocketHands collection to compare</param>
        /// <param name="arg2">Win Value</param>
        /// <returns>A PocketHands collection where each item has win odds greater than arg2</returns>
        static public PocketHands operator >(PocketHands arg1, double arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (PocketHands.WinOdds(mask) > arg2)
                    retval += mask;
            }
            return retval;
        }

        /// <summary>
        /// Compares each element of the PocketHands argument to see if
        /// it's win percentage is greater than or equal to arg2.
        /// </summary>
        /// <param name="arg1">PocketHands collection to compare</param>
        /// <param name="arg2">Win Value</param>
        /// <returns>A PocketHands collection where each item has win odds greater than or equal arg2</returns>
        static public PocketHands operator >=(PocketHands arg1, double arg2)
        {
            PocketHands retval = new PocketHands();

            foreach (ulong mask in arg1)
            {
                if (PocketHands.WinOdds(mask) >= arg2)
                    retval += mask;
            }
            return retval;
        }


        /// <summary>
        /// This operator returns all PocketHands that are in arg and are less than or equal to the
        /// 169 mask type specified in the argument type.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        static public PocketHands operator <=(PocketHands arg, Hand.PocketHand169Enum type)
        {
            PocketHands retval = new PocketHands();
            for (int i = (int)type; i < 169; i++)
            {
                retval += _Pocket169Combinations[i];
            }
            return arg & retval;
        }

        /// <summary>
        /// This operator returns all PocketHands that are in arg and are less than to the
        /// 169 mask type specified in the argument type.
        /// </summary>
        /// <param name="arg1"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        static public PocketHands operator <(PocketHands arg1, Hand.PocketHand169Enum type)
        {
            PocketHands retval = new PocketHands();

            for (int i = ((int)type) + 1; i < 169; i++)
            {
                retval += _Pocket169Combinations[i];
            }
            return arg1 & retval;
        }


        /// <summary>
        /// This operator returns all PocketHands that are in arg and are greater than to the
        /// 169 mask type specified in the argument type.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        static public PocketHands operator >(PocketHands arg, Hand.PocketHand169Enum type)
        {
            PocketHands retval = new PocketHands();

            for (int i = ((int)type) - 1; i >= 0; i--)
            {
                retval += _Pocket169Combinations[i];
            }

            return arg & retval;
        }

        /// <summary>
        /// This operator returns all PocketHands that are in arg and are greater than or equal to the
        /// 169 mask type specified in the argument type.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        static public PocketHands operator >=(PocketHands arg, Hand.PocketHand169Enum type)
        {
            PocketHands retval = new PocketHands();

            for (int i = (int)type; i >= 0; i--)
            {
                retval += _Pocket169Combinations[i];
            }

            return arg & retval;
        }

        /// <summary>
        /// This operator returns all PocketHands that are in arg and are less than the
        /// sklansky group type specified in the argument type.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="rank"></param>
        /// <returns></returns>
        static public PocketHands operator <(PocketHands arg, GroupTypeEnum rank)
        {
            PocketHands retval = new PocketHands();

            for (int i = ((int)rank) + 1; i < (int)GroupTypeEnum.Group8; i++)
            {
                retval += _PocketGroupings[i];
            }

            return arg & retval;
        }

        /// <summary>
        /// This operator returns all PocketHands that are in arg and are less than or equal to the
        /// sklansky group type specified in the argument type.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="rank"></param>
        /// <returns></returns>
        static public PocketHands operator <=(PocketHands arg, GroupTypeEnum rank)
        {
            PocketHands retval = new PocketHands();

            for (int i = (int)rank; i <= (int) GroupTypeEnum.Group8; i++)
            {
                retval += _PocketGroupings[i];
            }

            return arg & retval;
        }

        /// <summary>
        /// This operator returns all PocketHands that are in arg and are greter than the
        /// sklansky group type specified in the argument type.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="rank"></param>
        /// <returns></returns>
        static public PocketHands operator >(PocketHands arg, GroupTypeEnum rank)
        {
            PocketHands retval = new PocketHands();

            for (int i = ((int)rank) - 1; i >= (int) GroupTypeEnum.Group1; i--)
            {
                retval += _PocketGroupings[i];
            }

            return arg & retval;
        }

        /// <summary>
        /// This operator returns all PocketHands that are in arg and are greater than or equal to the
        /// sklansky group type specified in the argument type.
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="rank"></param>
        /// <returns></returns>
        static public PocketHands operator >=(PocketHands arg, GroupTypeEnum rank)
        {
            PocketHands retval = new PocketHands();

            for (int i = (int)rank; i >= (int) GroupTypeEnum.Group1; i--)
            {
                retval += _PocketGroupings[i];
            }

            return arg & retval;
        }
        #endregion

        #region Text Operators
        /// <summary>
        /// 
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="s"></param>
        /// <returns></returns>
        static internal PocketHands LT(PocketHands arg, string s)
        {
            if (FindFixCard169(s))
            {
                return arg < (Hand.PocketHand169Enum)Card169Max(s);
            }
           
            System.Diagnostics.Debug.Assert(false);
            return null;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="s"></param>
        /// <returns></returns>
        static internal PocketHands LE(PocketHands arg, string s)
        {
            if (FindFixCard169(s))
            {
                return arg <= (Hand.PocketHand169Enum)Card169Min(s);
            }

            System.Diagnostics.Debug.Assert(false);
            return null;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="s"></param>
        /// <returns></returns>
        static internal PocketHands GT(PocketHands arg, string s)
        {
            if (FindFixCard169(s))
            {
                return arg > (Hand.PocketHand169Enum)Card169Min(s);
            }
           
            System.Diagnostics.Debug.Assert(false);
            return null;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="arg"></param>
        /// <param name="s"></param>
        /// <returns></returns>
        static internal PocketHands GE(PocketHands arg, string s)
        {
            if (FindFixCard169(s))
            {
                return arg >= (Hand.PocketHand169Enum)Card169Max(s);
            }
           
            System.Diagnostics.Debug.Assert(false);
            return null;
        }
        #endregion

        #endregion   

        #region IEnumerable Members
        /// <summary>
        /// This method allows only one of each representative type of pocket mask to
        /// be iterated through. It might be preferred of over Hand.Hands() if you wish
        /// to reduce the number of pocket hands considered, but still cover all of the types of 
        /// pocket hands.
        /// </summary>
        /// <code>
        /// using System;
        /// using System.Collections.Generic;
        /// using Private.Logic.Poker.Probability;
        /// 
        /// namespace ConsoleApplication1
        /// {
        ///     class Program
        ///     {
        ///         static void Main(string[] args)
        ///         {
        ///             int count = 0;
        ///             foreach (ulong mask in PocketHands.Hands169())
        ///             {
        ///                 count++;
        ///             }
        ///             // Prints out 169
        ///             Console.WriteLine("count {0}", count);
        ///         }
        ///     }
        /// }
        /// </code>
        /// <param name="shared">The cards must be in the pocket mask</param>
        /// <param name="dead">These cards must not be in the pocket mask</param>
        /// <returns></returns>
        static public IEnumerable<ulong> Hands169(ulong shared, ulong dead)
        {
            foreach (ulong[] list in Hand.Pocket169Table)
            {
                foreach (ulong mask in list)
                {
                    if ((mask & shared) == shared && (mask & dead) == 0)
                    {
                        yield return mask;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// This method allows only one of each representative type of pocket mask to
        /// be iterated through. It might be preferred of over Hand.Hands() if you wish
        /// to reduce the number of pocket hands considered, but still cover all of the types of 
        /// pocket hands.
        /// </summary>
        /// <code>
        /// using System;
        /// using System.Collections.Generic;
        /// using Private.Logic.Poker.Probability;
        /// 
        /// namespace ConsoleApplication1
        /// {
        ///     class Program
        ///     {
        ///         static void Main(string[] args)
        ///         {
        ///             int count = 0;
        ///             foreach (ulong mask in PocketHands.Hands169())
        ///             {
        ///                 count++;
        ///             }
        ///             // Prints out 169
        ///             Console.WriteLine("count {0}", count);
        ///         }
        ///     }
        /// }
        /// </code>
        /// <param name="dead">These cards must not be in the pocket mask</param>
        /// <returns></returns>
        static public IEnumerable<ulong> Hands169(ulong dead)
        {
            return Hands169(0UL, dead);
        }

        /// <summary>
        /// This method allows only one of each representative type of pocket mask to
        /// be iterated through. It might be preferred of over Hand.Hands() if you wish
        /// to reduce the number of pocket hands considered, but still cover all of the types of 
        /// pocket hands.
        /// </summary>
        /// <code>
        /// using System;
        /// using System.Collections.Generic;
        /// using Private.Logic.Poker.Probability;
        /// 
        /// namespace ConsoleApplication1
        /// {
        ///     class Program
        ///     {
        ///         static void Main(string[] args)
        ///         {
        ///             int count = 0;
        ///             foreach (ulong mask in PocketHands.Hands169())
        ///             {
        ///                 count++;
        ///             }
        ///             // Prints out 169
        ///             Console.WriteLine("count {0}", count);
        ///         }
        ///     }
        /// }
        /// </code>
        /// <returns></returns>
        static public IEnumerable<ulong> Hands169()
        {
            return Hands169(0UL, 0UL);
        }

        /// <summary>
        /// This method iterates through the all of the hands with the specified number of
        /// cards that contain one of the sets of cards in the list array. 
        /// </summary>
        /// <param name="list">The list of cards sets (probably pocket hands) that one of which must be in the returned mask</param>
        /// <param name="dead">A mask of the cards that must not be in the returned mask</param>
        /// <param name="numberOfCards">The number of cards in the returned mask</param>
        /// <returns></returns>
        private static IEnumerable<ulong> Hands(ulong[] list, ulong dead, int numberOfCards)
        {
            ulong shared = 0UL;

            if (list == null || list.Length == 0)
                yield break;

            foreach (ulong mask in list)
            {
                shared |= mask;
            }

            foreach (ulong mask in Hand.Hands(0UL, dead, numberOfCards))
            {
                if ((mask & shared) != 0)
                {
                    foreach (ulong pocket in list)
                    {
                        if ((pocket & mask) == pocket)
                        {
                            yield return mask;
                            break;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Iterates through all of the hands that contain a pocket mask from the set
        /// pocket hands defined in the query string.
        /// </summary>
        /// <param name="query">A pocket mask query string</param>
        /// <param name="dead">A mask of cards that must no be included in the returned mask</param>
        /// <param name="numberOfCards">The number of cards in the returned mask</param>
        /// <returns>Hand mask</returns>
        /// <example>
        /// <code>
        /// using System;
        /// using Private.Logic.Poker.Probability;
        /// 
        /// namespace ConsoleApplication1
        /// {
        ///     class Program
        ///     {
        ///         static void Main(string[] args)
        ///         {
        ///             long count = 0;
        ///             foreach (ulong mask in PocketHands.Hands("AKs", 0UL, 7))
        ///             {
        ///                 count++;
        ///             }
        ///             Console.WriteLine("count: {0}", count);
        ///          }
        ///     }
        /// }
        /// </code>
        /// </example>
        public static IEnumerable<ulong> Hands(string query, ulong dead, int numberOfCards)
        {
            ulong[] list = new ulong[0];
            try
            {
                list = PocketHands.Query(query);
            }
            catch
            {
            }
            return Hands(list, dead, numberOfCards);
        }

        /// <summary>
        /// This method makes it possible to use the foreach statement
        /// on this class. This method has a typed return value so that 
        /// box/unbox is not needed while iterating.
        /// </summary>
        /// <returns></returns>
        public IEnumerator<ulong> GetEnumerator()
        {
            foreach (ulong mask in list)
                yield return mask;
        }

        /// <summary>
        /// This method makes it possible to use the foreach statement
        /// on this class.  
        /// </summary>
        /// <returns></returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        #endregion

        #region Basic Overrides

        /// <summary>
        /// Converts a PocketHands collection into a ulong[].
        /// </summary>
        /// <returns></returns>
        public ulong[] ToArray()
        {
            return list.ToArray();
        }

        /// <summary>
        /// Returns the number of ulong values in the pocket mask collection.
        /// </summary>
        public int Count
        {
            get { return list.Count; }
        }

        /// <summary>
        /// Returns the value associated with a specific index of the collection.
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public ulong this[int index]
        {
            get { return list[index]; }
            set { list[index] = value; }
        }

        /// <summary>
        /// Returns true if the collection already contains the specified mask.
        /// </summary>
        /// <param name="mask"></param>
        /// <returns></returns>
        public bool Contains(ulong mask)
        {
            return list.Contains(mask);
        }

        /// <summary>
        /// Calculate a hash code for the collection
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            int hash = 0;
            foreach (ulong mask in list)
            {
                hash += mask.GetHashCode();
            }

            return hash;
        }

        /// <summary>
        /// Checks this pocket mask collection with another for equality.
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            if (!(obj is PocketHands)) return false;

            PocketHands p = (PocketHands)obj;
            if (p.Count != Count) return false;

            foreach (ulong mask in list)
            {
                if (!p.Contains(mask)) return false;
            }
            return true;
        }

        /// <summary>
        /// Implicit cast
        /// </summary>
        /// <param name="p"></param>
        /// <returns></returns>
        public static implicit operator ulong[](PocketHands p)
        {
            return p.ToArray();
        }
        #endregion
    }
}
