using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Messaging;
using System.Security.Policy;
using System.Text;
using System.Threading;

using Shared.Channels;
using Shared.Channels.GenuineTcp;
using Shared.Channels.BroadcastEngine;
using Shared.Channels.DotNetRemotingLayer;

namespace Service.Channels
{
    public partial class Server : Component, IBroadcastServer
    {
        public Server()
        {
            InitializeComponent();
        }

        public Server(IContainer container)
        {
            container.Add(this);
            InitializeComponent();
        }

        public static void Daemonize()
        {
            EventLog evtLog = new EventLog("Application", ".", "Ludus Channels");
            StringBuilder evtEntry = new StringBuilder("Service::Channels::Server::Daemonize()" + Environment.NewLine);

            try
            {
                IDictionary props = new Hashtable();
                props["name"] = "gtcpd";
                props["prefix"] = "gtcpd";
                props["priority"] = "100";
                props["port"] = "48889";
//                props["connecttimeout"] = "3600000";
                props["persistentconnectionsendpingafterinactivity"] = "30000";

                // Create with the default formatters.
                BinaryServerFormatterSinkProvider srv = new BinaryServerFormatterSinkProvider();
                BinaryClientFormatterSinkProvider clnt = new BinaryClientFormatterSinkProvider();
                GenuineTcpChannel channel = new GenuineTcpChannel(props, clnt, srv);
                ChannelServices.RegisterChannel(channel, false);
                WellKnownServiceTypeEntry remoteType = new WellKnownServiceTypeEntry(typeof(Shared.Channels.GenuineTcp.GenuineTcpChannel), "Server", WellKnownObjectMode.Singleton);
                RemotingConfiguration.RegisterWellKnownServiceType(remoteType);
                GenuineGlobalEventProvider.GenuineChannelsGlobalEvent += new GenuineChannelsGlobalEventHandler(Service.Channels.Server.EventHandler);

                // Bind the server.
                RemotingServices.Marshal(new Server(), "Server");

                evtEntry.Append(Environment.NewLine + "Server has been started.");
                evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.SuccessAudit);
            }
            catch (Exception ex)
            {
                evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
            }
            finally
            {
                evtLog.Close();
            }
        }
        /// <summary>
        /// Catches Genuine Channels events and removes client session when user disconnects.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public static void EventHandler(Object sender, GenuineEventArgs e)
        {
            EventLog evtLog = new EventLog("Application", ".", "Ludus Channels");
            StringBuilder evtEntry = new StringBuilder("Service::Channels::Server::EventHandler()" + Environment.NewLine);

            if (e.SourceException == null)
            {
                evtEntry.Append(Environment.NewLine + String.Format("Global event: {0}\r\nRemote host: {1}", e.EventType, e.HostInformation == null ? "<unknown>" : e.HostInformation.ToString()));
            }
            else
            {
                evtEntry.Append(Environment.NewLine + String.Format("Global event: {0}\r\nRemote host: {1}\r\nException: {2}", e.EventType, e.HostInformation == null ? "<unknown>" : e.HostInformation.ToString(), e.SourceException));
            }

            if (e.EventType == GenuineEventType.GeneralConnectionClosed)
            {
                // the client disconnected
                String nickname = e.HostInformation["Nickname"] as String;

                if (nickname != null)
                {
                    evtEntry.Append(Environment.NewLine + String.Format("Client \"{0}\" has been disconnected.", nickname));
                }
            }

//            evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Information);
            evtLog.Close();
        }

        /// <summary>
        /// This example was designed to have the only chat room.
        /// </summary>
        public static ServerChat GlobalRoom = new ServerChat();

        /// <summary>
        /// Logs into the chat room.
        /// </summary>
        /// <param name="nickname">Nickname.</param>
        /// <returns>Chat room interface.</returns>
        public IBroadcastIt JoinDialog(String nickname)
        {
            GlobalRoom.AttachClient(nickname);
            GenuineUtility.CurrentSession["Nickname"] = nickname;
            return GlobalRoom;
        }

        /// <summary>
        /// This is to insure that when created as a Singleton, the first instance never dies, regardless of the expired time.
        /// </summary>
        /// <returns></returns>
        public override Object InitializeLifetimeService()
        {
            return null;
        }
    }
    /// <summary>
    /// Represents a chat room.
    /// </summary>
    public class ServerChat : MarshalByRefObject, IBroadcastIt
    {
        /// <summary>
        /// Constructs ServerChat instance.
        /// </summary>
        public ServerChat()
        {
            // Bind server's methods
            this._dispatcher.BroadcastCallFinishedHandler += new BroadcastCallFinishedHandler(this.BroadcastCallFinishedHandler);
            this._dispatcher.CallIsAsync = true;
        }

        /// <summary>
        /// Chat members.
        /// </summary>
        private Dispatcher _dispatcher = new Dispatcher(typeof(IBroadcastMessenger));

        /// <summary>
        /// Attaches the client.
        /// </summary>
        /// <param name="nickname">Nickname.</param>
        public void AttachClient(String nickname)
        {
            String receiverUri = GenuineUtility.FetchCurrentRemoteUri() + "/Messenger";
            IBroadcastMessenger iBroadcastMessenger = (IBroadcastMessenger)Activator.GetObject(typeof(IBroadcastMessenger), receiverUri);

            this._dispatcher.Add((MarshalByRefObject)iBroadcastMessenger);

            GenuineUtility.CurrentSession["Nickname"] = nickname;
        }

        /// <summary>
        /// Sends message to all clients.
        /// </summary>
        /// <param name="message">Message to send.</param>
        /// <returns>Number of clients having received this message.</returns>
        public void SendMessage(Object message)
        {
            // Fetch the nickname
            String nickname = GenuineUtility.CurrentSession["Nickname"] as String;
            IBroadcastMessenger iBroadcastMessenger = (IBroadcastMessenger)this._dispatcher.TransparentProxy;

            iBroadcastMessenger.ReceiveMessage(message, nickname);
        }

        /// <summary>
        /// Called by broadcast dispatcher when all calls are performed.
        /// Does not undertake any actions.
        /// </summary>
        /// <param name="dispatcher">Source dipatcher.</param>
        /// <param name="message">Source message.</param>
        /// <param name="resultCollector">Call results.</param>
        public void BroadcastCallFinishedHandler(Dispatcher dispatcher, IMessage message, ResultCollector resultCollector)
        {
        }
    }
}
