/*
 * Chris Ragland
 * November 14th 2008
 * 
 * Level.cs
 * 
 * The main class that drives the game. Contains all of the other objects
 * in the level and updates them. Determines if collisions occur and if so
 * it causes things to happen. Also renders the entire level.
 * 
 */

using System;
using System.Collections.Generic;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

namespace SeniorProject
{
	public class Level
	{
		#region Instanced Variables
		protected int levelNumber;
		protected string levelName;
		protected Background background = null;
		protected Player player = null;
		protected List<Target> targets;
		protected List<BounceWall> bouncyWalls;
		protected List<CollideWall> collideWalls;
		protected Bullet bullet = null;
		protected bool gravityChangeable = false;

		protected bool gravityAutoChange = false;
		protected double timeLeftCurrentGravity = -1;
		protected Queue<Vector2> gravityAutoChangeValues;
		protected Queue<Vector2> gravityAutoChangeBaseValues;

		protected float velocity;
		protected int gravity = -1;
		protected int gravityBase = -1;
		protected float gravityValue = -1;
		protected int time = -1;
		protected int stringTime = -1;
		protected int switchLimit = -1;
		protected int switchLimitBase = -1;

		protected double timer;
		protected double timeLeft = -1;

		protected float playerAngle = 0.0f;

		protected bool levelOver = false;
		protected int score = 0;
		#endregion

		#region Properties
		public int LevelNumber
		{
			get { return levelNumber; }
		}
		#endregion

		public Level(LevelBuilder lb)
		{
			levelNumber = lb.LevelNumber;
			levelName = lb.LevelTitle;
			player = new Player(lb.PlayerLocation);

			targets = new List<Target>();
			bouncyWalls = new List<BounceWall>();
			collideWalls = new List<CollideWall>();

			Vector4[] bounce = lb.BouncyWalls.ToArray();
			int[] a = lb.BounceDirectionPerWall.ToArray();
			int size = bounce.Length;

			for(int i = 0; i < size; i++)
			{
				Vector4 vector = bounce[i];
				bouncyWalls.Add(new BounceWall(new Vector2(vector.X, vector.Y), new Rectangle((int)vector.X, (int)vector.Y, (int)vector.W, (int)vector.Z), a[i]));
			}

			foreach(Vector2 vector in lb.TargetLocations)
				targets.Add(new Target(vector));
			foreach(Vector4 vector in lb.CollideWalls)
				collideWalls.Add(new CollideWall(new Vector2(vector.X, vector.Y), new Rectangle((int)vector.X, (int)vector.Y, (int)vector.W, (int)vector.Z)));

			gravityChangeable = (lb.GravitySwitchable == 1 ? true : false);
			gravityAutoChange = (lb.GravityAutoSwitch == 1 ? true : false);

			gravityAutoChangeValues = new Queue<Vector2>();
			gravityAutoChangeBaseValues = new Queue<Vector2>();
			foreach(Vector2 vector in lb.GravityAutoSwitchTimesAndValues)
			{
				gravityAutoChangeValues.Enqueue(vector);
				gravityAutoChangeBaseValues.Enqueue(vector);
			}

			velocity = lb.Velocity;
			gravity = lb.Gravity;
			gravityBase = gravity;
			gravityValue = lb.GravityValue;
			time = lb.Time;
			stringTime = time - 6;
			if(gravityAutoChangeValues.Count > 0)
				timeLeftCurrentGravity = (time - gravityAutoChangeValues.Peek().X);

			switchLimit = 0;
			switchLimitBase = lb.SwitchLimit;
			switchLimit = switchLimitBase;


			SeniorProject.Controller.Gravity = gravity;
			background = new Background();
		}

		public double Update(TimeSpan elapsed)
		{
			#region Time Update Section
			timer += elapsed.Milliseconds;

			if(timer > 600)
			{
				timer = 0;
				time--;
			}
			#endregion

			#region Controller/Gravity Update Region

			if(gravityAutoChange)
			{
				if(time < timeLeftCurrentGravity)
				{
					gravityAutoChangeValues.Enqueue(gravityAutoChangeValues.Dequeue());
					timeLeftCurrentGravity = (time - gravityAutoChangeValues.Peek().X);
					gravity = (int)gravityAutoChangeValues.Peek().Y;
				}
			}
			else
			{
				if(gravity != SeniorProject.Controller.Gravity)
				{
					if(switchLimit > 0 || switchLimitBase == -1)
					{
						gravity = SeniorProject.Controller.Gravity;
						switchLimit--;
					}
					else if(bullet == null)
					{
						gravity = gravityBase;
						SeniorProject.Controller.Gravity = gravityBase;
					}
				}
			}
			#endregion

			#region Bullet/Collision Update Region
			bool bouncedXAtAll = false;
			bool bouncedYAtAll = false;
			Vector2 tempBulletPositionVector;

			if(bullet != null)
			{
				foreach(BounceWall wall in bouncyWalls)
				{
					if(wall.Rectangle.Intersects(bullet.Rectangle))
					{
						SeniorProject.Sounds.PlayCue("bounce");
						tempBulletPositionVector = bullet.Position;
						if((wall.BounceDirection == 2 || wall.BounceDirection == 4) && bullet.bouncingX() == false)
						{
							if(wall.BounceDirection == 4)
								tempBulletPositionVector.X = (wall.Position.X + wall.Rectangle.Width);
							else
								tempBulletPositionVector.X = (wall.Position.X - 5);
							bullet.bounce(wall.BounceDirection);
							bouncedXAtAll = true;
						}
						else if((wall.BounceDirection == 1 || wall.BounceDirection == 3) && bullet.bouncingY() == false)
						{
							if(wall.BounceDirection == 1)
								tempBulletPositionVector.Y = (wall.Position.Y - 5);
							else
								tempBulletPositionVector.Y = (wall.Position.Y + wall.Rectangle.Height);
							bullet.bounce(wall.BounceDirection);
							bouncedYAtAll = true;
						}
						bullet.Position = tempBulletPositionVector;
					}
				}

				if(!bouncedXAtAll)
					bullet.setNotBouncingX();

				if(!bouncedYAtAll)
					bullet.setNotBouncingY();

				foreach(CollideWall wall in collideWalls)
				{
					if(wall.Rectangle.Intersects(bullet.Rectangle))
					{
						SeniorProject.Sounds.PlayCue("collide");
						bullet = null;
						break;
					}
				}
				if(bullet != null)
				{
					foreach(Target target in targets)
					{
						if(target.Rectangle.Intersects(bullet.Rectangle))
						{
							SeniorProject.Sounds.PlayCue("target");
							score += time;
							targets.Remove(target);
							bullet = null;
							break;
						}
					}
				}
			}

			#endregion

			#region Player/Bullet Shooting Update Region
			playerAngle = (float)SeniorProject.Controller.Angle;

			if(SeniorProject.Controller.Shoot)
			{
				SeniorProject.Sounds.PlayCue("shot");
				bullet = new Bullet(player.Position, SeniorProject.Controller.ShootVector, velocity, gravityValue);
				if(gravityChangeable)
				{
					switchLimit = switchLimitBase;
					gravity = gravityBase;
					SeniorProject.Controller.Gravity = gravityBase;
				}
			}
			#endregion

			return time;
		}

		/*
		 * Method called sixty times a second to draw the game.
		 * Drawing takes place back to front to do correct depth
		 * tracking.
		 */
		public void Render()
		{
			background.Render(gravity);

			if(bullet != null)
			{
				bullet.Render();
				bullet.Update(gravity);
			}
			player.Render(playerAngle);
			foreach(Target target in targets)
				target.Render();
			foreach(BounceWall wall in bouncyWalls)
				wall.Render();
			foreach(CollideWall wall in collideWalls)
				wall.Render();

			if(stringTime < time)
			{
				SeniorProject.SpriteBatch.DrawString(SeniorProject.FontTwo, levelName, new Vector2(40, 80), Color.Gold);
			}
		}

		public bool LevelFinished()
		{
			if(targets.Count == 0)
				levelOver = true;
			else
				levelOver = false;
			return levelOver;
		}

		public int LevelScore()
		{
			return score;
		}
	}
}