﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GP4File
{
    /**
     * The class ReadGP4File will parse a file of the GuitarPro 4 format
     */
    class GP4InputStream : GPInputStream
    {
        private static readonly string[] SupportedVersions = {
            "FICHIER GUITAR PRO v.4.00", "FICHIER GUITAR PRO v4.00", 
            "FICHIER GUITAR PRO v4.06",  "FICHIER GUITAR PRO L4.06" };

        private GPSong piece;

        /// <summary>
        /// Creates a new GP4InputStream from an existing BinaryReader
        /// </summary>
        /// <param name="fIn">The stream used to create a new stream with</param>
        public GP4InputStream(BinaryReader fIn): base(fIn)
        {}

        /// <summary>
        /// Creates a new GP4InputSTream by cloning an existing GPInputStream
        /// </summary>
        /// <param name="gpIn">GPInputStream to clone</param>
        public GP4InputStream(GPInputStream gpIn): base(gpIn)
        {}

        /// <summary>
        /// Checks if the string version is supported. Something like "gp4" is expected
        /// to be passed.
        /// </summary>
        /// <param name="version">The string to test</param>
        /// <returns>Whether or not the file type is supported</returns>
        public static bool supportedVersion(string version)
        {
            bool correct = false;
            for (int i = 0; (i < SupportedVersions.Length) && !correct; i++)
            {
                correct = version.Equals(SupportedVersions[i]);
            }
            return correct;
        }

        /// <summary>
        /// Reads a musical piece from the input stream.
        /// </summary>
        /// <returns>The new song created by the file</returns>
        public GPSong readPiece()
        {
            piece = new GPSong();

            // Read the version
            if(Version.Equals(""))
                Version = readStringByte(30);

            piece.setVersion(Version);

            // Read the title
            piece.setTitle(readStringIntegerPlusOne());

            // Read the subtitle
            piece.setSubtitle(readStringIntegerPlusOne());

            // Read the interpret
            piece.setInterpret(readStringIntegerPlusOne());

            // Read the album title
            piece.setAlbum(readStringIntegerPlusOne());

            // Read the author of the song
            piece.setAuthorSong(readStringIntegerPlusOne());

            // Read who owns the copyright to the music
            piece.setCopyright(readStringIntegerPlusOne());

            // Read who authored this GuitarPro file
            piece.setAuthorPiece(readStringIntegerPlusOne());

            // Read any special instructions
            piece.setInstructions(readStringIntegerPlusOne());

            // Read all the notes
            int nbNotes = readInt();
            string note = "";
            for(int i = 0; i < nbNotes; i++)
            {
                note += readStringIntegerPlusOne();
                note += "\n";
            }
            piece.setNote(note);

            // Read the triplet feel
            piece.tripletFeel = readBoolean();
            //MessageBox.Show(piece.tripletFeel.ToString());

            // Read the lyrics to the song
            GPTrackLyrics lyrics = new GPTrackLyrics();
            lyrics.setTrackNumber(readInt());
            for(int i = 0; i < 5; i++)
            {
                lyrics.setMeasureNumber(i, readInt());
                lyrics.setLine(i, readStringInteger());
            }
            piece.setLyrics(lyrics);

            // Read the tempo of the song
            piece.setTempo(readInt());

            // Read which key the piece is in
            piece.setKey(GPKey.valueOf(readByte()));

            // Read which octave the piece is in
            piece.setOctave(readInt());

            // Read which MIDI channels are used in the piece
            for(int i = 0; i < 64; i++)
                piece.setChannels(i, readMidiChannel());

            // Read in the number of measures and tracks
            int numberOfMeasures = readInt();
            int numberOfTracks = readInt();

            // Read in all the piece's measures
            List<GPMeasure> measures = piece.getMeasures();
            GPMeasure measure;
            if(numberOfMeasures > 0)
            {
                measure = readMeasure(null); // no previous measure
                measure.MeasureNumber = 1;
                measures.Add(measure);

                for (int i = 1; i < numberOfMeasures; i++)
                {
                    GPMeasure previous = measures.ElementAt(i - 1);
                    measure = readMeasure(previous);
                    measure.MeasureNumber = i + 1;
                    measures.Add(measure);
                }
            }

            // Read in all the piece's tracks
            List<GPTrack> tracks = piece.getTracks();
            for(int i = 0; i < numberOfTracks; i++)
                tracks.Add(readTrack());

            // Read measures-tracks pairs of the piece
            List<GPMeasureTrackPair> measuresTracksPairs = piece.getMeasuresTracksPairs();
            for(int i = 0; i < numberOfMeasures; i++)
                for(int j = 0; j < numberOfTracks; j++)
                {
                    GPMeasureTrackPair mtp = readMeasureTrackPair();
                    measuresTracksPairs.Add(mtp);
                }

            return piece;
        }

        /// <summary>
        /// Reads a GuitarPro beat
        /// </summary>
        /// <returns>The beat that was read</returns>
        private GPBeat readBeat()
        {
            GPBeat beat = new GPBeat();
            int header = readUnsignedByte();

            // Read in beat status information
            if ((header & 0x40) != 0)
            {
                int beatStatus = readUnsignedByte();
                beat.IsEmpty = (beatStatus == 0x00);
                beat.IsRestBeat = (beatStatus == 0x02);
            }

            // Dotted notes
            beat.DottedNotes = ((header & 0x01) != 0);

            // Beat duration
            GPDuration duration = GPDuration.valueOf(readByte());
            beat.Duration = duration;

            // N-Tuplet
            if ((header & 0x20) != 0)
                beat.NTuplet = readInt();

            // Chord diagram
            if ((header & 0x02) != 0)
                beat.ChordDiagram = readChordDiagram();

            // Text
            if ((header & 0x04) != 0)
                beat.text = readStringIntegerPlusOne();

            // Effects on the beat
            if ((header & 0x08) != 0)
                beat.effects = readEffectsOnBeat();

            // Mix table change
            if ((header & 0x10) != 0)
                beat.mixTableChange = readMixTableChange();

            // Finds out which strings are to be played
            int stringsPlayed = readUnsignedByte();
            int numberOfStrings = 0;
            for (int i = 0; i < 7; i++)
                if ((stringsPlayed & (1 << i)) != 0)
                {
                    numberOfStrings++;
                    beat.setStrings(i, true);
                }

            // Gets the corresponding notes
            List<GPNote> notes = beat.Notes;
            for (int i = 0; i < numberOfStrings; i++)
                notes.Add(readNote());

            return beat;
        }

        /// <summary>
        /// Reads a GuitarPro color object
        /// </summary>
        /// <returns>The GuitarPro color</returns>
        private GPColor readColor()
        {
            GPColor color = new GPColor
                                {
                                    red = readUnsignedByte(),
                                    green = readUnsignedByte(),
                                    blue = readUnsignedByte()
                                };

            try
            {
                read(); // white byte is always 0x00
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading color");
                Environment.Exit(0);
            }

            return color;
        }

        /// <summary>
        /// Read a GuiatrPro file marker
        /// </summary>
        /// <returns>The GuitarPro marker read</returns>
        private GPMarker readMarker()
        {
            GPMarker marker = new GPMarker
                                  {
                                      Name = readStringIntegerPlusOne(), 
                                      Color = readColor()
                                  };

            return marker;
        }

        /// <summary>
        /// Read a GuitarPro measure
        /// </summary>
        /// <param name="previous">The previous measure read</param>
        /// <returns>The new measure read</returns>
        private GPMeasure readMeasure(GPMeasure previous)
        {
            GPMeasure measure = new GPMeasure(previous);
            try
            {
                int header = readUnsignedByte();

                // Numerator
                if ((header & 0x01) != 0)
                    measure.Numerator = readByte();

                // Denominator
                if ((header & 0x02) != 0)
                    measure.Denominator = readByte();

                // Beginning of repeat
                measure.repeatStart = ((header & 0x04) != 0);

                // End of repeat
                if ((header & 0x08) != 0)
                    measure.NumberOfRepititions = readByte();

                // Number of alternate endings
                if ((header & 0x10) != 0)
                    measure.NumberOfAlternateEndings = readByte();

                // Marker
                if ((header & 0x20) != 0)
                    measure.marker = readMarker();

                // Tonality
                if ((header & 0x40) != 0)
                {
                    int type = readByte();
                    measure.Tonality = GPKey.valueOf(readByte(), type);
                }

                // Presence of a double bar
                measure.hasDoubleBar = ((header & 0x80) != 0);
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading measure");
                Environment.Exit(0);
            }

            return measure;
        }

        /// <summary>
        /// Reads a GuitarPro measure-track pair
        /// </summary>
        /// <returns>The new measure-track pair read in</returns>
        private GPMeasureTrackPair readMeasureTrackPair()
        {
            GPMeasureTrackPair mtp = new GPMeasureTrackPair();
            List<GPBeat> beats = mtp.Beats;

            try
            {
                int numberOfBeats = readInt();
                for (int i = 0; i < numberOfBeats; i++)
                {
                    GPBeat beat = readBeat();
                    beats.Add(beat);
                }
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading measure track pair");
                Environment.Exit(0);
            }

            return mtp;
        }

        /// <summary>
        /// Reads a GuitarPro MIDI Channel from the stream
        /// </summary>
        /// <returns>The MIDI Channel read in</returns>
        private GPMIDIChannel readMidiChannel()
        {
            GPMIDIChannel channel = new GPMIDIChannel();
            byte[] b = { 0, 0 };

            try
            {
                channel.Instrument = readInt();
                channel.Volume = readByte();
                channel.Balance = readByte();
                channel.Chorus = readByte();
                channel.Reverb = readByte();
                channel.Phaser = readByte();
                channel.Tremolo = readByte();

                // Backward compatibility with version 3.0
                read(ref b);
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading MIDI Channel");
                Environment.Exit(0);
            }

            return channel;
        }

        /// <summary>
        /// Reads a GuiatPro note
        /// </summary>
        /// <returns>The note read in</returns>
        private GPNote readNote()
        {
            GPNote note = new GPNote();
            try
            {
                int header = readUnsignedByte();

                // Note status
                note.IsAccentuated = ((header & 0x40) != 0);
                note.IsDotted = ((header & 0x02) != 0);
                note.IsGhostNote = ((header & 0x04) != 0);

                // Note type
                if ((header & 0x20) != 0)
                {
                    int noteType = readUnsignedByte();
                    note.IsTieNote = (noteType == 0x02);
                    note.IsDeadNote = (noteType == 0x03);
                }

                // Note duration
                if ((header & 0x01) != 0)
                {
                    // catch the case where a duration of 6 is possible.
                    // it is a special value in Guitar Pro that is not understood
                    try
                    {
                        //note.Duration = GPDuration.valueOf(readByte());
                        //GPDuration duration = new GPDuration();
                        note.Duration = GPDuration.valueOf(readByte());
                    }
                    catch (Exception)
                    {
                        MessageBox.Show("Error reading note duration");
                        Environment.Exit(0);
                    }
                    note.NTuplet = readByte();
                }

                // Note dynamic
                if ((header & 0x10) != 0)
                {
                    //note.Dynamic = GPDynamic.valueOf(readByte());
                    //GPDynamic dyn = new GPDynamic();
                    note.Dynamic = GPDynamic.valueOf(readByte());
                }

                // Fret number
                if ((header & 0x20) != 0)
                    note.FretNumber = readByte();

                // Fingering
                if ((header & 0x80) != 0)
                {
                    note.FingeringLeftHand = GPFingering.valueOf(readByte());
                    note.FingeringRightHand = GPFingering.valueOf(readByte());
                }

                // Effects on the note
                if ((header & 0x08) != 0)
                    note.Effects = readEffectsOnNote();
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading duration");
                Environment.Exit(0);
            }
            return note;
        }

        /// <summary>
        /// Reads a GuitarPro track
        /// </summary>
        /// <returns>The new track read in</returns>
        private GPTrack readTrack()
        {
            GPTrack track = new GPTrack();

            try
            {
                int header = readUnsignedByte();

                // Flags
                track.isDrumsTrack = ((header & 0x01) != 0);
                track.is12StringedGuitarTrack = ((header & 0x02) != 0);
                track.isBanjoTrack = ((header & 0x04) != 0);

                // Name
                track.Name = readStringByte(40);

                // Number of strings
                track.NumberOfStrings = readInt();

                // Tuning of the strings
                for(int i = 0; i < 7; i++)
                    track.setStringsTuning(i, readInt());

                // Port
                track.Port = readInt();

                // Channel
                track.Channel = readInt();

                // Channel effects
                track.ChannelEffects = readInt();

                // Number of frets
                track.NumberOfFrets = readInt();

                // Height of the capo
                track.Capo = readInt();

                // Track's color
                track.Color = readColor();
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading track");
                Environment.Exit(0);
            }

            return track;
        }

        /// <summary>
        /// Reads a GuitarPro chord type
        /// </summary>
        /// <returns>The chord type read in</returns>
        private GPChordType readChordType()
        {
            try
            {
                return (GPChordType.valueOf(readUnsignedByte()));
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading chord type");
                Environment.Exit(0);
            }

            // To get to compile
            MessageBox.Show("Error reading chord type");
            return null;
        }

        /// <summary>
        /// Reads a GuitarPro root note
        /// </summary>
        /// <returns>The root note read in</returns>
        private GPChordNote readRoot()
        {
            try
            {
                return GPChordNote.valueOf(readByte());
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading root");
                Environment.Exit(0);
            }

            // To get to compile
            MessageBox.Show("Error reading root2");
            return null;
        }

        /// <summary>
        /// Reads a GuitarPro file tonality type
        /// </summary>
        /// <param name="numBytes">The expected number of bytes of the tonality type in the binary file</param>
        /// <returns>The tonality type read in</returns>
        private GPTonalityType readTonalityType(int numBytes)
        {
            GPTonalityType tt = null;

            try
            {
                switch (numBytes)
                {
                    case 1:
                        tt = GPTonalityType.valueOf(readUnsignedByte());
                        break;
                    case 4:
                        tt = GPTonalityType.valueOf(readInt());
                        break;
                    default:
                        MessageBox.Show("Problem reading tonality type");
                        Environment.Exit(0);
                        break;
                }
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading tonality type");
                Environment.Exit(0);
            }

            return tt;
        }

        /// <summary>
        /// Reads a GuitarPro chord diagram from file
        /// </summary>
        /// <returns>The chord diagram read in</returns>
        private GPChordDiagram readChordDiagram()
        {
            GPChordDiagram cd = new GPChordDiagram();

            try
            {
                int header = readUnsignedByte();

                if ((header & 0x01) == 0)
                {
                    // TODO: Problems sometimes encountered with GP3 diagrams inside GP4 files
                    MessageBox.Show("The chord diagrams in this file are of type GP3 instead of " +
                        "GP4, which is expected. Please try another file");
                    Environment.Exit(0);
                }

                // Sharpness of chord
                cd.Sharp = readBoolean();

                // Ignore 3 bytes
                skip(3);

                // Root
                cd.Root = readRoot();

                // Chord type
                cd.ChordType = readChordType();

                // 9, 11, 13
                cd.NineElevenThirteen = readUnsignedByte();

                // Bass
                cd.Bass = GPChordNote.valueOf(readInt());

                // Tonality type
                cd.TonalityType = readTonalityType(4);

                // Added note
                cd.AddedNote = readUnsignedByte();

                // Name
                cd.Name = readStringByte(20);

                // Ignore 2 bytes
                skip(2);

                // Tonality 5
                cd.TonalityFive = readTonalityType(1);
                // Tonality 9
                cd.TonalityNine = readTonalityType(1);
                // Tonality 11
                cd.TonalityEleven = readTonalityType(1);

                // Base fret
                cd.BaseFret = readInt();

                // Frets of chord
                for (int i = 1; i <= 7; i++)
                    cd.setFret(i, readInt());

                // Number of barres
                cd.NumBarres = readUnsignedByte();

                // Frets of the barres
                for (int i = 1; i <= 5; i++)
                    cd.setFretOfBarre(i, readUnsignedByte());
                // Start of barres
                for (int i = 1; i <= 5; i++)
                    cd.setBarreStart(i, readUnsignedByte());
                // End of barres
                for (int i = 1; i <= 5; i++)
                    cd.setBarreEnd(i, readUnsignedByte());

                // Skip these fields: Omission1, Omission3, Omission5, Omission7,
                // Omission9, Omission 11, Omission13, Blank 
                //(1 byte each)
                skip(8);

                // Fingering
                for (int i = 1; i <= 7; i++)
                    cd.setFingering(i, readByte());

                // Chord fingering displayed
                cd.ChordFingeringDisplayed = readBoolean();

                return cd;
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading chord diagram");
                Environment.Exit(0);
            }

            return cd;
        }

        /// <summary>
        /// The grace notes are stored in the file with 4 variables, written in the
        /// following order:
        /// 
        /// Fret (byte) - The feirst number the grace note is made from
        /// 
        /// Dynamic (byte) - The grace note dynamic is coded like so: 1 = PPP, 2 = PP,
        /// 3 = P, 4 = MP, 5 = MF, 6 = F, 7 = FF, 8 = FFF. The default value is 6.
        /// 
        /// Transition (byte) - This variable determines the transition type used to 
        /// make the grace note: 0 = none, 1 = slide, 2 = bend, 3 = hammer.
        /// 
        /// Duration (byte) - Determines the grace note duration, coded this way: 1 = 1/16,
        /// 2 = 1/24th, 3 = 1/32nd
        /// </summary>
        /// <returns>The grace note read in</returns>
        private GPGraceNote readGraceNote()
        {
            GPGraceNote gn = new GPGraceNote();
            byte[] b = new byte[4];

            try
            {
                for (int i = 0; i < 4; i++)
                    b[i] = (byte)readUnsignedByte();

                gn.Fret = b[0];

                gn.Dynamic = GPDynamic.valueOf(b[1]);

                gn.Transition = GPGraceNoteTransition.valueOf(b[2]);

                /**
                 * TODO update the SPECIFICATION No other values are accepted, these
                 * number are related to the position of the note in the GraceNote
                 * dialog (reversed to other uses of the Duration) - 1: Thirty-second
                 * note. - 2: Sixteenth note. - 3: Eight note
                 */
                gn.Duration = GPDuration.valueOf(3 - b[3]);
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading grace note");
                Environment.Exit(0);
            }
            return gn;
        }

        /// <summary>
        /// Reads a bend point from the stream
        /// </summary>
        /// <returns>The bend point read in</returns>
        private GPBendPoint readBendPoint()
        {
            GPBendPoint bendPoint = new GPBendPoint();

            try
            {
                bendPoint.Position = readInt(); // Aboslute time position
                bendPoint.Value = readInt(); // Vertical height
                GPVibrato vib = GPVibrato.valueOf(readByte());
                bendPoint.Vibrato = vib;
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading bend point");
                Environment.Exit(0);
            }
            return bendPoint;
        }

        /// <summary>
        /// Reads a bend from the stream
        /// </summary>
        /// <returns>The bend read in</returns>
        private GPBend readBend()
        {
            GPBend bend = new GPBend();
            
            GPBendType type = GPBendType.valueOf(readByte());

            try
            {
                bend.Type = type;
                bend.setValue(readInt());

                // Reads the points
                int numPoints = readInt();
                List<GPBendPoint> points = bend.Points;
                for (int i = 0; i < numPoints; i++)
                    points.Add(readBendPoint());
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading bend");
                Environment.Exit(0);
            }

            return bend;
        }

        /// <summary>
        /// Reads the effects on the current note (bend, slide, etc)
        /// </summary>
        /// <returns>The effects read in</returns>
        private GPEffectsOnNote readEffectsOnNote()
        {
            GPEffectsOnNote eon = new GPEffectsOnNote();
            try
            {
                int header1 = readUnsignedByte();
                int header2 = readUnsignedByte();

                // Whether or not a bend is present
                if ((header1 & 0x01) != 0)
                    eon.bend = readBend();

                // Interpretation of the first header
                // Grace note present
                if ((header1 & 0x10) != 0)
                    eon.graceNote = readGraceNote();

                // Tremolo picking present
                if ((header2 & 0x04) != 0)
                    eon.tremoloPicking = GPDuration.valueOf(readUnsignedByte());

                // Slide from the current note present
                if ((header2 & 0x08) != 0)
                {
                    eon.slide = GPSlide.valueOf(readByte());
                }

                // Harmonic note present. This causes the effects on beat to be set
                if ((header2 & 0x10) != 0)
                {
                    eon.harmonic = GPHarmonic.valueOf(readByte());
                }

                // Trill present
                if ((header2 & 0x20) != 0)
                {
                    eon.trill = new GPTrill
                                    {
                                        Fret = readByte(), 
                                        Period = GPDuration.valueOf(readByte())
                                    };
                }

                // Let-ring present
                if ((header1 & 0x08) != 0)
                    eon.letRing = true;

                // Hammer-on or pull-off from current note present
                if ((header1 & 0x02) != 0)
                    eon.hammerOnPullOff = true;

                // Interpretation of the second header
                // Left hand vibrato present. This causes the effects on beat to be set
                if ((header2 & 0x40) != 0)
                    eon.leftHandVibrato = true;

                // Palm mute present
                if ((header2 & 0x02) != 0)
                    eon.palmMute = true;

                // Note played staccato
                if ((header2 & 0x01) != 0)
                    eon.staccato = true;
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading effects on note");
                Environment.Exit(0);
            }

            return eon;
        }

        /// <summary>
        /// Reads the effecs on the current bent from the stream
        /// </summary>
        /// <returns>The effects on the beat</returns>
        private GPEffectsOnBeat readEffectsOnBeat()
        {
            GPEffectsOnBeat eob = new GPEffectsOnBeat();

            try
            {
                int header1 = readUnsignedByte();
                int header2 = readUnsignedByte();

                // Tapping/slapping/popping effect
                if ((header1 & 0x20) != 0)
                {
                    int effect = readUnsignedByte();

                    if (effect == 1)
                        eob.tapping = true;
                    else if (effect == 2)
                        eob.slapping = true;
                    else if (effect == 3)
                        eob.popping = true;
                    else
                    {
                        MessageBox.Show("Error reading effects on beat tapping/slapping/popping");
                        Environment.Exit(0);
                    }
                }

                // Tremolo bar effect
                if ((header2 & 0x04) != 0)
                    eob.tremoloBar = readBend();

                // Stroke effect
                if ((header1 & 0x40) != 0)
                {
                    // Upstroke
                    int durationValue = readByte();
                    if (durationValue != 0)
                    {
                        // Converts value to a Duration value.
                        eob.upStroke = GPDuration.valueOf(6 - durationValue);
                    }

                    // Downstroke
                    durationValue = readByte();
                    if (durationValue != 0)
                        // Converts value to a Duration value
                        eob.downStroke = GPDuration.valueOf(6 - durationValue);
                }

                // Rasgueado
                if ((header2 & 0x01) != 0)
                    eob.hasRasgueado = true;

                // Pickstroke
                if ((header2 & 0x02) != 0)
                {
                    eob.pickStrokes = GPPickStroke.valueOf(readByte());
                }
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading effects on beat");
                Environment.Exit(0);
            }

            return eob;
        }

        /// <summary>
        /// Reads a GuitarPro mix table change event
        /// </summary>
        /// <returns>The mix table change read in</returns>
        private GPMixTableChange readMixTableChange()
        {
            int[] pos = new int[8];

            // For easier processing, create an array
            GPMixTableElement[] elements = new GPMixTableElement[8];

            for(int i = 0; i < 8; i++)
                elements[i] = new GPMixTableElement();

            // Sets all the values
            GPMixTableChange mtc = new GPMixTableChange
                                       {
                                           instrument = elements[0],
                                           volume = elements[1],
                                           balance = elements[2],
                                           chorus = elements[3],
                                           reverb = elements[4],
                                           phaser = elements[5],
                                           tremolo = elements[6],
                                           tempo = elements[7]
                                       };

            try
            {
                int n = 0;
                int aux;
                int i;
                for(i = 0; i < 7; i++)
                {
                    aux = readByte();
                    if((i != 0) && (aux != -1))
                    {
                        pos[n] = i;
                        n++;
                    }
                    elements[i].NewValue = aux;
                }

                // The tempo field is different (expects an integer)
                aux = readInt();
                if(aux != -1)
                {
                    pos[n] = i;
                    n++;
                }

                elements[7].NewValue = aux;

                // Skip the instrument field

                for(i = 0; i < n; i++)
                {
                    aux = readByte();
                    if(elements[pos[i]].NewValue != -1)
                        elements[pos[i]].ChangeDuration = aux;
                }

                int applyToAllTracks = readUnsignedByte();

                // The instrument and the tempo are not affected
                for(i = 0; i < 6; i++)
                {
                    if((applyToAllTracks & (1 << i)) != 0)
                        elements[i+1].applyToAllTracks = true;
                }

                // The tempo always applies to all the tracks
                elements[7].applyToAllTracks = true;
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading mix table change");
                Environment.Exit(0);
            }

            return mtc;
        }
    }
}
