﻿using System;
using System.Windows.Forms;
using System.Drawing;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GUI
{
    /// <summary>
    /// Class that draws a musical measure on the screen
    /// </summary>
    abstract class Bar: Panel
    {
        // If the this if the first bar of a song, its width must be increased
        // so we can add the timing signature to the left. This variable saves
        // that extra width
        private int timeSignatureWidth;

        // If the bar is the first of the song
        private bool isFirstBar;
        // If the bar is the last bar of the track
        private bool isLastBar;

        private readonly DisplayOptions displayOptions;

        // Do we display the time signature in this bar?
        private bool displayTimeSignature;

        // The minimum x coordinate to start drawing (we cannot draw a note
        // on the line separator between measures)
        private int xMin;

        // The numerator of the key signature
        private string numeratorKeySig;
        // The denominator of the key signature
        private string denominatorKeySig;

        // The measure that this bar is associated with
        private GPMeasure measure;

        // The GUI bar panel this bar that will contain this bar
        private readonly BarPanel barPanel;

        // The colors we will use to draw
        private readonly Color foreground = Color.Black;
        private readonly Color background = Color.White;

        /// <summary>
        /// Default constructor to use
        /// </summary>
        /// <param name="dos">The display options to be shown on this object</param>
        /// <param name="bp">The BarPanel that this Bar will be long to</param>
        protected Bar(DisplayOptions dos, BarPanel bp)
        {
            displayOptions = dos;
            barPanel = bp;

            setColors();

            measure = null;
            isFirstBar = false;
            isLastBar = false;
            displayTimeSignature = true;

            Visible = true;
        }

        /// <summary>
        /// Sets the background and foreground colors of this bar. Needed to be put in a method
        /// because of the way C# handles hierarchy of derived classes.
        /// </summary>
        private void setColors()
        {
            BackColor = background;
            ForeColor = foreground;
        }

        /// <summary>
        /// Returns true if this time signature is equal to the bar
        /// </summary>
        /// <param name="b">The bar being tested against the time signature</param>
        /// <returns>Whether or not the time signuate of this bar panel equals the bar argument</returns>
        public bool equalTimeSignature(Bar b)
        {
            bool equal = false;

            if (b != null)
            {
                if ((measure != null) && (b.measure != null))
                {
                    equal = (measure.Numerator == b.measure.Numerator);
                    equal = equal && (measure.Denominator == b.measure.Denominator);
                }
                else if ((measure == null) && (b.measure == null))
                    equal = true;
            }
            return equal;
        }

        /// <summary>
        /// This method assumes that the measure is not null
        /// </summary>
        /// <param name="g">The graphics being used to test the width of a string (if not null)</param>
        /// <returns>The width of the time signature</returns>
        private int calculateTimeSigWidth(Graphics g)
        {
            int toReturn;

            int numerator = measure.Numerator;
            int denominator = measure.Denominator;

            if (g == null)
            {
                int max = Math.Max(numerator, denominator);
                toReturn = max <= 9 ? 14 : 22;
            }
            else
            {
                // Measure the width of the time signature so we make sure we draw enough room
                int numWidth = (int)g.MeasureString(numeratorKeySig, BarPanel.timeSignatureFont).Width;
                int denWidth = (int)g.MeasureString(denominatorKeySig, BarPanel.timeSignatureFont).Width;
                toReturn = Math.Max(numWidth, denWidth);
            }

            numeratorKeySig = numerator.ToString();
            denominatorKeySig = denominator.ToString();

            return toReturn;
        }

        /// <summary>
        /// Calculate how much to increase the width of a bar because of repetitions
        /// </summary>
        /// <returns>The number of pixels to increase the width of the bar by</returns>
        private int calculateWidthInc()
        {
            int incrementWidth = 0;

            // The diameter of a repeat circle (something drawn in the bar panel).
            // This circle will force the entire measure to be wider

            /**
             * Since the hightest value of lineSpacing is currently 10, then d = 5 + 1
             * assuming this expression:
             *      d = lineSpacing/2 + 1
             * then the width of the tablature and standard bar will not match
             */
            int d = displayOptions.LineSpacing + 1;

            if (isFirstBar)
                incrementWidth = 3;
            if (isLastBar)
                incrementWidth += 2;

            if (measure != null)
            {
                int numRepetitions = measure.NumberOfRepititions;
                if ((!isLastBar) && (measure.hasDoubleBar))
                    incrementWidth += 2;

                if (measure.repeatStart)
                {
                    if (!isFirstBar)
                        incrementWidth += 3;
                    incrementWidth += d / 2 + d;
                }

                if (numRepetitions > 0)
                {
                    if (!isLastBar)
                        incrementWidth += 2;
                    incrementWidth += d / 2 + d;
                }

                if (displayTimeSignature)
                {
                    timeSignatureWidth = calculateTimeSigWidth(null);
                    incrementWidth += timeSignatureWidth;
                }
            }

            return incrementWidth;
        }

        /// <summary>
        /// What to do when we paint this bar
        /// </summary>
        /// <param name="e">The event arguments when the paint call is made</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            drawMeasure(e.Graphics);
        }

        /// <summary>
        /// Draws as many horizontal lines as necessary, separated by lineSpacing pixels
        /// 
        /// IMPORTANT: Any changes made to this method should be mirrored in calculateWidth()
        /// </summary>
        /// <param name="g">The graphics object used to draw the measure</param>
        private void drawMeasure(Graphics g)
        {
            int yCoord;

            Brush brush = new SolidBrush(foreground);
            Pen penColor = new Pen(new SolidBrush(foreground));
            Point p1 = new Point();
            Point p2 = new Point();

            const int xCoord = 0;
            int w = Width - 1;
            int maxX = w;
            int aboveText = displayOptions.TopOffset - 7;

            // Draw the measure number in red
            string aux = measure.MeasureNumber.ToString();
            g.DrawString(aux, BarPanel.measureNumberFont, new SolidBrush(Color.Red), new PointF(xCoord, (float)aboveText-10));

            // Draw horizontal lines of measure
            for (int i = 0; i < barPanel.lines; i++)
            {
                yCoord = displayOptions.TopOffset + i * displayOptions.LineSpacing;

                p1.X = xCoord; p1.Y = yCoord;
                p2.X = maxX; p2.Y = yCoord;
                g.DrawLine(penColor, p1, p2);
            }

            // Draw vertical lines of measure
            yCoord = displayOptions.TopOffset;
            if (displayOptions.ExtendTop)
                yCoord = 0;

            int yCoord2 = displayOptions.TopOffset + barPanel.h;
            if (displayOptions.ExtendBottom)
                yCoord2 = yCoord2 + displayOptions.BottomOffset;

            // The line at the left of the bar
            xMin = 0;

            p1.X = 0; p1.Y = yCoord;
            p2.X = 0; p2.Y = yCoord2;
            g.DrawLine(penColor, p1, p2);

            if (isFirstBar)
            {
                // The lines that represent the first bar
                p1.X = 1; p1.Y = yCoord;
                p2.X = 1; p2.Y = yCoord2;
                g.DrawLine(penColor, p1, p2);

                p1.X = 3; p1.Y = yCoord;
                p1.X = 3; p1.Y = yCoord2;
                g.DrawLine(penColor, p1, p2);
                xMin = 3;
            }

            int dx = 0;
            if (isLastBar)
            {
                dx = 1;

                // The lines that represent the last bar
                p1.X = w; p1.Y = yCoord;
                p2.X = w; p2.Y = yCoord2;
                g.DrawLine(penColor, p1, p2);

                p1.X = w - dx; p1.Y = yCoord;
                p2.X = w - dx; p2.Y = yCoord2;
                g.DrawLine(penColor, p1, p2);

                p1.X = w - dx - 2; p1.Y = yCoord;
                p2.X = w - dx - 2; p2.Y = yCoord2;
                g.DrawLine(penColor, p1, p2);
            }

            if (measure == null) return;

            int d = displayOptions.LineSpacing / 2 + 1;
            int numRep = measure.NumberOfRepititions;

            if ((!isLastBar) && (measure.hasDoubleBar))
            {
                p1.X = w - 2; p1.Y = yCoord;
                p2.X = w - 2; p2.Y = yCoord2;
                g.DrawLine(penColor, p1, p2);
            }

            int yAux;
            if (measure.repeatStart)
            {
                if (!isFirstBar)
                {
                    p1.X = 1; p1.Y = yCoord2;
                    p2.X = 1; p2.Y = yCoord2;
                    g.DrawLine(penColor, p1, p2);
                    xMin = 3;
                }

                yAux = (displayOptions.LineSpacing * (barPanel.lines - 2)) / 2 + displayOptions.TopOffset;
                g.FillEllipse(brush, new Rectangle(xMin + d / 2, yAux - d / 2 + 1, d, d));
                g.FillEllipse(brush, new Rectangle(xMin + d / 2, yAux + displayOptions.LineSpacing - d / 2 + 1, d, d));
                xMin = xMin + d / 2 + d;
            }

            if (numRep > 0)
            {
                if (!isLastBar)
                {
                    p1.X = w; p1.Y = yCoord;
                    p2.X = w; p2.Y = yCoord2;
                    g.DrawLine(penColor, p1, p2);
                }

                yAux = (displayOptions.LineSpacing * (barPanel.lines - 2)) / 2 + displayOptions.TopOffset;
                g.FillEllipse(brush, w - 2 - d - d / 2, yAux - d / 2 + 1, d, d);
                g.FillEllipse(brush, w - 2 - d - d / 2, yAux + displayOptions.LineSpacing - d / 2 + 1, d, d);

                // Draw the number of repetitions
                aux = "" + numRep + "x";
                p1.X = w - dx - 3; p1.Y = aboveText;
                g.DrawString(aux, BarPanel.repeatFont, brush, p1);
            }

            if (!displayTimeSignature) return;

            yCoord = barPanel.H / 2 + displayOptions.TopOffset;

            xMin += 5;
                    
            // Get the ascent of font
            Font f = BarPanel.timeSignatureFont;
            float fontHeight = f.GetHeight(g);
                    
            int dy = (int)Math.Floor((fontHeight / f.FontFamily.GetLineSpacing(f.Style)) * f.FontFamily.GetCellAscent(f.Style));

            timeSignatureWidth = calculateTimeSigWidth(g);

            p1.X = xMin; 
            p1.Y = yCoord - 20;
            p2.X = xMin; 
            p2.Y = yCoord + dy - 20;

            g.DrawString(numeratorKeySig, BarPanel.timeSignatureFont, brush, p1);
            g.DrawString(denominatorKeySig, BarPanel.timeSignatureFont, brush, p2);

            xMin = xMin + timeSignatureWidth;
        }

        /// <summary>
        /// Returns the current width of this component
        /// </summary>
        /// <returns>The width of this bar</returns>
        public int getWidth()
        {
            int inc = calculateWidthInc();

            return calculateWidth(inc);
        }

        protected abstract int calculateWidth(int inc);
        public abstract MusicCursorPosition getMusicCursorPosition(int beat);
        public abstract MusicCursorPosition getMusicCursorPosition(MusicCursor musicCursor);
        public abstract MusicCursorPosition snapX(int x, MusicCursor musicCursor);

        #region Getters and setters

        public bool IsFirstBar
        {
            set { isFirstBar = value; }
        }
        public bool IsLastBar
        {
            set { isLastBar = value; }
        }
        protected DisplayOptions DispOptions
        {
            get { return displayOptions; }
        }
        public bool DisplayTimeSignature
        {
            set { displayTimeSignature = value; }
        }
        protected int XMin
        {
            get { return xMin; }
        }

        public GPMeasure Measure
        {
            set { measure = value; }
        }

        protected BarPanel BPanel
        {
            get { return barPanel; }
        }
        public int TopOffset
        {
            set
            {
                if (value >= 0)
                    displayOptions.TopOffset = value;
            }
        }
        public int BottomOffset
        {
            set
            {
                if (value >= 0)
                    displayOptions.BottomOffset = value;
            }
        }
        public bool ExtendBottom
        {
            set { displayOptions.ExtendBottom = value; }
        }
        public bool ExtendTop
        {
            set { displayOptions.ExtendTop = value; }
        }
        #endregion
    }
}
