﻿using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GUI
{
    /// <summary>
    /// Extends Bar and holds a MeasureTrackPair
    /// 
    /// Music is represented as an array of Tracks and Measures. Tracks are represtented as
    /// rows where measures are columns
    /// 
    /// A measure-track pair is the intersection of a row and column. For instance, measure 2
    /// on the guitar track, drum track, and bass track.
    /// </summary>
    abstract class BarMtp: Bar
    {
        #region Class variables

        // Are silences going to be added for the first time

        // Are effects on notes going to be added for the first time
        private bool addEONs;

        // The font ascent of the fret font
        private int fontAscent;

        private GPMeasureTrackPair mtp;
        private GPDuration minDuration;

        // The minimum width of the bar
        private const int MIN_WIDTH = 50;
        private int minWidth;

        #endregion

        /// <summary>
        /// Creates a new BarTablature, related to the BarPanel argument
        /// </summary>
        /// <param name="currentDispOps">The display options to use with this BarMTP</param>
        /// <param name="bp">The BarPanel this BarMTP is associated with</param>
        protected BarMtp(DisplayOptions currentDispOps, BarPanel bp)
            : base(currentDispOps, bp)
        {
            mtp = null;
            AddSilence = true;
            addEONs = true;
            minWidth = MIN_WIDTH;
            minDuration = GPDuration.SIXTY_FOURTH;

            Visible = true;
        }

        /// <summary>
        /// Returns the current position of the music cursor along this
        /// </summary>
        /// <param name="musicCursor">The MusicCursor we're using to look up</param>
        /// <returns>Where the music cursor is in this BarMPT</returns>
        public override MusicCursorPosition getMusicCursorPosition(MusicCursor musicCursor)
        {
            Point p = new Point(0, 1);
            MusicCursorPosition toReturn = new MusicCursorPosition
                                               {
                                                   Point = p,
                                                   Width = getWidth() + 2*musicCursor.LeftOffset
                                               };

            toReturn.setSong(mtp);
            toReturn.Control = this;
            toReturn.BeatNum = 0;

            return toReturn;
        }
        
        /// <summary>
        /// Snaps the music cursor position to a particular place in the BarMTP
        /// </summary>
        /// <param name="xx">The location to snap to</param>
        /// <param name="musicCursor">The parent object reference</param>
        /// <returns></returns>
        public override MusicCursorPosition snapX(int xx, MusicCursor musicCursor)
        {
            bool found = false;

            int xMin = calculateXMin();

            int currentX = 0;
            Point p = new Point {Y = 2};
            MusicCursorPosition toReturn = new MusicCursorPosition();

            int offset = musicCursor.LeftOffset;

            if (mtp.Beats.Count > 0)
            {
                int countBeat = 1;
                int x = xx + offset;

                foreach (GPBeat beats in mtp.Beats)
                {
                    if (found)
                        break;

                    GPBeat current = beats;
                    int currentWidth = calculateWidthOfBeat(current);
                    int currentX2 = currentX + currentWidth;
                    found = ((currentX + xMin <= x) && (x <= currentX2 + xMin));
                    if (found)
                    {
                        toReturn.setSong(current);
                        toReturn.BeatNum = countBeat;
                        toReturn.Control = this;
                    }
                    p.X = currentX + xMin;
                    toReturn.Point = p;
                    toReturn.Width = currentWidth;

                    currentX = currentX2;
                    countBeat++;
                }
            }
            if (!found)
                toReturn = getMusicCursorPosition(musicCursor);

            return toReturn;
        }

        /// <summary>
        /// Returns where the music cursor position is relative to a beat number
        /// </summary>
        /// <param name="beat">The beat being used as reference</param>
        /// <returns>The position of the music cursor in relation to the beat argument</returns>
        public override MusicCursorPosition getMusicCursorPosition(int beat)
        {
            MusicCursorPosition mcp = new MusicCursorPosition();
            Point p = new Point();

            if (mtp.Beats.Count > 0)
            {
                p.X = calculateXMin();
                GPBeat current = null;
                int cw = 0;
                int count = 0;

                foreach (GPBeat beats in mtp.Beats)
                {
                    if (count >= beat)
                        break;
                    
                    current = beats;
                    cw = calculateWidthOfBeat(current);

                    p.X += cw;
                    count++;
                }
                p.Y = 1;
                mcp.Point = p;
                mcp.setSong(current);
                mcp.BeatNum = count;
                mcp.Width = cw;
            }

            return mcp;
        }

        /// <summary>
        /// Calculate what the minimum position of the x coordinate is of this BarMtp
        /// </summary>
        /// <returns>The X value that represents the minimum (furthest left) location the X can be</returns>
        private int calculateXMin()
        {
            // The current position depends on what was drawn before (time sig, repeat, etc)
            int currX = XMin;

            // Leave enough room for a grace note to the left of the first beat
            currX += DispOptions.NoteSpacing / 2;

            // Center the note
            currX += DispOptions.NoteSpacing / 2;

            return currX;
        }

        /// <summary>
        /// Calculates the width of a particular beat. For instance, a half rest would take up half
        /// the bar
        /// </summary>
        /// <param name="beat">The beat to calculate the width of</param>
        /// <returns>The actual width of the beat</returns>
        protected int calculateWidthOfBeat(GPBeat beat)
        {
            GPDuration d;

            int width = 0;

            if (beat != null)
            {
                d = beat.Duration;
                int difference = minDuration.getIndex() - d.getIndex();
                int factor = 1;
                for (int i = 0; i < difference; i++)
                    factor *= 2;
                width = factor * DispOptions.NoteSpacing;

                // This is done here because width is already in pixels
                // making w/2 so as not to be affected by rounding
                if (beat.DottedNotes)
                    width = width + (width / 2);
            }

            return width;
        }

        /// <summary>
        /// Calculate the width of a list of beats
        /// </summary>
        /// <param name="beats">The list of beats to calculate the total width of</param>
        /// <returns>The width of all the beats in the list</returns>
        private int calculateWidth(IList<GPBeat> beats)
        {
            int width = 0;

            for (int i = 0; i < beats.Count; i++)
            {
                GPBeat beat = beats[i];
                width = width + calculateWidthOfBeat(beat);
            }

            return width;
        }

        /// <summary>
        /// Calculates the width of a measure track pair based on a bar
        /// </summary>
        /// <param name="inc">The width of the bar that called this method</param>
        /// <returns>The width of the BarMtp</returns>
        protected override int calculateWidth(int inc)
        {
            int width = inc;

            if (mtp != null)
            {
                List<GPBeat> beats = mtp.Beats;
                if (beats != null)
                {
                    // Adds space for a potential grace note in the first beat
                    width += DispOptions.NoteSpacing / 2;
                    width += calculateWidth(beats);

                    // Add space at the end
                    width += DispOptions.NoteSpacing / 2;
                }
                if (width < minWidth)
                    width = minWidth;
            }

            return width;
        }

        /// <summary>
        /// Returns the minimum duration of a measure-track pair. This method is used
        /// for proportional spacing
        /// </summary>
        /// <returns>The minimum duration of a MTP</returns>
        public GPDuration minDurationOfBeats()
        {
            GPDuration d;
            GPBeat beat;
            List<GPBeat> beats;

            GPDuration min = GPDuration.WHOLE;

            if (mtp != null)
            {
                beats = mtp.Beats;

                if (beats != null)
                {
                    for (int i = 0; i < beats.Count; i++)
                    {
                        beat = beats[i];
                        d = beat.Duration;
                        if (d.compareTo(min) < 0)
                            min = d;
                    }
                }
            }
            return min;
        }

        /// <summary>
        /// Paints the vertical and horizontal rhythm lines. It is assumed
        /// that the beat is not null
        /// </summary>
        /// <param name="g">The graphics object to use to pain the rhythm</param>
        /// <param name="x">The x coordinate of the rhythm</param>
        /// <param name="beat">The beat to be painted</param>
        private void drawRhythm(Graphics g, int x, GPBeat beat)
        {
            GPDuration d = beat.Duration;

            if (d.Equals(GPDuration.WHOLE)) return;

            int lowest = DispOptions.TopOffset + BPanel.H + 2 * DispOptions.LineSpacing;

            // Draw the vertical rhythm line
            Color rhythmColor = DispOptions.RhythmColors.getColor(d);
            Brush b = new SolidBrush(rhythmColor);
            Pen pen = new Pen(b);

            if (beat.IsNoteBeat)
            {
                // Only beats that have notes have vertical lines.
                // Silences and empty beats do not.
            }
            if (beat.DottedNotes)
                g.DrawRectangle(pen, new Rectangle(x + 2, lowest + 2, 1, 1));
        }

        /// <summary>
        /// Draws a beaton the MTP
        /// </summary>
        /// <param name="g">The graphics object used to draw the beat</param>
        /// <param name="currentX">The current x coordinate of the beat</param>
        /// <param name="beat">The beat to draw</param>
        /// <param name="pbi">Information regarding this painted beat</param>
        private void drawBeat(Graphics g, int currentX, GPBeat beat, DrawnBeatInfo pbi)
        {
            int stringNum;
            GPNote note;
            bool first = true;

            for (stringNum = 6; stringNum >= 0; stringNum--)
            {
                bool played = beat.isStringPlayed(stringNum);

                if (!played) continue;

                int y = calculateHeightOfString(stringNum);

                // If this is the first string of the beat (the highest) then 
                // draw the line from the highest string to the bottom
                if (first)
                {
                    drawRhythm(g, currentX, beat);
                    first = false;
                }

                note = pbi.Note;

                if (note.Effects != null)
                {
                    // First draw the effects on the note, then the note to
                    // make sure that the fret number is always on top
                    drawEffects(currentX, y, note.Effects, pbi.EONIndex, beat);
                    pbi.EONIndex++;
                }

                // Set the colors according to the options
                Color beatColor = DispOptions.FretColors.getColor(beat.Duration);
                drawNote(g, currentX, y, note, beatColor);
            }
        }

        /// <summary>
        /// Calculates the minimum  value for drawing objects
        /// </summary>
        /// <returns>The minimum x value</returns>
        private int calculateMinX()
        {
            // The current position depends on what was drawn before (repeats, time sigs, etc)
            int currentX = XMin;
            // Make sure enough room is left for grace notes to the left of the first beat
            currentX += DispOptions.NoteSpacing / 2;
            // Center the note
            currentX += DispOptions.NoteSpacing / 2;

            return currentX + 10;
        }

        /// <summary>
        /// Draws a measure track pair
        /// </summary>
        /// <param name="g">The graphics object used to draw the MTP</param>
        private void drawMeasureTrackPair(Graphics g)
        {
            List<GPBeat> beats;
            GPBeat beat;
            DrawnBeatInfo pbi;

            if (mtp == null) return;

            beats = mtp.Beats;
            Font f = BarPanel.notesFont;

            float fontHeight = f.GetHeight(g);
            fontAscent = (int)(fontHeight / f.FontFamily.GetLineSpacing(f.Style)) * f.FontFamily.GetCellAscent(f.Style);
                
            GPBeat prev = null;
            int silencesCount = 0;
            pbi = new DrawnBeatInfo();
            int currentX = calculateMinX();
            int max = beats.Count;

            int currBeat;
            for (currBeat = 0; currBeat < max; currBeat++)
            {
                beat = beats[currBeat];
                pbi.Notes = beat.Notes;

                currentX += calculateWidthOfBeat(prev);
                if (beat.IsRestBeat)
                {
                    drawSilence(currentX, beat, silencesCount);
                    drawRhythm(g, currentX, beat);
                    silencesCount++;
                }
                else
                {
                    // It is a beat with notes
                    drawBeat(g, currentX, beat, pbi);
                }
                prev = beat;
            }
            AddSilence = false;
            addEONs = false;
        }

        /// <summary>
        /// What to do when this BarMtp is called
        /// </summary>
        /// <param name="e">The event arguments passed when this event occurs</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (mtp != null)
                drawMeasureTrackPair(e.Graphics);
        }

        /// <summary>
        /// Saves various information about a particular beat
        /// </summary>
        private class DrawnBeatInfo
        {
            // A list of notes
            private List<GPNote> notes;

            private int noteIndex;
            private int eonIndex;

            public DrawnBeatInfo()
            {
                notes = null;
                noteIndex = 0;
                eonIndex = 0;
            }


            #region Getters and setters
            public List<GPNote> Notes
            {
                set { notes = value; noteIndex = 0; }
            }
            public GPNote Note
            {
                get
                {
                    GPNote toReturn = null;
                    if (notes != null)
                    {
                        toReturn = notes[noteIndex];
                        noteIndex++;
                    }
                    return toReturn;
                }
            }

            public int EONIndex
            {
                get { return eonIndex; }
                set { eonIndex = value; }
            }
            #endregion
        }

        protected abstract int calculateHeightOfString(int s);
        protected abstract void drawNote(Graphics g, int currentX, int y, GPNote note, Color noteolor);
        protected abstract void drawSilence(int currX, GPBeat beat, int contSilences);
        protected abstract void drawEffects(int currX, int y, GPEffectsOnNote eon, int countEON, GPBeat beat);

        #region Getters and setters
        public GPMeasureTrackPair MeasureTrackPair
        {
            protected get { return mtp; }
            set { mtp = value; }
        }

        protected int FontAscent
        {
            get { return fontAscent; }
        }
        public GPDuration MinDuration
        {
            set
            {
                if (value == null) return;

                minDuration = value;
                minWidth = 5;
            }
        }
        public int MinWidth
        {
            set { minWidth = value; }
        }

        protected bool AddEONs
        {
            get { return addEONs; }
        }

        protected bool AddSilence { get; private set; }

        #endregion
    }
}
