﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GUI
{
    // Contains any type of bar
    sealed class BarPanel : FlowLayoutPanel//Panel //: OptionsDisplay
    {
        // The fonts for various parts of the bar
        public static Font repeatFont;
        public static Font notesFont;
        public static Font timeSignatureFont;
        public static Font measureNumberFont;

        public int lines;
        public int h;
        private DisplayOptions displayOptions;
        private List<Bar> barList;

        public BarPanel()
        {
            FlowDirection = FlowDirection.LeftToRight;
        }

        public BarPanel(DisplayOptions currentOptions)
        {
            FlowDirection = FlowDirection.TopDown;

            if (currentOptions != null)
            {
                displayOptions = currentOptions;

                BackColor = Color.White;
                ForeColor = Color.Black;

                repeatFont = new Font("Arial", 12);
                repeatFont = new Font(repeatFont, FontStyle.Bold);
                repeatFont = new Font(repeatFont, FontStyle.Italic);

                notesFont = new Font("Arial", 8);
                notesFont = new Font(notesFont, FontStyle.Bold);

                timeSignatureFont = new Font("ITCCenturyBook", 14);
                timeSignatureFont = new Font(timeSignatureFont, FontStyle.Bold);

                measureNumberFont = new Font("Arial", 9);

                barList = new List<Bar>();
            }
            else
            {
                MessageBox.Show("Display options in BarPanel is null");
                Environment.Exit(0);
            }
        }

        /// <summary>
        /// Returns the music cursor position in this bar panel
        /// </summary>
        /// <param name="numMeasure">The measure number to look for the music cursor</param>
        /// <param name="musicCursor">The music cursor to look for</param>
        /// <returns>The position of the music cursor</returns>
        public MusicCursorPosition getMusicCursorPosition(int numMeasure, MusicCursor musicCursor)
        {
            Bar b;
            MusicCursorPosition mcp = null;
            Point p;

            if ((numMeasure >= 1) && (numMeasure <= barList.Count))
            {
                b = barList[numMeasure - 1];
                mcp = b.getMusicCursorPosition(musicCursor);
                p = mcp.Point;
                p.X += b.Location.X;
                mcp.Point = p;
            }

            return mcp;
        }

        /// <summary>
        /// Adds a bar to this bar panel
        /// </summary>
        /// <param name="b">The bar to add to this bar panel</param>
        public void addBar(Bar b)
        {
            if (b != null)
            {
                barList.Add(b);
                Controls.Add(b);
                b.Visible = true;

                Width += b.Width;
            }
            else
            {
                MessageBox.Show("The Bar passed is null (BarPanel.addBar)");
                Environment.Exit(0);
            }

        }

        #region Getters and setters

        public int Lines
        {
            get { return lines; }
            private set 
            { 
                lines = value;
                h = (lines - 1) * displayOptions.LineSpacing;
            }
        }
        public int H
        {
            get { return h; }
        }

        public int NoteSpacing
        {
            get { return displayOptions.NoteSpacing; }
            set { displayOptions.NoteSpacing = value; }
        }
        public List<GPMeasure> Measures
        {
            set
            {
                Bar currBar;

                if (value == null) return;

                Bar prevBar = null;

                if (value.Count == barList.Count)
                {
                    for (int i = 0; i < value.Count; i++)
                    {
                        currBar = barList[i];
                        GPMeasure m = value[i];
                        currBar.Measure = m;

                        // If the time signature doesn't change, don't display it
                        currBar.DisplayTimeSignature = !currBar.equalTimeSignature(prevBar);
                        prevBar = currBar;
                    }
                }
                else
                {
                    MessageBox.Show("The number of measures and bars don't match (BarPanel.Measures)");
                    Environment.Exit(0);
                }
            }
        }

        public void setMeasureTrackPair(int measure, GPMeasureTrackPair mtp)
        {
            BarMtp bmtp;

            if (mtp != null)
            {
                if (measure < barList.Count)
                {
                    try
                    {
                        bmtp = (BarMtp)barList[measure];
                        bmtp.MeasureTrackPair = mtp;
                    }
                    catch (InvalidCastException)
                    {
                        MessageBox.Show("setMeasureTrackPair must be called on a BarPanel that contains BarMtp objects");
                        Environment.Exit(0);
                    }
                }
            }
        }

        public Bar getBar(int measure)
        {
            Bar b = null;

            try
            {
                b = barList[measure];
            }
            catch (InvalidCastException)
            {
                MessageBox.Show("BarPanel.barList contains an element that could not be cast into Bar");
                Environment.Exit(0);
            }

            return b;
        }

        public bool ExtendBottom
        {
            set
            {
                displayOptions.ExtendBottom = value;

                foreach (Bar bars in barList)
                    bars.ExtendBottom = value;
            }
        }

        public bool ExtendTop
        {
            set
            {
                displayOptions.ExtendTop = value;

                foreach (Bar bars in barList)
                    bars.ExtendTop = value;
            }
        }
        public Color BackgroundColor
        {
            set { BackColor = value; }
        }
        public Color ForegroundColor
        {
            set { ForeColor = value; }
        }
        public GPTrack Track
        {
            set 
            { 
                if (value != null) 
                    Lines = value.NumberOfStrings; 
            }
        }
        #endregion
    }
}
